﻿// $Id$
// CYPAC Generic application
// 2000/09/06 by T.Kudou <kudou@cypac.co.jp>

#ifndef INCLUDE_ccc_application_Application_h
#define INCLUDE_ccc_application_Application_h

#include <ccc/base/base.h>

CCC_NAMESPACE_START(CCC);

/*!
 * \brief オプション文字列と、機能番号のペアを表現するための構造体です。
 */
struct AppArg
{
  /*!
   * \brief オプション文字列
   */
  const char* option_str;
  /*!
   * \brief オプションに割り付ける機能番号
   *
   * enumなどで表現することをお勧めします。
   */
  int option_num;
 public:
  /*!
   * \brief オプション文字列を得ます。
   * \return オプション文字列
   */
  const char* getOptionStr() { return option_str; }
  /*!
   * \brief 機能番号を得ます。
   * \return 機能番号
   */
  int getOptionNum() { return option_num; }
  /*!
   * \brief 構造体のデータが空かどうかを調べます。
   * \retval true 空
   * \retval false 空ではありません
   */
  bool eofP() { return option_str == 0; }
};

/*!
 * \brief アプリケーションクラス
 *
 * 引数の解析と、アプリケーションの基本動作を提供します。<br>
 * 実際のアプリケーションでは、このクラスを継承したクラスを
 * 用いて実装することを想定しています。
 */
class Application
  : public Object
{
 protected:
  /*!
   * 引数の数
   */
  int argc;
  /*!
   * 引数文字列のポインタのポインタ
   */
  char** argv;
  
 protected:
  /*!
   * 引数の数を得ます。
   * \return 引数の数
   */
  int getArgc() { return argc; }
  /*!
   * 引数文字列のポインタのポインタを得ます。
   * \return 引数文字列のポインタのポインタ
   */
  char** getArgv() { return argv; }
  /*!
   * n番目の引数を得ます。
   * このメソッドの内部で引数の上限チェックは行ないません。
   * \return 引数
   */
  char* getArg(int n) { return argv[n]; }
  /*!
   * 引数の解析を行ないます。
   * \retval true 正常に解析が行なえたことを示します。
   * \retval false 正常に解析が行なえなかったことを示します。
   */
  virtual bool parseArgument();
  /*!
   * Applicationクラスを継承したクラスで、このメソッドは
   * 上書きされることを前提としています。
   * 引数の機能番号と、現在解析中の引数位置の情報を使って、
   * アプリケーション内にオプションの情報をセットします。
   * 引数を2つ取るオプションの場合には、iの値をこのメソッド内
   * でインクリメントすることで対応します。
   * \param n 機能番号
   * \param i 引数リスト中における現在の引数解析位置
   * \param eq_str =文字列形式の追加オプション文字列
   * \retval true 正常に引数解析が行なえたことを示します。
   * \retval false 正常に引数解析が行なえなかったことを示します。
   */
  virtual bool checkArgument(int n, int& i, char* eq_str);
  /*!
   * 引数の機能番号と、オプション文字列を使ってアプリケーション内に
   * オプション情報をセットします。オプション文字列を使用した場合には、
   * opt_used_pをtrueにセットします。
   * 
   * \param n 機能番号
   * \param argn 現在解析中の引数番号(1〜)
   * \param arg 現在解析中の引数
   * \param opt 現在解析中の引数に対するオプション文字列
   * \param opt_used_p 引数を使用した場合はtrueがセットします。
   * \retval true 正常に引数解析が行なえたことを示します。
   * \retval false 正常に引数解析が行なえなかったことを示します。
   */
  virtual bool checkArgument(int n, char* arg, char* opt, bool& opt_used_p);
  /*!
   * Applicationクラスを継承したクラスで、このメソッドは
   * 上書きされることを前提としています。
   * AppArgの構造体に引数オプションと、機能番号をセットした
   * データの配列を返します。
   * 配列の最後には{0,0}を用意してください。<br>
   * 引数解析では、この構造体のデータを元に、checkArgument()メソッド
   * が呼び出されます。
   * \return AppArgの構造体
   */
  virtual AppArg* getAppArg();
 public:
  /*!
   * コンストラクタ
   */
  Application();
  /*!
   * \brief コンストラクタ
   *
   * mainに渡された引数をセットし、このクラスで
   * 引数の内容を解析します。
   * \param argc 引数の数
   * \param argv 引数文字列のポインタのポインタ
   */
  Application(int argc, char** argv);
  /*!
   * デストラクタ
   */
  virtual ~Application();
  /*!
   * アプリケーションの主動作を実行します。
   * \return 終了コード、正常終了を0としますが、別の意味づけを行なっても構いません。
   */
  virtual int run();
  /*!
   * アプリケーションを途中で終了します。
   * このメソッドはrun()メソッドの実行中に利用できます。
   * \param ret 終了コード、run()の返り値となります。
   */
  virtual void exit(int ret);
  /*!
   * アプリケーションを初期化します。
   * \retval true 正常に初期化できたことを示します。
   * \retval false 正常に初期化できなかったことを示します。
   */
  virtual bool initialize();
  /*!
   * アプリケーションの終了処理を行ないます。
   * \param ret run()メソッドの返した終了コードを渡します。
   * \return 終了コード、通常引数のretの値をそのまま返しますが、終了処理で
   * エラーが発生した場合などでは別の値を返すことができます。
   */
  virtual int unInitialize(int ret);
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_application_Application_h */
