﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.

#include <ccc/base/AVLDict.h>

CCC_NAMESPACE_START(CCC);

// ------------------------------------------------------------
// class InnerAVLIterator

InnerAVLIterator::InnerAVLIterator(InnerAVLNode* top)
{
  InnerAVLIterator::top = top;
  ptr = 0;
  rewind();
}

void*
InnerAVLIterator::current() const
{
  return ptr ? ptr->getElement() : 0;
}

void*
InnerAVLIterator::next()
{
  InnerAVLNode* ret = ptr;
  if (ptr)
  {
    ptr = ptr->getNext();
  }
  return ret ? ret->getElement() : 0;
}

void*
InnerAVLIterator::prev()
{
  InnerAVLNode* ret = ptr;
  if (ptr)
  {
    ptr = ptr->getPrev();
  }
  return ret ? ret->getElement() : 0;
}

void*
InnerAVLIterator::rewind()
{
  if (top)
  {
    ptr = top->getTreeLeft();
    return ptr ? ptr->getElement() : 0;
  }
  ptr = 0;
  return 0;
}

void*
InnerAVLIterator::unwind()
{
  if (top)
  {
    ptr = top->getTreeRight();
    return ptr ? ptr->getElement() : 0;
  }
  ptr = 0;
  return 0;
}

bool
InnerAVLIterator::validPositionP() const
{
  return !!ptr;
}

Size
InnerAVLIterator::getPosition() const
{
  Size pos = 0;
  if (ptr && top)
  {
    InnerAVLNode* p = top->getTreeLeft();
    while (p)
    {
      pos++;
      if (p == ptr)
      {
	break;
      }
      p = p->getNext();
    }
  }
  return pos;
}

// ------------------------------------------------------------
// class InnerAVLNode

InnerAVLNode::InnerAVLNode(void* element, InnerAVLNode* parent,
			   InnerAVLNode* left, InnerAVLNode* right)
{
  InnerAVLNode::element = element;
  InnerAVLNode::parent = parent;
  InnerAVLNode::left = left;
  InnerAVLNode::right = right;
  InnerAVLNode::height = 1;
}

InnerAVLNode* 
InnerAVLNode::getTreeLeft() const
{
  if (!left)
  {
    return (InnerAVLNode*)this;
  }
  InnerAVLNode* node;
  for (node = left;
       node->getLeft() != 0;
       node = node->getLeft())
  {
    ;
  }
  return node;
}

InnerAVLNode* 
InnerAVLNode::getTreeRight() const
{
  if (!right)
  {
    return (InnerAVLNode*)this;
  }
  InnerAVLNode* node;
  for (node = right;
       node->getRight() != 0;
       node = node->getRight())
  {
    ;
  }
  return node;
}

InnerAVLNode* 
InnerAVLNode::getPrev() const
{
  if (left)
  {
    return left->getTreeRight();
  }
  if (rightP())
  {
    return getParent();
  }
  for (InnerAVLNode* node = getParent();
       node != 0;
       node = node->getParent())
  {
    if (node->rightP())
    {
      return node->getParent();
    }
  }
  return 0;
}

InnerAVLNode* 
InnerAVLNode::getNext() const
{
  if (right)
  {
    return right->getTreeLeft();
  }
  if (leftP())
  {
    return getParent();
  }
  for (InnerAVLNode* node = getParent();
       node != 0;
       node = node->getParent())
  {
    if (node->leftP())
    {
      return node->getParent();
    }
  }
  return 0;
}

InnerAVLNode* 
InnerAVLNode::unLink()
{
  if (!left)
  {
    if (!right)
    {
      changeParentLink(0);
      return 0;
    }
    else 
    {
      changeParentLink(right);
      right->setParent(parent);
      return right;
    }
  }
  else if (!right)
  {
    changeParentLink(left);
    left->setParent(parent);
    return left;
  }
  else 
  {
    // left != 0 && right != 0
    if (!left->right)
    {
      InnerAVLNode* node = left;
      changeParentLink(node);
      node->setParent(parent);
      node->right = right;
      right->setParent(node);
      return node;
    }
    if (!right->left)
    {
      InnerAVLNode* node = right;
      changeParentLink(node);
      node->setParent(parent);
      node->left = left;
      left->setParent(node);
      return node;
    }
    InnerAVLNode* node = left->right->getTreeRight();
    InnerAVLNode* update = node->getParent();
    InnerAVLNode* node_l = node->left;
    node->changeParentLink(node_l);
    if (node_l)
    {
      node_l->setParent(update);
    }
    while (update != this)
    {
      update->updateTreeHeight();
      update = update->getParent();
    }
    changeParentLink(node);
    node->setParent(parent);
    node->right = right;
    right->setParent(node);
    node->left = left;
    left->setParent(node);
    return node;
  }
}

void 
InnerAVLNode::changeParentLink(InnerAVLNode* new_node)
{
  if (parent)
  {
    if (parent->left == this)
    {
      parent->left = new_node;
    }
    else if (parent->right == this)
    {
      parent->right = new_node;
    }
  }
}

void 
InnerAVLNode::deleteTree()
{
  for (InnerAVLNode* node = getLeafNode();
       node != 0;
       node = getLeafNode())
  {
    node->changeParentLink(0);
    delete node;
  }
  changeParentLink(0);
  delete this;
}

InnerAVLNode* 
InnerAVLNode::getLeafNode() const
{
  if (!getChildNode())
  {
    return 0;
  }
  InnerAVLNode* node;
  for (node = getChildNode();
       node->getChildNode() != 0;
       node = node->getChildNode())
  {
    ;
  }
  return node;
}

InnerAVLNode*
InnerAVLNode::balanceTree(InnerAVLNode* root)
{
  for (InnerAVLNode* node = this;
       node != 0;
       node = node->getParent())
  {
    unsigned int left_height = (node->getLeft() == 0) ? 0 : node->getLeft()->height;
    unsigned int right_height = (node->getRight() == 0) ? 0 : node->getRight()->height;
    unsigned int new_height = getMax(left_height, right_height) + 1;
    unsigned int diff_height = getDiff(left_height, right_height);
    if ((diff_height <= 1) && (new_height == node->height))
    {
      break;
    }
    node->height = new_height;
    if (diff_height > 1)
    {
      if (node == root)
      {
	root = node = node->rotation();
      }
      else 
      {
	node = node->rotation();
      }
    }
  }
  return root;
}

InnerAVLNode* 
InnerAVLNode::rotation()
{
  unsigned int ll = (getLeft() == 0) ? 0 : ((getLeft()->getLeft() == 0) ? 0 :  getLeft()->getLeft()->height);
  unsigned int lr = (getLeft() == 0) ? 0 : ((getLeft()->getRight() == 0) ? 0 :  getLeft()->getRight()->height);
  unsigned int rl = (getRight() == 0) ? 0 :((getRight()->getLeft() == 0) ? 0 : getRight()->getLeft()->height);
  unsigned int rr = (getRight() == 0) ? 0 : ((getRight()->getRight() == 0) ? 0 :  getRight()->getRight()->height);
  unsigned int max_height = getMax(ll, lr, rl, rr);
  if (max_height == ll)
  {
    return leftSingleRotation();
  }
  else if (max_height == lr)
  {
    return leftDoubleRotation();
  }
  else if (max_height == rl)
  {
    return rightDoubleRotation();
  }
  else 
  {
    return rightSingleRotation();
  }
}

InnerAVLNode* 
InnerAVLNode::leftSingleRotation()
{
  InnerAVLNode* parent = getParent();
  InnerAVLNode* l  = getLeft();
  InnerAVLNode* lr = l->getRight();
  
  this->changeParentLink(l);
  l->setParent(parent);
  
  l->setRight(this);
  this->setParent(l);
  
  this->setLeft(lr);
  if (lr != 0)
  {
    lr->setParent(this);
  }
  
  this->updateTreeHeight();
  l->updateTreeHeight();
  
  return l;
}

InnerAVLNode* 
InnerAVLNode::leftDoubleRotation()
{
  InnerAVLNode* parent = getParent();
  InnerAVLNode* l = getLeft();
  InnerAVLNode* lr = l->getRight();
  InnerAVLNode* lrl = lr->getLeft();
  InnerAVLNode* lrr = lr->getRight();
  
  this->changeParentLink(lr);
  lr->setParent(parent);
  
  lr->setLeft(l);
  l->setParent(lr);
  
  lr->setRight(this);
  this->setParent(lr);
  
  l->setRight(lrl);
  if (lrl != 0)
  {
    lrl->setParent(l);
  }
  
  this->setLeft(lrr);
  if (lrr != 0)
  {
    lrr->setParent(this);
  }
  
  l->updateTreeHeight();
  this->updateTreeHeight();
  lr->updateTreeHeight();
  
  return lr;
}

InnerAVLNode* 
InnerAVLNode::rightSingleRotation()
{
  InnerAVLNode* parent = getParent();
  InnerAVLNode* r  = getRight();
  InnerAVLNode* rl = r->getLeft();
  
  this->changeParentLink(r);
  r->setParent(parent);
  
  r->setLeft(this);
  this->setParent(r);
  
  this->setRight(rl);
  if (rl != 0)
  {
    rl->setParent(this);
  }
  
  this->updateTreeHeight();
  r->updateTreeHeight();
  
  return r;
}

InnerAVLNode* 
InnerAVLNode::rightDoubleRotation()
{
  InnerAVLNode* parent = getParent();
  InnerAVLNode* r = getRight();
  InnerAVLNode* rl = r->getLeft();
  InnerAVLNode* rll = rl->getLeft();
  InnerAVLNode* rlr = rl->getRight();
  
  this->changeParentLink(rl);
  rl->setParent(parent);
  
  rl->setLeft(this);
  this->setParent(rl);
  
  rl->setRight(r);
  r->setParent(rl);
  
  this->setRight(rll);
  if (rll != 0)
  {
    rll->setParent(this);
  }
  
  r->setLeft(rlr);
  if (rlr != 0)
  {
    rlr->setParent(r);
  }
  
  r->updateTreeHeight();
  this->updateTreeHeight();
  rl->updateTreeHeight();
  
  return rl;
}

void 
InnerAVLNode::updateTreeHeight()
{
  unsigned int right = (getRight() == 0) ? 0 : getRight()->height;
  unsigned int left  = (getLeft() == 0) ? 0 : getLeft()->height;
  height = getMax(right, left) + 1;
}

// ------------------------------------------------------------
// class InnerAVL

InnerAVL::InnerAVL()
{
  root = 0;
}

InnerAVL::~InnerAVL()
{
  clear();
}

bool
InnerAVL::emptyP() const
{
  return !root;
}

Size
InnerAVL::number() const
{
  Size num = 0;
  if (root)
  {
    InnerAVLNode* p = root->getTreeLeft();
    while (p)
    {
      num++;
      p = p->getNext();
    }
  }
  return num;
}

void
InnerAVL::clear()
{
  if (root != 0)
  {
    root->deleteTree();
    root = 0;
  }
}

void 
InnerAVL::deleteNode(InnerAVLNode* node)
{
  if ((root != 0) && (node != 0)) 
  {
    InnerAVLNode* parent = node->getParent();
    InnerAVLNode* new_node = node->unLink();
    if (new_node != 0) 
    {
      new_node->updateTreeHeight();
    }
    if (root == node) 
    {
      root = new_node;
    }
    delete node;
    if (parent != 0) 
    {
      root = parent->balanceTree(root);
    }
  }
}

CCC_NAMESPACE_END(CCC);
