﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.

#ifndef INCLUDE_ccc_base_AVLDict_h
#define INCLUDE_ccc_base_AVLDict_h

#include <ccc/base/Dict.h>

CCC_NAMESPACE_START(CCC)

class InnerAVLNode
  : public Object
{
  void* element;
  InnerAVLNode* parent;
  InnerAVLNode* left;
  InnerAVLNode* right;
  unsigned int height;
  
 private:
  InnerAVLNode* rotation();
  InnerAVLNode* leftSingleRotation();
  InnerAVLNode* leftDoubleRotation();
  InnerAVLNode* rightSingleRotation();
  InnerAVLNode* rightDoubleRotation();

 public:
  InnerAVLNode(void* element, InnerAVLNode* parent = 0,
	       InnerAVLNode* left = 0, InnerAVLNode* right = 0);
  ~InnerAVLNode() { ; }
  void* getElement() const { return element; }
  void setElement(void* element) { InnerAVLNode::element = element; }
  unsigned int getHeight() const { return height; }
  InnerAVLNode* getParent() const { return parent; }
  void setParent(InnerAVLNode* parent) { InnerAVLNode::parent = parent; }
  InnerAVLNode* getLeft() const { return left; }
  void setLeft(InnerAVLNode* left) { InnerAVLNode::left = left; }
  InnerAVLNode* getRight() const { return right; }
  void setRight(InnerAVLNode* right) { InnerAVLNode::right = right; }
  bool leftP() const { return !parent ? false : (parent->getLeft() == this); }
  bool rightP() const { return !parent ? false : (parent->getRight() == this); }
  InnerAVLNode* getTreeLeft() const;
  InnerAVLNode* getTreeRight() const;
  InnerAVLNode* getPrev() const;
  InnerAVLNode* getNext() const;
  InnerAVLNode* unLink();
  void changeParentLink(InnerAVLNode* new_node);
  void deleteTree();
  InnerAVLNode* getLeafNode() const;
  InnerAVLNode* getChildNode() const { return(left != 0) ? left : right; }
  InnerAVLNode* balanceTree(InnerAVLNode* root); // return new root
  void updateTreeHeight();
};

class InnerAVL
  : public Object
{
  InnerAVLNode* root;

 public:
  InnerAVL();
  ~InnerAVL();

  bool emptyP() const;
  Size number() const;
  void clear();
  void deleteNode(InnerAVLNode* node);
  InnerAVLNode* getRoot() const { return root; }
  void setRoot(InnerAVLNode* new_root) { root = new_root; }
};

class InnerAVLIterator
  : public Object
{
  InnerAVLNode* top;
  InnerAVLNode* ptr;
  
 public:
  InnerAVLIterator(InnerAVLNode* top);
  void* current() const;
  void* next();
  void* prev();
  void* rewind();
  void* unwind();
  bool validPositionP() const;
  Size getPosition() const;
};

template <class Element, class Key> class AVLDictIterator;	// forward ref

template <class Element, class Key>
class AVLDict			//! AVLDict
  : public Dict<Element, Key>
{
  InnerAVL avl;

 public:
  AVLDict() { }
  virtual ~AVLDict() { }

  // Collection interface
  virtual bool emptyP() const { return avl.emptyP(); }
  virtual Size number() const { return avl.number(); }
  virtual bool add(Element* element);
  virtual bool remove(Element* element);
  virtual void clear() { avl.clear(); }
  virtual bool elementP(Element* element) const;
  virtual Iterator<Element>* createIterator() const;

  // Set interface
  //virtual bool remove(Set<Element>* s);
  //virtual bool add(Set<Element>* s);

  // Dict interface
  virtual Element* search(const Key* key) const;

 protected:
  virtual const Key* getKey(const Element* element) const = 0;
  virtual int compare(const Key* key1, const Key* key2) const = 0;

  friend class AVLDictIterator<Element, Key>;
};

template <class Element, class Key>
class AVLDictIterator
  : public Iterator<Element>
{
  InnerAVLIterator inner;

 public:
  AVLDictIterator(AVLDict<Element, Key>* dic) : inner(dic->avl.getRoot()) { }
  AVLDictIterator(InnerAVLNode* top) : inner(top) { }
  virtual ~AVLDictIterator() { };
  virtual Element* current() const { return (Element*)inner.current(); }
  virtual Element* next() { return (Element*)inner.next(); }
  virtual Element* prev() { return (Element*)inner.prev(); }
  virtual Element* rewind() { return (Element*)inner.rewind(); }
  virtual Element* unwind() { return (Element*)inner.unwind(); }
  virtual bool validPositionP() const { return inner.validPositionP(); }
  virtual Size getPosition() const { return inner.getPosition(); }
};

template <class Element, class Key>
bool
AVLDict<Element, Key>::add(Element* element)
{
  InnerAVLNode* node = avl.getRoot();
  while (node)
  {
    int cmp = this->compareX((Element*)node->getElement(), element);
    if (cmp < 0)
    {
      if (node->getRight() == 0)
      {
	node->setRight(new InnerAVLNode((void*)element, node));
	avl.setRoot(node->balanceTree(avl.getRoot()));
	return true;
      }
      node = node->getRight();
    }
    else if (cmp > 0)
    {
      if (node->getLeft() == 0)
      {
	node->setLeft(new InnerAVLNode((void*)element, node));
	avl.setRoot(node->balanceTree(avl.getRoot()));
	return true;
      }
      node = node->getLeft();
    }
    else
    {
      return false;
    }
  }
  avl.setRoot(new InnerAVLNode((void*)element));
  return true;
}

template <class Element, class Key>
bool
AVLDict<Element, Key>::remove(Element* element)
{
  InnerAVLNode* node = avl.getRoot();
  while (node)
  {
    int cmp = this->compareX((Element*)node->getElement(), element);
    if (cmp < 0)
    {
      node = node->getRight();
    }
    else if (cmp > 0)
    {
      node = node->getLeft();
    }
    else
    {
      avl.deleteNode(node);
      return true;
    }
  }
  return false;
}

template <class Element, class Key>
bool
AVLDict<Element, Key>::elementP(Element* element) const
{
  InnerAVLNode* node = avl.getRoot();
  while (node)
  {
    int cmp = this->compareX((Element*)node->getElement(), element);
    if (cmp < 0)
    {
      node = node->getRight();
    }
    else if (cmp > 0)
    {
      node = node->getLeft();
    }
    else
    {
      return true;
    }
  }
  return false;
}

template <class Element, class Key>
Iterator<Element>*
AVLDict<Element, Key>::createIterator() const
{
  return new AVLDictIterator<Element, Key>(avl.getRoot());
}

template <class Element, class Key>
Element*
AVLDict<Element, Key>::search(const Key* key) const
{
  InnerAVLNode* node = avl.getRoot();
  while (node != 0)
  {
    int cmp = this->compareWithElement((Element*)node->getElement(), key);
    if (cmp < 0)
    {
      node = node->getRight();
    }
    else if (cmp > 0)
    {
      node = node->getLeft();
    }
    else
    {
      return (Element*)node->getElement();
    }
  }
  return 0;
}

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_AVLDict_h */
