﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.

#ifndef INCLUDE_ccc_base_Allocator_h
#define INCLUDE_ccc_base_Allocator_h

#include <ccc/base/base.h>
#include <stddef.h>

CCC_NAMESPACE_START(CCC)

/*!
 * \ja
 * このクラスはメモリを動的に管理します。
 * \ja_end
 * \en
 * This class manages memory dynamically.
 * \en_end
 */
class Allocator
  : public Object
{
  /*!
   * \ja
   * 現在確保しているメモリへのポインタ
   * \ja_end
   * \en
   * A pointer to the memory which is currently allocated.
   * \en_end
   */
  void* mem;
  /*!
   * \ja
   * 現在確保されているメモリサイズ
   * \ja_end
   * \en
   * Size of the memory which is currently allocated.
   * \en_end
   */
  Size size;
  /*!
   * \ja
   * 現在使用中のメモリサイズ
   * \ja_end
   * \en
   * Size of the memory which  is currently used.
   * \en_end
   */
  Size used_size;
 public:
  /*!
   * \ja
   * コンストラクタ
   * \param size 最初に確保するメモリのサイズ
   * \ja_end
   * \en
   * constructor
   * \param size size of initially allocated memory
   * \en_end
   */
  Allocator(Size size = 16);
  /*!
   * \ja
   * コンストラクタ<br>
   * 初期化時に引数で渡された Allocator のメモリをコピーします。
   * \param m コピーを作成する Allocator への参照
   * \ja_end
   * \en
   * constructor
   * This constructor will copy the memory of Allocator
   * which is passed as an argument during the initialization stage.
   * \param m reference to the Allocator which is copied
   * \en_end
   */
  Allocator(Allocator& m);
  /*!
   * \ja
   * デストラクタ<br>
   * このオブジェクトが管理していたメモリは、デストラクタの呼び出しに
   * ともない解放されます。
   * \ja_end
   * \en
   * destructor
   * The memory which is managed by this object will be freed
   * by the calling this destructor.
   * \en_end
   */
  ~Allocator();
  /*!
   * \ja
   * このメソッドはこのオブジェクトが管理しているメモリの先頭アドレスを返します。
   * \return 管理中のメモリの先頭アドレス
   * \ja_end
   * \en
   * This method returns the first address of the memory which is managed by this object.
   * \return the first address of the memory which is being managed
   * \en_end
   */
  void* getMem() const { return mem; }
  /*!
   * \ja
   * このメソッドはこのオブジェクトが管理しているメモリのサイズを返します。
   * \return 管理中のメモリのサイズ
   * \ja_end
   * \en
   * This method returns the size of the memory this object is managing.
   * \return the size of the memory which is being managed
   * \en_end
   */
  Size getSize() const { return size; }
  /*!
   * \ja
   * メモリのサイズを引数で渡したメモリサイズより大きく拡張します。
   * 管理中のメモリのサイズが引数に渡されたサイズよりも大きい場合には何も行ないません。
   * 管理中のメモリのサイズが引数に渡されたサイズよりも小さい場合には、メモリの拡張を行ないます。
   * メモリの拡張の際に可能であれば、メモリの先頭アドレスが変化しないように拡張します。
   * しかし管理中のメモリのすぐ後ろが別の目的で利用中の場合には、引数で指定された大きさの
   * メモリを再度割り当てて、従来のメモリ内容を新しいメモリにコピーします。
   * \param size メモリサイズ
   * \ja_end
   * \en
   * This method expands the memory size larger than that passed as an argument.
   * This method does nothing if the size of the managed memory is larger than that
   * passed as an argument. This method expands the memory if the size of the memory
   * being managed is smaller than that passed as an argument.
   * If possible, the first address of the memory is not changed when expanding the
   * memory. If the rear of the managed memory is being used for another purpose,
   * this method allocates again the memory of the size specified by the argument
   * and copies the previous memory image to the new memory.
   * \param size memory size
   * \en_end
   */
  void expand(Size size);
  /*!
   * \ja
   * このメソッドは引数で渡したメモリサイズに管理中のメモリサイズを縮小します。
   * このメソッドは現在実装されておらず、呼び出しても何も行ないません。
   * \param size メモリサイズ
   * \ja_end
   * \en
   * This method shrinks the managed memory to the size passed by the argument.
   * This method is not implemented now, and nothing happens even if it is called.
   * \param size memory size
   * \en_end
   */
  void shrink(Size size);
  /*!
   * \ja
   * このメソッドは現在使用中のメモリサイズを取得します。
   * \return 使用中のメモリサイズ
   * \ja_end
   * \en
   * This method returns the size of the memory which is currently used.
   * \return the size of the memory which is currently used
   * \en_end
   */
  Size getUsedSize() const { return used_size; };
  /*!
   * \ja
   * このメソッドは現在利用中のメモリサイズを設定します。
   * \param n 設定するメモリサイズ
   * \ja_end
   * \en
   * This method set the size of the memory which is currently used.
   * \param n  the memory size being set.
   * \en_end
   */
  void setUsedSize(Size n);
  /*!
   * \ja
   * このメソッドは現在利用中のメモリサイズを引数の数だけ増やします。
   * \param add_size メモリサイズの増分
   * \ja_end
   * \en
   * This method add the size of the memory which is currently used.
   * \param size  the memory size being added.
   * \en_end
   */
  Size addUsedSize(Size add_size);
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_Allocator_h */
