﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.

#ifndef INCLUDE_ccc_base_ArrayList_h
#define INCLUDE_ccc_base_ArrayList_h

#include <ccc/base/XList.h>

CCC_NAMESPACE_START(CCC)

// $Id$
//
//! \class ArrayList
//! \brief eXogenous array list
//!
//! \verbatim buffer -> [*][*][*][*][-][ ][ ] \endverbatim
//
// class InnerArrayList
// class InnerArrayListIterator
// class XArraryList<T>
// class XArraryListIterator<T>

/*!
 * \brief InnerArrayList
 *
 * There are at least one more buffers for using the sentinel algorithm.
 */
class InnerArrayList
  : public InnerList
{
  //! buffer
  typedef void*(* buffer_t)[];
  buffer_t buffer;

  //! size of the buffer
  Size buffer_size;

  //! current used buffer size
  Size used_size;

  //! default initial buffer size
  enum { default_initial_buffer_size = 10 };

  //! buffer realloc additional size
  enum { extra_realloc_size = 10 };

  //! for to manage buffer size
  void checkBufferSize(Size new_size);
  bool validPositionP(Size pos) { return 0 < pos && pos <= used_size; }

public:
  InnerArrayList();
  //! constructor with inital buffer size
  InnerArrayList(Size initial_buffer_size);
  ~InnerArrayList();

  void assign(InnerArrayList& l);
  bool emptyP() const;
  Size number() const;
  void clear();
  void setUsedSize(Size s) { used_size = s; }

  void* vector(Size n) const;
  Size getPosition(void* element) const;
  void* prev(void* item) const;
  void* next(void* item) const;
  void* head() const;
  void* tail() const;
  bool elementP(void* element) const;
  void push(void* item);
  void* pop();
  void inject(void* item);
  void* eject();

  bool insertBefore(Size pos, void* item);
  bool insertBefore(void* pos, void* item);
  bool insertAfter(Size pos, void* item);
  bool insertAfter(void* pos, void* item);
  bool remove(void* pos);
  bool remove(Size pos);
  bool exchangeElement(Size pos, void* item);
  bool exchangeElement(void* pos, void* item);
  bool swapElement(void* i1, void* i2);
  bool swapElement(Size n1, Size n2);
  bool removeAllBefore(Size pos);
  bool removeAllBefore(void* pos);
  bool removeAllAfter(Size pos);
  bool removeAllAfter(void* pos);
  bool removeAllBetween(Size pos1, Size pos2);
  bool removeAllBetween(void* pos1, void* pos2);

  /*!
   * 配列バッファを返します。
   * \return 配列バッファ
   */
  void** getArrayBuffer();

  friend class InnerArrayListIterator;
};

class InnerArrayListIterator	//! InnerArrayListIterator
  : public InnerIterator
{
  //! iteration position
  Size pos;

public:
  InnerArrayListIterator(const InnerArrayList* l);
  void* current() const;
  void* next();
  void* prev();
  void* rewind();
  void* unwind();
  bool validPositionP() const;
  Size getPosition() const;	//!< valid position: return >= 1
};

template <class Element> class ArrayListIterator;

template <class Element>
class ArrayList
  : public XList<Element>
{
  InnerArrayList ial;

 public:
  ArrayList() : ial() { }
  /*!
   * コンストラクタ
   * \param initial_buffer_size 初期配列サイズ
   */
  ArrayList(Size initial_buffer_size) : ial(initial_buffer_size) { }
  /*!
   * コンストラクタ
   * \param initial_buffer_size 初期配列サイズ
   * \param initial_used_isze 初期使用データ数
   */
  ArrayList(Size initial_buffer_size, Size initial_used_isze)
    : ial(initial_buffer_size)
  {
    ial.setUsedSize(initial_used_isze);
  }
  virtual ~ArrayList() { }
  const InnerArrayList* getInnerArrayList() const { return &ial; }
  void assign(ArrayList& l) { ial.assign(l.ial); }
  void operator = (ArrayList& l) { assign(l); };

  /*!
   * \ja
   * \name Collectionインタフェース
   * \ja_end
   * \en
   * \name Collection interface
   * \en_end
   */
  //@{
  virtual bool emptyP() const { return ial.emptyP(); }
  virtual Size number() const { return ial.number(); }
  virtual bool add(Element* element) { ial.inject((void*)element); return true; }
  virtual void clear() { ial.clear(); }
  virtual void setUsedSize(Size s) { ial.setUsedSize(s); }
  virtual bool elementP(Element* element) const { return ial.elementP((void*)element); };
  virtual Iterator<Element>* createIterator() const;
  //@}

  /*!
   * \ja
   * \name Listインタフェース
   * \ja_end
   * \en
   * \name List interface
   * \en_end
   */
  //@{
  virtual Element* vector(Size vec_pos) const { return (Element*)ial.vector(vec_pos); }
  virtual Size getPosition(Element* element) const { return ial.getPosition((void*)element); }
  virtual Element* prev(Element* element) const { return (Element*)ial.prev((void*)element); }
  virtual Element* next(Element* element) const { return (Element*)ial.next((void*)element); }
  virtual Element* head() const { return (Element*)ial.head(); }
  virtual Element* tail() const { return (Element*)ial.tail(); }
  virtual void push(Element* element) { ial.push((void*)element); }
  virtual Element* pop() { return (Element*)ial.pop(); }
  virtual void inject(Element* element) { ial.inject((void*)element); }
  virtual Element* eject() { return (Element*)ial.eject(); }
  virtual bool insertBefore(Size pos, Element* element) { return ial.insertBefore(pos, (void*)element); }
  virtual bool insertBefore(Element* pos, Element* element) { return ial.insertBefore((void*)pos, (void*)element); }
  virtual bool insertAfter(Size pos, Element* element) { return ial.insertAfter(pos, (void*)element); }
  virtual bool insertAfter(Element* pos, Element* element) { return ial.insertAfter((void*)pos, (void*)element); }
  virtual bool remove(Element* pos) { return ial.remove((void*)pos); }
  virtual bool remove(Size pos) { return ial.remove(pos); }
  virtual bool exchangeElement(Size pos, Element* element) { return ial.exchangeElement(pos, (void*)element); }
  virtual bool exchangeElement(Element* pos, Element* element) { return ial.exchangeElement((void*)pos, (void*)element); }
  virtual bool swapElement(Element* i1, Element* i2) { return ial.swapElement((void*)i1, (void*)i2); }
  virtual bool swapElement(Size n1, Size n2) { return ial.swapElement(n1, n2); }
  virtual bool removeAllBefore(Size pos) { return ial.removeAllBefore(pos); }
  virtual bool removeAllBefore(Element* pos) { return ial.removeAllBefore((void*)pos); }
  virtual bool removeAllAfter(Size pos) { return ial.removeAllAfter(pos); }
  virtual bool removeAllAfter(Element* pos) { return ial.removeAllAfter((void*)pos); }
  virtual bool removeAllBetween(Size pos1, Size pos2) { return ial.removeAllBetween(pos1, pos2); }
  virtual bool removeAllBetween(Element* pos1, Element* pos2) { return ial.removeAllBetween((void*)pos1, (void*)pos2); }
  //@}
  /*!
   * 配列バッファを返します。
   * \return 配列バッファ
   */
  Element** getArrayBuffer() { return (Element**)ial.getArrayBuffer(); }
};

template <class Element>
class ArrayListIterator		//! ArrayListIterator
  : public Iterator<Element>
{
  InnerArrayListIterator ili;

 public:
  ArrayListIterator(const InnerArrayList* l) : ili(l) { }
  ArrayListIterator(const InnerArrayList& l) : ili(&l) { }
  ArrayListIterator(const ArrayList<Element>* l) : ili(l->getInnerArrayList()) { }
  ArrayListIterator(const ArrayList<Element>& l) : ili(l.getInnerArrayList()) { }
  virtual ~ArrayListIterator() { }
  Element* current() const { return (Element*)ili.current(); }
  Element* next() { return (Element*)ili.next(); }
  Element* prev() { return (Element*)ili.prev(); }
  Element* rewind() { return (Element*)ili.rewind(); }
  Element* unwind() { return (Element*)ili.unwind(); }
  bool validPositionP() const { return ili.validPositionP(); }
  Size getPosition() const { return ili.getPosition(); }
};

template <class Element>
Iterator<Element>*
ArrayList<Element>::createIterator() const
{
  return new ArrayListIterator<Element>(this);
}

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_ArrayList_h */
