﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.
// expression of double exogenous list

#include <ccc/base/DLinkList.h>
#include <assert.h>

CCC_NAMESPACE_START(CCC);

// ------------------------------------------------------------
// class InnerDLinkListNode

class InnerDLinkListNode
  : public InnerListNode
{
  void* element;
  InnerDLinkListNode* next;
  InnerDLinkListNode* prev;
  
 public:
  InnerDLinkListNode(void* element = 0);
  void* getElement() const { return element; }
  void setElement(void* element) { InnerDLinkListNode::element = element; }
  InnerDLinkListNode* getNext() const { return next; }
  InnerDLinkListNode* getPrev() const { return prev; }
  void setNext(InnerDLinkListNode* next);
  void setPrev(InnerDLinkListNode* prev);
};

InnerDLinkListNode::InnerDLinkListNode(void* element)
{
  InnerDLinkListNode::element = element;
  next = this;
  prev = this;
}

inline void
InnerDLinkListNode::setNext(InnerDLinkListNode* next_node)
{
  next = next_node;
  if (next_node)
  {
    next_node->prev = this;
  }
}

inline void
InnerDLinkListNode::setPrev(InnerDLinkListNode* prev_node)
{
  prev = prev_node;
  if (prev_node)
  {
    prev_node->next = this;
  }
}


// ------------------------------------------------------------
// class InnerDLinkList

InnerDLinkList::InnerDLinkList()
{
  sentinel = new InnerDLinkListNode();
  CHECKVALIDITY();
}

InnerDLinkList::InnerDLinkList(InnerDLinkList& l)
{
  sentinel = new InnerDLinkListNode();
  for (InnerDLinkListNode* node = l.getFirst(); 
       node != l.sentinel; 
       node = node->getNext())
  {
    InnerDLinkListNode* new_node = new InnerDLinkListNode(node->getElement());
    insert(sentinel->getPrev(), sentinel, new_node);
  }
  CHECKVALIDITY();
}

InnerDLinkList::~InnerDLinkList()
{
  clear();
  delete sentinel;
}

void
InnerDLinkList::insert(InnerDLinkListNode* prev, InnerDLinkListNode* next, InnerDLinkListNode* node)
{
  prev->setNext(node);
  next->setPrev(node);
}

void
InnerDLinkList::clear()
{
  InnerDLinkListNode* node = getFirst();
  while (node != sentinel)
  {
    InnerDLinkListNode* next = node->getNext();
    delete node;
    node = next;
  }
  sentinel->setNext(sentinel);
  CHECKVALIDITY();
}

InnerDLinkListNode*
InnerDLinkList::getFirst() const
{
  return sentinel->getNext();
}

InnerDLinkListNode*
InnerDLinkList::getLast() const
{
  return sentinel->getPrev();
}

InnerDLinkListNode*
InnerDLinkList::getNode(Size pos) const
{
  InnerDLinkListNode* node = sentinel;
  if (pos > 0)
  {
    for (node = getFirst();
	 node != sentinel;
	 node = node->getNext())
    {
      if (--pos == 0)
      {
	break;
      }
    }
  }
  return node;
}

InnerDLinkListNode*
InnerDLinkList::getNode(void* pos) const
{
  InnerDLinkListNode* node = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  for (node = getFirst();
       node->getElement() != pos;
       node = node->getNext())
  {
    ;
  }
  sentinel->setElement(0);
  return node;
}

void
InnerDLinkList::getBand(InnerDLinkListNode** beg_node, InnerDLinkListNode** end_node, void* pos1, void* pos2) const
{
  InnerDLinkListNode* node = getFirst();
  InnerDLinkListNode* pos1_node = 0;
  InnerDLinkListNode* pos2_node = 0;
  bool pos1_is_beg_p = false;
  while (node != sentinel)
  {
    if (!pos1_node &&
	node->getElement() == pos1)
    {
      pos1_node = node;
    }
    if (!pos2_node &&
	node->getElement() == pos2)
    {
      pos2_node = node;
      if (pos1_node)
      {
	pos1_is_beg_p = true;
      }
    }
    node = node->getNext();
  }
  if (pos1_node && pos2_node)
  {
    if (pos1_is_beg_p)
    {
      *beg_node = pos1_node;
      *end_node = pos2_node;
    }
    else
    {
      *beg_node = pos2_node;
      *end_node = pos1_node;
    }
  }
  else
  {
    *beg_node = sentinel;
    *end_node = sentinel;
  }
}

void
InnerDLinkList::assign(InnerDLinkList& l)
{
  clear();
  for (InnerDLinkListNode* node = l.getFirst(); 
       node != l.sentinel; 
       node = node->getNext())
  {
    InnerDLinkListNode* new_node = new InnerDLinkListNode(node->getElement());
    insert(sentinel->getPrev(), sentinel, new_node);
  }
  CHECKVALIDITY();
}

bool
InnerDLinkList::emptyP() const
{
  return getFirst() == sentinel;
}

Size
InnerDLinkList::number() const
{
  Size num = 0;
  for (InnerDLinkListNode* node = getFirst();
       node != sentinel;
       node = node->getNext())
  {
    num++;
  }
  return num;
}

void*
InnerDLinkList::vector(Size vec_pos) const
{
  for (InnerDLinkListNode* node = getFirst();
       node != sentinel;
       node = node->getNext())
  {
    if (vec_pos-- == 0)
    {
      return node->getElement();
    }
  }
  return 0;
}

Size
InnerDLinkList::getPosition(void* pos) const
{
  InnerDLinkListNode* node = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  Size n_pos = 1;
  for (node = getFirst();
       node->getElement() != pos;
       node = node->getNext())
  {
    n_pos ++;
  }
  sentinel->setElement(0);
  if (node == sentinel)
  {
    n_pos = 0;
  }
  return n_pos;
}

void*
InnerDLinkList::prev(void* element) const
{
  InnerDLinkListNode* node = getNode(element);
  if ((node != sentinel) &&
      (node->getPrev() != sentinel))
  {
    return node->getPrev()->getElement();
  }
  return 0;
}

void*
InnerDLinkList::next(void* element) const
{
  InnerDLinkListNode* node = getNode(element);
  if ((node != sentinel) &&
      (node->getNext() != sentinel))
  {
    return node->getNext()->getElement();
  }
  return 0;
}

void*
InnerDLinkList::head() const
{
  return getFirst()->getElement();
}

void*
InnerDLinkList::tail() const
{
  return getLast()->getElement();
}

bool
InnerDLinkList::elementP(void* pos) const
{
  InnerDLinkListNode* node = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  for (node = getFirst();
       node->getElement() != pos;
       node = node->getNext())
  {
    ;
  }
  sentinel->setElement(0);
  return (node != sentinel) ? true : false;
}

void
InnerDLinkList::push(void* element)
{
  InnerDLinkListNode* new_node = new InnerDLinkListNode(element);
  insert(sentinel, getFirst(), new_node);
  CHECKVALIDITY();
}

void*
InnerDLinkList::pop()
{
  if (getFirst() == sentinel)
  {
    return 0;
  }
  InnerDLinkListNode* pop_node = getFirst();
  InnerDLinkListNode* new_first = pop_node->getNext();
  sentinel->setNext(new_first);
  void* element = pop_node->getElement();
  delete pop_node;
  CHECKVALIDITY();
  return element;
}

void
InnerDLinkList::inject(void* element)
{
  InnerDLinkListNode* new_node = new InnerDLinkListNode(element);
  insert(getLast(), sentinel, new_node);
  CHECKVALIDITY();
}

void*
InnerDLinkList::eject()
{
  if (getFirst() == sentinel)
  {
    return 0;
  }
  InnerDLinkListNode* eject_node = getLast();
  InnerDLinkListNode* new_last = eject_node->getPrev();
  sentinel->setPrev(new_last);
  void* element = eject_node->getElement();
  delete eject_node;
  CHECKVALIDITY();
  return element;
}

bool
InnerDLinkList::insertBefore(InnerDLinkListNode* node, void* element)
{
  bool ret = false;
  if (node != sentinel)
  {
    InnerDLinkListNode* new_node = new InnerDLinkListNode(element);
    insert(node->getPrev(), node, new_node);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::insertBefore(Size pos, void* element)
{
  return insertBefore(getNode(pos), element);
}

bool
InnerDLinkList::insertBefore(void* pos, void* element)
{
  return insertBefore(getNode(pos), element);
}

bool
InnerDLinkList::insertAfter(InnerDLinkListNode* node, void* element)
{
  bool ret = false;
  if (node != sentinel)
  {
    InnerDLinkListNode* new_node = new InnerDLinkListNode(element);
    insert(node, node->getNext(), new_node);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::insertAfter(Size pos, void* element)
{
  return insertAfter(getNode(pos), element);
}

bool
InnerDLinkList::insertAfter(void* pos, void* element)
{
  return insertAfter(getNode(pos), element);
}

bool
InnerDLinkList::remove(InnerDLinkListNode* node)
{
  bool ret = false;
  if (node != sentinel)
  {
    InnerDLinkListNode* prev_node = node->getPrev();
    InnerDLinkListNode* next_node = node->getNext();
    prev_node->setNext(next_node);
    delete node;
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::remove(void* pos)
{
  return remove(getNode(pos));
}

bool
InnerDLinkList::remove(Size pos)
{
  return remove(getNode(pos));
}

bool
InnerDLinkList::exchangeElement(InnerDLinkListNode* node, void* element)
{
  bool ret = false;
  if (node != sentinel)
  {
    node->setElement(element);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::exchangeElement(Size pos, void* element)
{
  return exchangeElement(getNode(pos), element);
}

bool
InnerDLinkList::exchangeElement(void* pos, void* element)
{
  return exchangeElement(getNode(pos), element);
}

bool
InnerDLinkList::swapElement(InnerDLinkListNode* node1, InnerDLinkListNode* node2)
{
  bool ret = false;
  if ((node1 != sentinel) &&
      (node2 != sentinel))
  {
    void* swap_element = node1->getElement();
    node1->setElement(node2->getElement());
    node2->setElement(swap_element);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::swapElement(void* i1, void* i2)
{
  return swapElement(getNode(i1), getNode(i2));
}

bool
InnerDLinkList::swapElement(Size n1, Size n2)
{
  return swapElement(getNode(n1), getNode(n2));
}

bool
InnerDLinkList::removeAllBefore(InnerDLinkListNode* before_node)
{
  bool ret = false;
  if (before_node != sentinel)
  {
    InnerDLinkListNode* node = getFirst();
    while (node != before_node)
    {
      InnerDLinkListNode* next = node->getNext();
      delete node;
      node = next;
    }
    sentinel->setNext(before_node);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::removeAllBefore(Size pos)
{
  return removeAllBefore(getNode(pos));
}

bool
InnerDLinkList::removeAllBefore(void* pos)
{
  return removeAllBefore(getNode(pos));
}

bool
InnerDLinkList::removeAllAfter(InnerDLinkListNode* after_node)
{
  bool ret = false;
  if (after_node != sentinel)
  {
    InnerDLinkListNode* node = after_node->getNext();
    while (node != sentinel)
    {
      InnerDLinkListNode* next = node->getNext();
      delete node;
      node = next;
    }
    sentinel->setPrev(after_node);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::removeAllAfter(Size pos)
{
  return removeAllAfter(getNode(pos));
}

bool
InnerDLinkList::removeAllAfter(void* pos)
{
  return removeAllAfter(getNode(pos));
}

bool
InnerDLinkList::removeAllBetween(InnerDLinkListNode* node1, InnerDLinkListNode* node2)
{
  bool ret = false;
  if ((node1 != sentinel) &&
      (node2 != sentinel))
  {
    InnerDLinkListNode* node1_prev = node1->getPrev();
    InnerDLinkListNode* node2_next = node2->getNext();
    InnerDLinkListNode* node = node1;
    while (node != node2_next)
    {
      InnerDLinkListNode* next = node->getNext();
      delete node;
      node = next;
    }
    node1_prev->setNext(node2_next);
    ret = true;
  }
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::removeAllBetween(Size pos1, Size pos2)
{
  Size beg_pos = pos1;
  Size end_pos = pos2;
  if (pos1 > pos2)
  {
    beg_pos = pos2;
    end_pos = pos1;
  }
  bool ret = removeAllBetween(getNode(beg_pos), getNode(end_pos));
  CHECKVALIDITY();
  return ret;
}

bool
InnerDLinkList::removeAllBetween(void* pos1, void* pos2)
{
  InnerDLinkListNode* beg_node;
  InnerDLinkListNode* end_node;
  getBand(&beg_node, &end_node, pos1, pos2);
  return removeAllBetween(beg_node, end_node);
}

#ifndef NDEBUG
void
InnerDLinkList::checkValidity() const
{
  assert(sentinel != 0);

  if (sentinel->getNext() == sentinel)
  {
    assert(sentinel->getPrev() == sentinel);
  }
  if (sentinel->getPrev() == sentinel)
  {
    assert(sentinel->getNext() == sentinel);
  }
  
  InnerDLinkListNode* node = getFirst();
  InnerDLinkListNode* prev_node = sentinel;
  int next_n = 0;
  while (node != sentinel)
  {
    next_n++;
    assert(node->getPrev() != 0);
    assert(node->getNext() != 0);
    assert(prev_node == node->getPrev());
    prev_node = node;
    node = node->getNext();
  }
  node = getLast();
  InnerDLinkListNode* next_node = sentinel;
  int prev_n = 0;
  while (node != sentinel)
  {
    prev_n++;
    assert(node->getPrev() != 0);
    assert(node->getNext() != 0);
    assert(next_node == node->getNext());
    next_node = node;
    node = node->getPrev();
  }
  assert(next_n == prev_n);
}
#endif /* NDEBUG */


// ------------------------------------------------------------
// class InnerDLinkIterator

InnerDLinkIterator::InnerDLinkIterator(const InnerDLinkList* l)
 : InnerIterator(l)
{
  node = getFirst();
}

InnerDLinkListNode*
InnerDLinkIterator::getFirst() const
{
  return getInnerDLinkList()->getFirst();
}

InnerDLinkListNode*
InnerDLinkIterator::getLast() const
{
  return getInnerDLinkList()->getLast();
}

void*
InnerDLinkIterator::current() const
{
  assert((node != getSentinel()) || (node->getElement() == 0));
  return node->getElement();
}

void*
InnerDLinkIterator::next()
{
  void* ret = current();
  node = node->getNext();
  return ret;
}

void*
InnerDLinkIterator::prev()
{
  void* ret = current();
  node = node->getPrev();
  return ret;
}

void*
InnerDLinkIterator::rewind()
{
  node = getFirst();
  return node->getElement();
}

void*
InnerDLinkIterator::unwind()
{
  node = getLast();
  return node->getElement();
}

bool
InnerDLinkIterator::validPositionP() const
{
  return node != getSentinel();
}

Size
InnerDLinkIterator::getPosition() const
{
  Size n = 0;
  InnerDLinkListNode* x = getFirst();
  while (x != getSentinel())
  {
    n++;
    if (x == node)
    {
      break;
    }
    x = x->getNext();
  }
  return n;
}

CCC_NAMESPACE_END(CCC);
