﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.

#ifndef INCLUDE_ccc_base_DLinkList_h
#define INCLUDE_ccc_base_DLinkList_h

#include <ccc/base/XList.h>

CCC_NAMESPACE_START(CCC)

// exogenous double pointer list
// template <class Element> class DLinkList;
// template <class Element> class DLinkIterator;
//
//! \class DLinkList
//! exogenous double pointer list
//!
//! \verbatim
//!    +--------------------------------+
//!    |                                |
//!    v                                v
//!   [*]<-->[*]<-->[*]<--...-->[*]<-->[0] <-- sentinel
//!    |      |      |           |
//!    v      v      v           v
//! Element Element Element   Element
//! \endverbatim

class InnerDLinkListNode;

class InnerDLinkList		//! InnerDLinkList
  : public InnerList
{
  friend class InnerDLinkIterator;

  InnerDLinkListNode* sentinel;

 private:
  void insert(InnerDLinkListNode* prev, InnerDLinkListNode* next, InnerDLinkListNode* node);
  InnerDLinkListNode* getFirst() const;
  InnerDLinkListNode* getLast() const;
  InnerDLinkListNode* getNode(Size pos) const;
  InnerDLinkListNode* getNode(void* pos) const;
  void getBand(InnerDLinkListNode** beg_node, InnerDLinkListNode** end_node, void* pos1, void* pos2) const;
  bool insertBefore(InnerDLinkListNode* node, void* element);
  bool insertAfter(InnerDLinkListNode* node, void* element);
  bool remove(InnerDLinkListNode* node);
  bool exchangeElement(InnerDLinkListNode* node, void* element);
  bool swapElement(InnerDLinkListNode* node1, InnerDLinkListNode* node2);
  bool removeAllBefore(InnerDLinkListNode* node);
  bool removeAllAfter(InnerDLinkListNode* node);
  bool removeAllBetween(InnerDLinkListNode* node1, InnerDLinkListNode* node2);

 public:
  InnerDLinkList();
  InnerDLinkList(InnerDLinkList& l);
  ~InnerDLinkList();

  void assign(InnerDLinkList& l);
  bool emptyP() const;
  Size number() const;
  void clear();

  void* vector(Size vec_pos) const;
  Size getPosition(void* element) const;
  void* prev(void* element) const;
  void* next(void* element) const;
  void* head() const;
  void* tail() const;
  bool elementP(void* element) const;
  void push(void* element);
  void* pop();
  void inject(void* element);
  void* eject();

  bool insertBefore(Size pos, void* element);
  bool insertBefore(void* pos, void* element);
  bool insertAfter(Size pos, void* element);
  bool insertAfter(void* pos, void* element);
  bool remove(void* pos);
  bool remove(Size pos);
  bool exchangeElement(Size pos, void* element);
  bool exchangeElement(void* pos, void* element);
  bool swapElement(void* i1, void* i2);
  bool swapElement(Size n1, Size n2);
  bool removeAllBefore(Size pos);
  bool removeAllBefore(void* pos);
  bool removeAllAfter(Size pos);
  bool removeAllAfter(void* pos);
  bool removeAllBetween(Size pos1, Size pos2);
  bool removeAllBetween(void* pos1, void* pos2);

#ifndef NDEBUG
  void checkValidity() const;
#  define CHECKVALIDITY() checkValidity()
#else /* NDEBUG */
#  define CHECKVALIDITY()
#endif /* NDEBUG */
};

class InnerDLinkIterator
  : public InnerIterator
{
  InnerDLinkListNode* node;

 private:
  InnerDLinkList* getInnerDLinkList() const { return (InnerDLinkList*)getInnerList(); }
  InnerDLinkListNode* getFirst() const;
  InnerDLinkListNode* getLast() const;
  InnerDLinkListNode* getSentinel() const { return getInnerDLinkList()->sentinel; }

 public:
  InnerDLinkIterator(const InnerDLinkList* l);
  void* current() const;
  void* next();
  void* prev();
  void* rewind();
  void* unwind();
  bool validPositionP() const;
  Size getPosition() const; // valid position: return >= 1
};

template <class Element>
class DLinkList			//! DLinkList
  : public XList<Element>
{
  InnerDLinkList idll;

 public:
  DLinkList() : idll() { }
  ~DLinkList() { }
  const InnerDLinkList* getInnerDLinkList() const { return &idll; }
  void assign(DLinkList& l) { idll.assign(l.idll); }
  void operator = (DLinkList& l) { assign(l); };

  // Collection interface
  virtual bool emptyP() const { return idll.emptyP(); }
  virtual Size number() const { return idll.number(); }
  virtual bool add(Element* element) { idll.inject((void*)element); return true; }
  virtual void clear() { idll.clear(); }
  virtual bool elementP(Element* element) const { return idll.elementP((void*)element); };
  virtual Iterator<Element>* createIterator() const;

  // List interface
  virtual Element* vector(Size vec_pos) const { return (Element*)idll.vector(vec_pos); }
  virtual Size getPosition(Element* element) const { return idll.getPosition((void*)element); }
  virtual Element* prev(Element* element) const { return (Element*)idll.prev((void*)element); }
  virtual Element* next(Element* element) const { return (Element*)idll.next((void*)element); }
  virtual Element* head() const { return (Element*)idll.head(); }
  virtual Element* tail() const { return (Element*)idll.tail(); }
  virtual void push(Element* element) { idll.push((void*)element); }
  virtual Element* pop() { return (Element*)idll.pop(); }
  virtual void inject(Element* element) { idll.inject((void*)element); }
  virtual Element* eject() { return (Element*)idll.eject(); }
  virtual bool insertBefore(Size pos, Element* element) { return idll.insertBefore(pos, (void*)element); }
  virtual bool insertBefore(Element* pos, Element* element) { return idll.insertBefore((void*)pos, (void*)element); }
  virtual bool insertAfter(Size pos, Element* element) { return idll.insertAfter(pos, (void*)element); }
  virtual bool insertAfter(Element* pos, Element* element) { return idll.insertAfter((void*)pos, (void*)element); }
  virtual bool remove(Element* pos) { return idll.remove((void*)pos); }
  virtual bool remove(Size pos) { return idll.remove(pos); }
  virtual bool exchangeElement(Size pos, Element* element) { return idll.exchangeElement(pos, (void*)element); }
  virtual bool exchangeElement(Element* pos, Element* element) { return idll.exchangeElement((void*)pos, (void*)element); }
  virtual bool swapElement(Element* i1, Element* i2) { return idll.swapElement((void*)i1, (void*)i2); }
  virtual bool swapElement(Size n1, Size n2) { return idll.swapElement(n1, n2); }
  virtual bool removeAllBefore(Size pos) { return idll.removeAllBefore(pos); }
  virtual bool removeAllBefore(Element* pos) { return idll.removeAllBefore((void*)pos); }
  virtual bool removeAllAfter(Size pos) { return idll.removeAllAfter(pos); }
  virtual bool removeAllAfter(Element* pos) { return idll.removeAllAfter((void*)pos); }
  virtual bool removeAllBetween(Size pos1, Size pos2) { return idll.removeAllBetween(pos1, pos2); }
  virtual bool removeAllBetween(Element* pos1, Element* pos2) { return idll.removeAllBetween((void*)pos1, (void*)pos2); }
};

template <class Element>
class DLinkIterator
  : public Iterator<Element>
{
  InnerDLinkIterator idli;

 public:
  DLinkIterator(const InnerDLinkList* l) : idli(l) { }
  DLinkIterator(const InnerDLinkList& l) : idli(&l) { }
  DLinkIterator(const DLinkList<Element>* l) : idli(l->getInnerDLinkList()) { }
  DLinkIterator(const DLinkList<Element>& l) : idli(l.getInnerDLinkList()) { }
  virtual ~DLinkIterator() { }
  Element* next() { return (Element*)idli.next(); }
  Element* current() const { return (Element*)idli.current(); }
  Element* prev() { return (Element*)idli.prev(); }
  Element* rewind() { return (Element*)idli.rewind(); }
  Element* unwind() { return (Element*)idli.unwind(); }
  bool validPositionP() const { return idli.validPositionP(); }
  Size getPosition() const { return idli.getPosition(); }
};

template <class Element>
Iterator<Element>*
DLinkList<Element>::createIterator() const
{
  return new DLinkIterator<Element>(this);
}

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_DLinkList_h */
