﻿// $Id$

#if defined(_MSC_VER) && (_MSC_VER >= 1300)
#include <WTypes.h>
#include <basetsd.h>
#endif

#include <stdio.h>
#include <ccc/base/Formatter.h>
#include <ccc/base/StringIFlow.h>
#include <ccc/base/StringOFlow.h>
#include <ccc/base/TextReader.h>
#include <ccc/base/TextWriter.h>
#include <ccc/base/StringIFlow.h>
#include <ccc/base/cstring.h>

CCC_NAMESPACE_START(CCC)

template <typename T>
Formatter<T>::Formatter()
{
}

template <typename T>
Formatter<T>::Formatter(const T* fmt)
{
  fmt_str.assign(fmt);
}

template <typename T>
Formatter<T>::Formatter(const T* fmt, Size len)
{
  fmt_str.assign(fmt, len);
}

template <typename T>
Formatter<T>::Formatter(const TString<T>* fmt)
{
  fmt_str.assign(*fmt);
}

template <typename T>
Formatter<T>::Formatter(const TString<T>& fmt)
{
  fmt_str.assign(fmt);
}

template <typename T>
Formatter<T>::~Formatter()
{
}

template <typename T>
void
Formatter<T>::clear()
{
  fmt_str.clear();
  clearArgs();
}

template <typename T>
void
Formatter<T>::clearArgs()
{
  FmtArg<T>* fa;
  while ((fa = args.pop()))
  {
    delete fa;
  }
}

template <typename T>
void
Formatter<T>::setFormat(const T* fmt)
{
  fmt_str.assign(fmt);
}

template <typename T>
void
Formatter<T>::setFormat(const T* fmt, Size len)
{
  fmt_str.assign(fmt, len);
}

template <typename T>
void
Formatter<T>::setFormat(const TString<T>* fmt)
{
  fmt_str.assign(*fmt);
}

template <typename T>
void
Formatter<T>::setFormat(const TString<T>& fmt)
{
  fmt_str.assign(fmt);
}

template <typename T>
void
Formatter<T>::addFormat(const T* fmt)
{
  fmt_str.add(fmt);
}

template <typename T>
void
Formatter<T>::addFormat(const T* fmt, Size len)
{
  fmt_str.add(fmt, len);
}

template <typename T>
void
Formatter<T>::addFormat(const TString<T>* fmt)
{
  fmt_str.add(*fmt);
}

template <typename T>
void
Formatter<T>::addFormat(const TString<T>& fmt)
{
  fmt_str.add(fmt);
}

template <typename T>
FmtArg<T>*
Formatter<T>::setArg(FmtArg<T> arg)
{
  FmtArg<T>* fa = new FmtArg<T>(arg);
  args.inject(fa);
  return fa;
}

template <typename T>
void
Formatter<T>::changeArgN(Size n, FmtArg<T> arg) CCC_RAISES(FmtException)
{
  FmtArg<T>* fa = args.access(n);
  if (!fa)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::NO_APPLICABLE_ARG);
  }
  *fa = arg;
}

template <typename T>
FmtArg<T>* Formatter<T>::setVP(void* addr)
{
  FmtArg<T>* fa = new FmtArg<T>(addr);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setSC(signed char c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setUC(unsigned char c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setSSI(signed short int c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setUSI(unsigned short int c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setSI(signed int c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setUI(unsigned int c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setSLI(signed long int c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setULI(unsigned long int c)
{
  FmtArg<T>* fa = new FmtArg<T>(c);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setTP(const T* s)
{
  FmtArg<T>* fa = new FmtArg<T>((T*)s);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setSTP(const TString<T>* s)
{
  FmtArg<T>* fa = new FmtArg<T>(s);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setSTR(TString<T>& s)
{
  FmtArg<T>* fa = new FmtArg<T>(s);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setD(double d)
{
  FmtArg<T>* fa = new FmtArg<T>(d);
  args.inject(fa);
  return fa;
}

template <typename T>
FmtArg<T>* Formatter<T>::setF(float f)
{
  FmtArg<T>* fa = new FmtArg<T>(f);
  args.inject(fa);
  return fa;
}

// template <typename T>
// FmtArg<T>* Formatter<T>::setLD(long double ld)
// {
//   FmtArg<T>* fa = new FmtArg<T>(ld);
//   args.inject(fa);
//   return fa;
// }

template <typename T>
FmtArg<T>* Formatter<T>::setIP(int* p)
{
  FmtArg<T>* fa = new FmtArg<T>(p);
  args.inject(fa);
  return fa;
}

template <typename T>
Size
Formatter<T>::writeTo(TString<T>* str, bool clear_p) CCC_RAISES(FmtException)
{
  TString<T>* result = makeString(clear_p);
  str->clear();
  str->assign(*result);
  Size len = result->getLength();
  delete result;
  return len;
}

template <typename T>
Size
Formatter<T>::writeTo(TString<T>& str, bool clear_p) CCC_RAISES(FmtException)
{
  TString<T>* result = makeString(clear_p);
  str.clear();
  str.assign(*result);
  Size len = result->getLength();
  delete result;
  return len;
}

template <typename T>
Size
Formatter<T>::writeTo(T* str, Size size, bool clear_p) CCC_RAISES(FmtException)
{
  TString<T>* result = makeString(clear_p);
  Size len = result->getLength();
  if (len + 1 < size)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::INSUFFICIENT_DISTINATION_MEMORY);
  }
  memCpy(str, result->getStartPtr(), len);
  str[len] = (T)'\0';
  delete result;
  return len;
}

template <typename T>
Size
Formatter<T>::writeTo(OFlow* oflow, bool clear_p) CCC_RAISES(FmtException)
{
  throw FmtException(__FILE__, __LINE__, FmtException::NOT_SUPPORTED);
  return 0;
}

template <typename T>
Size
Formatter<T>::addTo(TString<T>* str, bool clear_p) CCC_RAISES(FmtException)
{
  TString<T>* result = makeString(clear_p);
  str->add(*result);
  Size len = result->getLength();
  delete result;
  return len;
}

template <typename T>
Size
Formatter<T>::addTo(TString<T>& str, bool clear_p) CCC_RAISES(FmtException)
{
  TString<T>* result = makeString(clear_p);
  str.add(*result);
  Size len = result->getLength();
  delete result;
  return len;
}

template <typename T>
Size
Formatter<T>::addTo(T* str, Size size, bool clear_p) CCC_RAISES(FmtException)
{
  TString<T>* result = makeString(clear_p);
  Size str_len = strLen(str);
  Size len = result->getLength();
  if (str_len + len + 1 < size)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::INSUFFICIENT_DISTINATION_MEMORY);
  }
  memCpy(str + str_len, result->getStartPtr(), len);
  str[str_len + len] = (T)'\0';
  delete result;
  return len;
}

/*!
 * \ja
 * printf形式のフォーマットを処理するために使用する構造体です。
 * \ja_end
 */
struct PrintfFmt
{
  /*!
   *  変換文字
   *
   * d, i : int 符合付き10進数<br>
   * o : int 符合無し8進数<br>
   * x, X : int 符合無し16進数, Xは大文字表記<br>
   * u : int 符合無し10進数<br>
   * c : unsigned char に変換された後の単一文字<br>
   * s : 文字列<br>
   * f : double [-]mmm.dddの形の10進数、dの数は精度で指定される<br>
   * e, E: double [-]m.dddddd(e|E)(+|-)xx, dの数は精度で指定される<br>
   * g, G: double 指数が-4よりも小さいか、あるいは精度より大きいか等しい
   *       ときには\%e, \%Eが使用される。そうでないときには\%f<br>
   * p : ポインタとしての印字(処理系依存)<br>
   * n : int これまで書き出された文字の数<br>
   * \% : \%を印字
   */
  char conv;

  /* フラグ */
  int argn;			/*!< {数字}: 引き数の番号 */
  bool left_alinement_p;	/*!< -: 左揃え */
  bool with_sign_p;		/*!< +: 符合付き */
  bool with_space_p;		/*!< ' ': スペース付き */
  bool fill_with_zero_p;	/*!< 0: 0で埋める */
  bool other_desc_p;		/*!< #: 他の出力形式 */
  int min_field_width;		/*!< 数字 : 最小のフィールド幅 */
  bool min_field_width_p;	/*!<  : フィールド幅が指定された */
  int precision;		/*!< 数字 : 精度 */
  bool precision_p;		/*!<  : 精度が指定された */
  bool h_p;			/*!< h: short指定 */
  bool l_p;			/*!< l: long指定 */
  bool ld_p;			/*!< L: long double指定 */

 public:
  void clear()
  {
    conv = '\0';
    argn = 0;
    left_alinement_p = false;
    with_sign_p = false;
    with_space_p = false;
    fill_with_zero_p = false;
    other_desc_p = false;
    min_field_width = 0;
    min_field_width_p = false;
    precision = 1;	/* デフォルトの精度は1 (実数値), 浮動小数の時は6なので注意 */
    precision_p = false;
    h_p = false;
    l_p = false;
    ld_p = false;
  }
};

template <typename T>
TString<T>*
Formatter<T>::makeString(bool clear_p) CCC_RAISES(FmtException)
{
  T num[] =
  {
    (T)'0', (T)'1', (T)'2', (T)'3', (T)'4', (T)'5', (T)'6', (T)'7', (T)'8', (T)'9', 0
  };
  TString<T>* result = new TString<T>;
  StringIFlow<T> si(&fmt_str);
  TextReader<T> reader(&si);
  try
  {
    int argn = 1;
    for (;;)
    {
      T c = reader.getChar();
      if (c == (T)'%')
      {
	/* 構文解析 */
	// %[{argn}][-][+][ ][0][#][width][.precision]type
	PrintfFmt fmt;
	TString<T>* x;
	Size len;
	fmt.clear();
	fmt.argn = argn++;
	if (reader.readChar((T)'{'))
	{
	  x = reader.skipCandidates(num);
	  len = x->getLength();
	  if (len > 0)
	  {
	    T* s = x->getStartPtr();
	    fmt.argn = 0;
	    while (len-- > 0)
	    {
	      int n = (int)(*s++ - (T)'0');
	      fmt.argn *= 10;
	      fmt.argn += n;
	    }
	  }
	  delete x;
	  reader.readChar((T)'}');
	}
	fmt.left_alinement_p = reader.readChar((T)'-');
	fmt.with_sign_p = reader.readChar((T)'+');
	fmt.with_space_p = reader.readChar((T)' ');
	fmt.fill_with_zero_p = reader.readChar((T)'0');
	fmt.other_desc_p = reader.readChar((T)'#');
	if (reader.readChar((T)'*'))
	{
	  FmtArg<T>* arg = args.access(fmt.argn);
	  if (!arg)
	  {
	    throw FmtException(__FILE__, __LINE__, FmtException::NO_APPLICABLE_ARG);
	  }
	  if (arg->getType() != FmtArg<T>::SIGNED_INT)
	  {
	    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
	  }
	  fmt.min_field_width = arg->si_val;
	  fmt.min_field_width_p = true;
	  fmt.argn++;
	  argn++;
	}
	else
	{
	  x = reader.skipCandidates(num);
	  len = x->getLength();
	  if (len > 0)
	  {
	    T* s = x->getStartPtr();
	    fmt.min_field_width = 0;
	    fmt.min_field_width_p = true;
	    while (len-- > 0)
	    {
	      int n = (int)(*s++ - (T)'0');
	      fmt.min_field_width *= 10;
	      fmt.min_field_width += n;
	    }
	  }
	  delete x;
	}
	if (reader.readChar((T)'.'))
	{
	  fmt.precision_p = true;
	  if (reader.readChar((T)'*'))
	  {
	    FmtArg<T>* arg = args.access(fmt.argn);
	    if (!arg)
	    {
	      throw FmtException(__FILE__, __LINE__, FmtException::NO_APPLICABLE_ARG);
	    }
	    if (arg->getType() != FmtArg<T>::SIGNED_INT)
	    {
	      throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
	    }
	    fmt.precision = arg->si_val;
	    fmt.argn++;
	    argn++;
	  }
	  else
	  {
	    fmt.precision = 0;	/* 省略時は精度0 */
	    x = reader.skipCandidates(num);
	    len = x->getLength();
	    if (len > 0)
	    {
	      T* s = x->getStartPtr();
	      while (len-- > 0)
	      {
		int n = (int)(*s++ - (T)'0');
		fmt.precision *= 10;
		fmt.precision += n;
	      }
	    }
	    delete x;
	  }
	}
	c = reader.getChar();
	if (c > (T)0x7e)
	{
	  /* 不正な変換指定子 */
	  throw FmtException(__FILE__, __LINE__, FmtException::INVALID_FORMAT_SPECIFIER);
	}
	bool long_p = false;
	if (c == (T)'l')
	{
	  long_p = true;
	  c = reader.getChar();
	  if (c > (T)0x7e)
	  {
	    /* 不正な変換指定子 */
	    throw FmtException(__FILE__, __LINE__, FmtException::INVALID_FORMAT_SPECIFIER);
	  }
	}
	fmt.conv = (char)c;
	/* 出力の生成 */
	FmtArg<T>* arg = args.access(fmt.argn);
	if (!arg)
	{
	  throw FmtException(__FILE__, __LINE__, FmtException::NO_APPLICABLE_ARG);
	}
	switch (fmt.conv)
	{
	 case 'd':
	 case 'i':
	  if (long_p)
	  {
	    makeLI(arg, &fmt, result);
	  }
	  else
	  {
	    makeI(arg, &fmt, result);
	  }
	  break;
	 case 'o':
	  makeO(arg, &fmt, result);
	  break;
	 case 'x':
	 case 'X':
	  makeX(arg, &fmt, result, fmt.conv == 'X');
	  break;
	 case 'u':
	  if (long_p)
	  {
	    makeLU(arg, &fmt, result);
	  }
	  else
	  {
	    makeU(arg, &fmt, result);
	  }
	  break;
	 case 'c':
	  makeC(arg, &fmt, result);
	  break;
	 case 's':
	  makeS(arg, &fmt, result);
	  break;
	 case 'f':
	  makeF(arg, &fmt, result);
	  break;
	 case 'e':
	 case 'E':
	  makeE(arg, &fmt, result, fmt.conv == 'E');
	  break;
	 case 'g':
	 case 'G':
	  makeG(arg, &fmt, result, fmt.conv == 'G');
	  break;
	 case 'p':
	  makeP(arg, &fmt, result);
	  break;
	 case 'n':
	  makeN(arg, &fmt, result);
	  break;
	 case '%':
	  result->add((T)'%');
	  break;
	 default:	/* 不正な変換指定子 */
	  throw FmtException(__FILE__, __LINE__, FmtException::INVALID_FORMAT_SPECIFIER);
	  break;
	}
      }
      else
      {
	result->add(c);
      }
    }
  }
  catch (IOException& ioe)
  {
    if (ioe.errorNum() != IOException::READ_BEYOND_THE_EOF)
    {
      delete result;
      throw ioe;
    }
  }
  if (clear_p)
  {
    clear();
  }
  return result;
}

static int
getDigit(int n)
{
  // for to support INT_MIN n
  unsigned int nx = (unsigned int)n;
  int digit = 1;
  if (n < 0)
  {
    nx = -n;
  }
  while (nx > 9)
  {
    nx /= 10;
    digit++;
  }
  return digit;
}

static int
getDigitL(signed long int n)
{
  unsigned long int nx = (unsigned long int)n;
  int digit = 1;
  if (n < 0)
  {
    nx = -n;
  }
  while (nx > 9)
  {
    nx /= 10;
    digit++;
  }
  return digit;
}

template <typename T>
static void
putDigit(TString<T>* out, unsigned int n)
{
  if (n > 9)
  {
    putDigit(out, n / 10);
  }
  n %= 10;
  char c = '0' + (char)n;
  out->add((T)c);
}

template <typename T>
static void
putDigitL(TString<T>* out, unsigned long int n)
{
  if (n > 9)
  {
    putDigitL(out, n / 10);
  }
  n %= 10;
  char c = '0' + (char)n;
  out->add((T)c);
}

template <typename T>
void
Formatter<T>::makeI(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::SIGNED_INT)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  int n = arg->si_val;
  int digit = getDigit(n);
  int min_field_width = fmt->min_field_width;
  if (n >= 0)
  {
    if (fmt->with_sign_p)
    {
      tmp.add((T)'+');
      min_field_width--;
    }
    else if (fmt->with_space_p)
    {
      tmp.add((T)' ');
      min_field_width--;
    }
    if (!fmt->left_alinement_p)
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      else if (fmt->fill_with_zero_p)
      {
	while (digit < min_field_width)
	{
	  tmp.add((T)'0');
	  min_field_width--;
	}
      }
      if ((n != 0) || (fmt->precision > 0))
      {
	putDigit(&tmp, (unsigned int)n);
      }
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
      result->add(tmp);
    }
    else
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      if ((n != 0) || (fmt->precision > 0))
      {
	putDigit(&tmp, (unsigned int)n);
      }
      result->add(tmp);
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
    }
  }
  else
  {
    n = -n;
    tmp.add((T)'-');
    min_field_width--;
    if (!fmt->left_alinement_p)
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      else if (fmt->fill_with_zero_p)
      {
	while (digit < min_field_width)
	{
	  tmp.add((T)'0');
	  min_field_width--;
	}
      }
      putDigit(&tmp, (unsigned int)n);
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
      result->add(tmp);
    }
    else
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      putDigit(&tmp, (unsigned int)n);
      result->add(tmp);
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
    }
  }
}

template <typename T>
void
Formatter<T>::makeLI(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::SIGNED_LONG_INT)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  signed long int n = arg->sli_val;
  int digit = getDigitL(n);
  int min_field_width = fmt->min_field_width;
  if (n >= 0)
  {
    if (fmt->with_sign_p)
    {
      tmp.add((T)'+');
      min_field_width--;
    }
    else if (fmt->with_space_p)
    {
      tmp.add((T)' ');
      min_field_width--;
    }
    if (!fmt->left_alinement_p)
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      else if (fmt->fill_with_zero_p)
      {
	while (digit < min_field_width)
	{
	  tmp.add((T)'0');
	  min_field_width--;
	}
      }
      if ((n != 0) || (fmt->precision > 0))
      {
	putDigitL(&tmp, (unsigned long int)n);
      }
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
      result->add(tmp);
    }
    else
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      if ((n != 0) || (fmt->precision > 0))
      {
	putDigitL(&tmp, (unsigned long int)n);
      }
      result->add(tmp);
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
    }
  }
  else
  {
    n = -n;
    tmp.add((T)'-');
    min_field_width--;
    if (!fmt->left_alinement_p)
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      else if (fmt->fill_with_zero_p)
      {
	while (digit < min_field_width)
	{
	  tmp.add((T)'0');
	  min_field_width--;
	}
      }
      putDigit(&tmp, (unsigned int)n);
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
      result->add(tmp);
    }
    else
    {
      if (fmt->precision_p)
      {
	while (fmt->precision > digit)
	{
	  tmp.add((T)'0');
	  fmt->precision--;
	}
      }
      putDigit(&tmp, (unsigned int)n);
      result->add(tmp);
      while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
      {
	result->add((T)' ');
	fmt->min_field_width--;
      }
    }
  }
}

/*!
 * 数値の桁数を得ます。
 * \param n 数値
 * \param step 進数
 * \return 桁数
 */
static int
getDigit(Size n, Size step)
{
  Size digit = 1;
  while (n >= step)
  {
    n /= step;
    digit++;
  }
  return (int)digit;
}

static int
getDigitL(unsigned long int n, Size step)
{
  Size digit = 1;
  while (n >= step)
  {
    n /= step;
    digit++;
  }
  return (int)digit;
}

template <typename T>
static void
putDigit(TString<T>* out, unsigned int n, unsigned int step)
{
  if (n > step - 1)
  {
    putDigit(out, n / step, step);
  }
  n %= step;
  char c = '0' + (char)n;
  out->add((T)c);
}

template <typename T>
static void
putDigitL(TString<T>* out, unsigned long int n, unsigned int step)
{
  if (n > step - 1)
  {
    putDigitL(out, n / step, step);
  }
  n %= step;
  char c = '0' + (char)n;
  out->add((T)c);
}

template <typename T>
void
Formatter<T>::makeO(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::UNSIGNED_INT)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  unsigned int n = arg->ui_val;
  int digit = getDigit(n, 8);
  int min_field_width = fmt->min_field_width;
  if ((fmt->other_desc_p) && (n != 0))
  {
    tmp.add((T)'0');
    min_field_width--;
    fmt->precision--;
  }
  if (!fmt->left_alinement_p)
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    else if (fmt->fill_with_zero_p)
    {
      while (digit < min_field_width)
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit(&tmp, n, 8);
    }
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit(&tmp, n, 8);
    }
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

template <typename T>
static void
putDigit16(TString<T>* out, Size n, bool upper_p)
{
  if (n > 15)
  {
    putDigit16(out, n / 16, upper_p);
  }
  n %= 16;
  char c;
  if (n <= 9)
  {
    c = '0' + (char)n;
  }
  else if (upper_p)
  {
    c = 'A' + (char)(n - 10);
  }
  else
  {
    c = 'a' + (char)(n - 10);
  }
  out->add((T)c);
}

template <typename T>
void
Formatter<T>::makeX(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result, bool upper_p) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::UNSIGNED_INT)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  unsigned int n = arg->ui_val;
  int digit = getDigit(n, 16);
  int min_field_width = fmt->min_field_width;
  if ((fmt->other_desc_p) && (n != 0))
  {
    tmp.add((T)'0');
    if (upper_p)
    {
      tmp.add((T)'X');
    }
    else
    {
      tmp.add((T)'x');
    }
    min_field_width -= 2;
  }
  if (!fmt->left_alinement_p)
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    else if (fmt->fill_with_zero_p)
    {
      while (digit < min_field_width)
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit16(&tmp, n, upper_p);
    }
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit16(&tmp, n, upper_p);
    }
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

template <typename T>
void
Formatter<T>::makeU(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::UNSIGNED_INT)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  unsigned int n = arg->ui_val;
  int digit = getDigit(n, 10);
  int min_field_width = fmt->min_field_width;
  //if ((fmt->other_desc_p) && (n != 0))
  //{
  //  tmp.add((T)'0');
  //  min_field_width--;
  //  fmt->precision--;
  //}
  if (!fmt->left_alinement_p)
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    else if (fmt->fill_with_zero_p)
    {
      while (digit < min_field_width)
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit(&tmp, n, 10);
    }
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit(&tmp, n, 10);
    }
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

template <typename T>
void
Formatter<T>::makeLU(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::UNSIGNED_LONG_INT)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  unsigned long int n = arg->uli_val;
  int digit = getDigitL(n, 10);
  int min_field_width = fmt->min_field_width;
  //if ((fmt->other_desc_p) && (n != 0))
  //{
  //  tmp.add((T)'0');
  //  min_field_width--;
  //  fmt->precision--;
  //}
  if (!fmt->left_alinement_p)
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    else if (fmt->fill_with_zero_p)
    {
      while (digit < min_field_width)
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigitL(&tmp, n, 10);
    }
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigitL(&tmp, n, 10);
    }
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

template <typename T>
void
Formatter<T>::makeC(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  TString<T> tmp;
  T c = 0;
  if (arg->getType() == FmtArg<T>::SIGNED_CHAR)
  {
    c = (T)arg->sc_val;
  }
  else if (arg->getType() == FmtArg<T>::UNSIGNED_CHAR)
  {
    c = (T)arg->uc_val;
  }
  else if (arg->getType() == FmtArg<T>::SIGNED_SHORT_INT)
  {
    c = (T)arg->ssi_val;
  }
  else if (arg->getType() == FmtArg<T>::UNSIGNED_SHORT_INT)
  {
    c = (T)arg->usi_val;
  }
  else if (arg->getType() == FmtArg<T>::SIGNED_INT)
  {
    c = (T)arg->si_val;
  }
  else if (arg->getType() == FmtArg<T>::UNSIGNED_INT)
  {
    c = (T)arg->ui_val;
  }
  else
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  const int digit = 1;
  int min_field_width = fmt->min_field_width;
  if (!fmt->left_alinement_p)
  {
    if (fmt->fill_with_zero_p)
    {
      while (digit < min_field_width)
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    tmp.add((T)c);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    tmp.add(c);
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

template <typename T>
void
Formatter<T>::makeS(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  TString<T> tmp;
  TString<T> src;
  if (arg->getType() == FmtArg<T>::T_PTR)
  {
    src.assign(arg->tp_val);
  }
  else if (arg->getType() == FmtArg<T>::STRING_T_PTR)
  {
    src.assign(*arg->stp_val);
  }
  else
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  Size digit = src.getLength();
  if (fmt->precision_p)
  {
    if (digit > static_cast<Size>(fmt->precision))
    {
      src.remove(fmt->precision, digit - fmt->precision);
      digit = fmt->precision;
    }
  }
  int min_field_width = fmt->min_field_width;
  if (!fmt->left_alinement_p)
  {
    if (fmt->fill_with_zero_p)
    {
      while (digit < static_cast<Size>(min_field_width))
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    tmp.add(src);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    tmp.add(src);
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

static char*
setPrintfFmt(char* p, PrintfFmt* fmt)
{
  *p++ = '%';
  if (fmt->left_alinement_p)
  {
    *p++ = '-';
  }
  if (fmt->with_sign_p)
  {
    *p++ = '+';
  }
  if (fmt->with_space_p)
  {
    *p++ = ' ';
  }
  if (fmt->fill_with_zero_p)
  {
    *p++ = '0';
  }
  if (fmt->other_desc_p)
  {
    *p++ = '#';
  }
  if (fmt->min_field_width_p)
  {
    p += sprintf(p, "%d", fmt->min_field_width);
  }
  if (fmt->precision_p)
  {
    *p++ = '.';
    p += sprintf(p, "%d", fmt->precision);
  }
  return p;
}

template <typename T>
void
Formatter<T>::makeF(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  double c;
  if (arg->getType() == FmtArg<T>::DOUBLE)
  {
    c = arg->d_val;
  }
  else if (arg->getType() == FmtArg<T>::FLOAT)
  {
    c = (double)arg->f_val;
  }
  else
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  if (!fmt->precision_p)
  {
    fmt->precision = 6;
  }
  char fmt_str[256];
  char* p = setPrintfFmt(fmt_str, fmt);
  *p++ = 'f';
  *p = '\0';
  char out[256];
  sprintf(out, fmt_str, c);
  p = out;
  while (*p)
  {
    result->add((T)*p++);
  }
}

template <typename T>
void
Formatter<T>::makeE(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result, bool upper_p) CCC_RAISES(FmtException)
{
  double c;
  if (arg->getType() == FmtArg<T>::DOUBLE)
  {
    c = arg->d_val;
  }
  else if (arg->getType() == FmtArg<T>::FLOAT)
  {
    c = (double)arg->f_val;
  }
  else
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  if (!fmt->precision_p)
  {
    fmt->precision = 6;
  }
  char fmt_str[256];
  char* p = setPrintfFmt(fmt_str, fmt);
  *p++ = upper_p ? 'E' : 'e';
  *p = '\0';
  char out[256];
  sprintf(out, fmt_str, c);
  p = out;
  while (*p)
  {
    result->add((T)*p++);
  }
}

template <typename T>
void
Formatter<T>::makeG(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result, bool upper_p) CCC_RAISES(FmtException)
{
  double c;
  if (arg->getType() == FmtArg<T>::DOUBLE)
  {
    c = arg->d_val;
  }
  else if (arg->getType() == FmtArg<T>::FLOAT)
  {
    c = (double)arg->f_val;
  }
  else
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  if (!fmt->precision_p)
  {
    fmt->precision = 6;
  }
  char fmt_str[256];
  char* p = setPrintfFmt(fmt_str, fmt);
  *p++ = upper_p ? 'G' : 'g';
  *p = '\0';
  char out[256];
  sprintf(out, fmt_str, c);
  p = out;
  while (*p)
  {
    result->add((T)*p++);
  }
}

template <typename T>
void
Formatter<T>::makeP(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  Size n;
//#if defined(_MSC_VER) && (_MSC_VER >= 1300)
//  if (arg->getType() == FmtArg<T>::VOID_PTR)
//  {
//    n = PtrToUlong(arg->vp_val);
//  }
//  else if (arg->getType() == FmtArg<T>::INT_PTR)
//  {
//    n = PtrToUlong(arg->ip_val);
//  }
//  else if (arg->getType() == FmtArg<T>::T_PTR)
//  {
//    n = PtrToUlong(arg->tp_val);
//  }
//  else if (arg->getType() == FmtArg<T>::STRING_T_PTR)
//  {
//    n = PtrToUlong(arg->stp_val);
//  }
//#else
  if (arg->getType() == FmtArg<T>::VOID_PTR)
  {
    n = reinterpret_cast<Size>(arg->vp_val);
  }
  else if (arg->getType() == FmtArg<T>::INT_PTR)
  {
    n = reinterpret_cast<Size>(arg->ip_val);
  }
  else if (arg->getType() == FmtArg<T>::T_PTR)
  {
    n = reinterpret_cast<Size>(arg->tp_val);
  }
  else if (arg->getType() == FmtArg<T>::STRING_T_PTR)
  {
    n = reinterpret_cast<Size>(arg->stp_val);
  }
//#endif
  else
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  TString<T> tmp;
  int digit = getDigit(n, 16);
  int min_field_width = fmt->min_field_width;
  tmp.add((T)'0');
  tmp.add((T)'x');
  min_field_width -= 2;
  if (!fmt->left_alinement_p)
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    else if (fmt->fill_with_zero_p)
    {
      while (digit < min_field_width)
      {
	tmp.add((T)'0');
	min_field_width--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit16(&tmp, n, false);
    }
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
    result->add(tmp);
  }
  else
  {
    if (fmt->precision_p)
    {
      while (fmt->precision > digit)
      {
	tmp.add((T)'0');
	fmt->precision--;
      }
    }
    if ((n != 0) || (fmt->precision > 0))
    {
      putDigit16(&tmp, n, false);
    }
    result->add(tmp);
    while (tmp.getLength() < static_cast<Size>(fmt->min_field_width))
    {
      result->add((T)' ');
      fmt->min_field_width--;
    }
  }
}

template <typename T>
void
Formatter<T>::makeN(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException)
{
  if (arg->getType() != FmtArg<T>::INT_PTR)
  {
    throw FmtException(__FILE__, __LINE__, FmtException::DIFFERENT_TYPE_ARG);
  }
  int* p = arg->ip_val;
  *p = static_cast<int>(result->getLength());
}

template <>
Size
Formatter<Int8>::writeTo(OFlow* oflow, bool clear_p) CCC_RAISES(FmtException)
{
  TString<Int8>* result = makeString(clear_p);
  Size len = result->getLength();
  Size put_size;
  oflow->putInt8Block(len, result->getStartPtr(), put_size);
  delete result;
  return put_size;
}

template <>
Size
Formatter<UInt16>::writeTo(OFlow* oflow, bool clear_p) CCC_RAISES(FmtException)
{
  TString<UInt16>* result = makeString(clear_p);
  Size len = result->getLength();
  Size put_size;
  oflow->putUInt16Block(len, result->getStartPtr(), put_size);
  delete result;
  return put_size;
}

template <>
Size
Formatter<UInt32>::writeTo(OFlow* oflow, bool clear_p) CCC_RAISES(FmtException)
{
  TString<UInt32>* result = makeString(clear_p);
  Size len = result->getLength();
  Size put_size;
  oflow->putUInt32Block(len, result->getStartPtr(), put_size);
  delete result;
  return put_size;
}

template <>
Size
Formatter<Int8>::writeTo(FILE* file, bool clear_p) CCC_RAISES(FmtException)
{
  TString<Int8>* result = makeString(clear_p);
  Size len = result->getLength();
  Size put_size = static_cast<Size>(fwrite(result->getStartPtr(), len * sizeof(Int8), 1, file)) * len;
  delete result;
  return put_size;
}

template <>
Size
Formatter<UInt16>::writeTo(FILE* file, bool clear_p) CCC_RAISES(FmtException)
{
  TString<UInt16>* result = makeString(clear_p);
  Size len = result->getLength();
  Size put_size = static_cast<Size>(fwrite(result->getStartPtr(), len * sizeof(UInt16), 1, file)) * len;
  delete result;
  return put_size;
}

template <>
Size
Formatter<UInt32>::writeTo(FILE* file, bool clear_p) CCC_RAISES(FmtException)
{
  TString<UInt32>* result = makeString(clear_p);
  Size len = result->getLength();
  Size put_size = static_cast<Size>(fwrite(result->getStartPtr(), len * sizeof(UInt32), 1, file)) * len;
  delete result;
  return put_size;
}

template class Formatter<Int8>;
template class Formatter<UInt16>;
template class Formatter<UInt32>;

CCC_NAMESPACE_END(CCC)
