﻿// $Id$

#ifndef INCLUDE_ccc_base_Formatter_h
#define INCLUDE_ccc_base_Formatter_h

#include <stdio.h>
#include <ccc/base/base.h>
#include <ccc/base/FmtException.h>
#include <ccc/base/FmtArg.h>
#include <ccc/base/TString.h>
#include <ccc/base/Flow.h>
#include <ccc/base/OFlow.h>
#include <ccc/base/LinkList.h>

CCC_NAMESPACE_START(CCC)

// forward ref.
struct PrintfFmt;

/*!
 * フォーマット出力を行なうクラスです。<br>
 * 書式文字列としてはC言語の標準ライブラリのprintfでサポートされている
 * 書式に加えて{番号}を使って(1から始まる正数値)何目の引数かを指定する
 * 書式がサポートされています。<br>
 * <br>
 * 例:<br>
 * int<br>
 * main()<br>
 * {<br>
 *   Formatter<Int8> f("%{2}5.3d %{1}d");<br>
 *   f.setSI(0);<br>
 *   f.setSI(10);<br>
 *   BString s;<br>
 *   f.writeTo(s);<br>
 *   printf("[%s]\n", s.getCString());<br>
 *   return 0;<br>
 * }<br>
 *<br>
 * 出力は[  010 0]となります。
 */
template <typename T>
class Formatter
  : public Object
{
 protected:
  /*!
   * フォーマット文字列
   */
  TString<T> fmt_str;
  /*!
   * 引数リスト
   */
  LinkList<FmtArg<T> > args;

 public:
  /*!
   * コンストラクタ
   */
  Formatter();
  /*!
   * コンストラクタ
   * \param fmt 書式文字列
   */
  Formatter(const T* fmt);
  /*!
   * コンストラクタ
   * \param fmt 書式文字列
   * \param len 書式文字列の長さ
   */
  Formatter(const T* fmt, Size len);
  /*!
   * コンストラクタ
   * \param fmt 書式文字列
   */
  Formatter(const TString<T>* fmt);
  /*!
   * コンストラクタ
   * \param fmt 書式文字列
   */
  Formatter(const TString<T>& fmt);
  /*!
   * デストラクタ
   */
  virtual ~Formatter();
  // ------------------------------------------------------------
  /* クリア */
  /*!
   * 書式と引数情報をクリアします。
   */
  void clear();
  /*!
   * TODO: UNITTEST<br>
   * 引数情報のみをクリアします。
   * フォーマット情報は以前に指定したものを継続して利用できます。
   */
  void clearArgs();
  // ------------------------------------------------------------
  /* 書式文字列のセット */
  /*!
   * TODO: UNITTEST<br>
   * 書式文字列をセットします。
   * \param fmt 書式文字列
   */
  void setFormat(const T* fmt);
  /*!
   * TODO: UNITTEST<br>
   * 書式文字列をセットします。
   * \param fmt 書式文字列
   * \param len 書式文字列の長さ
   */
  void setFormat(const T* fmt, Size len);
  /*!
   * TODO: UNITTEST<br>
   * 書式文字列をセットします。
   * \param fmt 書式文字列
   */
  void setFormat(const TString<T>* fmt);
  /*!
   * TODO: UNITTEST<br>
   * 書式文字列をセットします。
   * \param fmt 書式文字列
   */
  void setFormat(const TString<T>& fmt);
  /*!
   * TODO: UNITTEST<br>
   * 既にセットされている書式文字列の後ろに書式文字列を追加します。
   * \param fmt 書式文字列
   */
  void addFormat(const T* fmt);
  /*!
   * TODO: UNITTEST<br>
   * 既にセットされている書式文字列の後ろに書式文字列を追加します。
   * \param fmt 書式文字列
   * \param len 書式文字列の長さ
   */
  void addFormat(const T* fmt, Size len);
  /*!
   * TODO: UNITTEST<br>
   * 既にセットされている書式文字列の後ろに書式文字列を追加します。
   * \param fmt 書式文字列
   */
  void addFormat(const TString<T>* fmt);
  /*!
   * TODO: UNITTEST<br>
   * 既にセットされている書式文字列の後ろに書式文字列を追加します。
   * \param fmt 書式文字列
   */
  void addFormat(const TString<T>& fmt);
  // ------------------------------------------------------------
  /* 引数のセット */
  /*!
   * TODO: UNITTEST<br>
   * 引数をセットします。<br>
   * FmtArg&lt;T&gt;を渡しますが、FmtArg&lt;T&gt;のコンストラクタには主たる型を引数とする
   * ものが用意されているので、これらの型を直接引数として指定できます。これら以外の型の引数を指定する
   * 際には明示的なキャストを行なうようにしてください。
   * \param arg セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatter
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setArg(FmtArg<T> arg);
  /*!
   * void*の引数をセットします。
   * \param addr セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setVP(void* addr);
  /*!
   * signed charの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setSC(signed char c);
  /*!
   * unsigned charの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setUC(unsigned char c);
  /*!
   * signed short intの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setSSI(signed short int c);
  /*!
   * unsigned short intの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setUSI(unsigned short int c);
  /*!
   * signed intの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setSI(signed int c);
  /*!
   * unsigned intの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setUI(unsigned int c);
  /*!
   * signed long intの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setSLI(signed long int c);
  /*!
   * unsigned long intの引数をセットします。
   * \param c セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setULI(unsigned long int c);
  /*!
   * T*の引数をセットします。
   * \param s セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setTP(const T* s);
  /*!
   * TString<T>*の引数をセットします。
   * \param s セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setSTP(const TString<T>* s);
  /*!
   * TString<T>&の引数をセットします。
   * \param s セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setSTR(TString<T>& s);
  /*!
   * doubleの引数をセットします。
   * \param d セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setD(double d);
  /*!
   * floatの引数をセットします。
   * \param f セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setF(float f);
  /*!
   * long doubleの引数をセットします。
   * \param ld セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  //FmtArg<T>* setLD(long double ld);
  /*!
   * int*の引数をセットします。
   * \param p セットする値
   * \return 追加されたFmtArg&lt;T&gt;へのポインタが返ります。このオブジェクトはFormatterが
   * の引数がクリアされるまで有効なオブジェクトです。引数をセットした後に、このオブジェクト
   * のメソッドを呼び出して値を変更することができます。
   */
  FmtArg<T>* setIP(int* p);
  /*!
   * TODO: UNITTEST<br>
   * 既にセットされている引数の内容を、FmtArg&lt;T&gt;で指定したものに置き換えます。
   * \param n 何番目の引数かを示します。
   * \param arg 置き換えるFmtArg&lt;T&gt;を指定します。FmtArg&lt;T&gt;型の引数の扱いについては、
   * setArg(FmtArg<T> arg)を参照してください。
   */
  void changeArgN(Size n, FmtArg<T> arg) CCC_RAISES(FmtException);
  // ------------------------------------------------------------
  /* 出力 */
  /*!
   * TODO: UNITTEST<br>
   * 文字列にフォーマット結果を出力します。
   * \param str 出力先文字列
   * \param clear_p 出力後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size writeTo(TString<T>* str, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * 文字列にフォーマット結果を出力します。
   * \param str 出力先文字列
   * \param clear_p 出力後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size writeTo(TString<T>& str, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * 文字列にフォーマット結果を出力します。
   * \param str 出力先バッファ
   * \param size 出力先バッファのサイズ
   * \param clear_p 出力後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size writeTo(T* str, Size size, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * 出力フローにフォーマット結果を出力します。
   * \param oflow 出力先フロー
   * \param clear_p 出力後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size writeTo(OFlow* oflow, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * ファイルにフォーマット結果を出力します。
   * \param file 出力先フロー
   * \param clear_p 出力後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size writeTo(FILE* file, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * 文字列にフォーマット結果を追記します。
   * \param str 出力先文字列
   * \param clear_p 実行後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size addTo(TString<T>* str, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * 文字列にフォーマット結果を追記します。
   * \param str 出力先文字列
   * \param clear_p 実行後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size addTo(TString<T>& str, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * 文字列にフォーマット結果を追記します。
   * \param str 出力先バッファ
   * \param size 出力先バッファサイズ
   * \param clear_p 実行後に書式と引数情報をクリアするかどうかを指定します。
   * \return 出力した文字数
   */
  Size addTo(T* str, Size size, bool clear_p = true) CCC_RAISES(FmtException);
  /*!
   * TODO: UNITTEST<br>
   * フォーマット文字列を解釈して、フォーマット済文字列を生成します。
   * \param clear_p 実行後に書式と引数情報をクリアするかどうかを指定します。
   * \return フォーマット済文字列
   */
  virtual TString<T>* makeString(bool clear_p = true) CCC_RAISES(FmtException);

 private:
  void makeI(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeLI(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeO(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeX(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result, bool upper_p) CCC_RAISES(FmtException);
  void makeU(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeLU(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeC(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeS(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeF(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeE(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result, bool upper_p) CCC_RAISES(FmtException);
  void makeG(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result, bool upper_p) CCC_RAISES(FmtException);
  void makeP(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
  void makeN(FmtArg<T>* arg, PrintfFmt* fmt, TString<T>* result) CCC_RAISES(FmtException);
};

typedef Formatter<Int8> Fmt8;
typedef Formatter<UInt16> Fmt16;
typedef Formatter<UInt32> Fmt32;

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_Formatter_h */
