﻿// $Id$

#ifndef INCLUDE_ccc_base_IFilter_h
#define INCLUDE_ccc_base_IFilter_h

#include <ccc/base/IFlow.h>

CCC_NAMESPACE_START(CCC)

/*!
 * \ja
 * \brief 入力用フィルタ
 *
 * 各入力用フィルタ用クラスの継承元となるクラスです。
 * \ja_end
 */
class IFilter
  : public IFlow
{
 protected:
  /*!
   * \ja
   * フィルタ元のIFlowを示します。
   * \ja_end
   */
  IFlow* src;
  
  /*!
   * \ja
   * フィルタ元の入力フローがサポートすべき型
   * \ja_end
   */
  IOType src_io_type;
 public:
  /*!
   * \ja
   * コンストラクタ
   * \param my_io_type このフィルタの出力がサポートする型
   * \param src_io_type フィルタの入力フローがサポートすべき型
   * \param src フィルタ元IFlow
   * \ja_end
   */
  IFilter(IOType my_io_type, IOType src_io_type, IFlow* src = 0);
  /*!
   * \ja
   * デストラクタ
   * \ja_end
   */
  virtual ~IFilter();
  /*!
   * \ja
   * フィルタ元のIFlowをセットします。
   * \param src フィルタ元のIFlow
   * \ja_end
   */
  virtual void setSrc(IFlow* src);
  /*!
   * \ja
   * フィルタ元のIFlowを取り出します。
   * \return フィルタ元のIFlow
   * \ja_end
   */
  IFlow* getSrc() { return src; }
  /*!
   * \ja
   * このフィルタがフィルタ元として前提とするIO型を返します。
   * \return サポートする型
   * \ja_end
   */
  virtual IOType getSrcIOType() { return src_io_type; }
  /*!
   * \ja
   * このフィルタがフィルタ元として前提とするIO型と、
   * フィルタ元がサポートする型を結合した型を返します。
   * この値が0(IOTYPE_NULL)であることは、IOが行えないフィルタ元
   * のIFlowを適応していることを示します。
   * フィルタ元がセットされていない時には0(IOTYPE_NULL)を返します。
   * \return サポートする型
   * \ja_end
   */
  virtual IOType getSrcCombinationIOType();
  /*!
   * \ja
   * フィルタ元とフィルタの入力が前提とする型が合致するかを
   * フィルタのチェーンをたどり再帰的に調べます。
   * \retval true 合致する
   * \retval false 合致しない
   * \ja_end
   */
  virtual bool validCombinationP();
  /*!
   * \ja
   * フィルタを停止します。
   * フィルタ元との接続を解除します。
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool close();
  /*!
   * \ja
   * フィルタ元が設定されており、フィルタが機能する状態になっているかを調べます。
   * \retval true 機能状態
   * \retval false 機能状態になっていません
   * \ja_end
   */
  virtual bool openedP();
  //virtual IOType getIOType();
  /*!
   * \ja
   * このフィルタを初期状態にし、フィルタ元に対してもrewind()を呼び出して
   * その結果を返します。
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool rewind();
  /*!
   * \ja
   * フィルタがフローの先頭状態になっているかどうかを返します。
   * このフィルタのステータスと、フィルタ元のtopP()の結果の合成値
   * が返ります。
   * \retval true 先頭
   * \retval false 先頭ではない。
   * \ja_end
   */
  virtual bool topP();
  /*!
   * \ja
   * Int8のデータを取り出します。
   * \return 取り出した値
   * \exception IOException
   * \ja_end
   */
  virtual Int8 getInt8() CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt16のデータを取り出します。
   * \return 取り出した値
   * \exception IOException
   * \ja_end
   */
  virtual UInt16 getUInt16() CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt32 のデータを取り出します。
   * \return 取り出した値
   * \exception IOException
   * \ja_end
   */
  virtual UInt32 getUInt32() CCC_RAISES(IOException);
  /*!
   * \ja
   * Int8のデータブロックを取り出します。
   * get_sizeに読み込めたデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの保存先
   * \param get_size 読み込んだデータ数
   * \ja_end
   */
  virtual void getInt8Block(Size block_size, Int8* ptr, Size& get_size) CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt16のデータブロックを取り出します。
   * get_sizeに読み込めたデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの保存先
   * \param get_size 読み込んだデータ数
   * \ja_end
   */
  virtual void getUInt16Block(Size block_size, UInt16* ptr, Size& get_size) CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt32のデータブロックを取り出します。
   * get_sizeに読み込めたデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの保存先
   * \param get_size 読み込んだデータ数
   * \ja_end
   */
  virtual void getUInt32Block(Size block_size, UInt32* ptr, Size& get_size) CCC_RAISES(IOException);
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_IFilter_h */
