﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.
// expression of single exogenous list

#include <ccc/base/LinkList.h>

CCC_NAMESPACE_START(CCC);

// ------------------------------------------------------------
// class InnerLinkListNode

class InnerLinkListNode
  : public InnerListNode
{
  void* element;
  InnerLinkListNode* next;
  
 public:
  InnerLinkListNode(void* element = 0, InnerLinkListNode* next = 0);
  void* getElement() const { return element; }
  void setElement(void* element) { InnerLinkListNode::element = element; }
  InnerLinkListNode* getNext() const { return next; }
  void setNext(InnerLinkListNode* next) { InnerLinkListNode::next = next; }
};

InnerLinkListNode::InnerLinkListNode(void* element_, InnerLinkListNode* next_)
{
  element = element_;
  if (next_ == 0)
  {
    next_ = this;
  }
  next = next_;
}


// ------------------------------------------------------------
// class InnerLinkList

InnerLinkList::InnerLinkList()
{
  first = sentinel = new InnerLinkListNode();
}

InnerLinkList::InnerLinkList(InnerLinkList& l)
{
  first = sentinel = new InnerLinkListNode();
  for (InnerLinkListNode* node = l.first; 
       node != l.sentinel; 
       node = node->getNext())
  {
    sentinel->setElement(node->getElement());
    InnerLinkListNode* new_sentinel = new InnerLinkListNode();
    sentinel->setNext(new_sentinel);
    setSentinel(new_sentinel);
  }
}

InnerLinkList::~InnerLinkList()
{
  clear();
  delete sentinel;
}

void
InnerLinkList::setSentinel(InnerLinkListNode* n)
{
  sentinel = n;
  n->setNext(n);
  n->setElement(0);
}

void
InnerLinkList::clear()
{
  InnerLinkListNode* node = first;
  while (node != sentinel)
  {
    InnerLinkListNode* next = node->getNext();
    delete node;
    node = next;
  }
  first = sentinel;
}

InnerLinkListNode*
InnerLinkList::getNode(Size pos) const
{
  InnerLinkListNode* node = sentinel;
  if (pos > 0)
  {
    for (node = first;
	 node != sentinel;
	 node = node->getNext())
    {
      if (--pos == 0)
      {
	break;
      }
    }
  }
  return node;
}

InnerLinkListNode*
InnerLinkList::getNode(void* pos) const
{
  InnerLinkListNode* node = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  for (node = first;
       node->getElement() != pos;
       node = node->getNext())
  {
    ;
  }
  sentinel->setElement(0);
  return node;
}

void
InnerLinkList::getNodeAndPrev(InnerLinkListNode** node_rtn,
					     InnerLinkListNode** prev_rtn, Size pos) const
{
  InnerLinkListNode* node = sentinel;
  InnerLinkListNode* prev = sentinel;
  if (pos > 0)
  {
    for (node = first;
	 node != sentinel;
	 node = node->getNext())
    {
      if (--pos == 0)
      {
	break;
      }
      prev = node;
    }
  }
  *node_rtn = node;
  *prev_rtn = prev;
}

void
InnerLinkList::getNodeAndPrev(InnerLinkListNode** node_rtn,
					     InnerLinkListNode** prev_rtn, void* pos) const
{
  InnerLinkListNode* node = sentinel;
  InnerLinkListNode* prev = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  for (node = first;
       node->getElement() != pos;
       node = node->getNext())
  {
    prev = node;
  }
  sentinel->setElement(0);
  *node_rtn = node;
  *prev_rtn = prev;
}

void
InnerLinkList::getBand(InnerLinkListNode** beg_node,
				      InnerLinkListNode** end_node, void* pos1, void* pos2) const
{
  InnerLinkListNode* node = first;
  InnerLinkListNode* pos1_node = 0;
  InnerLinkListNode* pos2_node = 0;
  bool pos1_is_beg_p = false;
  while (node != sentinel)
  {
    if (!pos1_node &&
	node->getElement() == pos1)
    {
      pos1_node = node;
    }
    if (!pos2_node &&
	node->getElement() == pos2)
    {
      pos2_node = node;
      if (pos1_node)
      {
	pos1_is_beg_p = true;
      }
    }
    node = node->getNext();
  }
  if (pos1_node && pos2_node)
  {
    if (pos1_is_beg_p)
    {
      *beg_node = pos1_node;
      *end_node = pos2_node;
    }
    else
    {
      *beg_node = pos2_node;
      *end_node = pos1_node;
    }
  }
  else
  {
    *beg_node = sentinel;
    *end_node = sentinel;
  }
}

void
InnerLinkList::assign(InnerLinkList& l)
{
  clear();
  for (InnerLinkListNode* node = l.first;
       node != l.sentinel;
       node = node->getNext())
  {
    sentinel->setElement(node->getElement());
    InnerLinkListNode* new_sentinel = new InnerLinkListNode();
    sentinel->setNext(new_sentinel);
    setSentinel(new_sentinel);
  }
}

bool
InnerLinkList::emptyP() const
{
  return first == sentinel;
}

Size
InnerLinkList::number() const
{
  Size num = 0;
  for (InnerLinkListNode* node = first;
       node != sentinel;
       node = node->getNext())
  {
    num++;
  }
  return num;
}

void*
InnerLinkList::vector(Size vec_pos) const
{
  for (InnerLinkListNode* node = first;
       node != sentinel;
       node = node->getNext())
  {
    if (vec_pos-- == 0)
    {
      return node->getElement();
    }
  }
  return 0;
}

Size
InnerLinkList::getPosition(void* pos) const
{
  InnerLinkListNode* node = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  Size n_pos = 1;
  for (node = first;
       node->getElement() != pos;
       node = node->getNext())
  {
    n_pos ++;
  }
  sentinel->setElement(0);
  if (node == sentinel)
  {
    n_pos = 0;
  }
  return n_pos;
}

void*
InnerLinkList::prev(void* element) const
{
  InnerLinkListNode* prev = 0;
  sentinel->setElement(element);
  InnerLinkListNode* node;
  for (node = first;
       node->getElement() != element;
       node = node->getNext())
  {
    prev = node;
  }
  if ((prev != 0) && (node != sentinel))
  {
    return prev->getElement();
  }
  return 0;
}

void*
InnerLinkList::next(void* element) const
{
  const InnerLinkListNode* node = getNode(element);
  if ((node != sentinel) &&
      (node->getNext() != sentinel))
  {
    return node->getNext()->getElement();
  }
  return 0;
}

void*
InnerLinkList::head() const
{
  if (first == sentinel)
  {
    return 0;
  }
  return first->getElement();
}

void*
InnerLinkList::tail() const
{
  InnerLinkListNode* node;
  for (node = first;
       node->getNext() != sentinel;
       node = node->getNext())
  {
    ;
  }
  if (node != sentinel)
  {
    return node->getElement();
  }
  return 0;
}

bool
InnerLinkList::elementP(void* pos) const
{
  InnerLinkListNode* node = sentinel;
  sentinel->setElement(pos); // for sentinel loop optimization
  Size n_pos = 1;
  for (node = first;
       node->getElement() != pos;
       node = node->getNext())
  {
    n_pos ++;
  }
  sentinel->setElement(0);
  return (node != sentinel) ? true : false;
}

void
InnerLinkList::push(void* element)
{
  first = new InnerLinkListNode(element, first);
}

void*
InnerLinkList::pop()
{
  if (first == sentinel)
  {
    return 0;
  }
  InnerLinkListNode* node = first;
  first = first->getNext();
  void* element = node->getElement();
  delete node;
  return element;
}

void
InnerLinkList::inject(void* element)
{
  sentinel->setElement(element);
  InnerLinkListNode* new_sentinel = new InnerLinkListNode();
  sentinel->setNext(new_sentinel);
  setSentinel(new_sentinel);
}

void*
InnerLinkList::eject()
{
  if (first == sentinel)
  {
    return 0;
  }
  InnerLinkListNode* node;
  for (node = first;
       node->getNext() != sentinel; 
       node = node->getNext())
  {
    ;
  }
  void* ret = node->getElement();
  delete sentinel;
  setSentinel(node);
  return ret;
}

bool
InnerLinkList::insertBefore(InnerLinkListNode* node,
					   InnerLinkListNode* prev, void* element)
{
  if (node != sentinel)
  {
    if (prev == sentinel) // node == first
    {
      first = new InnerLinkListNode(element, node);
    }
    else
    {
      prev->setNext(new InnerLinkListNode(element, node));
    }
    return true;
  }
  return false;
}

bool
InnerLinkList::insertBefore(Size pos, void* element)
{
  InnerLinkListNode* node;
  InnerLinkListNode* prev;
  getNodeAndPrev(&node, &prev, pos);
  return insertBefore(node, prev, element);
}

bool
InnerLinkList::insertBefore(void* pos, void* element)
{
  InnerLinkListNode* node;
  InnerLinkListNode* prev;
  getNodeAndPrev(&node, &prev, pos);
  return insertBefore(node, prev, element);
}

bool
InnerLinkList::insertAfter(InnerLinkListNode* node, void* element)
{
  if (node != sentinel)
  {
    node->setNext(new InnerLinkListNode(element, node->getNext()));
    return true;
  }
  return false;
}

bool
InnerLinkList::insertAfter(Size pos, void* element)
{
  return insertAfter(getNode(pos), element);
}

bool
InnerLinkList::insertAfter(void* pos, void* element)
{
  return insertAfter(getNode(pos), element);
}

bool
InnerLinkList::remove(InnerLinkListNode* node)
{
  if (node != sentinel)
  {
    InnerLinkListNode* next = node->getNext();
    if (next == sentinel)
    {
      setSentinel(node);
    }
    else
    {
      // for to aboid previous searching, use node as a next node.
      node->setElement(next->getElement());
      node->setNext(next->getNext());
    }
    delete next;
    return true;
  }
  return false;
}

bool
InnerLinkList::remove(void* pos)
{
  return remove(getNode(pos));
}

bool
InnerLinkList::remove(Size pos)
{
  return remove(getNode(pos));
}

bool
InnerLinkList::exchangeElement(InnerLinkListNode* node, void* element)
{
  if (node != sentinel)
  {
    node->setElement(element);
    return true;
  }
  return false;
}

bool
InnerLinkList::exchangeElement(Size pos, void* element)
{
  return exchangeElement(getNode(pos), element);
}

bool
InnerLinkList::exchangeElement(void* pos, void* element)
{
  return exchangeElement(getNode(pos), element);
}

bool
InnerLinkList::swapElement(InnerLinkListNode* node1,
					  InnerLinkListNode* node2)
{
  if ((node1 != sentinel) &&
      (node2 != sentinel))
  {
    void* swap_element = node1->getElement();
    node1->setElement(node2->getElement());
    node2->setElement(swap_element);
    return true;
  }
  else
  {
    return false;
  }
}

bool
InnerLinkList::swapElement(void* i1, void* i2)
{
  return swapElement(getNode(i1), getNode(i2));
}

bool
InnerLinkList::swapElement(Size n1, Size n2)
{
  return swapElement(getNode(n1), getNode(n2));
}

bool
InnerLinkList::removeAllBefore(InnerLinkListNode* before_node)
{
  if (before_node != sentinel)
  {
    InnerLinkListNode* node = first;
    while (node != before_node)
    {
      InnerLinkListNode* next = node->getNext();
      delete node;
      node = next;
    }
    first = before_node;
    return true;
  }
  return false;
}

bool
InnerLinkList::removeAllBefore(Size pos)
{
  return removeAllBefore(getNode(pos));
}

bool
InnerLinkList::removeAllBefore(void* pos)
{
  return removeAllBefore(getNode(pos));
}

bool
InnerLinkList::removeAllAfter(InnerLinkListNode* after_node)
{
  if (after_node != sentinel)
  {
    InnerLinkListNode* node = after_node->getNext();
    while (node != sentinel)
    {
      InnerLinkListNode* next = node->getNext();
      delete node;
      node = next;
    }
    after_node->setNext(sentinel);
    return true;
  }
  return false;
}

bool
InnerLinkList::removeAllAfter(Size pos)
{
  return removeAllAfter(getNode(pos));
}

bool
InnerLinkList::removeAllAfter(void* pos)
{
  return removeAllAfter(getNode(pos));
}

bool
InnerLinkList::removeAllBetween(InnerLinkListNode* node1,
					       InnerLinkListNode* node2)
{
  if ((node1 != sentinel) &&
      (node2 != sentinel))
  {
    InnerLinkListNode* node2_next = node2->getNext();
    InnerLinkListNode* node = node1->getNext();
    while (node != node2_next)
    {
      InnerLinkListNode* next = node->getNext();
      delete node;
      node = next;
    }
    if (node2_next == sentinel)
    {
      setSentinel(node1);
    }
    else
    {
      node1->setElement(node2_next->getElement());
      node1->setNext(node2_next->getNext());
    }
    delete node2_next;
    return true;
  }
  return false;
}

bool
InnerLinkList::removeAllBetween(Size pos1, Size pos2)
{
  Size beg_pos = pos1;
  Size end_pos = pos2;
  if (pos1 > pos2)
  {
    beg_pos = pos2;
    end_pos = pos1;
  }
  return removeAllBetween(getNode(beg_pos), getNode(end_pos));
}

bool
InnerLinkList::removeAllBetween(void* pos1, void* pos2)
{
  InnerLinkListNode* beg_node;
  InnerLinkListNode* end_node;
  getBand(&beg_node, &end_node, pos1, pos2);
  return removeAllBetween(beg_node, end_node);
}


// ------------------------------------------------------------
// class InnerLinkIterator

InnerLinkIterator::InnerLinkIterator(const InnerLinkList* l)
 : InnerIterator(l)
{
  node = getFirst();
}

InnerLinkListNode*
InnerLinkIterator::getLast() const
{
  InnerLinkListNode* x = getFirst();
  while (x->getNext() != getSentinel())
  {
    x = x->getNext();
  }
  return x;
}

void*
InnerLinkIterator::current() const
{
  if (node == getSentinel())
  {
    return 0;
  }
  else
  {
    return node->getElement();
  }
}

void*
InnerLinkIterator::next()
{
  void* ret = current();
  node = node->getNext();
  return ret;
}

void*
InnerLinkIterator::prev()
{
  if (getSentinel() == node)
  {
    return 0;
  }
  InnerLinkListNode* prev = getFirst();
  if (prev == node)
  {
    void* ret = node->getElement();
    node = getSentinel();
    return ret;
  }
  while (prev->getNext() != getSentinel())
  {
    if (prev->getNext() == node)
    {
      void* ret = node->getElement();
      node = prev;
      return ret;
    }
    prev = prev->getNext();
  }
  return 0;
}

void*
InnerLinkIterator::rewind()
{
  node = getFirst();
  return node->getElement();
}

void*
InnerLinkIterator::unwind()
{
  node = getLast();
  return node->getElement();
}

bool
InnerLinkIterator::validPositionP() const
{
  return node != getSentinel();
}

Size
InnerLinkIterator::getPosition() const
{
  Size n = 0;
  InnerLinkListNode* x = getFirst();
  while (x != getSentinel())
  {
    n++;
    if (x == node)
    {
      break;
    }
    x = x->getNext();
  }
  return n;
}

CCC_NAMESPACE_END(CCC);
