﻿// $Id$
// Copyright (C) 1992, 1993, 1994, 1999 by T.Kudou. All rights reserved.
// Copyright (C) 2000 CYPAC Co.,Inc. All rights reserved.

#ifndef INCLUDE_ccc_base_LinkList_h
#define INCLUDE_ccc_base_LinkList_h

#include <ccc/base/XList.h>

CCC_NAMESPACE_START(CCC)

// eXogenous single pointer list
// template <class Element> class LinkList;
// template <class Element> class LinkIterator;
//
//! \class LinkList
//! exogenous single pointer list
//!
//! \verbatim
//!                                         +---+
//!                                         v   |
//! first -->[*]-->[*]-->[*]-->...-->[*]-->[-]--+ <-- sentinel
//!           |     |     |           |
//!           v     v     v           v
//!      Element Element Element    Element
//! \endverbatim

class InnerLinkListNode;

class InnerLinkList		//! InnerLinkList
  : public InnerList
{
  friend class InnerLinkIterator;
  
  InnerLinkListNode* first;
  InnerLinkListNode* sentinel;

 private:
  void setSentinel(InnerLinkListNode* n);
  InnerLinkListNode* getNode(Size pos) const;
  InnerLinkListNode* getNode(void* pos) const;
  void getNodeAndPrev(InnerLinkListNode** node,
		      InnerLinkListNode** prev, Size pos) const;
  void getNodeAndPrev(InnerLinkListNode** node,
		      InnerLinkListNode** prev, void* pos) const;
  void getBand(InnerLinkListNode** beg_node,
	       InnerLinkListNode** end_node,
	       void* pos1, void* pos2) const;

  bool insertBefore(InnerLinkListNode* node,
		    InnerLinkListNode* prev, void* element);
  bool insertAfter(InnerLinkListNode* node, void* element);
  bool remove(InnerLinkListNode* node);
  bool exchangeElement(InnerLinkListNode* node, void* element);
  bool swapElement(InnerLinkListNode* node1,
		   InnerLinkListNode* node2);
  bool removeAllBefore(InnerLinkListNode* node);
  bool removeAllAfter(InnerLinkListNode* node);
  bool removeAllBetween(InnerLinkListNode* node1,
			InnerLinkListNode* node2);
  
 public:
  InnerLinkList();
  InnerLinkList(InnerLinkList& l);
  ~InnerLinkList();

  void assign(InnerLinkList& l);
  bool emptyP() const;
  Size number() const;
  void clear();

  void* vector(Size vec_pos) const;
  Size getPosition(void* element) const;
  void* prev(void* element) const;
  void* next(void* element) const;
  void* head() const;
  void* tail() const;
  bool elementP(void* element) const;
  void push(void* element);
  void* pop();
  void inject(void* element);
  void* eject();

  bool insertBefore(Size pos, void* element);
  bool insertBefore(void* pos, void* element);
  bool insertAfter(Size pos, void* element);
  bool insertAfter(void* pos, void* element);
  bool remove(void* pos);
  bool remove(Size pos);
  bool exchangeElement(Size pos, void* element);
  bool exchangeElement(void* pos, void* element);
  bool swapElement(void* i1, void* i2);
  bool swapElement(Size n1, Size n2);
  bool removeAllBefore(Size pos);
  bool removeAllBefore(void* pos);
  bool removeAllAfter(Size pos);
  bool removeAllAfter(void* pos);
  bool removeAllBetween(Size pos1, Size pos2);
  bool removeAllBetween(void* pos1, void* pos2);
};

class InnerLinkIterator
  : public InnerIterator
{
  InnerLinkListNode* node;

 private:
  const InnerLinkList* getInnerLinkList() const { return (const InnerLinkList*)getInnerList(); }
  InnerLinkListNode* getFirst() const { return getInnerLinkList()->first; }
  InnerLinkListNode* getLast() const;
  InnerLinkListNode* getSentinel() const { return getInnerLinkList()->sentinel; }

 public:
  InnerLinkIterator(const InnerLinkList* l);
  void* current() const;
  void* next();
  void* prev();
  void* rewind();
  void* unwind();
  bool validPositionP() const;
  Size getPosition() const; // valid position: return >= 1
};

template <class Element>
class LinkList			//! LinkList
  : public XList<Element>
{
  InnerLinkList ill;
  
 public:
  LinkList() : ill() { }
  ~LinkList() { }
  const InnerLinkList* getInnerLinkList() const { return &ill; }
  void assign(LinkList& l) { ill.assign(l.ill); }
  void operator = (LinkList& l) { assign(l); };

  // Collection interface
  virtual bool emptyP() const { return ill.emptyP(); }
  virtual Size number() const { return ill.number(); }
  virtual bool add(Element* element) { ill.inject((void*)element); return true; }
  virtual void clear() { ill.clear(); }
  virtual bool elementP(Element* element) const { return ill.elementP((void*)element); };
  virtual Iterator<Element>* createIterator() const;

  // List interface
  virtual Element* vector(Size vec_pos) const { return (Element*)ill.vector(vec_pos); }
  virtual Size getPosition(Element* element) const { return ill.getPosition((void*)element); }
  virtual Element* prev(Element* element) const { return (Element*)ill.prev((void*)element); }
  virtual Element* next(Element* element) const { return (Element*)ill.next((void*)element); }
  virtual Element* head() const { return (Element*)ill.head(); }
  virtual Element* tail() const { return (Element*)ill.tail(); }
  virtual void push(Element* element) { ill.push((void*)element); }
  virtual Element* pop() { return (Element*)ill.pop(); }
  virtual void inject(Element* element) { ill.inject((void*)element); }
  virtual Element* eject() { return (Element*)ill.eject(); }
  virtual bool insertBefore(Size pos, Element* element) { return ill.insertBefore(pos, (void*)element); }
  virtual bool insertBefore(Element* pos, Element* element) { return ill.insertBefore((void*)pos, (void*)element); }
  virtual bool insertAfter(Size pos, Element* element) { return ill.insertAfter(pos, (void*)element); }
  virtual bool insertAfter(Element* pos, Element* element) { return ill.insertAfter((void*)pos, (void*)element); }
  virtual bool remove(Element* pos) { return ill.remove((void*)pos); }
  virtual bool remove(Size pos) { return ill.remove(pos); }
  virtual bool exchangeElement(Size pos, Element* element) { return ill.exchangeElement(pos, (void*)element); }
  virtual bool exchangeElement(Element* pos, Element* element) { return ill.exchangeElement((void*)pos, (void*)element); }
  virtual bool swapElement(Element* i1, Element* i2) { return ill.swapElement((void*)i1, (void*)i2); }
  virtual bool swapElement(Size n1, Size n2) { return ill.swapElement(n1, n2); }
  virtual bool removeAllBefore(Size pos) { return ill.removeAllBefore(pos); }
  virtual bool removeAllBefore(Element* pos) { return ill.removeAllBefore((void*)pos); }
  virtual bool removeAllAfter(Size pos) { return ill.removeAllAfter(pos); }
  virtual bool removeAllAfter(Element* pos) { return ill.removeAllAfter((void*)pos); }
  virtual bool removeAllBetween(Size pos1, Size pos2) { return ill.removeAllBetween(pos1, pos2); }
  virtual bool removeAllBetween(Element* pos1, Element* pos2) { return ill.removeAllBetween((void*)pos1, (void*)pos2); }
};

template <class Element>
class LinkIterator
  : public Iterator<Element>
{
  InnerLinkIterator ili;

 public:
  LinkIterator(const InnerLinkList* l) : ili(l) { }
  LinkIterator(const InnerLinkList& l) : ili(&l) { }
  LinkIterator(const LinkList<Element>* l) : ili(l->getInnerLinkList()) { }
  LinkIterator(const LinkList<Element>& l) : ili(l.getInnerLinkList()) { }
  virtual ~LinkIterator() { }
  Element* current() const { return (Element*)ili.current(); }
  Element* next() { return (Element*)ili.next(); }
  Element* prev() { return (Element*)ili.prev(); }
  Element* rewind() { return (Element*)ili.rewind(); }
  Element* unwind() { return (Element*)ili.unwind(); }
  bool validPositionP() const { return ili.validPositionP(); }
  Size getPosition() const { return ili.getPosition(); }
};

template <class Element>
Iterator<Element>*
LinkList<Element>::createIterator() const
{
  return new LinkIterator<Element>(this);
}

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_LinkList_h */
