﻿// $Id$

#ifndef INCLUDE_ccc_base_List_h
#define INCLUDE_ccc_base_List_h

#include <ccc/base/Collection.h>
#include <assert.h>

CCC_NAMESPACE_START(CCC)

/*!
 * \ja
 * リストのインターフェースを提供します。
 * \ja_end
 */
template <class Element>
class List
  : public Collection<Element>
{
 public:
  /*!
   * \ja
   * コンストラクタ
   * \ja_end
   */
  List() { }
  /*!
   * \ja
   * デストラクタ
   * \ja_end
   */
  virtual ~List() { };

  // Collection interface
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual bool emptyP() const = 0;
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual Size number() const = 0;
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual bool add(Element* element) = 0;
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual void clear() = 0;
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual bool elementP(Element* element) const = 0;
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual Iterator<Element>* createIterator() const = 0;

  // List interface
  /*!
   * \ja
   * 0から始まる数字で要素にアクセスします。
   * 0から始まる数字で要素にアクセスするのは、このメソッドと[]演算子のみです。
   * vec_posに範囲外の数字を渡すと0が返ります。
   * \param vec_pos 0から始まる数字でアクセスする要素を指定します。
   * \return 要素へのポインタ
   * \ja_end
   */
  virtual Element* vector(Size vec_pos) const = 0;
  /*!
   * \ja
   * []演算子は、0から始まる数字で要素にアクセスできます。
   * vec_posに範囲外の数字を渡すと0が返ります。
   * \param vec_pos 0から始まる数字でアクセスする要素を指定します。
   * \return 要素へのポインタ
   * \ja_end
   */
  Element* operator [] (Size vec_pos) const { return vector(vec_pos); };
  /*!
   * \ja
   * 1から始まる数字で要素にアクセスします。
   * posに範囲外の数字を渡すと0が返ります。
   * \param pos 1から始まる位置番号でアクセスする要素を指定します。
   * \return 要素へのポインタ
   * \ja_end
   */
  virtual Element* access(Size pos) const { return vector(pos - 1); }
  /*!
   * \ja
   * 要素へのポインタを渡しリストの何番目に存在するかを調べます。
   * 同じ要素がリスト内に複数登録されている場合には、最初に見つかった位置が返ります。
   * 要素が見つからない場合には0が返ります。
   * \param element 要素へのポインタ
   * \return 位置番号
   * \ja_end
   */
  virtual Size getPosition(Element* element) const = 0;
  /*!
   * \ja
   * 要素へのポインタを渡し、一つ前の要素を取り出します。
   * 要素が見つからない場合には0が返ります。
   * \param element 要素へのポインタ
   * \return 一つ前の要素へのポインタ
   * \ja_end
   */
  virtual Element* prev(Element* element) const = 0;
  /*!
   * \ja
   * 要素へのポインタを渡し、一つ後ろの要素を取り出します。
   * 要素が見つからない場合には0が返ります。
   * \param element 要素へのポインタ
   * \return 一つ後ろの要素へのポインタ
   * \ja_end
   */
  virtual Element* next(Element* element) const = 0;
  /*!
   * \ja
   * リストの最初の要素を取り出します。
   * 要素が無い場合には0が返ります。
   * \return element 要素へのポインタ
   * \ja_end
   */
  virtual Element* head() const = 0;
  /*!
   * \ja
   * リストの最後の要素を取り出します。
   * 要素が無い場合には0が返ります。
   * \return 要素へのポインタ
   * \ja_end
   */
  virtual Element* tail() const = 0;
  // single element operations
  /*!
   * \ja
   * 要素をリストの先頭に入れます。
   * \param element 先頭に入れる要素
   * \ja_end
   */
  virtual void push(Element* element) = 0;
  /*!
   * \ja
   * 要素をリストの先頭から取り出します。
   * \return 先頭に入っていた要素
   * \ja_end
   */
  virtual Element* pop() = 0;
  /*!
   * \ja
   * 要素をリストの最後に追加します。
   * \param element 最後に追加する要素
   * \ja_end
   */
  virtual void inject(Element* element) = 0;
  /*!
   * \ja
   * 要素をリストの最後から取り出します。
   * \return 最後に入っていた要素
   * \ja_end
   */
  virtual Element* eject() = 0;
  /*!
   * \ja
   * 指定した位置の前に要素を挿入します。
   * \param pos 位置番号
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertBefore(Size pos, Element* element) = 0;
  /*!
   * \ja
   * 指定した要素の前に要素を挿入します。
   * \param pos 位置を示す要素
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertBefore(Element* pos, Element* element) = 0;
  /*!
   * \ja
   * 指定した位置の後ろに要素を挿入します。
   * \param pos 位置番号
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertAfter(Size pos, Element* element) = 0;
  /*!
   * \ja
   * 指定した要素の後ろに要素を挿入します。
   * \param pos 位置を示す要素
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertAfter(Element* pos, Element* element) = 0;
  /*!
   * \ja
   * 指定した要素をリストから外します。
   * \param pos リストから外す要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool remove(Element* pos) = 0;
  /*!
   * \ja
   * 指定した位置番号の要素をリストから外します。
   * \param pos 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool remove(Size pos) = 0;
  /*!
   * \ja
   * リスト内の指定した位置番号の要素を、引数で与えた要素と入れ替えます。
   * \param pos 位置番号
   * \param element 入れ替える要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool exchangeElement(Size pos, Element* element) = 0;
  /*!
   * \ja
   * リスト内の指定した要素を、引数で与えた要素と入れ替えます。
   * \param pos 入れ替え元の要素
   * \param element 入れ替え先の要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool exchangeElement(Element* pos, Element* element) = 0;
  /*!
   * \ja
   * リスト内の2つの要素を入れ替えます。
   * \param i1 入れ替える要素
   * \param i2 入れ替える要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool swapElement(Element* i1, Element* i2) = 0;
  /*!
   * \ja
   * リスト内の2つの要素を入れ替えます。
   * \param n1 入れ替える要素の位置番号
   * \param n2 入れ替える要素位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool swapElement(Size n1, Size n2) = 0;
  /*!
   * \ja
   * 指定した要素より前の要素をリストから全て外します。
   * posの要素はリストに残ります。
   * \param pos 指定した要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBefore(Size pos) = 0;
  /*!
   * \ja
   * 指定した要素より前の要素をリストから全て外します。
   * posの位置番号の要素はリストに残ります。
   * \param pos 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBefore(Element* pos) = 0;
  /*!
   * \ja
   * 指定した要素より後ろの要素をリストから全て外します。
   * posの要素はリストに残ります。
   * \param pos 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllAfter(Size pos) = 0;
  /*!
   * \ja
   * 指定した要素より後ろの要素をリストから全て外します。
   * posの位置番号の要素はリストに残ります。
   * \param pos 指定した要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllAfter(Element* pos) = 0;
  /*!
   * \ja
   * 指定した位置番号で挟まれた要素をリストから全て外します。
   * pos1, pos2の位置番号の要素はリストに残ります。
   * \param pos1 位置番号
   * \param pos2 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBetween(Size pos1, Size pos2) = 0;
  /*!
   * \ja
   * 指定した要素で挟まれた要素をリストから全て外します。
   * pos1, pos2の要素はリストに残ります。
   * \param pos1 要素
   * \param pos2 要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBetween(Element* pos1, Element* pos2) = 0;
};

/*!
 * \ja
 * リストクラスの実装用のクラスです。
 * \ja_end
 */
class InnerList
  : public Object
{
 public:
};

/*!
 * \ja
 * リストクラスの実装用のクラスです。
 * \ja_end
 */
class InnerListNode
  : public Object
{
 public:
};

/*!
 * \ja
 * リスト用のイタレータクラスの実装用のクラスです。
 * \ja_end
 */
class InnerIterator
  : public Object
{
  const InnerList* list;

 public:
  InnerIterator(const InnerList* l) : list(l) { }
  const InnerList* getInnerList() const { return list; }
};

/*!
 * リスト表現をプロキシを経由するインターフェースを実現します。
 */
template <class Element>
class ListProxy
  : public List<Element>
{
  /*!
   * プロキシ
   */
  List<Element>* proxy;
 public:
  /*!
   * コンストラクタ
   * \param proxy プロキシ
   */
  ListProxy()
  {
    proxy = 0;
  }
  /*!
   * コンストラクタ
   * \param proxy プロキシ
   */
  ListProxy(List<Element>* proxy)
  {
    ListProxy::proxy = proxy;
  }
  /*!
   * コピーコンストラクタ
   * proxyの複製を仮想的に生成する手段が必要ですが、まだ
   * 実現されていないため、このメソッドは呼び出し不可です。
   */
  ListProxy(ListProxy<Element>& a)
  {
    ListProxy::proxy = 0;
    assert(false);
  }
  /*!
   * デストラクタ
   * proxyはデストラクタで破棄されます。
   */
  virtual ~ListProxy()
  {
    delete proxy;
  }
  /*!
   * クラスの代入を行います。
   * \param l 代入対象
   */
  /*
   * TODO: proxyの複製を仮想的に生成する手段が必要なので、次のコードで
   * は問題が発生するため実装しません。
   */
  //void assign(ListProxy& l) { proxy = l.proxy; }
  /*!
   * クラスの代入を行います。
   * \param l 代入対象
   */
  /*
   * TODO: proxyの複製を仮想的に生成する手段が必要なので、次のコードで
   * は問題が発生するため実装しません。
   */
  //void operator = (ListProxy& l) { proxy = l.proxy; };
  // Collection interface
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual bool emptyP() const { return proxy->emptyP(); }
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual Size number() const { return proxy->number(); }
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual bool add(Element* element) { return proxy->add(element); }
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual void clear() { proxy->clear(); }
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual bool elementP(Element* element) const { return proxy->elementP(element); }
  /*!
   * \ja
   * Collectionクラスのドキュメントを参照してください。
   * \ja_end
   */
  virtual Iterator<Element>* createIterator() const { return proxy->createIterator(); }

  // List interface
  /*!
   * \ja
   * 0から始まる数字で要素にアクセスします。
   * 0から始まる数字で要素にアクセスするのは、このメソッドと[]演算子のみです。
   * vec_posに範囲外の数字を渡すと0が返ります。
   * \param vec_pos 0から始まる数字でアクセスする要素を指定します。
   * \return 要素へのポインタ
   * \ja_end
   */
  virtual Element* vector(Size vec_pos) const { return proxy->vector(vec_pos); }
  /*!
   * \ja
   * []演算子は、0から始まる数字で要素にアクセスできます。
   * vec_posに範囲外の数字を渡すと0が返ります。
   * \param vec_pos 0から始まる数字でアクセスする要素を指定します。
   * \return 要素へのポインタ
   * \ja_end
   */
  Element* operator [] (Size vec_pos) const { return proxy->vector(vec_pos); };
  /*!
   * \ja
   * 1から始まる数字で要素にアクセスします。
   * posに範囲外の数字を渡すと0が返ります。
   * \param pos 1から始まる位置番号でアクセスする要素を指定します。
   * \return 要素へのポインタ
   * \ja_end
   */
  virtual Element* access(Size pos) const { return proxy->vector(pos - 1); }
  /*!
   * \ja
   * 要素へのポインタを渡しリストの何番目に存在するかを調べます。
   * 同じ要素がリスト内に複数登録されている場合には、最初に見つかった位置が返ります。
   * 要素が見つからない場合には0が返ります。
   * \param element 要素へのポインタ
   * \return 位置番号
   * \ja_end
   */
  virtual Size getPosition(Element* element) const { return proxy->getPosition(element); }
  /*!
   * \ja
   * 要素へのポインタを渡し、一つ前の要素を取り出します。
   * 要素が見つからない場合には0が返ります。
   * \param element 要素へのポインタ
   * \return 一つ前の要素へのポインタ
   * \ja_end
   */
  virtual Element* prev(Element* element) const { return proxy->prev(element); }
  /*!
   * \ja
   * 要素へのポインタを渡し、一つ後ろの要素を取り出します。
   * 要素が見つからない場合には0が返ります。
   * \param element 要素へのポインタ
   * \return 一つ後ろの要素へのポインタ
   * \ja_end
   */
  virtual Element* next(Element* element) const { return proxy->next(element); }
  /*!
   * \ja
   * リストの最初の要素を取り出します。
   * 要素が無い場合には0が返ります。
   * \return element 要素へのポインタ
   * \ja_end
   */
  virtual Element* head() const { return proxy->head(); }
  /*!
   * \ja
   * リストの最後の要素を取り出します。
   * 要素が無い場合には0が返ります。
   * \return 要素へのポインタ
   * \ja_end
   */
  virtual Element* tail() const { return proxy->tail(); }
  // single element operations
  /*!
   * \ja
   * 要素をリストの先頭に入れます。
   * \param element 先頭に入れる要素
   * \ja_end
   */
  virtual void push(Element* element) { proxy->push(element); }
  /*!
   * \ja
   * 要素をリストの先頭から取り出します。
   * \return 先頭に入っていた要素
   * \ja_end
   */
  virtual Element* pop() { return proxy->pop(); }
  /*!
   * \ja
   * 要素をリストの最後に追加します。
   * \param element 最後に追加する要素
   * \ja_end
   */
  virtual void inject(Element* element) { proxy->inject(element); }
  /*!
   * \ja
   * 要素をリストの最後から取り出します。
   * \return 最後に入っていた要素
   * \ja_end
   */
  virtual Element* eject() { return proxy->eject(); }
  /*!
   * \ja
   * 指定した位置の前に要素を挿入します。
   * \param pos 位置番号
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertBefore(Size pos, Element* element) { return proxy->insertBefore(pos, element); }
  /*!
   * \ja
   * 指定した要素の前に要素を挿入します。
   * \param pos 位置を示す要素
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertBefore(Element* pos, Element* element) { return proxy->insertBefore(pos, element); }
  /*!
   * \ja
   * 指定した位置の後ろに要素を挿入します。
   * \param pos 位置番号
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertAfter(Size pos, Element* element) { return proxy->insertAfter(pos, element); }
  /*!
   * \ja
   * 指定した要素の後ろに要素を挿入します。
   * \param pos 位置を示す要素
   * \param element 挿入する要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool insertAfter(Element* pos, Element* element) { return proxy->insertAfter(pos, element); }
  /*!
   * \ja
   * 指定した要素をリストから外します。
   * \param pos リストから外す要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool remove(Element* pos) { return proxy->remove(pos); }
  /*!
   * \ja
   * 指定した位置番号の要素をリストから外します。
   * \param pos 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool remove(Size pos) { return proxy->remove(pos); }
  /*!
   * \ja
   * リスト内の指定した位置番号の要素を、引数で与えた要素と入れ替えます。
   * \param pos 位置番号
   * \param element 入れ替える要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool exchangeElement(Size pos, Element* element) { return proxy->exchangeElement(pos, element); }
  /*!
   * \ja
   * リスト内の指定した要素を、引数で与えた要素と入れ替えます。
   * \param pos 入れ替え元の要素
   * \param element 入れ替え先の要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool exchangeElement(Element* pos, Element* element) { return proxy->exchangeElement(pos, element); }
  /*!
   * \ja
   * リスト内の2つの要素を入れ替えます。
   * \param i1 入れ替える要素
   * \param i2 入れ替える要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool swapElement(Element* i1, Element* i2) { return proxy->swapElement(i1, i2); }
  /*!
   * \ja
   * リスト内の2つの要素を入れ替えます。
   * \param n1 入れ替える要素の位置番号
   * \param n2 入れ替える要素位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool swapElement(Size n1, Size n2) { return proxy->swapElement(n1, n2); }
  /*!
   * \ja
   * 指定した要素より前の要素をリストから全て外します。
   * posの要素はリストに残ります。
   * \param pos 指定した要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBefore(Size pos) { return proxy->removeAllBefore(pos); }
  /*!
   * \ja
   * 指定した要素より前の要素をリストから全て外します。
   * posの位置番号の要素はリストに残ります。
   * \param pos 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBefore(Element* pos) { return proxy->removeAllBefore(pos); }
  /*!
   * \ja
   * 指定した要素より後ろの要素をリストから全て外します。
   * posの要素はリストに残ります。
   * \param pos 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllAfter(Size pos) { return proxy->removeAllAfter(pos); }
  /*!
   * \ja
   * 指定した要素より後ろの要素をリストから全て外します。
   * posの位置番号の要素はリストに残ります。
   * \param pos 指定した要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllAfter(Element* pos) { return proxy->removeAllAfter(pos); }
  /*!
   * \ja
   * 指定した位置番号で挟まれた要素をリストから全て外します。
   * pos1, pos2の位置番号の要素はリストに残ります。
   * \param pos1 位置番号
   * \param pos2 位置番号
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBetween(Size pos1, Size pos2) { return proxy->removeAllBetween(pos1, pos2); }
  /*!
   * \ja
   * 指定した要素で挟まれた要素をリストから全て外します。
   * pos1, pos2の要素はリストに残ります。
   * \param pos1 要素
   * \param pos2 要素
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool removeAllBetween(Element* pos1, Element* pos2) { return proxy->removeAllBetween(pos1, pos2); }
  /*!
   * プロキシを取得します。
   * \return プロキシ
   */
  List<Element>* getProxy() { return proxy; }
  /*!
   * プロキシを設定します。
   */
  void setProxy(List<Element>* proxy) { ListProxy<Element>::proxy = proxy; }
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_List_h */
