﻿// $Id$

#ifndef INCLUDE_ccc_base_OFilter_h
#define INCLUDE_ccc_base_OFilter_h

#include <ccc/base/OFlow.h>

CCC_NAMESPACE_START(CCC)

/*!
 * 出力用フィルタ
 * 各入力用フィルタ用クラスの継承元となるクラスです。
 */
class OFilter
  : public OFlow
{
  /*!
   * \ja
   * データの送り先のOFlowを示します。
   * \ja_end
   */
  OFlow* dest;

  /*!
   * \ja
   * フィルタ先のI/O可能型
   * \ja_end
   */
  IOType dest_io_type;

 public:
  /*!
   * \ja
   * コンストラクタ
   * \param my_io_type このフィルタの入力がサポートする型
   * \param dest_io_type フィルタの出力がサポートする型
   * \param dest フィルタ先OFlow
   * \ja_end
   */
  OFilter(IOType my_io_type, IOType dest_io_type, OFlow* dest = 0);
  /*!
   * \ja
   * デストラクタ
   * \ja_end
   */
  virtual ~OFilter();
  /*!
   * \ja
   * データの送り先のOFlowをセットします。
   * \param dest データの送り先OFlow
   * \ja_end
   */
  virtual void setDest(OFlow* dest);
  /*!
   * \ja
   * データの送り先のOFlowを取り出します。
   * \return データの送り先OFlow
   * \ja_end
   */
  OFlow* getDest() { return dest; }
  /*!
   * このフィルタがフィルタ先として前提とするIO型を返します。
   * \return サポートする型
   */
  IOType getDestIOType() { return dest_io_type; }
  /*!
   * \ja
   * このフィルタがフィルタ先として前提とするIO型と、
   * フィルタ先がサポートする型を結合した型を返します。
   * この値が0(IOTYPE_NULL)であることは、IOが行えないフィルタ先
   * のOFlowを適応していることを示します。
   * フィルタ先がセットされていない時には0(IOTYPE_NULL)を返します。
   * \return サポートする型
   * \ja_end
   */
  IOType getDestCombinationIOType();
  /*!
   * \ja
   * フィルタを停止します。
   * フィルタ先との接続を解除します。
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool close();
  /*!
   * \ja
   * フィルタ先が設定されており、フィルタが機能する状態になっているかを調べます。
   * \retval true 機能状態
   * \retval false 機能状態になっていません
   * \ja_end
   */
  virtual bool openedP();
  /*!
   * \ja
   * このフィルタを初期状態にし、フィルタ先に対してもrewind()を呼び出して
   * その結果を返します。
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool rewind();
  /*!
   * \ja
   * フィルタがフローの先頭状態になっているかどうかを返します。
   * このフィルタのステータスと、フィルタ先のtopP()の結果の合成値
   * が返ります。
   * \retval true 先頭
   * \retval false 先頭ではない。
   * \ja_end
   */
  virtual bool topP();
  /*!
   * \ja
   * バッファリング中のデータを出力に完全に書出します。
   * \ja_end
   */
  virtual void flush() CCC_RAISES(IOException);
  /*!
   * \ja
   * フィルタ先とフィルタの出力が前提とする型が合致するかを
   * フィルタのチェーンをたどり再帰的に調べます。
   * \retval true 合致する
   * \retval false 合致しない
   * \ja_end
   */
  virtual bool validCombinationP();
  /*!
   * \ja
   * Int8のデータを出力します。
   * \param c 出力するデータ
   * \exception IOException
   * \ja_end
   */
  virtual void putInt8(Int8 c) CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt16 のデータを出力します。
   * \param c 出力するデータ
   * \exception IOException
   * \ja_end
   */
  virtual void putUInt16(UInt16 c) CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt32 のデータを出力します。
   * \param c 出力するデータ
   * \exception IOException
   * \ja_end
   */
  virtual void putUInt32(UInt32 c) CCC_RAISES(IOException);
  /*!
   * \ja
   * Int8のデータブロックを出力します。
   * put_sizeに出力したデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの送り元
   * \param put_size 出力したデータ数
   * \ja_end
   */
  virtual void putInt8Block(Size block_size, const Int8* ptr, Size& put_size) CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt16のデータブロックを出力します。
   * put_sizeに出力したデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの送り元
   * \param put_size 出力したデータ数
   * \ja_end
   */
  virtual void putUInt16Block(Size block_size, const UInt16* ptr, Size& put_size) CCC_RAISES(IOException);
  /*!
   * \ja
   * UInt32のデータブロックを出力します。
   * put_sizeに出力したデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの送り元
   * \param put_size 出力したデータ数
   * \ja_end
   */
  virtual void putUInt32Block(Size block_size, const UInt32* ptr, Size& put_size) CCC_RAISES(IOException);
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_OFilter_h */
