﻿// $Id$

#include <assert.h>
#include <string.h>
#include <ccc/base/TString.h>
#include <ccc/base/cstring.h>
#include <ccc/base/StringOFlow.h>
#include <ccc/base/TextWriter.h>

CCC_NAMESPACE_START(CCC);

const Size initial_alloc_size = 10;

template <typename T>
TString<T>::TString()
 : TSubString<T>(),
   mem(initial_alloc_size * sizeof(T))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = 0;
}

template <typename T>
TString<T>::TString(const T* c_str)
 : TSubString<T>(),
   mem(c_str ? (strLen(c_str) * sizeof(T)) : (initial_alloc_size * sizeof(T)))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  if (c_str)
  {
    TSubString<T>::to = strLen(c_str);
    memCpy(getMem(), c_str, TSubString<T>::to);
  }
  else
  {
    TSubString<T>::to = 0;
  }
}

template <typename T>
TString<T>::TString(const T* c_str, Size size)
 : TSubString<T>(),
   mem(c_str ? (size * sizeof(T)) : (initial_alloc_size * sizeof(T)))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  if (c_str)
  {
    TSubString<T>::to = size;
    memCpy(getMem(), c_str, TSubString<T>::to);
  }
  else
  {
    TSubString<T>::to = 0;
  }
}

template <typename T>
TString<T>::TString(const TSubString<T>& ss)
 : TSubString<T>(),
   mem(ss.getLength() * sizeof(T))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  if ((TSubString<T>::to = ss.getLength()) != 0)
  {
    memCpy(getMem(), ss.getStartPtr(), ss.getLength());
  }
}

template <typename T>
TString<T>::TString(const TSubString<T>* ss)
 : TSubString<T>(),
   mem(ss ? (ss->getLength() * sizeof(T)) : (initial_alloc_size * sizeof(T)))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  if (ss)
  {
    if ((TSubString<T>::to = ss->getLength()))
    {
      memCpy(getMem(), ss->getStartPtr(), ss->getLength());
    }
  }
  else
  {
    TSubString<T>::to = 0;
  }
}

template <typename T>
TString<T>::TString(const TString<T>& s)
 : TSubString<T>(),
   mem(s.getLength() * sizeof(T))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  if ((TSubString<T>::to = s.getLength()) != 0)
  {
    memCpy(getMem(), s.getStartPtr(), s.getLength());
  }
}

template <typename T>
TString<T>::TString(const TString<T>* s)
 : TSubString<T>(),
   mem(s ? (s->getLength() * sizeof(T)) : (initial_alloc_size * sizeof(T)))
{
  TStringPtr<T>::str = this;
  TStringPtr<T>::pos = 0;
  if (s)
  {
    if ((TSubString<T>::to = s->getLength()))
    {
      memCpy(getMem(), s->getStartPtr(), s->getLength());
    }
  }
  else
  {
    TSubString<T>::to = 0;
  }
}

template <typename T>
TString<T>::~TString()
{
  ; // mem destructor is called implicitly
}

template <typename T>
Size
TString<T>::getPos() const
{
  return 0;
}

template <typename T>
Size
TString<T>::getTo() const
{
  return TSubString<T>::to - TStringPtr<T>::pos;
}

template <typename T>
T*
TString<T>::getStart() const
{
  return getMem() + TStringPtr<T>::pos;
}

template <typename T>
T*
TString<T>::getEnd() const
{
  return getMem() + TSubString<T>::to;
}

template <typename T>
T*
TString<T>::getStartPtr() const
{
  return getStart();
}

template <typename T>
T*
TString<T>::getEndPtr() const
{
  return getEnd();
}

template <typename T>
T&
TString<T>::operator [] (Size n) const
{
  assert(n < TSubString<T>::to - TStringPtr<T>::pos);
  return getStart()[n];
}

template <typename T>
void
TString<T>::clear()
{
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = 0;
  shrink(initial_alloc_size);
}

template <typename T>
void
TString<T>::clear(T c, Size size)
{
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = 0;
  if (getSize() < size)
  {
    expand(size);
  }
  memset(getStart(), c, size * sizeof(T));
}

template <typename T>
void
TString<T>::assign(const T* c_str)
{
  Size l = strLen(c_str);
  expand(l);
  memCpy(getMem(), c_str, l);
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = l;
}

template <typename T>
TString<T>
TString<T>::operator = (const T* c_str)
{
  assign(c_str);
  return *this;
}

template <typename T>
void
TString<T>::assign(const T* str, Size l)
{
  expand(l);
  memCpy(getMem(), str, l);
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = l;
}

template <typename T>
void
TString<T>::assign(const TString<T>& ss)
{
  expand(ss.getLength());
  memCpy(getMem(), ss.getStartPtr(), ss.getLength());
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = ss.getLength();
}

template <typename T>
void
TString<T>::assign(const TString<T>* ss)
{
  expand(ss->getLength());
  memCpy(getMem(), ss->getStartPtr(), ss->getLength());
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = ss->getLength();
}

template <typename T>
TString<T>
TString<T>::operator = (const TString<T>& ss)
{
  assign(ss);
  return *this;
}

template <typename T>
void
TString<T>::assign(const TSubString<T>& ss)
{
  expand(ss.getLength());
  memCpy(getMem(), ss.getStartPtr(), ss.getLength());
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = ss.getLength();
}

template <typename T>
void
TString<T>::assign(const TSubString<T>* ss)
{
  expand(ss->getLength());
  memCpy(getMem(), ss->getStartPtr(), ss->getLength());
  TStringPtr<T>::pos = 0;
  TSubString<T>::to = ss->getLength();
}

template <typename T>
TString<T>
TString<T>::operator = (const TSubString<T>& ss)
{
  assign(ss);
  return *this;
}

template <typename T>
TString<T>
TString<T>::operator + (T c) const
{
  TString<T> ret(*this);
  ret += c;
  return ret;
}

template <typename T>
TString<T>
TString<T>::operator + (const TSubString<T>& ss) const
{
  TString<T> ret(*this);
  ret += ss;
  return ret;
}

template <typename T>
void
TString<T>::add(T c)
{
  if (TSubString<T>::to < getSize())
  {
    TSubString<T>::to++;
  }
  else
  {
    if (TStringPtr<T>::pos > 0)
    {
      T* s = getStart();
      memMove(s - 1, s, TSubString<T>::getLength());
      TStringPtr<T>::pos --;
    }
    else
    {
      expand(TSubString<T>::to + 1);
      TSubString<T>::to++;
    }
  }
  *(getEnd() - 1) = c;
}

template <typename T>
void
TString<T>::add(const T* c_str)
{
  Size l = strLen(c_str);
  if (TSubString<T>::to + l < getSize())
  {
    memCpy(getEnd(), c_str, l);
    TSubString<T>::to += l;
  }
  else
  {
    if (TStringPtr<T>::pos > l)
    {
      T* s = getStart();
      memMove(s - l, s, TSubString<T>::getLength());
      TStringPtr<T>::pos -= l;
      memCpy(getEnd() - l, c_str, l);
    }
    else
    {
      expand(TSubString<T>::to + l);
      memCpy(getEnd(), c_str, l);
      TSubString<T>::to += l;
    }
  }
}

template <typename T>
void
TString<T>::add(const T* c_str, Size l)
{
  if (TSubString<T>::to + l < getSize())
  {
    memCpy(getEnd(), c_str, l);
    TSubString<T>::to += l;
  }
  else
  {
    if (TStringPtr<T>::pos > l)
    {
      T* s = getStart();
      memMove(s - l, s, TSubString<T>::getLength());
      TStringPtr<T>::pos -= l;
      memCpy(getEnd() - l, c_str, l);
    }
    else
    {
      expand(TSubString<T>::to + l);
      memCpy(getEnd(), c_str, l);
      TSubString<T>::to += l;
    }
  }
}

template <typename T>
void
TString<T>::add(const TSubString<T>& ss)
{
  if (TSubString<T>::to + ss.getLength() < getSize())
  {
    memCpy(getEnd(), ss.getStartPtr(), ss.getLength());
    TSubString<T>::to += ss.getLength();
  }
  else
  {
    if (TStringPtr<T>::pos > ss.getLength())
    {
      T* s = getStart();
      memMove(s - ss.getLength(), s, TSubString<T>::getLength());
      TStringPtr<T>::pos -= ss.getLength();
      memCpy(getEnd() - ss.getLength(), ss.getStartPtr(), ss.getLength());
    }
    else
    {
      expand(TSubString<T>::to + ss.getLength());
      memCpy(getEnd(), ss.getStartPtr(), ss.getLength());
      TSubString<T>::to += ss.getLength();
    }
  }
}

template <typename T>
void
TString<T>::add(const TSubString<T>* ss)
{
  if (TSubString<T>::to + ss->getLength() < getSize())
  {
    memCpy(getEnd(), ss->getStartPtr(), ss->getLength());
    TSubString<T>::to += ss->getLength();
  }
  else
  {
    if (TStringPtr<T>::pos > ss->getLength())
    {
      T* s = getStart();
      memMove(s - ss->getLength(), s, TSubString<T>::getLength());
      TStringPtr<T>::pos -= ss->getLength();
      memCpy(getEnd() - ss->getLength(), ss->getStartPtr(), ss->getLength());
    }
    else
    {
      expand(TSubString<T>::to + ss->getLength());
      memCpy(getEnd(), ss->getStartPtr(), ss->getLength());
      TSubString<T>::to += ss->getLength();
    }
  }
}

template <typename T>
bool
TString<T>::insert(Size ip, T c)
{
  if (ip > TSubString<T>::getLength())
  {
    return false;
  }
  for (;;)
  {
    bool l_shift = false;
    bool r_shift = false;
    if (TStringPtr<T>::pos >= 1)
    {
      // L has space
      if (TSubString<T>::to + 1 <= getSize())
      {
	// R free
	// L & R free, caliculate minimum move cost
	if (ip < TSubString<T>::getLength() / 2)
	{
	  // L shift
	  l_shift = true;
	}
	else
	{
	  // R shift
	  r_shift = true;
	}
      }
      else
      {
	// L shift
	l_shift = true;
      }
    }
    else if (TSubString<T>::to + 1 <= getSize())
    {
      // R shift
      r_shift = true;
    }
    else
    {
      // expand buffer
      expand(TSubString<T>::to + 1);
      // insert will be done on the next loop.
    }
    if (l_shift)
    {
      // L shift
      memMove(getStart() - 1,
	      getStart(),
	      ip);
      TStringPtr<T>::pos -= 1;
      memCpy(getStart() + ip, &c, 1);
      return true;
    }
    if (r_shift)
    {
      // R shift
      memMove(getStart() + ip + 1,
	      getStart() + ip,
	      TSubString<T>::getLength() - ip);
      memCpy(getStart() + ip, &c, 1);
      TSubString<T>::to += 1;
      return true;
    }
  }	    
  return true;
}

template <typename T>
bool
TString<T>::insert(Size ip, const T* c_str)
{
  if (ip > TSubString<T>::getLength())
  {
    return false;
  }
  for (;;)
  {
    Size l = strLen(c_str);
    bool l_shift = false;
    bool r_shift = false;
    if (TStringPtr<T>::pos >= l)
    {
      // L has space
      if (TSubString<T>::to + l <= getSize())
      {
	// R free
	// L & R free, caliculate minimum move cost
	if (ip < TSubString<T>::getLength() / 2)
	{
	  // L shift
	  l_shift = true;
	}
	else
	{
	  // R shift
	  r_shift = true;
	}
      }
      else
      {
	// L shift
	l_shift = true;
      }
    }
    else if (TSubString<T>::to + l <= getSize())
    {
      // R shift
      r_shift = true;
    }
    else
    {
      // expand buffer
      expand(TSubString<T>::to + l);
      // insert will be done on the next loop.
    }
    if (l_shift)
    {
      // L shift
      memMove(getStart() - l,
	      getStart(),
	      ip);
      TStringPtr<T>::pos -= l;
      memCpy(getStart() + ip, c_str, l);
      return true;
    }
    if (r_shift)
    {
      // R shift
      memMove(getStart() + ip + l,
	      getStart() + ip,
	      TSubString<T>::getLength() - ip);
      memCpy(getStart() + ip, c_str, l);
      TSubString<T>::to += l;
      return true;
    }
  }	    
  return true;
}

template <typename T>
bool
TString<T>::insert(Size ip, const TSubString<T>& ss)
{
  if (ip > TSubString<T>::getLength())
  {
    return false;
  }
  for (;;)
  {
    bool l_shift = false;
    bool r_shift = false;
    if (TStringPtr<T>::pos >= ss.getLength())
    {
      // L has space
      if (TSubString<T>::to + ss.getLength() <= getSize())
      {
	// R free
	// L & R free, caliculate minimum move cost
	if (ip < TSubString<T>::getLength() / 2)
	{
	  // L shift
	  l_shift = true;
	}
	else
	{
	  // R shift
	  r_shift = true;
	}
      }
      else
      {
	// L shift
	l_shift = true;
      }
    }
    else if (TSubString<T>::to + ss.getLength() <= getSize())
    {
      // R shift
      r_shift = true;
    }
    else
    {
      // expand buffer
      expand(TSubString<T>::to + ss.getLength());
      // insert will be done on the next loop.
    }
    if (l_shift)
    {
      // L shift
      memMove(getStart() - ss.getLength(),
	      getStart(),
	      ip);
      TStringPtr<T>::pos -= ss.getLength();
      memCpy(getStart() + ip, ss.getStartPtr(), ss.getLength());
      return true;
    }
    if (r_shift)
    {
      // R shift
      memMove(getStart() + ip + ss.getLength(),
	      getStart() + ip,
	      TSubString<T>::getLength() - ip);
      memCpy(getStart() + ip, ss.getStartPtr(), ss.getLength());
      TSubString<T>::to += ss.getLength();
      return true;
    }
  }
  return true;
}

template <typename T>
bool
TString<T>::insert(Size ip, const TSubString<T>* ss)
{
  if (ip > TSubString<T>::getLength())
  {
    return false;
  }
  for (;;)
  {
    bool l_shift = false;
    bool r_shift = false;
    if (TStringPtr<T>::pos >= ss->getLength())
    {
      // L has space
      if (TSubString<T>::to + ss->getLength() <= getSize())
      {
	// R free
	// L & R free, caliculate minimum move cost
	if (ip < TSubString<T>::getLength() / 2)
	{
	  // L shift
	  l_shift = true;
	}
	else
	{
	  // R shift
	  r_shift = true;
	}
      }
      else
      {
	// L shift
	l_shift = true;
      }
    }
    else if (TSubString<T>::to + ss->getLength() <= getSize())
    {
      // R shift
      r_shift = true;
    }
    else
    {
      // expand buffer
      expand(TSubString<T>::to + ss->getLength());
      // insert will be done on the next loop.
    }
    if (l_shift)
    {
      // L shift
      memMove(getStart() - ss->getLength(),
	      getStart(),
	      ip);
      TStringPtr<T>::pos -= ss->getLength();
      memCpy(getStart() + ip, ss->getStartPtr(), ss->getLength());
      return true;
    }
    if (r_shift)
    {
      // R shift
      memMove(getStart() + ip + ss->getLength(),
	      getStart() + ip,
	      TSubString<T>::getLength() - ip);
      memCpy(getStart() + ip, ss->getStartPtr(), ss->getLength());
      TSubString<T>::to += ss->getLength();
      return true;
    }
  }
  return true;
}

template <typename T>
bool
TString<T>::remove(Size from, Size len)
{
  Size s_len = TSubString<T>::to - TStringPtr<T>::pos;
  if (s_len < from)
  {
    return false;
  }
  if (len == 0)
  {
    return true;
  }
  if (s_len < from + len)
  {
    return false;
  }
  if (s_len == from)
  {
    return false;
  }
  if ((from == 0) && (len <= s_len))
  {
    TStringPtr<T>::pos += len;
    return true;
  }
  if ((s_len > from) && (s_len <= from + len)) 
  {
    TSubString<T>::to = TStringPtr<T>::pos + from;
    return true;
  }
  Size shift_size = (TSubString<T>::to - TStringPtr<T>::pos) - (from + len);
  if (from <= shift_size)
  {
    // move front memory block
    memMove(getStart() + len, getStart(), from);
    TStringPtr<T>::pos += len;
  }
  else
  {
    // move rear memory block
    memMove(getStart() + from, getStart() + from + len, shift_size);
    TSubString<T>::to -= len;
  }
  return true;
}

template <typename T>
bool
TString<T>::replace(Size from, Size len, const TSubString<T>& ss)
{
  remove(from, len);
  return insert(from, ss);
}

template <typename T>
bool
TString<T>::replace(Size from, Size len, const TSubString<T>* ss)
{
  remove(from, len);
  return insert(from, ss);
}

template <typename T>
bool
TString<T>::replace(Size from, Size len, const T* c_str)
{
  remove(from, len);
  return insert(from, c_str);
}

template <typename T>
T*
TString<T>::getCString() const
{
  TString<T>* me = (TString<T>*)this;	// for to break const
  if (me->to >= me->getSize())
  {
    if (me->pos > 0)
    {
      T* s = me->getStart();
      memMove(s - 1, s, me->getLength());
      me->pos--;
      me->to--;
    }
    else
    {
      me->expand(me->to + 1);
    }
  }
  *me->getEnd() = '\0';
  return me->getStart();
}

template <typename T>
bool
TString<T>::chop(T c)
{
  T* x = getStart();
  Size len = TSubString<T>::getLength();
  if ((len > 0) && x[len - 1] == c)
  {
    TSubString<T>::to--;
    return true;
  }
  return false;
}

template <typename T>
bool
TString<T>::chopHead(T c)
{
  T* x = getStart();
  Size len = TSubString<T>::getLength();
  if ((len > 0) && x[0] == c)
  {
    TSubString<T>::pos++;
    return true;
  }
  return false;
}

template <typename T>
Size
TString<T>::replaceAll(const TSubString<T>& from, const TSubString<T>& to)
{
  if (from.strCmp(to) == 0)
  {
    return 0;
  }
  Size n = 0;
  for (;;)
  {
    /* TODO:
     * 最初から検索しなおしているので、replace後から検索しなおすようにすれば
     * 高速になります。
     */
    Size pos = this->kmpMatch(from);
    if (pos == 0)
    {
      break;
    }
    replace(pos - 1, from.getLength(), to);
    n++;
  }
  return n;
}

template <typename T>
Size
TString<T>::replaceAll(const T* from, const T* to)
{
  TString<T> from_str(from);
  TString<T> to_str(to);
  return replaceAll(from_str, to_str);
}

template <typename T>
Size
TString<T>::printf(const TString<T>* fmt, ...) CCC_RAISES(IOException)
{
  va_list arg;
  va_start(arg, fmt);
  Size ret = vprintf(fmt, arg);
  va_end(arg);
  return ret;
}

template <typename T>
Size
TString<T>::printf(const T* fmt, ...) CCC_RAISES(IOException)
{
  va_list arg;
  va_start(arg, fmt);
  Size ret = vprintf(fmt, arg);
  va_end(arg);
  return ret;
}

template <typename T>
Size
TString<T>::printf(Size len, const T* fmt, ...) CCC_RAISES(IOException)
{
  va_list arg;
  va_start(arg, fmt);
  Size ret = vprintf(len, fmt, arg);
  va_end(arg);
  return ret;
}

template <typename T>
Size
TString<T>::vprintf(const TString<T>* fmt, va_list arg) CCC_RAISES(IOException)
{
  clear();
  StringOFlow<T> flow(this);
  TextWriter<T> writer(&flow);
  Size ret = writer.vprintf(fmt, arg);
  return ret;
}

template <typename T>
Size
TString<T>::vprintf(const T* fmt, va_list arg) CCC_RAISES(IOException)
{
  clear();
  StringOFlow<T> flow(this);
  TextWriter<T> writer(&flow);
  Size ret = writer.vprintf(fmt, arg);
  return ret;
}

template <typename T>
Size
TString<T>::vprintf(Size len, const T* fmt, va_list arg) CCC_RAISES(IOException)
{
  clear();
  StringOFlow<T> flow(this);
  TextWriter<T> writer(&flow);
  Size ret = writer.vprintf(len, fmt, arg);
  return ret;
}

template class TString<Int8>;
//template class TString<UInt8>;
template class TString<UInt16>;
template class TString<UInt32>;

CCC_NAMESPACE_END(CCC);
