﻿// $Id$

#ifndef INCLUDE_ccc_base_TString_h
#define INCLUDE_ccc_base_TString_h

#include <stdarg.h>
#include <ccc/base/TSubString.h>
#include <ccc/base/Allocator.h>
#include <ccc/base/IOException.h>

CCC_NAMESPACE_START(CCC)

/*!
 * \ja
 * 文字列クラス
 * \ja_end
 * \en
 * This class handles string.
 * \en_end
 */
template <typename T>
class TString
  : public TSubString<T>
{
  /*!
   * \ja
   * 文字列バッファ
   * \ja_end
   * \en
   * string buffer
   * \en_end
   */
  Allocator mem;
   
 public:
  /*!
   * \ja
   * コンストラクタ
   * \ja_end
   */
  TString();
  /*!
   * \ja
   * コンストラクタ<br>
   * 引数で渡した文字列の内容で生成する文字列の内容が初期化されます。
   * \param c_str ヌルで終端された文字列へのポインタ
   * \ja_end
   */
  TString(const T* c_str);
  /*!
   * \ja
   * コンストラクタ<br>
   * 引数で渡した文字列の内容で生成する文字列の内容が初期化されます。
   * \param c_str 文字列へのポインタ
   * \param size 文字数
   * \ja_end
   */
  TString(const T* c_str, Size size);
  /*!
   * \ja
   * コンストラクタ<br>
   * 引数で渡した部分文字列の内容で生成する文字列の内容が初期化されます。
   * \param ss 部分文字列
   * \ja_end
   */
  TString(const TSubString<T>& ss);
  /*!
   * \ja
   * コピーコンストラクタ
   * \param ss コピー元
   * \ja_end
   */
  TString(const TString<T>& ss);
  /*!
   * \ja
   * コンストラクタ<br>
   * 引数で渡した部分文字列の内容で生成する文字列の内容が初期化されます。
   * \param ss 部分文字列へのポインタ
   * \ja_end
   */
  TString(const TSubString<T>* ss);
  /*!
   * \ja
   * コンストラクタ<br>
   * 引数で渡した文字列の内容で生成する文字列の内容が初期化されます。
   * \param ss 文字列へのポインタ
   * \ja_end
   */
  TString(const TString<T>* ss);
  /*!
   * デストラクタ
   */
  virtual ~TString();
  /*!
   * \ja
   * 文字列イメージの開始位置のポインタを得ます。
   * \return 開始位置のポインタ
   * \ja_end
   */
  virtual T* getStartPtr() const;
  /*!
   * \ja
   * 文字列イメージの終了位置のポインタを得ます。
   * \return 終了位置のポインタ
   * \ja_end
   */
  virtual T* getEndPtr() const;

 private:
  /*!
   * \ja
   * 文字列イメージを保管しているメモリを取り出します。
   * \return 文字列イメージを保管しているメモリアドレス
   * \ja_end
   */
  T* getMem() const { return (T*)mem.getMem(); }
  /*!
   * \ja
   * 文字列イメージの保管メモリサイズを縮小します。
   * \param size 新たな保管メモリサイズ
   * \ja_end
   */
  void shrink(Size size) { mem.shrink(size * sizeof(T)); }
  /*!
   * \ja
   * 文字列イメージの保管メモリサイズを拡張します。
   * \param size 新たな保管メモリサイズ
   * \ja_end
   */
  void expand(Size size) { mem.expand(size * sizeof(T)); }
  /*!
   * \ja
   * 文字列イメージの保管メモリサイズを得ます。
   * \return 保管メモリサイズ
   * \ja_end
   */
  Size getSize() { return (mem.getSize() / sizeof(T)); }
  /*!
   * \ja
   * 文字列の開始位置のポインタを得ます。このクラス内でのみ使用します。
   * このメソッドはインラインであるため、上記のメソッドよりも若干速いです。
   * \ja_end
   * \en
   * Get this string start pointer. Class internal use only.
   * As this function is inline, so little bit faster than 
   * upper function.
   * \en_end
   */
  T* getStart() const;
  /*!
   * \ja
   * 文字列の開始位置のポインタを得ます。このクラス内でのみ使用します。
   * このメソッドはインラインであるため、上記のメソッドよりも若干速いです。
   * \ja_end
   * \en
   * Get this string end pointer. Class internal use only.
   * As this function is inline so little bit faster than
   * upper function.
   * \en_end
   */
  T* getEnd() const;

 public:
  /*!
   * \ja
   * 文字列の開始位置を得ます。
   * StringPtr, SubStringの同名のメソッドをオーバライドしています。
   * \return 常に0が返ります。
   * \ja_end
   */
  virtual Size getPos() const;
  /*!
   * \ja
   * 文字列の終了位置を得ます。
   * この位置が指し示す文字は、文字列の最後の文字の次の文字であり、文字列に含まれません。
   * \return 終了位置オフセット (0から始まる数値)
   * \ja_end
   */
  virtual Size getTo() const;
  /*!
   * \ja
   * []演算子
   * 文字列の各要素に対して、配列のようにして参照するインターフェースを提供します。
   * \param n 位置
   * \return 指定位置の文字への参照
   * \ja_end
   */
  virtual T& operator [] (Size n) const;
  /*!
   * \ja
   * 文字列をクリアします。クリア後は空文字列になります。
   * \ja_end
   */    
  void clear();
  /*!
   * \ja
   * 文字列をクリアします。
   * \param c この文字で文字列を埋めます。
   * \param size クリア後の文字列の長さです。
   * \ja_end
   */
  void clear(T c, Size size);
  /*!
   * \ja
   * ヌルで終端された文字列を代入します。
   * \param c_str ヌルで終端された文字列
   * \ja_end
   */
  void assign(const T* c_str);
  /*!
   * \ja
   * 文字列を代入します。
   * \param str 文字列
   * \param size 文字列の長さ
   * \ja_end
   */
  void assign(const T* str, Size size);
  /*!
   * \ja
   * =演算子、ヌルで終端された文字列を代入します。
   * また、代入後のオブジェクトを返します。
   * 代入後のオブジェクトを使用しないのならば、効率の良いため
   * assign(const T* c_str)の利用をお勧めします。
   * \param c_str ヌルで終端された文字列
   * \return 代入後のオブジェクト
   * \ja_end
   */
  TString operator = (const T* c_str);
  /*!
   * \ja
   * 文字列を代入します。
   * \param str 代入する文字列
   * \ja_end
   */
  void assign(const TString& str);
  /*!
   * \ja
   * 文字列を代入します。
   * \param str 代入する文字列
   * \ja_end
   */
  void assign(const TString* str);
  /*!
   * \ja
   * =演算子、文字列を代入します。
   * 代入後のオブジェクトを使用しないのならば、効率の良いため
   * assign(const String& str)の利用をお勧めします。
   * \param str 代入する文字列
   * \return 代入後のオブジェクト
   * \ja_end
   */
  TString operator = (const TString& str);
  /*!
   * \ja
   * 部分文字列を代入します。
   * \param ss 代入する部分文字列
   * \ja_end 
   */
  void assign(const TSubString<T>& ss);
  /*!
   * \ja
   * 部分文字列を代入します。
   * \param ss 代入する部分文字列
   * \ja_end 
   */
  void assign(const TSubString<T>* ss);
  /*!
   * \ja
   * =演算子、部分文字列を代入します。
   * 代入後のオブジェクトを使用しないのならば、効率の良いため
   * assign(const SubString& ss)の利用をお勧めします。
   * \return 代入後のオブジェクト
   * \ja_end
   */
  TString operator = (const TSubString<T>& ss);
  /*!
   * \ja
   * +演算子、文字列に文字を追加した文字列を作成します。
   * \param c 追加する文字
   * \return 代入後のStringオブジェクトを返します。
   * \ja_end
   */
  TString operator + (T c) const;
  /*!
   * \ja
   * +演算子、文字列に部分文字列を追加した文字列を作成します。
   * \param ss 追加する部分文字列
   * \return 代入後のStringオブジェクトを返します。
   * \ja_end
   */
  TString operator + (const TSubString<T>& ss) const;
  /*!
   * \ja
   * 文字列に文字を追加します。
   * \param c 追加する文字
   * \ja_end
   */
  void add(T c);
  /*!
   * \ja
   * +=演算子、文字列に文字を追加します。
   * \param c 追加する文字
   * \ja_end
   */
  void operator += (T c) { add(c); }
  /*!
   * \ja
   * 文字列にヌルで終端された文字列を追加します。
   * \param c_str ヌルで終端された文字列
   * \ja_end
   */
  void add(const T* c_str);
  /*!
   * \ja
   * +=演算子、文字列にヌルで終端された文字列を追加します。
   * \param c_str ヌルで終端された文字列
   * \ja_end
   */
  void operator += (const T* c_str) { add(c_str); }
  /*!
   * \ja
   * 文字列に文字列を追加します。
   * \param c_str 文字列へのポインタ
   * \param l 追加する文字列の長さ
   * \ja_end
   */
  void add(const T* c_str, Size l);
  /*!
   * \ja
   * 文字列に部分文字列を追加します。
   * \param ss 追加する部分文字列
   * \ja_end
   */
  void add(const TSubString<T>& ss);
  /*!
   * \ja
   * 文字列に部分文字列を追加します。
   * \param ss 追加する部分文字列
   * \ja_end
   */
  void add(const TSubString<T>* ss);
  /*!
   * \ja
   * +=演算子、文字列に部分文字列を追加します。
   * \param ss 追加する部分文字列
   * \ja_end
   */
  void operator += (const TSubString<T>& ss) { add(ss); }
  /*!
   * \ja
   * 文字列挿入
   * \param ip 挿入位置
   * \param ss 挿入する部分文字列
   * \retval true 挿入成功
   * \retval false 失敗、挿入位置が文字列の範囲外
   * \ja_end
   * \en
   * String insertion
   * \param ip insertion position
   * \param ss insert string
   * \retval true success
   * \retval false insertion failed, insertion point is out of string boundary
   * \en_end
   */
  bool insert(Size ip, const TSubString<T>& ss);
  /*!
   * \ja
   * 文字列挿入
   * \param ip 挿入位置
   * \param ss 挿入する部分文字列
   * \retval true 挿入成功
   * \retval false 失敗、挿入位置が文字列の範囲外
   * \ja_end
   * \en
   * String insertion
   * \param ip insertion position
   * \param ss insert string
   * \retval true success
   * \retval false insertion failed, insertion point is out of string boundary
   * \en_end
   */
  bool insert(Size ip, const TSubString<T>* ss);
  /*!
   * \ja
   * 文字列挿入
   * \param ip 挿入位置
   * \param c_str 挿入するヌル終端文字列
   * \retval true 挿入成功
   * \retval false 失敗、挿入位置が文字列の範囲外
   * \ja_end
   * \en
   * String insertion
   * \param ip insertion position
   * \param c_str insert null terminated string
   * \retval true success
   * \retval false insertion failed, insertion point is out of string boundary
   * \en_end
   */
  bool insert(Size ip, const T* c_str);
  /*!
   * \ja
   * 文字挿入
   * \param ip 挿入位置
   * \param c 挿入する文字
   * \retval true 挿入成功
   * \retval false 失敗、挿入位置が文字列の範囲外
   * \ja_end
   * \en
   * Character insertion
   * \param ip insertion position
   * \param c insertion character
   * \retval true success
   * \retval false insertion failed, insertion point is out of string boundary
   * \en_end
   */
  bool insert(Size ip, T c);
  /*!
   * \ja
   * 文字列を部分的に削除します。
   * \param pos 削除開始位置(文字列頭を0とするoffset)
   * \param len 削除する長さ
   * \retval true 削除成功
   * \retval false 失敗、削除範囲が文字列の範囲を越えています。
   * \ja_end
   */
  bool remove(Size pos, Size len);
  /*!
   * \ja
   * 文字列を部分的に置き換えます。
   * \param pos 置き換え開始位置
   * \param len 置き換え長さ
   * \param ss 置き換え対象の部分文字列
   * \retval true 置き換え成功
   * \retval false 置き換え失敗、削除範囲が文字列の範囲を越えています。
   * \ja_end
   */
  bool replace(Size pos, Size len, const TSubString<T>& ss);
  /*!
   * \ja
   * 文字列を部分的に置き換えます。
   * \param pos 置き換え開始位置
   * \param len 置き換え長さ
   * \param ss 置き換え対象の部分文字列
   * \retval true 置き換え成功
   * \retval false 置き換え失敗、削除範囲が文字列の範囲を越えています。
   * \ja_end
   */
  bool replace(Size pos, Size len, const TSubString<T>* ss);
  /*!
   * \ja
   * 文字列を部分的に置き換えます。
   * \param pos 置き換え開始位置
   * \param len 置き換え長さ
   * \param c_str 置き換え対象のヌル終端文字列
   * \retval true 置き換え成功
   * \retval false 置き換え失敗、削除範囲が文字列の範囲を越えています。
   * \ja_end
   */
  bool replace(Size pos, Size len, const T* c_str);
  /*!
   * \ja
   * ヌルで終端された文字列を得ます。
   * \return ヌルで終端された文字列の先頭アドレス
   * \ja_end
   * \en
   * get null terminated string
   * \return the first address of the null terminated string.
   * \en_end
   */
  T* getCString() const;
  /*!
   * TODO: UNITTEST
   * \ja
   * 指定された文字が文字列の終りにあったら削除します。
   * \param c 文字列の終りの存在をチェックする文字
   * \retval true 削除しました。
   * \retval false 削除しませんでした。
   * \ja_end
   * \en
   * Remove a charactor from the end of string, if the charactor is a specified charactor.
   * \param c specified charactor
   * \retval true removed
   * \retval false not removed
   * \en_end
   */
  bool chop(T c);
  /*!
   * TODO: UNITTEST
   * \ja
   * 指定された文字が文字列の最初にあったら削除します。
   * \param c チェックする文字
   * \retval true 削除しました。
   * \retval false 削除しませんでした。
   * \ja_end
   * \en
   * Remove a charactor from the start of string, if the charactor is a specified charactor.
   * \param c specified charactor
   * \retval true removed
   * \retval false not removed
   * \en_end
   */
  bool chopHead(T c);
  /*!
   * TODO: UNITTEST
   * \ja
   * 文字列中の置き換え元文字列で指定した文字列を全て、置き換え先文字列に置き換えます。
   * 注意:
   * 現在の実装では、置き換え先文字列に置き換え元と同じ文字列が含まれていると、処理が
   * 無限に行なわれてしまいます。
   * \param from 置き換え元文字列
   * \param to 置き換え先文字列
   * \return 置き換えた文字列の数
   * \ja_end
   */
  Size replaceAll(const TSubString<T>& from, const TSubString<T>& to);
  /*!
   * TODO: UNITTEST
   * \ja
   * 文字列中の置き換え元文字列で指定した文字列を全て、置き換え先文字列に置き換えます。
   * 注意:
   * 現在の実装では、置き換え先文字列に置き換え元と同じ文字列が含まれていると、処理が
   * 無限に行なわれてしまいます。
   * \param from 置き換え元文字列
   * \param to 置き換え先文字列
   * \return 置き換えた文字列の数
   * \ja_end
   */
  Size replaceAll(const T* from, const T* to);
  // ------------------------------------------------------------
  /* 書式付きデータ出力
   */
  /*!
   * 書式付きデータ出力を行ないます。書式の仕様はANSI Cのprintfの書式仕様に従います。
   * \param fmt 書式
   * \return 出力した文字数
   */
  virtual Size printf(const TString<T>* fmt, ...) CCC_RAISES(IOException);
  /*!
   * 書式付きデータ出力を行ないます。書式の仕様はANSI Cのprintfの書式仕様に従います。
   * \param fmt 書式
   * \return 出力した文字数
   */
  virtual Size printf(const T* fmt, ...) CCC_RAISES(IOException);
  /*!
   * 書式付きデータ出力を行ないます。書式の仕様はANSI Cのprintfの書式仕様に従います。
   * \param len 書式文字列の長さ
   * \param fmt 書式
   * \return 出力した文字数
   */
  virtual Size printf(Size len, const T* fmt, ...) CCC_RAISES(IOException);
  /*!
   * 書式付きデータ出力を行ないます。書式の仕様はANSI Cのprintfの書式仕様に従います。
   * \param fmt 書式
   * \param arg 引き数
   * \return 出力した文字数
   */
  virtual Size vprintf(const TString<T>* fmt, va_list arg) CCC_RAISES(IOException);
  /*!
   * 書式付きデータ出力を行ないます。書式の仕様はANSI Cのprintfの書式仕様に従います。
   * \param fmt 書式
   * \param arg 引き数
   * \return 出力した文字数
   */
  virtual Size vprintf(const T* fmt, va_list arg) CCC_RAISES(IOException);
  /*!
   * 書式付きデータ出力を行ないます。書式の仕様はANSI Cのprintfの書式仕様に従います。
   * \param len 書式文字列の長さ
   * \param fmt 書式
   * \param arg 引き数
   * \return 出力した文字数
   */
  virtual Size vprintf(Size len, const T* fmt, va_list arg) CCC_RAISES(IOException);
};

typedef TString<char> BString;
typedef TString<UInt16> DString;
typedef TString<UInt32> WString;

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_TString_h */
