﻿// $Id$

#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <ccc/base/TSubString.h>
#include <ccc/base/TString.h>
#include <ccc/base/cstring.h>

CCC_NAMESPACE_START(CCC)

template <typename T>
TSubString<T>::TSubString()
 : TStringPtr<T>()
{
  to = 0;
}

template <typename T>
TSubString<T>::TSubString(const TSubString<T>& ss)
 : TStringPtr<T>(ss)
{
  to = ss.to;
}

template <typename T>
TSubString<T>::TSubString(const TString<T>& s, Size offset, Size length)
 : TStringPtr<T>(s)
{
  to = s.to;
  if (offset <= s.getLength())
  {
    TStringPtr<T>::pos = offset;
    if (offset + length <= s.getLength())
    {
      to = offset + length;
    }
    else
    {
      assert(false);
      to = s.getLength();
    }
  }
  else
  {
    to = TStringPtr<T>::pos;
  }
  TStringPtr<T>::pos = offset;
}

template <typename T>
TSubString<T>::TSubString(const TString<T>& s, Size offset)
 : TStringPtr<T>(s)
{
  to = s.to;
  if (offset <= s.getLength())
  {
    TStringPtr<T>::pos = offset;
    to = s.getLength();
  }
  else
  {
    assert(false);
    to = TStringPtr<T>::pos;	// LATENT
  }
  TStringPtr<T>::pos = offset;
}

template <typename T>
TSubString<T>::~TSubString()
{
  ; // do nothing
}

template <typename T>
Size
TSubString<T>::getTo() const
{
  return to;
}

template <typename T>
T*
TSubString<T>::getEndPtr() const
{
  return TStringPtr<T>::getStartPtr() + to;
}

template <typename T>
bool
TSubString<T>::nilStrP() const
{
  return TStringPtr<T>::pos == to;
}

template <typename T>
Size
TSubString<T>::getLength() const
{
  return to - TStringPtr<T>::pos;
}

template <typename T>
int
TSubString<T>::strCmp(const TSubString& ss) const
{
  T* p = this->getStartPtr();
  Size len  = getLength();
  T* ss_p = ss.getStartPtr();
  Size ss_len  = ss.getLength();
  Size min_len = len < ss_len ? len : ss_len;
  for (Size i = 0; i < min_len; i++)
  {
    if (p[i] != ss_p[i])
    {
      return p[i] > ss_p[i] ? 1 : -1;
    }
  }
  return (len == ss_len) ? 0 : (len > ss_len) ? 1 : -1;
}

template <typename T>
int
TSubString<T>::strCmp(const TSubString* ss) const
{
  T* p = this->getStartPtr();
  Size len  = getLength();
  T* ss_p = ss->getStartPtr();
  Size ss_len  = ss->getLength();
  Size min_len = len < ss_len ? len : ss_len;
  for (Size i = 0; i < min_len; i++)
  {
    if (p[i] != ss_p[i])
    {
      return p[i] > ss_p[i] ? 1 : -1;
    }
  }
  return (len == ss_len) ? 0 : (len > ss_len) ? 1 : -1;
}

template <typename T>
int
TSubString<T>::strCmp(const T* s) const
{
  T* p = TStringPtr<T>::getStartPtr();
  Size len  = getLength();
  Size s_len  = strLen(s);
  Size min_len = len < s_len ? len : s_len;
  for (Size i = 0; i < min_len; i++)
  {
    if (p[i] != s[i])
    {
      return p[i] > s[i] ? 1 : -1;
    }
  }
  return (len == s_len) ? 0 : (len > s_len) ? 1 : -1;
}

template <typename T>
int
TSubString<T>::strNCaseCmp(const TSubString& ss) const
{
  T* p = TStringPtr<T>::getStartPtr();
  Size len  = getLength();
  T* ss_p = ss.getStartPtr();
  Size ss_len  = ss.getLength();
  Size min_len = len < ss_len ? len : ss_len;
  for (Size i = 0; i < min_len; i++)
  {
    T a = p[i];
    T b = ss_p[i];
    if ((a <= 0xff) && isupper((char)a))
    {
      a = tolower(a);
    }
    if ((b <= 0xff) && isupper((char)b))
    {
      b = tolower(b);
    }
    if (a != b)
    {
      return a > b ? 1 : -1;
    }
  }
  return (len == ss_len) ? 0 : (len > ss_len) ? 1 : -1;
}

template <typename T>
int
TSubString<T>::strNCaseCmp(const T* s) const
{
  T* p = TStringPtr<T>::getStartPtr();
  Size len  = getLength();
  Size s_len  = strLen(s);
  Size min_len = len < s_len ? len : s_len;
  for (Size i = 0; i < min_len; i++)
  {
    T a = p[i];
    T b = s[i];
    if ((a <= 0xff) && isupper((char)a))
    {
      a = tolower(a);
    }
    if ((b <= 0xff) && isupper((char)b))
    {
      b = tolower(b);
    }
    if (a != b)
    {
      return a > b ? 1 : -1;
    }
  }
  return (len == s_len) ? 0 : (len > s_len) ? 1 : -1;
}

template <typename T>
bool
TSubString<T>::operator == (const TSubString& ss) const
{
  return !strCmp(ss);
}

template <typename T>
bool
TSubString<T>::operator == (const T* s) const
{
  return !strCmp(s);
}

template <typename T>
bool
TSubString<T>::operator != (const TSubString& ss) const
{
  return !!strCmp(ss);
}

template <typename T>
bool
TSubString<T>::operator != (const T* s) const
{
  return !!strCmp(s);
}

template <typename T>
TSubString<T>
TSubString<T>::leftSubStr(Size l_pos) const
{
  TSubString ret(*this);
  if (l_pos > getLength())
  {
    ret.pos = ret.to = 0;
  }
  else
  {
    ret.to = ret.pos + l_pos;
  }
  return ret;
}

template <typename T>
TSubString<T>
TSubString<T>::rightSubStr(Size r_pos) const
{
  TSubString ret(*this);
  if (r_pos > getLength())
  {
    ret.pos = ret.to = 0;
  }
  else
  {
    ret.pos = ret.to - r_pos;
  }
  return ret;
}

template <typename T>
TSubString<T>
TSubString<T>::midSubStr(Size l_pos, Size r_pos) const
{
  TSubString ret(*this);   
  if (l_pos + r_pos > getLength())
  {
    ret.pos = ret.to = 0;
  }
  else
  {
    ret.pos += l_pos;
    ret.to -= r_pos;
  }
  return ret;
}

template <typename T>
void
TSubString<T>::toUpper()
{
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    if (islower(*p))
    {
      *p = toupper(*p);
    }
  }
}

template <typename T>
void
TSubString<T>::toLower()
{
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    if (isupper(*p))
    {
      *p = tolower(*p);
    }
  }
}

template <typename T>
void
TSubString<T>::fill(T c)
{
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    *p = c;
  }
}

template <typename T>
Size
TSubString<T>::simpleMatch(const TSubString<T>& ss) const
{
  Size n = getLength();        // target text length
  Size m = ss.getLength();     // pattern length
  if (n < m)
  {
    // pattern length is bigger than string
    return 0;
  }
  T* text = this->getStartPtr();    // target text
  T* patt = ss.getStartPtr(); // pattern
  Size i;
  for (i = 0; i <= n - m; i++)
  {
    Size j;
    for (j = 0; j < m; j++)
    {
      if (text[i + j] != patt[j])
      {
	// different
	break;
      }
    }
    if (j == m)
    {
      // same pattern
      return i + 1;
    }
  }
  // can't find pattern
  return 0;
}

template <typename T>
Size
TSubString<T>::kmpMatch(const TSubString<T>& ss) const
{
  Size n = getLength();        // target text length
  Size m = ss.getLength();     // pattern length
  if (n < m)
  {
    // pattern length is bigger than string
    return 0;
  }
  T* text = this->getStartPtr();    // target text
  T* patt = ss.getStartPtr(); // pattern
   
  // make up shift table
  Size* next = new Size[m + 1];  // table
  Size j = 0;
  next[0] = 0;
  next[1] = 0;
  Size i;
  for (i = 1; i < m; i++)
  {
    next[i + 1] = 0;
    while (patt[i] != patt[j])
    {
      if (j == 0)
      {
	goto L0;
      }
      j = next[j];
    }
    next[i + 1] = ++j;
  L0:;
  }
   
  // find
  j = 0;
  for (i = 0; i < n; i++)
  {
    while (text[i] != patt[j])
    {
      if (j == 0)
      {
	goto L1;
      }
      j = next[j];
    }
    if (++j == m)
    {
      // mattch
      return i - j + 2;
    }
  L1:;
  }
   
  delete[] next;
  if (j < m)
  {
    return 0;
  }
  else
  {
    return i - j + 1;
  }
}

template <typename T>
Size
TSubString<T>::bmMatch(const TSubString<T>& ss) const
{
  Size n = getLength();        // target text length
  Size m = ss.getLength();     // pattern length
  if (n < m)
  {
    // pattern length is bigger than string
    return 0;
  }
  T* text = TStringPtr<T>::getStartPtr();    // target text
  T* patt = ss.getStartPtr(); // pattern
   
  // make up skip table
  Size skip_tbl_size = 256;
  Size i;
  for (i = 1; i < sizeof(T); i++)
  {
    skip_tbl_size *= 256;
  }
  Size* skip = new Size[skip_tbl_size];
  for (unsigned int c = 0; c < skip_tbl_size; c++)
  {
    skip[c] = m;
  }
  for (i = 0; i < m; i++)
  {
    skip[patt[i]] = m - i - 1;
  }
   
  // find
  i = m - 1;
  while (i < n)
  {
    Size j = m - 1;
    while (text[i] == patt[j])
    {
      if (j == 0)
      {
	delete[] skip;
	return i + 1;
      }
      i--;
      j--;
    }
    i += (skip[text[i]] > (m - j)) ? skip[text[i]] : (m - j);
  }
  delete[] skip;
  return 0;
}

#if 0
template <>
Size
TSubString<UInt32>::bmMatch(const TSubString<UInt32>& ss) const
{
  /* bmMatchはUInt32の型に対してはskip表が4Gのサイズになるので
   * 使用すべきではない。
   */
  assert(false);
  return 0;
}
#endif

template <typename T>
Size
TSubString<T>::match(const TSubString<T>& ss) const
{
  return kmpMatch(ss);
}

template <typename T>
Size
TSubString<T>::strchar(T c) const
{
  Size pos = 1;
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    if (*p == c)
    {
      return pos;
    }
    pos++;
  }
  return 0;
}

template <typename T>
Size
TSubString<T>::strrchar(T c) const
{
  Size last_pos = 0;
  Size pos = 1;
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    if (*p == c)
    {
      last_pos = pos;
    }
    pos++;
  }
  return last_pos;
}

template <typename T>
int
TSubString<T>::getInt() const
{
  int ret = 0;
  bool minus_p = false;
  bool head_p = true;
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    int c = (int)*p;
    if (('0' <= c) && (c <= '9'))
    {
      ret = ret * 10 + (c - '0');
    }
    else if (head_p && (c == '+'))
    {
      // nothing
    }
    else if (head_p && (c == '-'))
    {
      minus_p = true;
    }
    else
    {
      // error
      return 0;
    }
    head_p = false;
  }
  if (minus_p)
  {
    ret = -ret;
  }
  return ret;
}

template <typename T>
unsigned int
TSubString<T>::getUInt() const
{
  unsigned int ret = 0;
  T* end = getEndPtr();
  for (T* p = TStringPtr<T>::getStartPtr(); p < end; p++)
  {
    int c = (int)*p;
    if (('0' <= c) && (c <= '9'))
    {
      ret = ret * 10 + (c - '0');
    }
    else
    {
      // error
      return 0;
    }
  }
  return ret;
}

template class TSubString<Int8>;
template class TSubString<UInt8>;
template class TSubString<UInt16>;
template class TSubString<UInt32>;

CCC_NAMESPACE_END(CCC)
