﻿// $Id$

#ifndef INCLUDE_ccc_base_TSubString_h
#define INCLUDE_ccc_base_TSubString_h

#include <ccc/base/TStringPtr.h>

CCC_NAMESPACE_START(CCC)

/*!
 * \ja
 * このクラスは部分文字列を示します。
 * \ja_end
 * \en
 * This class shows a substring.
 * \en_end
 */
template <typename T>
class TSubString
  : public TStringPtr<T>
{
 protected:
  /*!
   * \ja
   * 部分文字列の上限を示します。0から始まる数値です。
   * 部分文字列の下限はStringPtr&lt;T&gt;::posによって示されます。
   * toの示す位置は部分文字列最後の文字の次の位置を示します。
   * to == pos は空文字列を示します。
   * \ja_end
   * \en
   * the upper position, from 0 coordination.
   * lower position is pointed by StringPtr::pos.
   * the point index by 'to' is next position of the substring.
   * to == pos meens null string.
   * \en_end
   */
  Size to;
  /*!
   * \ja
   * コンストラクタ<br>
   * Stringのコンストラクタより呼ばれます。
   * \ja_end
   * \en
   * This method is called by the String constructor
   * \en_end
   */
  TSubString();

 public:
  /*!
   * \ja
   * コピーコンストラクタ
   * \param ss コピー対象の部分文字列
   * \ja_end
   */
  TSubString(const TSubString& ss);
  /*!
   * \ja
   * コンストラクタ
   * \param s 参照対象の文字列
   * \param offset オフセット位置。この位置より後ろが部分文字列として扱われます。
   * \ja_end
   */
  TSubString(const TString<T>& s, Size offset);
  /*!
   * \ja
   * コンストラクタ
   * \param s 参照対象の文字列
   * \param offset 開始オフセット位置
   * \param length 部分文字列の長さ
   * \ja_end
   */
  TSubString(const TString<T>& s, Size offset, Size length);
  /*!
   * \ja
   * デストラクタ
   * \ja_end
   */
  virtual ~TSubString();
  /*!
   * \ja
   * 部分文字列の開始位置を得ます。
   * \return 開始位置オフセット (0から始まる数値)
   * \ja_end
   */
  Size getFrom() const { return TStringPtr<T>::getPos(); }
  /*!
   * \ja
   * 部分文字列の終了位置を得ます。
   * この位置が指し示す文字は、部分文字列の最後の文字の次の文字であり、部分文字列に含まれません。
   * \return 終了位置オフセット (0から始まる数値)
   * \ja_end
   */
  virtual Size getTo() const;
  /*!
   * \ja
   * 部分文字列の終了位置のポインタを得ます。
   * \return 終了位置のポインタ
   * \ja_end
   */
  virtual T* getEndPtr() const;
  /*!
   * \ja
   * 空文字列かどうかを調べます。
   * \retval true 空文字列
   * \retval false 空文字列でない
   * \ja_end
   */
  bool nilStrP() const;
  /*!
   * \ja
   * 部分文字列の長さを得ます。
   * \return 長さ
   * \ja_end
   */
  Size getLength() const;
  /*!
   * \ja
   * 部分文字列を比較します。
   * \param ss 比較対象の部分文字列
   * \retval 1 ssよりも大きい
   * \retval 0 同じ
   * \retval -1 ssよりも小さい
   * \ja_end
   * \en
   * Compare substring.
   * \param ss target substring
   * \retval 1 larger than ss
   * \retval 0 same
   * \retval -1 smaller than ss
   * \en_end
   */
  int strCmp(const TSubString& ss) const;
  /*!
   * \ja
   * 部分文字列を比較します。
   * \param ss 比較対象の部分文字列
   * \retval 1 ssよりも大きい
   * \retval 0 同じ
   * \retval -1 ssよりも小さい
   * \ja_end
   * \en
   * Compare substring.
   * \param ss target substring
   * \retval 1 larger than ss
   * \retval 0 same
   * \retval -1 smaller than ss
   * \en_end
   */
  int strCmp(const TSubString* ss) const;
  /*!
   * \ja
   * 部分文字列を比較します。
   * \param s 比較対象のヌルで終端された文字列
   * \retval 1 ssよりも大きい
   * \retval 0 同じ
   * \retval -1 ssよりも小さい
   * \ja_end
   * \en
   * Compare substring.
   * \param s target string delimited with null.
   * \retval 1 larger than ss
   * \retval 0 same
   * \retval -1 smaller than ss
   * \en_end
   */
  int strCmp(const T* s) const;
  /*!
   * \ja
   * 大文字子文字の区別なく部分文字列を比較します。
   * \param ss 比較対象の部分文字列
   * \retval 1 ssよりも大きい
   * \retval 0 同じ
   * \retval -1 ssよりも小さい
   * \ja_end
   * \en
   * Compare substring.
   * \param ss target substring
   * \retval 1 larger than ss
   * \retval 0 same
   * \retval -1 smaller than ss
   * \en_end
   */
  int strNCaseCmp(const TSubString& ss) const;
  /*!
   * \ja
   * 大文字子文字の区別なく部分文字列を比較します。
   * \param s 比較対象のヌルで終端された文字列
   * \retval 1 ssよりも大きい
   * \retval 0 同じ
   * \retval -1 ssよりも小さい
   * \ja_end
   * \en
   * Compare substring.
   * \param s target string delimited with null.
   * \retval 1 larger than ss
   * \retval 0 same
   * \retval -1 smaller than ss
   * \en_end
   */
  int strNCaseCmp(const T* s) const;
  /*!
   * \ja
   * ==演算子、部分文字列の内容を比較します。
   * \param ss 比較対象
   * \retval true 同じ内容
   * \retval false 違う内容
   * \ja_end
   */
  bool operator == (const TSubString& ss) const;
  /*!
   * \ja
   * ==演算子、部分文字列とヌルで終端された文字列の内容を比較します。
   * \param s 比較対象のヌルで終端された文字列
   * \retval true 同じ内容
   * \retval false 違う内容
   * \ja_end
   */
  bool operator == (const T* s) const;
  /*!
   * \ja
   * !=演算子、部分文字列の内容を比較します。
   * \param ss 比較対象
   * \retval true 違う内容
   * \retval false 同じ内容
   * \ja_end
   */
  bool operator != (const TSubString& ss) const;
  /*!
   * \ja
   * !=演算子、部分文字列とヌルで終端された文字列の内容を比較します。
   * \param s 比較対象のヌルで終端された文字列
   * \retval true 違う内容
   * \retval false 同じ内容
   * \ja_end
   */
  bool operator != (const T* s) const;
  /*!
   * \ja
   * 左側の部分文字列を生成します。
   * \param l_str 生成する部分文字列の長さ
   * \ja_end
   * \en
   * Get left sub string.
   * \param l_str length
   * \en_end
   */
  TSubString leftSubStr(Size l_str) const;
  /*!
   * \ja
   * 右側の部分文字列を生成します。
   * \param r_str 生成する部分文字列の長さ
   * \ja_end
   * \en
   * Get right sub string.
   * \param r_str length
   * \en_end
   */
  TSubString rightSubStr(Size r_str) const;
  /*!
   * \ja
   * 指定した部分文字列を生成します。
   * \param l_str 左側からの位置
   * \param r_str 右側からの位置
   * \ja_end
   * \en
   * Get specified sub string
   * \param l_str length from left
   * \param r_str length from right
   * \en_end
   */
  TSubString midSubStr(Size l_str, Size r_str) const;
  /*!
   * \ja
   * シングルバイトのアルファベットの文字を大文字に変更します。
   * \ja_end
   * \en
   * change into the single bye alphabetical charactors to 
   * the uppper case charactor.
   * \en_end
   */
  void toUpper();
  /*!
   * \ja
   * シングルバイトのアルファベットの文字を子文字に変更します。
   * \ja_end
   * \en
   * change into the single bye alphabetical charactors to 
   * the lower case charactor.
   * \en_end
   */
  void toLower();
  /*!
   * \ja
   * 部分文字列を指定した文字で埋めます。
   * \param c 埋める文字
   * \ja_end
   */
  void fill(T c);
  /*!
   * \ja
   * 単純文字列照合をします。
   * 0が返った場合には該当する文字列が見つからなかったことを示します。
   * \param ss 照合対象
   * \return 照合位置(1から始まる数値)
   * \ja_end
   * \en
   * Simple string match.
   * If 0 is returned, it means that there isn't a matching position. 
   * \param ss target of the comparison
   * \return position - which number is started from 1.
   * \en_end
   */
  Size simpleMatch(const TSubString& ss) const;
  /*!
   * \ja
   * KMPアルゴリズムによる文字列照合を行ないます。
   * 0が返った場合には該当する文字列が見つからなかったことを示します。
   * \param ss 照合対象
   * \return 照合位置(1から始まる数値)
   * \ja_end
   * \en
   * Knuth-Morris-Pratt algorithm string match
   * If 0 is returned, it means that there isn't a matching position. 
   * \param ss target of the comparison
   * \return position - which number is started from 1.
   * \en_end
   */
  Size kmpMatch(const TSubString& ss) const;
  /*!
   * \ja
   * Boyer-Mooreアルゴリズムによる文字列照合を行ないます。
   * 0が返った場合には該当する文字列が見つからなかったことを示します。
   * このアルゴリズムは内部的にスキップ表を型Tで表現しうる数だけ用意するので、
   * 16ビットの文字列や、32ビットの文字列の場合にはスキップ表の作成に多くの
   * メモリやCPUタイムなどのリソースを消費するため、8ビットキャラクタ以外での
   * 利用はお勧めしません。
   * \param ss 照合対象
   * \return 照合位置(1から始まる数値)
   * \ja_end
   * \en
   * Boyer-Moore algorithm string match
   * If 0 is returned, it means that there isn't a matching position. 
   * \param ss target of the comparison
   * \return position - which number is started from 1.
   * \en_end
   */
  Size bmMatch(const TSubString& ss) const;
  /*!
   * \ja
   * 文字列照合をします。
   * 0が返った場合には該当する文字列が見つからなかったことを示します。
   * \param ss 照合対象
   * \return 照合位置(1から始まる数値)
   * \ja_end
   * \en
   * string match
   * If 0 is returned, it means that there isn't a matching position. 
   * \param ss target of the comparison
   * \return position - which number is started from 1.
   * \en_end
   */
  Size match(const TSubString& ss) const;
  /*!
   * \ja
   * 部分文字列から指定した文字の出現位置を調べます。
   * 0が返った場合には該当する文字列が見つからなかったことを示します。
   * \param ss 照合対象
   * \return 照合位置(1から始まる数値)
   * \ja_end
   */
  Size strchar(T c) const;
  /*!
   * \ja
   * 部分文字列から指定した文字が最後に出現する位置を調べます。
   * 0が返った場合には該当する文字列が見つからなかったことを示します。
   * \param ss 照合対象
   * \return 照合位置(1から始まる数値)
   * \ja_end
   */
  Size strrchar(T c) const;
  /*!
   * \ja
   * 数字と符合(+-)の文字のみから構成される部分文字列で表現されるintの値を得ます。
   * 数字以外の文字が入っていた場合や、空文字列の場合には0が返ります。
   * \return 数値
   * \ja_end
   */
  int getInt() const;
  /*!
   * \ja
   * 数字のみから構成される部分文字列で表現されるunsigned intの値を得ます。
   * 数字以外の文字が入っていた場合や、空文字列の場合には0が返ります。
   * \return 数値
   * \ja_end
   */
  unsigned int getUInt() const;
};

typedef TSubString<char> BSubString;
typedef TSubString<UInt16> DSubString;
typedef TSubString<UInt32> WSubString;

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_TSubString_h */
