﻿// $Id$

#include <ccc/base/TextReader.h>
#include <ccc/base/cstring.h>

CCC_NAMESPACE_START(CCC)

template <typename T>
TextReader<T>::TextReader(IFlow* src)
  : PushbackIFilter<T>(src), line_num(0)
{
  read_block_size = READ_BLOCK_SIZE;
  read_buf = new T[read_block_size];
}

template <typename T>
TextReader<T>::~TextReader()
{
  delete read_buf;
}

template <typename T>
void
TextReader<T>::setBlockSize(Size new_read_block_size)
{
  read_block_size = new_read_block_size;
  delete read_buf;
  read_buf = new T[read_block_size];
}

template <typename T>
bool
TextReader<T>::rewind()
{
  line_num = 0;
  return PushbackIFilter<T>::rewind();
}

template <typename T>
T
TextReader<T>::getChar() CCC_RAISES(IOException)
{
  if (PushbackIFilter<T>::getQueueLength() == 0)
  {
    // read block
    Size read_size;
    try
    {
      PushbackIFilter<T>::getBlock(read_block_size, read_buf, read_size);
    }
    catch (IOException ioe)
    {
      if (ioe.errorNum() != IOException::READ_BEYOND_THE_EOF)
      {
	throw ioe;
      }
    }
    int prev_line_num = line_num;
    Size i;
    for (i = 0; i < read_size; i++)
    {
      ungetChar(read_buf[read_size - 1 - i]);
    }
    line_num = prev_line_num;
  }
  T c = PushbackIFilter<T>::getChar();
  if (c == lf)
  {
    line_num++;
  }
  return c;
}

template <typename T>
void
TextReader<T>::ungetChar(T c) CCC_RAISES(IOException)
{
  if (!PushbackIFilter<T>::openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::EMPTY_FILTER_SRC);
  }
  if (c == lf)
  {
    line_num--;
  }
  this->push(c);
}

template <typename T>
void 
TextReader<T>::getBlock(Size block_size, T* ptr, Size& get_size) CCC_RAISES(IOException)
{
  get_size = 0;
  while (block_size-- > 0)
  {
    if ((*ptr++ = getChar()) == lf)
    {
      line_num++;
    }
    get_size++;
  }
}

template <typename T>
TString<T>*
TextReader<T>::readRest() CCC_RAISES(IOException)
{
  TString<T>* str = new TString<T>;
  try
  {
    for (;;)
    {
      T c = getChar();
      str->add(c);
    }
  }
  catch (IOException ioe)
  {
    if (ioe.errorNum() == IOException::READ_BEYOND_THE_EOF)
    {
      return str;
    }
    delete str;
    throw ioe;
  }
  return 0;	/* この行は実行されません。*/
}

template <typename T>
TString<T>*
TextReader<T>::readLine() CCC_RAISES(IOException)
{
  TString<T>* str = new TString<T>;
  try
  {
    T c;
    do
    {
      c = getChar();
      str->add(c);
    }
    while (c != lf);
  }
  catch (IOException ioe)
  {
    /* 最後の行末にLFが無い場合にもIOExceptionが発生している */
    if (ioe.errorNum() == IOException::READ_BEYOND_THE_EOF)
    {
      if (str->getLength() > 0)
      {
	/* 最後の行末にLFが無い場合 */
	return str;
      }
      else
      {
	delete str;
	return 0;
      }
    }
    delete str;
    throw ioe;
  }
  return str;
}

template <typename T>
TString<T>*
TextReader<T>::readDelimitedString(const TString<T>& delim) CCC_RAISES(IOException)
{
  Size delim_len = delim.getLength();
  if (delim_len == 0)
  {
    throw IOException(__FILE__, __LINE__, IOException::INVALID_ARG);
  }
  TString<T>* str = new TString<T>;
  Size match_pos = 0;
  T c;
  try
  {
    for (;;)
    {
      c = getChar();
      if (c == delim[match_pos])
      {
	match_pos++;
	if (match_pos == delim_len)
	{
	  return str;
	}
      }
      else
      {
	if (match_pos != 0)
	{
	  for (Size i = 0; i < match_pos; i++)
	  {
	    str->add(delim[i]);
	  }
	  match_pos = 0;
	}
	str->add(c);
      }
    }
  }
  catch (IOException ioe)
  {
    delete str;
    throw ioe;
  }
  assert(false);
  return 0;	// program never comes here.
}

template <typename T>
TString<T>*
TextReader<T>::readDelimitedString(const T* delim, Size delim_len) CCC_RAISES(IOException)
{
  if (delim_len == 0)
  {
    throw IOException(__FILE__, __LINE__, IOException::INVALID_ARG);
  }
  TString<T>* str = new TString<T>;
  Size match_pos = 0;
  T c;
  try
  {
    for (;;)
    {
      c = getChar();
      if (c == delim[match_pos])
      {
	match_pos++;
	if (match_pos == delim_len)
	{
	  return str;
	}
      }
      else
      {
	if (match_pos != 0)
	{
	  for (Size i = 0; i < match_pos; i++)
	  {
	    str->add(delim[i]);
	  }
	  match_pos = 0;
	}
	str->add(c);
      }
    }
  }
  catch (IOException ioe)
  {
    delete str;
    throw ioe;
  }
  assert(false);
  return 0;	// program never comes here.
}

template <typename T>
TString<T>*
TextReader<T>::readDelimitedString(const T* delim) CCC_RAISES(IOException)
{
  Size delim_len = strLen(delim);
  return readDelimitedString(delim, delim_len);
}

template <typename T>
TString<T>*
TextReader<T>::readDelimitedCharString(T delim) CCC_RAISES(IOException)
{
  TString<T>* str = new TString<T>;
  T c;
  try
  {
    for (;;)
    {
      c = getChar();
      if (c == delim)
      {
	return str;
      }
      else
      {
	str->add(c);
      }
    }
  }
  catch (IOException ioe)
  {
    delete str;
    throw ioe;
  }
  assert(false);
  return 0;	// program never comes here.
}

template <typename T>
TString<T>*
TextReader<T>::readDelimitedString(const LinkList<TString<T> >& delims) CCC_RAISES(IOException)
{
  Size delims_num = delims.number();
  if (delims_num == 0)
  {
    throw IOException(__FILE__, __LINE__, IOException::INVALID_ARG);
  }
  Size i;
  for (i = 0; i < delims_num; i++)
  {
    if (delims[i]->getLength() == 0)
    {
      throw IOException(__FILE__, __LINE__, IOException::INVALID_ARG);
    }
  }
  TString<T>** strs = new TString<T>*[delims_num];
  Size* delim_lens = new Size[delims_num];
  Size* matchs = new Size[delims_num];
  for (i = 0; i < delims_num; i++)
  {
    strs[i] = new TString<T>;
    delim_lens[i] = delims[i]->getLength();
    matchs[i] = 0;
  }
  T c;
  try
  {
    for (;;)
    {
      c = getChar();
      for (i = 0; i < delims_num; i++)
      {
	TString<T>& delim = *delims.vector(i);
	if (c == delim[matchs[i]])
	{
	  matchs[i]++;
	  if (matchs[i] == delim_lens[i])
	  {
	    TString<T>* str = strs[i];
	    strs[i] = 0;
	    for (Size j = 0; j < delims_num; j++)
	    {
	      delete strs[i];
	    }
            delete[] strs;
            delete[] delim_lens;
            delete[] matchs;
	    return str;
	  }
	}
	else
	{
	  if (matchs[i] != 0)
	  {
	    for (Size j = 0; j < matchs[i]; j++)
	    {
	      strs[i]->add(delim[j]);
	    }
	    matchs[i] = 0;
	  }
	  strs[i]->add(c);
	}
      }
    }
  }
  catch (IOException ioe)
  {
    for (Size j = 0; j < delims_num; j++)
    {
      delete strs[j];
    }
    delete[] strs;
    delete[] delim_lens;
    delete[] matchs;
    throw ioe;
  }
  assert(false);
  return 0;	// program never comes here.
}

template <typename T>
TString<T>*
TextReader<T>::readDelimitedString(const T** delims) CCC_RAISES(IOException)
{
  Size i = 0;
  while (delims[i])
  {
    i++;
  }
  Size delims_num = i;
  if (delims_num == 0)
  {
    throw IOException(__FILE__, __LINE__, IOException::INVALID_ARG);
  }
  for (i = 0; i < delims_num; i++)
  {
    if (strLen(delims[i]) == 0)
    {
      throw IOException(__FILE__, __LINE__, IOException::INVALID_ARG);
    }
  }
  TString<T>** strs = new TString<T>*[delims_num];
  Size* delim_lens = new Size[delims_num];
  Size* matchs = new Size[delims_num];
  for (i = 0; i < delims_num; i++)
  {
    strs[i] = new TString<T>;
    delim_lens[i] = strLen(delims[i]);
    matchs[i] = 0;
  }
  T c;
  try
  {
    for (;;)
    {
      c = getChar();
      for (i = 0; i < delims_num; i++)
      {
	const T* delim = delims[i];
	if (c == delim[matchs[i]])
	{
	  matchs[i]++;
	  if (matchs[i] == delim_lens[i])
	  {
	    TString<T>* str = strs[i];
	    strs[i] = 0;
	    for (Size j = 0; j < delims_num; j++)
	    {
	      delete strs[i];
	    }
            delete[] strs;
	    delete[] delim_lens;
	    delete[] matchs;
	    return str;
	  }
	}
	else
	{
	  if (matchs[i] != 0)
	  {
	    for (Size j = 0; j < matchs[i]; j++)
	    {
	      strs[i]->add(delim[j]);
	    }
	    matchs[i] = 0;
	  }
	  strs[i]->add(c);
	}
      }
    }
  }
  catch (IOException ioe)
  {
    for (Size j = 0; j < delims_num; j++)
    {
      delete strs[j];
    }
    delete[] strs;
    delete[] delim_lens;
    delete[] matchs;
    throw ioe;
  }
  assert(false);
  return 0;	// program never comes here.
}

template <typename T>
bool
TextReader<T>::readString(const TString<T>& s) CCC_RAISES(IOException)
{
  Size s_len = s.getLength();
  if (s_len == 0)
  {
    return false;
  }
  Size n = 0;
  for (;;)
  {
    T c = getChar();
    if (c == s[n])
    {
      n++;
      if (n == s_len)
      {
	return true;
      }
    }
    else
    {
      ungetChar(c);
      for (Size m = n; m > 0; m--)
      {
	ungetChar(s[m - 1]);
      }
      break;
    }
  }
  return false;
}

template <typename T>
bool
TextReader<T>::readString(const T* s) CCC_RAISES(IOException)
{
  Size s_len = strLen(s);
  if (s_len == 0)
  {
    return false;
  }
  Size n = 0;
  for (;;)
  {
    T c = getChar();
    if (c == s[n])
    {
      n++;
      if (n == s_len)
      {
	return true;
      }
    }
    else
    {
      ungetChar(c);
      for (Size m = n; m > 0; m--)
      {
	ungetChar(s[m - 1]);
      }
      break;
    }
  }
  return false;
}

template <typename T>
Size
TextReader<T>::readStrings(const LinkList<TString<T> >& ss) CCC_RAISES(IOException)
{
  Size ss_num = ss.number();
  if (ss_num == 0)
  {
    return 0;
  }
  Size* lens = new Size[ss_num];
  Size* matchs = new Size[ss_num];
  bool* candidates = new bool[ss_num];
  Size i;
  for (i = 0; i < ss_num; i++)
  {
    lens[i] = ss[i]->getLength();
    matchs[i] = 0;
    candidates[i] = true;
  }
  try
  {
    //Size n = 0;
    for (;;)
    {
      T c = getChar();
      for (i = 0; i < ss_num; i++)
      {
	TString<T>* s = ss[i];
	if (candidates[i] && 
	    (matchs[i] < lens[i]) &&
	    (c == (*s)[matchs[i]]))
	{
	  matchs[i]++;
	  if (matchs[i] == lens[i])
	  {
            delete[] lens;
	    delete[] matchs;
	    delete[] candidates;
	    return i + 1;
	  }
	}
	else
	{
	  candidates[i] = false;
	  Size m;
	  for (m = 0; m < ss_num; m++)
	  {
	    if (candidates[m])
	    {
	      break;
	    }
	  }
	  if (m == ss_num)
	  {
	    ungetChar(c);
	    for (Size m = matchs[i]; m > 0; m--)
	    {
	      ungetChar((*s)[m - 1]);
	    }
	    delete[] lens;
	    delete[] matchs;
	    delete[] candidates;
	    return 0;
	  }
	}
      }
    }
  }
  catch (IOException ioe)
  {
    delete[] lens;
    delete[] matchs;
    delete[] candidates;
    throw ioe;
  }
}

template <typename T>
Size
TextReader<T>::readStrings(const T** ss) CCC_RAISES(IOException)
{
  Size ss_num = 0;
  while (!!ss[ss_num])
  {
    ss_num++;
  }
  if (ss_num == 0)
  {
    return 0;
  }
  Size* lens = new Size[ss_num];
  Size* matchs = new Size[ss_num];
  bool* candidates = new bool[ss_num];
  Size i;
  for (i = 0; i < ss_num; i++)
  {
    lens[i] = strLen(ss[i]);
    matchs[i] = 0;
    candidates[i] = true;
  }
  try
  {
    //Size n = 0;
    for (;;)
    {
      T c = getChar();
      for (i = 0; i < ss_num; i++)
      {
	const T* s = ss[i];
	if (candidates[i] && 
	    (matchs[i] < lens[i]) &&
	    (c == s[matchs[i]]))
	{
	  matchs[i]++;
	  if (matchs[i] == lens[i])
	  {
	    delete[] lens;
	    delete[] matchs;
	    delete[] candidates;
	    return i + 1;
	  }
	}
	else
	{
	  candidates[i] = false;
	  Size m;
	  for (m = 0; m < ss_num; m++)
	  {
	    if (candidates[m])
	    {
	      break;
	    }
	  }
	  if (m == ss_num)
	  {
	    ungetChar(c);
	    for (Size m = matchs[i]; m > 0; m--)
	    {
	      ungetChar(s[m - 1]);
	    }
	    delete[] lens;
	    delete[] matchs;
	    delete[] candidates;
	    return 0;
	  }
	}
      }
    }
  }
  catch (IOException ioe)
  {
    delete[] lens;
    delete[] matchs;
    delete[] candidates;
    throw ioe;
  }
}

template <typename T>
bool
TextReader<T>::readChar(T c) CCC_RAISES(IOException)
{
  T r = getChar();
  if (r == c)
  {
    return true;
  }
  ungetChar(r);
  return false;
}

template <typename T>
TString<T>*
TextReader<T>::skipChar(T candidate) CCC_RAISES(IOException)
{
  TString<T>* str = new TString<T>;
  T c;
  try
  {
    while ((c = getChar()) == candidate)
    {
      str->add(c);
    }
    ungetChar(c);
  }
  catch (IOException ioe)
  {
    if (ioe.errorNum() == IOException::READ_BEYOND_THE_EOF)
    {
      return str;
    }
    delete str;
    throw ioe;
  }
  return str;
}

template <typename T>
TString<T>*
TextReader<T>::skipCandidates(const T* candidates, Size candidates_len) CCC_RAISES(IOException)
{
  TString<T>* str = new TString<T>;
  T c;
  try
  {
    for (;;)
    {
      c = getChar();
      const T* x = candidates;
      Size i;
      for (i = 0; i < candidates_len; i++)
      {
	if (*x++ == c)
	{
	  break;
	}
      }
      if (i == candidates_len)
      {
	break;
      }
      str->add(c);
    }
    ungetChar(c);
  }
  catch (IOException ioe)
  {
    if (ioe.errorNum() == IOException::READ_BEYOND_THE_EOF)
    {
      return str;
    }
    delete str;
    throw ioe;
  }
  return str;
}

template <typename T>
TString<T>*
TextReader<T>::skipCandidates(const T* candidates) CCC_RAISES(IOException)
{
  return skipCandidates(candidates, strLen(candidates));
}

template <typename T>
TString<T>*
TextReader<T>::skipCandidates(const TString<T>& candidates) CCC_RAISES(IOException)
{
  return skipCandidates(candidates.getStartPtr(), candidates.getLength());
}

template class TextReader<Int8>;
template class TextReader<UInt16>;
template class TextReader<UInt32>;

TextReader<Int8>* stdin_r8 = 0;
TextReader<UInt16>* stdin_r16 = 0;
TextReader<UInt32>* stdin_r32 = 0;

CCC_NAMESPACE_END(CCC)
