﻿// $Id$

#ifndef INCLUDE_ccc_base_TextReader_h
#define INCLUDE_ccc_base_TextReader_h

#include <ccc/base/PushbackIFilter.h>
#include <ccc/base/TString.h>
#include <ccc/base/LinkList.h>

CCC_NAMESPACE_START(CCC)

/*!
 * TextReaderクラスはフローに対する入力フィルタとして動作し、
 * テキストの入力インターフェースを提供します。
 * 改行コードをLFとして動作します。別の改行コードを扱う場合には、
 * 全段に適切な変換用のフィルタを設けるようにしてください。
 * テンプレート引数TはInt8, UInt16, UInt32のいずれかの型のみを
 * サポートします。
 */
template <typename T>
class TextReader
  : public PushbackIFilter<T>
{
  enum
  {
    /*!
     * 改行コード
     */
    lf = 0x0a,
    READ_BLOCK_SIZE = 1024 * 16,
  };
  /*!
   * 行番号
   */ 
  int line_num;
  /*!
   * 読み込みブロック用バッファ
   */
  T* read_buf;
  /*!
   * 読み込みブロックのバッファサイズ
   */
  Size read_block_size;
 public:
  /*!
   * コンストラクタ
   * \param src 入力元フロー
   */
  TextReader(IFlow* src = 0);
  /*!
   * デストラクタ
   */
  ~TextReader();
  /*!
   * ブロックサイズを指定します。
   * \param new_read_block_sizeブロックサイズ
   */
  void setBlockSize(Size new_read_block_size);
  /*!
   * \ja
   * このフィルタを初期状態にし、フィルタ先に対してもrewind()を呼び出して
   * その結果を返します。
   * \retval true 成功
   * \retval false 失敗
   * \ja_end
   */
  virtual bool rewind();
  // ------------------------------------------------------------
  /* 行番号管理 */
  /*!
   * フローの先頭の行番号を設定します。
   * デフォルト値は1です。
   * \param n 先頭の行番号
   */
  virtual void setTopLineNumber(int n) { line_num = n; }
  /*!
   * 現在の行番号を取得します。
   * \return 現在の行番号
   */
  virtual int getLineNumber() { return line_num; }
  // ------------------------------------------------------------
  /* 単位データ入力 */
  /*!
   * T型の値を読み込みます。
   * \return 読み込んだデータ
   */
  virtual T getChar() CCC_RAISES(IOException);
  /*!
   * T型の値を読み戻します。LFを戻した場合には、
   * 行番号が一つ前に戻ります。
   */
  virtual void ungetChar(T c) CCC_RAISES(IOException);
  /*!
   * 指定した数のデータを読み込みます。
   * get_sizeに読み込めたデータ数がセットされます。
   * \param block_size ブロックサイズ
   * \param ptr データの保存先
   * \param get_size 読み込んだデータ数
   */
  virtual void getBlock(Size block_size, T* ptr, Size& get_size) CCC_RAISES(IOException);
  /*!
   * 残りのデータを全て読み込みます。
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readRest() CCC_RAISES(IOException);
  // ------------------------------------------------------------
  /* 終端を指定する文字列の読み込み */
  /*!
   * TODO: フローの終りが改行で終わらない場合のスペック見直し
   *
   * 1行分のデータを読み込みます。
   * LFの改行コードを読み込んだ場合には結果にもLFが付加されます。
   * 入力フローが終了した場合には、結果にLFは付加されません。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readLine() CCC_RAISES(IOException);
  /*!
   * 指定した文字を終端文字として、文字列を読み込みます。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * 結果の文字列には終端文字は含まれません。
   * 入力フローの読み込み位置のデータが終端文字だった場合には、空文字列が返ります。
   * \param delim 終端文字
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readDelimitedCharString(T delim) CCC_RAISES(IOException);
  /*!
   * 指定した文字列を終端文字として、文字列を読み込みます。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * 結果の文字列には終端文字列は含まれません。
   * 入力フローの読み込み位置のデータが終端文字列だった場合には、空文字列が返ります。
   * 空文字列を終端文字列として指定した場合には、空文字列が返ります。
   * \param delim 終端文字列
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readDelimitedString(const TString<T>& delim) CCC_RAISES(IOException);
  /*!
   * 指定した文字列を終端文字として、文字列を読み込みます。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * 結果の文字列には終端文字列は含まれません。
   * 入力フローの読み込み位置のデータが終端文字列だった場合には、空文字列が返ります。
   * 空文字列を終端文字列として指定した場合には、空文字列が返ります。
   * \param delim 終端文字列(ヌル終端)
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readDelimitedString(const T* delim) CCC_RAISES(IOException);
  /*!
   * 指定した文字列を終端文字として、文字列を読み込みます。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * 結果の文字列には終端文字列は含まれません。
   * 入力フローの読み込み位置のデータが終端文字列だった場合には、空文字列が返ります。
   * \param delim 終端文字列
   * \param delim_length 終端文字列の長さ
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readDelimitedString(const T* delim, Size delim_length) CCC_RAISES(IOException);
  /*!
   * 複数の終端文字列のいずれかを終端として、文字列を読み込みます。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * 結果の文字列には終端文字列は含まれません。
   * 入力フローの読み込み位置のデータが終端文字列だった場合には、空文字列が返ります。
   * \param delims 終端文字列のリスト
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readDelimitedString(const LinkList<TString<T> >& delims) CCC_RAISES(IOException);
  /*!
   * 複数の終端文字列のいずれかを終端として、文字列を読み込みます。
   * 結果として返された文字列オブジェクトは呼出側が責任を持って不要になったらdeleteしてください。
   * 結果の文字列には終端文字列は含まれません。
   * 入力フローの読み込み位置のデータが終端文字列だった場合には、空文字列が返ります。
   * \param delims 終端文字列の配列、配列の最後の要素には0を設定します。
   * \return 読み込んだ文字列
   */
  virtual TString<T>* readDelimitedString(const T** delims) CCC_RAISES(IOException);
  // ------------------------------------------------------------
  /* 指定文字列の読み込み */
  /*!
   * 指定した文字列を読み込みます。
   * \param s 文字列
   * \retval true 読み込めた
   * \retval false 読み込めなかった
   */
  virtual bool readString(const TString<T>& s) CCC_RAISES(IOException);
  /*!
   * 指定した文字列を読み込みます。
   * \param s 文字列
   * \retval true 読み込めた
   * \retval false 読み込めなかった
   */
  virtual bool readString(const T* s) CCC_RAISES(IOException);
  /*!
   * 指定した複数の文字列のどれかを読み込みます。
   * \param ss 複数の文字列
   * \retval 0 読み込めなかった
   * \retval その他 リストの何番目の文字列が読み込まれたかを示す数値
   */
  virtual Size readStrings(const LinkList<TString<T> >& ss) CCC_RAISES(IOException);
  /*!
   * 指定した複数の文字列のどれかを読み込みます。
   * \param ss 複数の文字列の配列、配列の最後の要素には0を設定します。
   * \retval 0 読み込めなかった
   * \retval その他 何番目の文字列が読み込まれたかを示す数値(最初の文字列は1として表現される)
   */
  virtual Size readStrings(const T** ss) CCC_RAISES(IOException);
  // ------------------------------------------------------------
  /* 指定文字の読み込み */
  /*!
   * 指定した文字を読み込みます。
   * \param c 文字
   * \retval true 読み込めた
   * \retval false 読み込めなかった
   */
  virtual bool readChar(T c) CCC_RAISES(IOException);
  /*!
   * 読み込んだ文字が指定した文字の間は読み込みを行います。
   * 読み込みを行わなかった場合には、空文字列が返ります。
   * 途中でフローが終了した場合には、直前までに読み込まれた文字列を返します。
   * フローが終了している状態で、このメソッドを呼び出した場合には空文字列が返ります。
   * \param candidate 文字
   * \return 読み込んだ文字列
   */
  virtual TString<T>* skipChar(T candidate) CCC_RAISES(IOException);
  /*!
   * 読み込んだ文字が指定した文字列に含まれる文字のいずれかの間
   * は読み込みを行います。
   * 読み込みを行わなかった場合には、空文字列が返ります。
   * \param candidates 候補文字列
   * \param candidates_len 候補文字列の長さ
   * \return 読み込んだ文字列
   */
  virtual TString<T>* skipCandidates(const T* candidates, Size candidates_len) CCC_RAISES(IOException);
  /*!
   * 読み込んだ文字が指定した文字列に含まれる文字のいずれかの間
   * は読み込みを行います。
   * 読み込みを行わなかった場合には、空文字列が返ります。
   * \param candidates 候補文字列
   * \return 読み込んだ文字列
   */
  virtual TString<T>* skipCandidates(const T* candidates) CCC_RAISES(IOException);
  /*!
   * 読み込んだ文字が指定した文字列に含まれる文字のいずれかの間
   * は読み込みを行います。
   * 読み込みを行わなかった場合には、空文字列が返ります。
   * \param candidates 候補文字列
   * \return 読み込んだ文字列
   */
  virtual TString<T>* skipCandidates(const TString<T>& candidates) CCC_RAISES(IOException);
#if 0
  // ------------------------------------------------------------
  /* 数値の読み込み */
  /*!
   * intの数値を読み込みます。
   * [-+]?[0-9][0-9]*
   * 桁数が指定されない場合には、桁数はintで表現できる最大の桁数までとします。
   * それを越える数値の場合には、intの最大値もしくは最小値が返ります。
   * \param digit 桁数
   * \return 読み込んだ数値
   */
  virtual int readInt(int digit = 0) CCC_RAISES(IOException);
  /*!
   * unsigned intの数値を読み込みます。
   * [+]?[0-9][0-9]*
   * 桁数が指定されない場合には、桁数はunsigned intで表現できる最大の桁数までとします。
   * それを越える数値の場合には、unsigned intの最大値もしくは最小値が返ります。
   * \param digit 桁数
   * \return 読み込んだ数値
   */
  virtual unsigned int readUint(int digit = 0) CCC_RAISES(IOException);
  /*!
   * 0と1で表現された2進数の数値を読み込みます。
   * 桁数が指定されない場合には、桁数はunsigned intで表現できる最大の桁数までとします。
   * それを越える数値の場合には、unsigned intの最大値もしくは最小値が返ります。
   * \param digit 桁数
   * \return 読み込んだ数値
   */
  virtual unsigned int readBinary(int digit = 0) CCC_RAISES(IOException);
  /*!
   * [0-7]の文字で表現された8進数の数値を読み込みます。
   * 桁数が指定されない場合には、桁数はunsigned intで表現できる最大の桁数までとします。
   * それを越える数値の場合には、unsigned intの最大値もしくは最小値が返ります。
   * \param digit 桁数
   * \return 読み込んだ数値
   */
  virtual unsigned int readOctal(int digit = 0) CCC_RAISES(IOException);
  /*!
   * [0-9][a-fA-F]の文字で表現された16進数の数値を読み込みます。
   * 桁数が指定されない場合には、桁数はunsigned intで表現できる最大の桁数までとします。
   * それを越える数値の場合には、unsigned intの最大値もしくは最小値が返ります。
   * \param digit 桁数
   * \return 読み込んだ数値
   */
  virtual unsigned int readHex(int digit = 0) CCC_RAISES(IOException);
#endif
};

typedef TextReader<char> BTextReader;
typedef TextReader<UInt16> DTextReader;
typedef TextReader<UInt32> WTextReader;

extern TextReader<Int8>* stdin_r8;
extern TextReader<UInt16>* stdin_r16;
extern TextReader<UInt32>* stdin_r32;

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_base_TextReader_h */
