﻿/* $Id$ */

#ifndef INCLUDE_ccc_base_base_h
#define INCLUDE_ccc_base_base_h

/*!
 * \file ccc/base/base.h
 * \ja
 * このファイルはcccの基本的な型や関数を定義しています。
 * \ja_end
 * \en
 * This file defines the basic types and the functions of the ccc.
 * \en_end
 */

#include <ccc/base/config.h>

/*
 * for CPU bits
 */
#define CCC_16BITS 16
#define CCC_32BITS 32
#define CCC_64BITS 64
#define CCC_128BITS 128

/*
 * for ENDIAN
 */
#define	CCC_LITTLE_ENDIAN	1234	/*!< LSB first */
#define	CCC_BIG_ENDIAN		4321	/*!< MSB first */
#define	CCC_PDP_ENDIAN		3412	/*!< LSB first in word, MSW first in long */

/*
 * IDL I/O
 */
#define CCC_IN		/*!< IN */
#define CCC_OUT		/*!< OUT */
#define CCC_INOUT	/*!< IN and OUT */

/*!
 * \ja
 * cccバージョン文字列
 * \ja_end
 * \en
 * It is the version string of the libccc.
 * \en_end
 */
#define LIBCCC_VERSION_STR "3.0"

/*!
 * \def CCC_NAMESPACE_START(ns)
 * \ja
 * このマクロはネームスペースをサポートするC++コンパイラではnamespace ns {
 * に展開されます。ネームスペースをサポートしないC++コンパイラやC言語では
 * 空に展開されます。
 * ネームスペースをサポートしないC++のコンパイラ(Microsoft VC++ 6.0など)
 * のために、直接namespaceコマンドを使わずに、このマクロを使うようにします。
 * \ja_end
 * \en
 * With a C++ compiler which supports namespace, this macro is 
 * expanded into 'namespace ns {'. This macro is expanded into empty
 * by the namespace unsupported C++ compiler or by the C compiler.
 * With a C++ compiler which doesn't support namespace such as Microsoft
 * VC++ 6.0, use this macro and don't use namespace command directly.
 * \en_end
 */

/*!
 * \def CCC_NAMESPACE_END(ns)
 * \ja
 * このマクロはネームスペースをサポートするC++コンパイラでは}に展開されます。
 * \ja_end
 * \en
 * With a C++ compiler which supports namespace, this macro is 
 * expanded into '}'
 * \en_end
 */

/*!
 * \def CCC_NAMESPACE_USING(x)
 * \ja
 * このマクロはネームスペースをサポートするC++コンパイラではではusing x;
 * に展開されます。
 * \ja_end
 * \en
 * With a C++ compiler which suppports namespace, this macro is expanded
 * into 'using x;'.
 * \en_end
 */

/*! 
 * \def CCC_NAMESPACE_USING_NAMESPACE(ns)
 * \ja
 * このマクロはネームスペースをサポートするC++コンパイラではusing namespace ns;
 * に展開されます。
 * \ja_end
 * \en
 * With a C++ compiler which supports namespace, this macro is expanded
 * into 'using namespace ns;'.
 * \en_end
 */
#ifndef __cplusplus
#  define CCC_NAMESPACE_START(ns)
#  define CCC_NAMESPACE_END(ns)
#  define CCC_NAMESPACE_USING(x)
#  define CCC_NAMESPACE_USING_NAMESPACE(ns)
#else /* __cplusplus */
#  ifdef DO_NOT_USE_NAMESPACE
#    define CCC_NAMESPACE_START(ns)
#    define CCC_NAMESPACE_END(ns)
#    define CCC_NAMESPACE_USING(x)
#    define CCC_NAMESPACE_USING_NAMESPACE(ns)
#  else /* DO_NOT_USE_NAMESPACE */
#    define CCC_NAMESPACE_START(ns) namespace ns {
#    define CCC_NAMESPACE_END(ns) }
#    define CCC_NAMESPACE_USING(x) using x;
#    define CCC_NAMESPACE_USING_NAMESPACE(ns) using namespace ns;
#  endif /* DO_NOT_USE_NAMESPACE */
#endif /* __cplusplus */

#ifndef CCC_RAISES
/*!
 * replacement of throw
 */
#define CCC_RAISES(x) throw(x)
#endif /* CCC_RAISES */

/* NOTE:
 * CCC_* type will be used for "C" interface.
 *
 *		signed		unsigned
 * ---------------------------------------
 * 1 byte	Int8		UInt8
 * 2 byte	Int16		UInt16
 * 4 byte	Int32		UInt32
 * 8 byte	Int64		UInt64
 */
typedef char			CCC_Int8;	/*!< signed 8 bit type */
typedef unsigned char		CCC_UInt8;	/*!< unsigned 8 bit type */
typedef short int		CCC_Int16;	/*!< signed 16 bit type */
typedef unsigned short int	CCC_UInt16;	/*!< unsigned 16 bit type */
typedef int			CCC_Int32;	/*!< signed 32 bit type */
typedef unsigned int		CCC_UInt32;	/*!< unsigned 32 bit type */
typedef long long int		CCC_Int64;	/*!< signed 64 bit type */
typedef unsigned long long int	CCC_UInt64;	/*!< unsigned 64 bit type */

#ifndef CCC_Size
#  include <stddef.h>
#  define CCC_Size size_t		/*!< Size type */
#endif /* CCC_Size */

#ifndef __cplusplus
#ifndef	bool
typedef enum
{
  false = 0,
  true = 1,
} bool;
#endif	/* bool */
#endif /* __cplusplus */

#ifdef __cplusplus

#include <ccc/base/Ccc.h>
#include <ccc/base/SPtr.h>

CCC_NAMESPACE_START(CCC)

typedef CCC_Int8 Int8;		/*!< signed 8 bit type */
typedef CCC_UInt8 UInt8;	/*!< unsigned 8 bit type */
typedef CCC_Int16 Int16;	/*!< signed 16 bit type */ 
typedef CCC_UInt16 UInt16;	/*!< unsigned 16 bit type */
typedef CCC_Int32 Int32;	/*!< signed 32 bit type */
typedef CCC_UInt32 UInt32;	/*!< unsigned 32 bit type */
typedef CCC_Int64 Int64;	/*!< signed 64 bit type */
typedef CCC_UInt64 UInt64;	/*!< unsigned 64 bit type */
typedef CCC_Size Size;		/*!< Size type */

/*
 * for CCC Cocoa Extentions
 */
#ifdef __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__
/* Mac */
#  define CCC_COCOA
#  define CCC_COCOA_MAC
#endif
#ifdef __ENVIRONMENT_IPHONE_OS_VERSION_MIN_REQUIRED__
/* iPhone */
#  define CCC_COCOA
#  define CCC_COCOA_IPHONE
#  ifdef __arm__
#    define CCC_COCOA_IPHONE_ARM
#  endif /* __arm__ */
#  ifdef __i386__
#    define CCC_COCOA_IPHONE_386
#  endif /* __i386__ */
#endif

/*!
 * \ja
 * cccライブラリのスーパークラスです。<br>
 * 今の所、このクラスにはメンバやメソッドはありません。
 * \ja_end
 * \en
 * super class of ccc library.<br>
 * This class doesn't have any members or any methods so for.
 * \en_end
 */
class Object
{
};

/*!
 * \ja
 * cccライブラリのスーパー仮想クラスです。<br>
 * 仮想関数のデストラクタが提供されます。
 * \ja_end
 * \en
 * A virtual super class of ccc library.<br>
 * \en_end
 */
class VObject
  : public Object
{
 public:
  virtual ~VObject() = 0;
};

/*!
 * \ja
 * 2つの値から小さい方の値を得ます。
 * \ja_end
 * \en
 * This function gets the smaller value from the two values.
 * \en_end
 */
template <class T> inline T
getMin(const T a, const T b)
{
  return a < b ? a : b;
}

/*!
 * \ja
 * 3つの値から最も小さい値を得ます。
 * \ja_end
 * \en
 * This function gets the smallest value from the three values.
 * \en_end
 */
template <class T> inline T
getMin(const T a, const T b, const T c)
{
  return a < b ? (a < c ? a : c) : (b < c ? b : c);
}

/*!
 * \ja
 * 4つの値から最も小さい値を得ます。
 * \ja_end
 * \en
 * This function gets the smallest value from the four values.
 * \en_end
 */
template <class T> inline T
getMin(const T a, const T b, const T c, const T d)
{
  return getMin(getMin(a, b), getMin(c, d));
}

/*!
 * \ja
 * 5つの値から最も小さい値を得ます。
 * \ja_end
 * \en
 * This function gets the smallest value from the five values.
 * \en_end
 */
template <class T> inline T
getMin(const T a, const T b, const T c, const T d, const T e)
{
  return getMin(getMin(a, b), getMin(c, d, e));
}

/*!
 * \ja
 * 2つの値から大きい方の値を得ます。
 * \ja_end
 * \en
 * This function gets the larger value from the two values.
 * \en_end
 */
template <class T> inline T
getMax(const T a, const T b)
{
  return a > b ? a : b;
}

/*!
 * \ja
 * 3つの値から最も大きい値を得ます。
 * \ja_end
 * \en
 * This function gets the largest value from the three values.
 * \en_end
 */
template <class T> inline T
getMax(const T a, const T b, const T c)
{
  return a > b ? (a > c ? a : c) : (b > c ? b : c);
}

/*!
 * \ja
 * 4つの値から最も大きい値を得ます。
 * \ja_end
 * \en
 * This function gets the largest value from the four values.
 * \en_end
 */
template <class T> inline T
getMax(const T a, const T b, const T c, const T d)
{
  return getMax(getMax(a, b), getMax(c, d));
}

/*!
 * \ja
 * 5つの値から最も大きい値を得ます。
 * \ja_end
 * \en
 * This function gets the largest value from the five values.
 * \en_end
 */
template <class T> inline T
getMax(const T a, const T b, const T c, const T d, const T e)
{
  return getMax(getMax(a, b), getMax(c, d, e));
}

/*!
 * \ja
 * 2つの値の差を求めます。
 * \return 差の値、マイナスの値にはなりません。
 * \ja_end
 * \en
 * This function gets the difference of the two values.
 * \return difference, This value never becomes minus.
 * \en_end
 */
template <class T> inline T
getDiff(const T i, const T j)
{
  return (i > j) ? i - j : j - i;
}

/*!
 * \ja
 * 2つの値から、小さい方の値と大きい方の値を求めます。
 * \param beg_pos 小さい方の値をセットする参照
 * \param end_pos 大きい方の値をセットする参照
 * \param pos1 値1
 * \param pos2 値2
 * \ja_end
 * \en
 * This function gets the smaller value and the larger value of the two.
 * \param beg_pos reference, the smaller value is set.
 * \param end_pos reference, the larger value is set.
 * \param pos1 value 1
 * \param pos2 value 2
 * \en_end
 */
template <class T> void
getBand(T& beg_pos, T& end_pos, const T pos1, const T pos2)
{
  if (pos1 < pos2)
  {
    beg_pos = pos1;
    end_pos = pos2;
  }
  else
  {
    beg_pos = pos2;
    end_pos = pos1;
  }
}

/*!
 * \ja
 * 16ビット値の上位バイトと下位バイトを入れ換えます。
 * \param c 入れ換えを行ないたい 16 ビット値
 * \return 入れ換え後の値
 * \ja_end
 * \en
 * This function swaps a high byte with a low byte in a 16 bits value.
 * \param c 16 bits value which is to be swaped
 * \return swaped value
 * \en_end
 */
inline UInt16
swap_16(UInt16 c)
{
  UInt8 low = (UInt8)(0x00ffu & c);
  UInt8 hi = (UInt8)(c >> 8);
  return ((UInt16)hi) | (((UInt16)low) << 8);
}

/*!
 * \ja
 * 32ビット値の上位ワードと下位ワードを入れ換えます。
 * \param c 入れ換えを行ないたい 32 ビット値
 * \return 入れ換え後の値
 * \ja_end
 * \en
 * This function swaps a high word with a low word in a 32 bits value.
 * \param c 32 bits value which is to be swaped
 * \return swaped value
 * \en_end
 */
inline UInt32
swap_32(UInt32 c)
{
  UInt16 low = swap_16((UInt16)(0x0000ffffu & c));
  UInt16 hi = swap_16((UInt16)(c >> 16));
  return ((UInt32)hi) | (((UInt32)low) << 16);
}

CCC_NAMESPACE_END(CCC)

#endif /* __cplusplus */

#endif /* INCLUDE_ccc_base_base_h */
