﻿// $Id$

#ifndef INCLUDE_ccc_fetch_ContentType_h
#define INCLUDE_ccc_fetch_ContentType_h

#include <ccc/base/base.h>
#include <ccc/base/LinkList.h>
#include <ccc/fetch/ContentTypeParameterItem.h>

CCC_NAMESPACE_START(CCC)

/*!
 * MIMEのContent-Typeを表現するクラスです。
 * RFC2045, RFC2046
 * Content-Type:type/subtype*[;parameter]
 */
class ContentType
{
 public:
  /*!
   * Content-Typeの主形式を示します。
   * 注意: ライブラリの実装はこの順番に依存しているので、
   * 不用意に順番を変更しないでください。
   */
  enum MainType
  {
    /*!
     * 設定されていない状態を示します。
     */
    MAINTYPE_NOT_SET,
    /*!
     * テキスト
     */
    MAINTYPE_TEXT,
    /*!
     * メッセージ
     */
    MAINTYPE_MESSAGE,
    /*!
     * アプリケーション
     */
    MAINTYPE_APPLICATION,
    /*!
     * イメージ
     */
    MAINTYPE_IMAGE,
    /*!
     * 音声
     */
    MAINTYPE_AUDIO,
    /*!
     * 動画
     */
    MAINTYPE_VIDEO,
    /*!
     * マルチパート
     */
    MAINTYPE_MULTIPART,
    /*!
     * 上記の定義以外
     */
    MAINTYPE_OTHER,
  };
  /*!
   * Content-Typeの副形式を示します。
   * 注意: ライブラリの実装はこの順番に依存しているので、
   * 不用意に順番を変更しないでください。
   */
  enum SubType
  {
    /*!
     * 設定されていない状態を示します。
     */
    SUBTYPE_NOT_SET,
    // TEXT
    SUBTYPE_PLAIN,
    SUBTYPE_ENRICHED,
    // MESSAGE
    SUBTYPE_RFC822,
    SUBTYPE_PARTIAL,
    SUBTYPE_EXTARNAL_BODY,
    // APPLICATION
    SUBTYPE_OCTET_STREAM,
    SUBTYPE_POSTSCRIPT,
    // IMAGE
    SUBTYPE_JPEG,
    SUBTYPE_GIF,
    // AUDIO
    SUBTYPE_BASIC,
    // VIDEO
    SUBTYPE_MPEG,
    // MULTIPART
    SUBTYPE_MIXED,
    SUBTYPE_ALTERNATIVE,
    SUBTYPE_PARALLEL,
    SUBTYPE_DIGEST,
    // HTML
    SUBTYPE_HTML,
    // XML
    SUBTYPE_XML,
    // JavaScript
    SUBTYPE_JAVASCRIPT,
    // CSS
    SUBTYPE_CSS,
    /*!
     * 上記の定義以外
     */
    SUBTYPE_OTHER,
  };
 private:
  /*!
   * Content-Type:文字列
   */
  char* ct;
  /*!
   * 主形式
   */
  MainType main_type;
  /*!
   * 主形式の文字列
   */
  char* main_type_str;
  /*!
   * 副形式
   */
  SubType sub_type;
  /*!
   * 副形式の文字列
   */
  char* sub_type_str;
  /*!
   * パラメータの全文字列
   */
  char* parameter_str;
  /*!
   * パラメータを解析したリスト
   */
  LinkList<ContentTypeParameterItem> params;
 public:  
  /*!
   * コンストラクタ<br>
   * Content-Type:文字列が指定された場合には文字列を内部にセットし解析します。
   * \param ct Content-Type:文字列
   */
  ContentType(const char* ct = 0);
  /*!
   * デストラクタ
   */
  ~ContentType();
  /*!
   * Content-Type:文字列をセットし解析します。
   * \param ct Content-Type:文字列
   * \retval true 正しく解析できました。
   * \retval false 正しく解析できませんでした。
   */
  bool set(const char* ct);
  /*!
   * 設定されているContent-Type:文字列を返します。
   * \return Content-Type:文字列
   */
  const char* get() { return ct; }
  /*!
   * 主形式を返します
   * \return 主形式
   */
  MainType getMainType() const { return main_type; }
  /*!
   * 副形式を返します
   * \return 副形式
   */
  SubType getSubType() const { return sub_type; }
  /*!
   * 主形式の文字列を返します
   * \return 主形式文字列
   */
  const char* getMainTypeStr() { return main_type_str; }
  /*!
   * 副形式の文字列を返します
   * \return 副形式
   */
  const char* getSubTypeStr() { return sub_type_str; }
  /*!
   * パラメータの全文字列を返します。
   * \return パラメータ全文字列
   */
  const char* getParameter() { return parameter_str; }
  /*!
   * いくつのパラメータがあるかを得ます。
   * \return パラメータの数
   */
  const Size getParameterNumber() { return params.number(); }
  /*!
   * 引数で指定したパラメータを返します。
   * \param n パラメータの順番、0から始まる数字で指定します
   * \return ContentTypeParameterItem
   */
  const ContentTypeParameterItem* getNthParameter(Size n) { return params.vector(n); }
  /*!
   * パラメータでcharsetが指定された場合のCEIDを返します。
   * charsetが指定されていない場合やicemanで未サポートの
   * 文字セットエンコーディングが指定された場合にはCEID_NULL
   * が返ります。
   * \return CEID
   */
  //CEID getCharset();
  /*!
   * オブジェクトの状態を初期状態にします。
   */
  void clear();
 private:
  void setType();
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_fetch_ContentType_h */
