﻿// $Id$

#include <stdlib.h>
#include <string.h>
#include <ccc/base/TextWriter.h>
#include <ccc/base/TextReader.h>
#include <ccc/base/StringOFlow.h>
#include <ccc/base/NullOFlow.h>
#include <ccc/base/MemOFlow.h>
#include <ccc/posix/SocketIOFlow.h>
#include <ccc/fetch/Uri.h>
#include <ccc/fetch/HttpFetch.h>

CCC_NAMESPACE_START(CCC)

HttpFetch::HttpFetch()
{
  proxy = 0;
  proxy_port = 0;
  timeout = 10;
}

HttpFetch::~HttpFetch()
{
#ifdef _WIN32_WCE
  delete (void *) proxy;
#else /* _WIN32_WCE */
  delete proxy;
#endif /* _WIN32_WCE */
}

void
HttpFetch::setProxy(const char* proxy_)
{
#ifdef _WIN32_WCE
  delete (void *) proxy;
#else /* _WIN32_WCE */
  delete proxy;
#endif /* _WIN32_WCE */
  if (proxy_)
  {
    proxy = new char[strlen(proxy_) + 1];
    strcpy((char*)proxy, proxy_);
  }
  else
  {
    proxy = 0;
  }
}

void
HttpFetch::setProxyPort(int port)
{
  proxy_port = port;
}

void
HttpFetch::setTimeout(int timeout_)
{
  timeout = timeout_;
}

/*!
 * Get chunk data size from chunk line.
 * The chunk line is composed with hex numbers.
 */
static Size
getChunkSize(BString* line)
{
  Size ret = 0;
  char* p = line->getCString();
  char c;
  while ((c = *p++) != '\0')
  {
    Size k = 0;
    switch (c)
    {
     case '0':
     case '1':
     case '2':
     case '3':
     case '4':
     case '5':
     case '6':
     case '7':
     case '8':
     case '9':
      k = (Size)(c - '0');
      break;
      
     case 'a':
     case 'b':
     case 'c':
     case 'd':
     case 'e':
     case 'f':
      k = (Size)(c - 'a') + 10;
      break;

     case 'A':
     case 'B':
     case 'C':
     case 'D':
     case 'E':
     case 'F':
      k = (Size)(c - 'A') + 10;
      break;

     case 0x0d:
     case 0x0a:
      return ret;

     default:
      return 0;
    }
    ret = ret * 16 + k;
  }
  return ret;
}

HttpResponse*
HttpFetch::fetch(Uri* uri, HttpRequest* request, FetchStatus& status)
{
  HttpResponse* ret = fetch(uri, request, status, 0);
  return ret;
}

/*!
 * 指定サイズ分のデータを受信して、出力フローに書き込みます。
 * \param data_size データサイズ
 * \param r 受信元
 * \param out 出力先
 */
static void
receiveIntoOFlow(Size data_size, TextReader<Int8>* r, OFlow* out)
{
  const Size buffer_size = 1024 * 1024;	// 1MB
  Int8* buffer = new Int8[buffer_size];
  try
  {
    Size rest_size = data_size;
    while (rest_size > 0)
    {
      Size block_size = buffer_size;
      if (rest_size < buffer_size)
      {
	block_size = rest_size;
      }
      Size get_size = 0;
      r->getBlock(block_size, buffer, get_size);
      Size put_size = 0;
      out->putInt8Block(get_size, buffer, put_size);
      rest_size -= get_size;
    }
    delete[] buffer;
  }
  catch (IOException& ioe)
  {
    delete[] buffer;
    throw ioe;
  }
}

HttpResponse*
HttpFetch::fetch(Uri* uri, HttpRequest* request, FetchStatus& status, OFlow* out)
{
  HttpResponse* http_res = 0;
  if (uri->getScheme() != Uri::SCHEME_HTTP)
  {
    status = FS_URI_ERROR;
    return 0;
  }
  try
  {
    SocketIOFlow sock(IOTYPE_INT8);
    sock.setTimeoutSec(timeout);
    if (proxy && proxy_port != 0)
    {
      // use Proxy
      if (!sock.openAfInet(proxy_port, proxy))
      {
	status = FS_SOCKET_OPEN_ERROR;
	return 0;
      }
    }
    else
    {
      // direct connection
      if (!sock.openAfInet(uri->getPort(), uri->getHost()))
      {
	status = FS_SOCKET_OPEN_ERROR;
	return 0;
      }
    }
    // write header data
    {
      Iterator<HttpHeaderLine>* itr = request->makeHeaderLineIterator();
      HttpHeaderLine* hhl;
      while ((hhl = itr->next()))
      {
	BString* s = hhl->getHeaderLineString();
	sock.put(s);
	//sock.put("\r\n");
      }
      delete itr;
      sock.put("\r\n");
    }
    // write body data
    {
      Size put_size;
      sock.putInt8Block(request->getBodySize(), (const Int8*)request->getBodyData(), put_size);
    }
    
    // read response header
    TextReader<Int8> r(&sock);
    {
      BString* status_line = r.readLine();
      if (status_line == 0)
      {
	status = FS_INVALID_RESPONSE_HEADER_ERROR;
	return 0;
      }
      http_res = new HttpResponse();
      HttpHeaderLine* hhl = new HttpHeaderLine(status_line, HttpHeaderLine::FT_STATUS_LINE);
      http_res->add(hhl);
      delete status_line;
      BString* header_line = 0;
      for (;;)
      {
	/* 空行(CR+LRのみ)がくるまでヘッダを読み込みます。*/
	header_line = r.readLine();
	if (header_line == 0)
	{
	  delete http_res;
	  status = FS_INVALID_RESPONSE_HEADER_ERROR;
	  return 0;
	}
	if (header_line->strCmp("\r\n") == 0)
	{
	  delete header_line;
	  break;
	}
	hhl = new HttpHeaderLine(header_line);
	http_res->add(hhl);
	delete header_line;
      }
    }
    bool found_p = false;
    Size res_data_size = http_res->getContentLength(found_p);
    if (request->getRequestMethod() == RM_HEAD)
    {
      res_data_size = 0;
    }
    // read response data
    if (found_p && res_data_size > 0)
    {
      if (out)
      {
	receiveIntoOFlow(res_data_size, &r, out);
      }
      else
      {
	http_res->expandBody(res_data_size);
	Size get_size;
	r.getBlock(res_data_size, (Int8*)http_res->getBodyData(), get_size);
      }
    }
    else if (http_res->chunkedP())
    {
      /* Transfer-Encoding: chunkedが指定されたので、チャンクごとに読み込み */
      Size offset = 0;
      for (;;)
      {
	/* 1行バイト数を読み込み */
	BString* line = r.readLine();
	if (line->strCmp("\r\n") == 0)
	{
	  delete line;
	  line = r.readLine();
	}
	Size n = getChunkSize(line);
	//printf("chunk length:(%s)\n", line->getCString());
	if (n == 0)
	{
	  break;
	}
	delete line;
	if (out)
	{
	  receiveIntoOFlow(n, &r, out);
	}
	else
	{
	  /* 指定バイト数のデータを読み込み */
	  http_res->expandBody(offset + n);
	  Size get_size;
	  r.getBlock(n, (Int8*)http_res->getBodyData() + offset, get_size);
	  offset += n;
	}
      }
#if 0
      /* 結果の表示(デバック用) */
      {
	puts("--------------------------------------------------------------------------------\n");
	char* x = (char*)http_res->getBodyData();
	Size n;
	for (n = 0; n < offset; n++)
	{
	  putchar(*x++);
	}
	puts("\n");
	puts("--------------------------------------------------------------------------------\n");
      }
#endif
    }
  }
  catch (IOException& /*ioe*/)
  {
    delete http_res;
    status = FS_SOCKET_IO_ERROR;
    return 0;
  }
  status = FS_OK;
  return http_res;
}

CCC_NAMESPACE_END(CCC)
