﻿// $Id$

#include <string.h>
#include <unistd.h>
#include <ccc/base/base.h>
#include <ccc/base/TString.h>
#include <ccc/base/TextWriter.h>
#include <ccc/fetch/HttpFetch.h>
#include <ccc/fetch/Uri.h>
#include <ccc/http/HttpResponse.h>
#include <ccc/base/Formatter.h>
#include <ccc/http/HttpHeaderLine.h>
#include <ccc/fetch/HttpRangeFetch.h>

CCC_NAMESPACE_START(CCC)

HttpRangeFetch::HttpRangeFetch(int option)
{
  timeout = 10;
  chunk_size = 1024 * 100;	/* 100KB */
  user_agent.assign("CCC/3.1 (CYPAC)");
  proxy = 0;
  proxy_port = 0;
  retry_counts = DEFAULT_RETRY;
  this->option = option;
}

HttpRangeFetch::~HttpRangeFetch()
{
#ifdef _WIN32_WCE
  delete (void *) proxy;
#else /* _WIN32_WCE */
  delete proxy;
#endif /* _WIN32_WCE */
}

HttpRangeFetch::Status
HttpRangeFetch::getContentsSize(const char* url, BString* last_modified_timestamp, Size* data_size)
{
  HttpRangeFetch::Status status = RFS_OK;
  *data_size = 0;
  last_modified_timestamp->clear();
  HttpResponse* result = 0;
  do
  {
    // setup header
    Uri uri(url);
    HttpRequest request;
    BString request_line;
    {
      Fmt8 fmt("HEAD %s HTTP/1.1\r\n");
      if (option & O_PROXY_HOSTNAME_IN_URL)
      {
	fmt.setTP(url);
      }
      else
      {
	fmt.setTP(uri.getPath());
      }
      fmt.writeTo(request_line);
    }
    request.addMethod(&request_line);
    BString host_line;
    {
      Fmt8 fmt("Host: %s\r\n");
      fmt.setTP(uri.getHost());
      fmt.writeTo(host_line);
    }
    request.addHeader(&host_line);
    BString accept_line("Accept: */*\r\n");
    request.addHeader(&accept_line);
    BString accept_encoding("Accept-Encoding: identity\r\n");
    request.addHeader(&accept_encoding);
    BString user_agent_line;
    {
      Fmt8 fmt("User-Agent: %s\r\n");
      fmt.setTP(user_agent.getCString());
      fmt.writeTo(user_agent_line);
    }
    request.addHeader(&user_agent_line);
    // fetch
    HttpFetch http_fetch;
    http_fetch.setTimeout(timeout);
    if (proxy != 0)
    {
      http_fetch.setProxy(proxy);
      http_fetch.setProxyPort(proxy_port);
    }
    HttpFetch::FetchStatus fetch_status;
    result = http_fetch.fetch(&uri, &request, fetch_status);
    if (result == 0)
    {
      switch (fetch_status)
      {
       default:
       case HttpFetch::FS_OK:
	status = RFS_LOGICAL_ERROR;
	break;

       case HttpFetch::FS_URI_ERROR:
	status = RFS_URI_ERROR;
	break;

       case HttpFetch::FS_SOCKET_OPEN_ERROR:
       case HttpFetch::FS_SOCKET_IO_ERROR:
	status = RFS_NETWORK_ERROR;
	break;

       case HttpFetch::FS_INVALID_RESPONSE_HEADER_ERROR:
	status = RFS_HTTP_ERROR;
	break;
      }
      break;
    }
#if 0
    Iterator<HttpHeaderLine>* itr = result->makeHeaderLineIterator();
    HttpHeaderLine* line;
    while (line = itr->next())
    {
      //line->dump(stdout_w8);
      printf("%s", line->getHeaderLineString()->getCString());
    }
    delete itr;
#endif
    // Last-Modified: Thu, 17 Feb 2011 16:14:25 GMT
    HttpHeaderLine* last_modified = result->getTypedHeaderLine(HttpHeaderLine::FT_LAST_MODIFIED);
    if (last_modified == 0)
    {
      status = RFS_HTTP_ERROR;
      break;
    }
    last_modified_timestamp->assign(*last_modified->getValue2());
    bool found_p = false;
    Size size = result->getContentLength(found_p);
    if (!found_p)
    {
      status = RFS_HTTP_ERROR;
      size = 0;
    }
    *data_size = size;
  }
  while (false);
  delete result;
  return status;
}

HttpRangeFetch::Status
HttpRangeFetch::downloadChunk(const char* url, const BString* timestamp, FILE* fp, Size from, Size to)
{
  //printf("downloadChunk url:%s timestamp:%s from:%u to:%u\n", url, timestamp->getCString(), (unsigned int)from, (unsigned int)to);
  if (!startDownloadChunk(from, to))
  {
    return RFS_CANCELED;
  }
  HttpRangeFetch::Status status = RFS_OK;
  HttpResponse* result = 0;
  do
  {
    // setup header
    Uri uri(url);
    HttpRequest request;
    BString request_line;
    {
      Fmt8 fmt("GET %s HTTP/1.1\r\n");
      if (option & O_PROXY_HOSTNAME_IN_URL)
      {
	fmt.setTP(url);
      }
      else
      {
	fmt.setTP(uri.getPath());
      }
      fmt.writeTo(request_line);
    }
    request.addMethod(&request_line);
    BString host_line;
    {
      Fmt8 fmt("Host: %s\r\n");
      fmt.setTP(uri.getHost());
      fmt.writeTo(host_line);
    }
    request.addHeader(&host_line);
    BString accept_line("Accept: */*\r\n");
    request.addHeader(&accept_line);
    BString range;
    {
      Fmt8 fmt("Range: bytes=%u-%u\r\n");
      fmt.setUI((unsigned int)from);
      fmt.setUI((unsigned int)to);
      fmt.writeTo(range);
    }
    request.addHeader(&range);
    BString user_agent_line;
    {
      Fmt8 fmt("User-Agent: %s\r\n");
      fmt.setTP(user_agent.getCString());
      fmt.writeTo(user_agent_line);
    }
    request.addHeader(&user_agent_line);
    // fetch
    HttpFetch http_fetch;
    http_fetch.setTimeout(timeout);
    if (proxy != 0)
    {
      http_fetch.setProxy(proxy);
      http_fetch.setProxyPort(proxy_port);
    }
    HttpFetch::FetchStatus fetch_status;
    result = http_fetch.fetch(&uri, &request, fetch_status);
    if (result == 0)
    {
      switch (fetch_status)
      {
       default:
       case HttpFetch::FS_OK:
	status = RFS_LOGICAL_ERROR;
	break;

       case HttpFetch::FS_URI_ERROR:
	status = RFS_URI_ERROR;
	break;

       case HttpFetch::FS_SOCKET_OPEN_ERROR:
       case HttpFetch::FS_SOCKET_IO_ERROR:
	status = RFS_NETWORK_ERROR;
	break;

       case HttpFetch::FS_INVALID_RESPONSE_HEADER_ERROR:
	status = RFS_HTTP_ERROR;
	break;
      }
      break;
    }
#if 0
    Iterator<HttpHeaderLine>* itr = result->makeHeaderLineIterator();
    HttpHeaderLine* line;
    while (line = itr->next())
    {
      //line->dump(stdout_w8);
      printf("%s", line->getHeaderLineString()->getCString());
    }
    delete itr;
#endif
    // Last-Modified: Thu, 17 Feb 2011 16:14:25 GMT
    HttpHeaderLine* last_modified = result->getTypedHeaderLine(HttpHeaderLine::FT_LAST_MODIFIED);
    if (last_modified == 0)
    {
      status = RFS_HTTP_ERROR;
      break;
    }
    if (timestamp->strCmp(last_modified->getValue2()) != 0)
    {
      // the timestamp Last-Modified is not matched.
      status = RFS_DATA_UPDATED_ERROR;
      break;
    }
    Size size = result->getBodySize();
    if (size != (to - from + 1))
    {
      status = RFS_NETWORK_ERROR;
      break;
    }
    //printf("# body data size:%u\n", (unsigned int)size);
    if (size > 0)
    {
      size_t n = fwrite(result->getBodyData(), size, 1, fp);
      if (n != 1)
      {
	status = RFS_FILE_WRITE_ERROR;
	break;
      }
    }
    endDownloadChunk(from, to);
  }
  while (false);
  delete result;
  return status;
}

bool
HttpRangeFetch::startDownloadChunk(Size from, Size to)
{
  return true;
}

void
HttpRangeFetch::endDownloadChunk(Size from, Size to)
{
}

HttpRangeFetch::Status
HttpRangeFetch::fetch(const char* url, FILE* fp, BString* timestamp, Size* data_size)
{
  timestamp->clear();
  *data_size = 0;
  HttpRangeFetch::Status status;
  Size size;
  unsigned int retry = 0;
  do
  {
    if (retry != 0)
    {
      sleep(RETRY_SLEEP_SEC);
    }
    status = getContentsSize(url, timestamp, &size);
    if (++retry >= retry_counts)
    {
      break;
    }
  }
  while (status == RFS_NETWORK_ERROR);
  if (status != RFS_OK)
  {
    return status;
  }
  *data_size = size;
  Size num_of_full_chunk = size / chunk_size;
  Size rest_chunk_size = size % chunk_size;
  Size chunk;
  for (chunk = 0; chunk < num_of_full_chunk; chunk++)
  {
    Size from = chunk * chunk_size;
    Size to = (chunk + 1) * chunk_size - 1;
    retry = 0;
    do
    {
      if (retry != 0)
      {
	sleep(RETRY_SLEEP_SEC);
      }
      status = downloadChunk(url, timestamp, fp, from, to);
      if (++retry >= retry_counts)
      {
	break;
      }
    }
    while (status == RFS_NETWORK_ERROR);
    if (status != RFS_OK)
    {
      return status;
    }
  }
  if (rest_chunk_size > 0)
  {
    Size from = chunk * chunk_size;
    Size to = from + rest_chunk_size - 1;
    retry = 0;
    do
    {
      if (retry != 0)
      {
	sleep(RETRY_SLEEP_SEC);
      }
      status = downloadChunk(url, timestamp, fp, from, to);
      if (++retry >= retry_counts)
      {
	break;
      }
    }
    while (status == RFS_NETWORK_ERROR);
    if (status != RFS_OK)
    {
      return status;
    }
  }
  return RFS_OK;
}

HttpRangeFetch::Status
HttpRangeFetch::resumeFetch(const char* url, FILE* fp, const BString* timestamp, Size data_size, Size downloaded_size)
{
  BString current_timestamp;
  Size size;
  HttpRangeFetch::Status status;
  unsigned int retry = 0;
  do
  {
    if (retry != 0)
    {
      sleep(RETRY_SLEEP_SEC);
    }
    status = getContentsSize(url, &current_timestamp, &size);
    if (++retry >= retry_counts)
    {
      break;
    }
  }
  while (status == RFS_NETWORK_ERROR);
  if (status != RFS_OK)
  {
    return status;
  }
  if ((timestamp->strCmp(current_timestamp) != 0) ||
      (data_size != size))
  {
    return RFS_DATA_UPDATED_ERROR;
  }
  Size check_from = downloaded_size;
  Size num_of_full_chunk = size / chunk_size;
  Size rest_chunk_size = size % chunk_size;
  Size chunk;
  /* スタートチャンク */
  Size start_chunk_block_size = chunk_size - (downloaded_size % chunk_size);
  Size start_chunk_from = downloaded_size;
  Size start_chunk_to = downloaded_size + start_chunk_block_size - 1;
  Size start_chunk_num = (downloaded_size / chunk_size) + 1;
  Size to = 0;
  if (start_chunk_num <= num_of_full_chunk)
  {
//#ifdef DUMP
//    printf("< from:%u to:%u\n", (unsigned int)start_chunk_from, (unsigned int)start_chunk_to);
//#endif
    assert(start_chunk_from <= start_chunk_to);
    assert(start_chunk_from == check_from);
    retry = 0;
    do
    {
      if (retry != 0)
      {
	sleep(RETRY_SLEEP_SEC);
      }
      status = downloadChunk(url, timestamp, fp, start_chunk_from, start_chunk_to);
      if (++retry >= retry_counts)
      {
	break;
      }
    }
    while (status == RFS_NETWORK_ERROR);
    if (status != RFS_OK)
    {
      return status;
    }
    check_from = start_chunk_to + 1;
    to = start_chunk_to;
  }
  /* 固定サイズのチャンク */
  for (chunk = start_chunk_num; chunk < num_of_full_chunk; chunk++)
  {
    Size from = chunk * chunk_size;
    to = (chunk + 1) * chunk_size - 1;
//#ifdef DUMP
//    printf("= from:%u to:%u\n", (unsigned int)from, (unsigned int)to);
//#endif
    assert(from <= to);
    assert(from == check_from);
    retry = 0;
    do
    {
      if (retry != 0)
      {
	sleep(RETRY_SLEEP_SEC);
      }
      status = downloadChunk(url, timestamp, fp, from, to);
      if (++retry >= retry_counts)
      {
	break;
      }
    }
    while (status == RFS_NETWORK_ERROR);
    if (status != RFS_OK)
    {
      return status;
    }
    check_from = to + 1;
  }
  /* 最後のチャンク */
  if (downloaded_size >= num_of_full_chunk * chunk_size)
  {
    Size from = downloaded_size;
    to = size - 1;
    if (from <= to)
    {
//#ifdef DUMP
//      printf("@ from:%u to:%u\n", (unsigned int)from, (unsigned int)to);
//#endif
      assert(from <= to);
      assert(from == check_from);
      retry = 0;
      do
      {
	if (retry != 0)
	{
	  sleep(RETRY_SLEEP_SEC);
	}
	status = downloadChunk(url, timestamp, fp, from, to);
	if (++retry >= retry_counts)
	{
	  break;
	}
      }
      while (status == RFS_NETWORK_ERROR);
      if (status != RFS_OK)
      {
	return status;
      }
    }
  }
  else if (rest_chunk_size > 0)
  {
    Size from = chunk * chunk_size;
    to = from + rest_chunk_size - 1;
//#ifdef DUMP
//    printf("> from:%u to:%u\n", (unsigned int)from, (unsigned int)to);
//#endif
    assert(from <= to);
    assert(from == check_from);
    status = downloadChunk(url, timestamp, fp, from, to);
    if (status != RFS_OK)
    {
      return status;
    }
  }
  assert(to == size - 1);
  return RFS_OK;
}

void
HttpRangeFetch::setProxy(const char* proxy_)
{
#ifdef _WIN32_WCE
  delete (void *) proxy;
#else /* _WIN32_WCE */
  delete proxy;
#endif /* _WIN32_WCE */
  if (proxy_)
  {
    proxy = new char[strlen(proxy_) + 1];
    strcpy((char*)proxy, proxy_);
  }
  else
  {
    proxy = 0;
  }
}

void
HttpRangeFetch::setProxyPort(int port)
{
  proxy_port = port;
}

CCC_NAMESPACE_END(CCC)
