﻿// $Id$

#ifndef INCLUDE_ccc_fetch_HttpRangeFetch_h
#define INCLUDE_ccc_fetch_HttpRangeFetch_h

#include <stdio.h>
#include <ccc/base/base.h>
#include <ccc/base/TString.h>

CCC_NAMESPACE_START(CCC)

/*!
 * HTTPのRangeリクエストを使ってダウンロードする機能を提供します。
 */
class HttpRangeFetch
{
 public:
  /*!
   * エラー等のステータス
   */
  enum Status
  {
    RFS_OK,			/* 正常 */
    RFS_URI_ERROR,		/* URIのエラー(HTTPではない場合等) */
    RFS_NETWORK_ERROR,		/* ネットワークのエラー */
    RFS_HTTP_ERROR,		/* HTTPプロトコルのエラー */
    RFS_FILE_WRITE_ERROR,	/* ファイル書き込みエラー */
    RFS_DATA_UPDATED_ERROR,	/* データが更新されています。*/
    RFS_LOGICAL_ERROR,		/* その他のロジックエラー */
    RFS_CANCELED,		/* キャンセルされました。*/
  };
  enum
  {
    DEFAULT_RETRY = 3,		/* デフォルトのリトライ数 */
    RETRY_SLEEP_SEC = 2,	/* リトライごとの待ち時間(秒) */
  };
  /*!
   * オプション
   */
  enum Option
  {
    /*!
     * オプション無し
     */
    O_NONE = 0,
    /*!
     * proxyに対してのGET, HEADリクエストURLにホスト名を含めます。
     */
    O_PROXY_HOSTNAME_IN_URL = 0x01,
  };
 private:
  /*!
   * タイムアウト(秒)
   */
  int timeout;
  /*!
   * チャンクサイズ(Bytes)
   */
  Size chunk_size;
  /*!
   * User-Agent: の文字列
   */
  BString user_agent;
  /*!
   * プロキシアドレス
   */
  const char* proxy;
  /*!
   * プロキシのポート番号
   */
  int proxy_port;
  /*!
   * リトライ数
   */
  unsigned int retry_counts;
  /*!
   * 動作オプション
   */
  int option;
 public:
  /*!
   * コンストラクタ
   * \param option 動作オプション、enum Optionのビットパターンで
   * 指定します。
   */
  HttpRangeFetch(int option);
  /*!
   * デストラクタ
   */
  ~HttpRangeFetch();
  /*!
   * リクエストのタイムアウト(秒)を設定します。
   * \param t タイムアウト(秒)
   */
  void setTimeout(int t) { timeout = t; }
  /*!
   * チャンクサイズを指定します。
   * \param size チャンクサイズ(Bytes)
   */
  void setChunkSize(Size size) { chunk_size = size; }
  /*!
   * User-Agentの文字列を指定します。
   * \param str 'User-Agent: 'の後ろに指定する文字列
   */
  void setUserAgent(const char* str) { user_agent.assign(str); }
  /*!
   * リトライ数をセットします。
   * \param n リトライ数
   */
  void setRetryCount(unsigned n) { retry_counts = n; }
  /*
   * HEADリクエストで、データのサイズを確認します。
   * \param url URL
   * \param timestamp URLのリソースのLast-Modified情報がセットされます。
   * \return データのサイズ, 0の場合はエラーを意味します。
   */
  Status getContentsSize(const char* url, BString* timestamp, Size* data_size);
  /*
   * Rangeリクエストで、チャンクデータをダウンロードします。
   * \param url URL
   * \param timestamp URLのリソースのLast-Modified情報, 取得したLast-Modifiedが異なった場合はエラーになります。
   * \param fp データ保存先ファイル
   * \param from データの取得開始オフセット
   * \param to データの終了位置(このオフセットのデータまで取得されます)
   * \retval true 成功
   * \retval false 失敗
   */
  Status downloadChunk(const char* url, const BString* timestamp, FILE* fp, Size from, Size to);
  /*!
   * チャンクのダウンロード開始を通知します。
   * クラスを継承しオリジナルの処理を実装することを想定しています。
   * \param from データの取得開始オフセット
   * \param to データの終了位置(このオフセットのデータまで取得されます)
   * \retval false チャンクのダウンロードを中止します。(RFS_CANCELEDが返ります)
   * \retval true ダウンロードを行います。
   */
  virtual bool startDownloadChunk(Size from, Size to);
  /*!
   * チャンクのダウンロード終了を通知します。
   * クラスを継承しオリジナルの処理を実装することを想定しています。
   * \param from データの取得開始オフセット
   * \param to データの終了位置(このオフセットのデータまで取得されます)
   */
  virtual void endDownloadChunk(Size from, Size to);
  /*!
   * データをRangeリクエストを使ってダウンロードします。
   * \param url URL
   * \param fp データの保存先ファイル
   * \param timestamp データのタイムスタンプ
   * \param data_size データのサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  Status fetch(const char* url, FILE* fp, BString* timestamp, Size* data_size);
  /*!
   * データのダウンロードをRangeリクエストを使って再開します。
   * \param url URL
   * \param fp データの保存先ファイル
   * \param timestamp 前回の元タイムスタンプ
   * \param data_size 前回のデータサイズ
   * \param downloaded_size ダウンロード済みのサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  Status resumeFetch(const char* url, FILE* fp, const BString* timestamp, Size data_size, Size downloaded_size);
  /*!
   * プロキシのアドレスを指定します。
   * 文字列は内部でコピーされます。
   * \param proxy アドレス
   */
  void setProxy(const char* proxy);
  /*!
   * プロキシのポート番号
   * \param port ポート番号
   */
  void setProxyPort(int port);
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_fetch_HttpRangeFetch_h */
