﻿// $Id$

#ifndef INCLUDE_ccc_fetch_Uri_h
#define INCLUDE_ccc_fetch_Uri_h

#include <ccc/base/base.h>
#include <ccc/base/TString.h>

CCC_NAMESPACE_START(CCC)

/*!
 * URIを解析するクラスです。
 * URIについての詳細はRFC1630に記載されています。
 *
 * 現在HTTP,FILE,DEBUGのURIのみがサポートされています。
 */
class Uri
{
 public:
  /*!
   * オブジェクトにアクセスする方法を示します。
   */
  enum Scheme
  {
    /*!
     * スキーム不明
     */
    SCHEME_UNKNOWN,
    /*!
     * デバック用
     */
    SCHEME_DEBUG,
    // RFC1738
    /*!
     * FTP
     */
    SCHEME_FTP,
    /*!
     * HTTP
     */
    SCHEME_HTTP,
    /*!
     * The Gopher protocol
     */
    SCHEME_GOPHER,
    /*!
     * Electronic mail address
     */
    SCHEME_MAILTO,
    /*!
     * USENET news
     */
    SCHEME_NEWS,
    /*!
     * USENET news using NNTP access
     */
    SCHEME_NNTP,
    /*!
     * Reference to interactive sessions
     */
    SCHEME_TELNET,
    /*!
     * Wide Area Information servers
     */
    SCHEME_WAIS,
    /*!
     * Host-specific file names
     */
    SCHEME_FILE,
    /*!
     * Prospero Directory Service
     */
    SCHEME_PROSPERO,
    /*!
     * HTTPS
     */
    SCHEME_HTTPS,
    // RFC2111
    /*!
     * Content ID
     * cid:content-id
     */
    SCHEME_CID,
    /*!
     * Message ID
     * mid:message-id
     */
    SCHEME_MID,
    /*!
     * JavaScript
     */
    SCHEME_JAVASCRIPT,
  };

 private:
  /*!
   * スキーム
   */
  Scheme scheme;
  /*!
   * 解析対象のURIをコピーしたもの
   */
  char* uri;
  /*!
   * ポート
   */
  int port;
  /*!
   * 接続先ホスト
   */
  char* host;
  /*!
   * ファイルのパス
   */
  char* path;

 public:
  /*!
   * コンストラクタ
   * \param uri 解析するURI
   */
  Uri(const char* uri = 0);
  /*!
   * コンストラクタ
   * \param uri 解析するURI
   * \param parent_uri 親URI
   */
  Uri(const char* uri, const char* parent_uri);
  /*!
   * コンストラクタ
   * \param uri 解析するURI
   * \param parent_uri 親URI
   */
  Uri(const char* uri, const Uri* parent_uri);
  /*!
   * デストラクタ
   */
  ~Uri();
  /*!
   * 解析結果のプロトコルのプロトコルを返します。
   * \return プロトコル
   */
  Scheme getScheme() const { return scheme; }
  /*!
   * 解析結果のプロトコルのプロトコル名を返します。
   * \return プロトコル名文字列
   */
  const char* getSchemeName();
  /*!
   * URI
   * \return URI
   */
  const char* getUri() const { return uri; }
  /*!
   * 解析結果のポート番号
   */
  int getPort() const { return port; }
  /*!
   * 解析結果のホスト文字列
   */
  const char* getHost() const { return host; }
  /*!
   * 解析結果のファイルのパス
   */
  const char* getPath() const { return path; }
  /*!
   * URIをセットし解析します。
   * \param uri URI
   */
  void setUri(const char* uri);
  /*!
   * URIをセットし解析します。
   * \param uri URI
   * \param parent_uri 親URI
   */
  void setUri(const char* uri, const char* parent_uri);
  /*!
   * URIをセットし解析します。
   * \param uri URI
   * \param parent_uri 親URI
   */
  void setUri(const char* uri, const Uri* parent_uri);
  /*!
   * 正しいURI形式かどうかを調べます。
   * \retval true 正しいURI
   * \retval false 不正なURI
   */
  bool validUriP() const;
  /*!
   * HTTPのURI形式かどうかを調べます。
   * \retval true HTTPのURI
   * \retval false HTTPのURIではない
   */
  bool validHttpUriP() const;
  /*!
   * HTTPSのURI形式かどうかを調べます。
   * \retval true HTTPのURI
   * \retval false HTTPのURIではない
   */
  bool validHttpsUriP() const;
  /*!
   * ファイルのURI形式かどうかを調べます。
   * \retval true ファイルのURI
   * \retval false ファイルのURIではない
   */
  bool validFileUriP() const;
  /*!
   * デバック用URI形式かどうかを調べます。
   * \retval true デバック用URI
   * \retval false デバック用URIではない
   */
  bool validDebugUriP() const;
  /*!
   * パスが絶対パスかどうかを調べます。
   * \return ture 絶対パス
   * \return valse 相対パス
   */
  bool absolutePathP() const;
#if 0
  /* TODO:
   * mailtoとnews以外の共通的なシンタックス
   * //<user>:<password>@<host>:<port>/<url-path>
   */
  const char* getUserName();
  const char* getPassword();
#endif
  /*!
   * オブジェクトの状態を初期状態にします。
   */
  void clear();
  /*!
   * 正規化した後のURI文字列を取得します。
   * 返り値は呼び出し側で削除してください。
   * \return URI文字列
   */
  CCC::BString* getFormalUri() const;
  /*!
   * 正規化した後のURI文字列を#REF以降を除いた形式で取得します。
   * 返り値は呼び出し側で削除してください。
   * \return URI文字列
   */
  CCC::BString* getFormalUriWithoutLocalRef() const;
  /*!
   * 正規化した後のURI文字列の#REF以降を取得します。
   * 返り値は呼び出し側で削除してください。
   * \return ローカルの参照文字列
   */
  CCC::BString* getLocalRef() const;
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_fetch_Uri_h */
