﻿// $Id$

#include <assert.h>
#include <ccc/file/FileIFlow.h>
#include <ccc/file/PathString.h>

CCC_NAMESPACE_START(CCC)

FileIFlow::FileIFlow()
  : IFlow(IOTYPE_ANY)
{
  fp = 0;
  attached_p = false;
}

FileIFlow::~FileIFlow()
{
  if (fp)
  {
    close();
  }
}

bool
FileIFlow::open(const PathString* path, bool binary_p)
{
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  return open((wchar_t*)path->getCString(), binary_p);
#  else /* _WIN32 */
  assert(false);
  return false;	/* not supported */
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
  return open(path->getCString(), binary_p);
#endif /* CCC_UTF16_PATHSTRING */
}

bool
FileIFlow::open(const char* path, bool binary_p)
{
  fp = fopen(path, binary_p ? "rb" : "r");
  setTopP(true);
  return !!fp;
}

#ifdef _WIN32
bool
FileIFlow::open(const wchar_t* path, bool binary_p)
{
  wchar_t mode_binary[] = { 'r', 'b', 0 };
  wchar_t mode_normal[] = { 'r', 0 };
  fp = _wfopen(path, binary_p ? mode_binary : mode_normal);
  setTopP(true);
  return !!fp;
}
#endif /* _WIN32 */

void
FileIFlow::attach(FILE* fp_)
{
  fp = fp_;
  attached_p = true;
  setTopP(false);
}

bool
FileIFlow::close()
{
  if (!openedP())
  {
    return false;
  }
  bool ret = true;
  if (!attached_p)
  {
    ret = !fclose(fp);
  }
  fp = 0;
  return ret;
}

Int8
FileIFlow::getInt8() CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  int c = fgetc(fp);
  if (c == EOF)
  {
    throw IOException(__FILE__, __LINE__, IOException::READ_BEYOND_THE_EOF);
  }
  setTopP(false);
  return (Int8)c;
}

UInt16
FileIFlow::getUInt16() CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  // native endian
  UInt16 ret;
  UInt8* p = (UInt8*)&ret;
  size_t i;
  for (i = 0; i < sizeof(UInt16); i++)
  {
    int c = fgetc(fp);
    if (c == EOF)
    {
      throw IOException(__FILE__, __LINE__, IOException::READ_BEYOND_THE_EOF);
    }
    *p++ = (UInt8)c;
  }
  setTopP(false);
  return ret;
}

UInt32
FileIFlow::getUInt32() CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  // native endian
  UInt32 ret;
  UInt8* p = (UInt8*)&ret;
  size_t i;
  for (i = 0; i < sizeof(UInt32); i++)
  {
    int c = fgetc(fp);
    if (c == EOF)
    {
      throw IOException(__FILE__, __LINE__, IOException::READ_BEYOND_THE_EOF);
    }
    *p++ = (UInt8)c;
  }
  setTopP(false);
  return ret;
}

void
FileIFlow::getInt8Block(Size block_size, Int8* ptr, Size& get_size) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  get_size = (Size)fread((void*)ptr, 1, block_size, fp);
  if (get_size != block_size)
  {
    if (feof(fp))
    {
      throw IOException(__FILE__, __LINE__, IOException::READ_BEYOND_THE_EOF);
    }
    throw IOException(__FILE__, __LINE__, IOException::FILE_IO_ERROR);
  }
  setTopP(false);
}

void
FileIFlow::getUInt16Block(Size block_size, UInt16* ptr, Size& get_size) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  get_size = (Size)fread((void*)ptr, sizeof(UInt16), block_size, fp);
  if (get_size != block_size)
  {
    if (feof(fp))
    {
      throw IOException(__FILE__, __LINE__, IOException::READ_BEYOND_THE_EOF);
    }
    throw IOException(__FILE__, __LINE__, IOException::FILE_IO_ERROR);
  }
  setTopP(false);
}

void
FileIFlow::getUInt32Block(Size block_size, UInt32* ptr, Size& get_size) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  get_size = (Size)fread((void*)ptr, sizeof(UInt32), block_size, fp);
  if (get_size != block_size)
  {
    if (feof(fp))
    {
      throw IOException(__FILE__, __LINE__, IOException::READ_BEYOND_THE_EOF);
    }
    throw IOException(__FILE__, __LINE__, IOException::FILE_IO_ERROR);
  }
  setTopP(false);
}

bool
FileIFlow::rewind()
{
  if (!openedP())
  {
    return false;
  }
  assert(fp);
  int ret = fseek(fp, 0L, SEEK_SET);
  if (ret != 0)
  {
    return false;
  }
  return IFlow::rewind();
}

bool
FileIFlow::openedP()
{
  return fp != 0;
}

CCC_NAMESPACE_END(CCC)
