﻿// $Id$

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef _WIN32
// for Windows
#  include <windows.h>
#  include <wchar.h>
#  include <io.h>
#else /* _WIN32 */
// for POSIX system
#  include <dirent.h>
#  include <unistd.h>
#  include <pwd.h>
#  include <grp.h>
#  include <dirent.h>
#endif /* _WIN32 */

#include <ccc/file/FileManager.h>
#include <ccc/base/Formatter.h>
#include <ccc/base/ArrayList.h>
#include <ccc/iceman/Iceman.h>

CCC_NAMESPACE_START(CCC);

bool
FileManager::fileExistsAtPath(const PathString* path, bool* dir_p)
{
#ifdef _WIN32
#  ifdef CCC_UTF16_PATHSTRING
  DWORD dwAttrib = GetFileAttributesW((wchar_t*)path->getCString());
#  else /* CCC_UTF16_PATHSTRING */
  DWORD dwAttrib = GetFileAttributesA(path->getCString());
#  endif /* CCC_UTF16_PATHSTRING */
  if (dwAttrib != INVALID_FILE_ATTRIBUTES)
  {
    if (dwAttrib & FILE_ATTRIBUTE_DIRECTORY)
    {
      *dir_p = true;
      return true;
    }
    *dir_p = false;
    return true;
  }
  return false;
#else /* _WIN32 */
  *dir_p = false;
  struct stat info;
  int ret = stat(path->getCString(), &info);
  if (ret != 0)
  {
    return false;
  }
  if (S_ISDIR(info.st_mode))
  {
    *dir_p = true;
    return true;
  }
  /* in case of regular file, pipe, special file, sybolic link or socket */
  return true;
#endif /* _WIN32 */
}

bool
FileManager::copyFile(const PathString* src, const PathString* dest, Status* status)
{
  const PathChar* from = src->getCString();
  const PathChar* to = dest->getCString();
  bool ret_value = false;
  int from_fd = 0;
  int to_fd = 0;
  const off_t buf_size = 1024 * 1024;	/* 1M */
  PathChar* buf = new PathChar[(size_t)buf_size];
  do
  {
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
    from_fd = _wopen((wchar_t*)from, _O_RDONLY);
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
    from_fd = open(from, O_RDONLY);
#endif /* CCC_UTF16_PATHSTRING */
    if (from_fd < 0)
    {
      //printf("can not open from file:%s\n", from);
      *status = FMS_ERROR_ITEM_CAN_NOT_OPEN;
      break;
    }
    struct stat from_info;
    if (fstat(from_fd, &from_info) != 0)
    {
      //printf("can not get information of from file:%s\n", from);
      *status = FMS_ERROR_ITEM_CAN_NOT_GET_STATUS;
      break;
    }
    off_t from_size = from_info.st_size;
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
    to_fd = _wopen((wchar_t*)to, _O_WRONLY | _O_CREAT | _O_TRUNC, from_info.st_mode);
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
#  ifdef _WIN32
    to_fd = _open(to, _O_WRONLY | _O_CREAT | _O_TRUNC, from_info.st_mode);
#  else /* _WIN32 */
    to_fd = open(to, O_WRONLY | O_CREAT | O_TRUNC, from_info.st_mode);
#  endif /* _WIN32 */
#endif /* CCC_UTF16_PATHSTRING */
    if (to_fd < 0)
    {
      //printf("can not open to file:%s\n", to);
      *status = FMS_ERROR_DEST_CAN_NOT_OPEN;
      break;
    }
    while (from_size != 0)
    {
      off_t block_size = buf_size;
      if (from_size < buf_size)
      {
	block_size = from_size;
      }
#ifdef _WIN32
      int read_size = _read(from_fd, buf, block_size);
#else /* _WIN32 */
      ssize_t read_size = read(from_fd, buf, block_size);
#endif /* _WIN32 */
      if (read_size != block_size)
      {
	if (read_size == 0)
	{
	  *status = FMS_ERROR_ITEM_READ;
	  break;
	}
	block_size = read_size;
      }
#ifdef _WIN32
      int write_size = _write(to_fd, buf, block_size);
#else /* _WIN32 */
      ssize_t write_size = write(to_fd, buf, block_size);
#endif /* _WIN32 */
      if (write_size != block_size)
      {
	//printf("write error\n");
	*status = FMS_ERROR_DEST_WRITE;
	break;
      }
      from_size -= block_size;
    }
    if (from_size == 0)
    {
      ret_value = true;
      *status = FMS_SUCCESS;
    }
  }
  while (false);
  if (from_fd > 0)
  {
#ifdef _WIN32
    _close(from_fd);
#else /* _WIN32 */
    close(from_fd);
#endif /* _WIN32 */
  }
  if (to_fd > 0)
  {
#ifdef _WIN32
    _close(to_fd);
#else /* _WIN32 */
    close(to_fd);
#endif /* _WIN32 */
  }
  delete [] buf;
  return ret_value;
}

bool
FileManager::copyDirectory(const PathString* src, const PathString* dest, Status* status)
{
  bool dir_p;
  if (!FileManager::fileExistsAtPath(src, &dir_p))
  {
    *status = FMS_ERROR_ITEM_DOES_NOT_EXIST;
    return false;
  }
  if (!dir_p)
  {
    *status = FMS_ERROR_ITEM_IS_NOT_DIRECTORY;
    return false;
  }
  // TODO: not yet implemented!
  *status = FMS_ERROR_OTHER;
  return false;
}

bool
FileManager::removeFile(const PathString* path, Status* status)
{
  bool dir_p;
  if (!FileManager::fileExistsAtPath(path, &dir_p))
  {
    *status = FMS_ERROR_ITEM_DOES_NOT_EXIST;
    return false;
  }
  if (dir_p)
  {
    *status = FMS_ERROR_ITEM_IS_NOT_FILE;
    return false;
  }
#ifdef _WIN32
#  ifdef CCC_UTF16_PATHSTRING
  if (::DeleteFileW((wchar_t*)path->getCString()) == 0)
#else /* CCC_UTF16_PATHSTRING */
  if (::DeleteFileA(path->getCString()) != 0)
#endif /* CCC_UTF16_PATHSTRING */
#else /* _WIN32 */
  if (unlink(path->getCString()) != 0)
#endif /* _WIN32 */
  {
    *status = FMS_ERROR_OTHER;	// TODO: more detail error detection is needed.
    return false;
  }
  *status = FMS_SUCCESS;
  return true;
}

bool
FileManager::removeDirectory(const PathString* path, Status* status)
{
  bool dir_p;
  if (!FileManager::fileExistsAtPath(path, &dir_p))
  {
    *status = FMS_ERROR_ITEM_DOES_NOT_EXIST;
    return false;
  }
  if (!dir_p)
  {
    *status = FMS_ERROR_ITEM_IS_NOT_DIRECTORY;
    return false;
  }
#ifdef _WIN32
#  ifdef CCC_UTF16_PATHSTRING
  PathString pattern(*path);
  pattern.add(directory_separator);
  pattern.add('*');
  WIN32_FIND_DATAW fdata;
  HANDLE hfind = ::FindFirstFileW((wchar_t*)pattern.getCString(), &fdata);
  if (hfind == INVALID_HANDLE_VALUE)
  {
    return false;
  }
  bool ret = true;
  do
  {
    PathString sub_path(*path);
    sub_path.appendPathComponent((PathChar*)fdata.cFileName);
    if (fdata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
    {
      if ((wcscmp(L".", fdata.cFileName) != 0) &&
	  (wcscmp(L"..", fdata.cFileName) != 0))
      {
	if (!removeDirectory(&sub_path, status))
	{
	  ret = false;
	  break;
	}
      }
    }
    else
    {
      if (!removeFile(&sub_path, status))
      {
	ret = false;
	*status = FMS_ERROR_ITEM_CAN_NOT_REMOVE;
	break;
      }
    }
  }
  while (::FindNextFileW(hfind, &fdata));
  ::FindClose(hfind);
  if (ret)
  {
    ret = ::RemoveDirectoryW((wchar_t*)path->getCString()) != 0;
    if (!ret)
    {
      *status = FMS_ERROR_ITEM_CAN_NOT_REMOVE;
    }
  }
  *status = FMS_SUCCESS;
  return ret;
#  else /* CCC_UTF16_PATHSTRING */
#    error TODO
#  endif /* CCC_UTF16_PATHSTRING */
#else /* _WIN32 */
  DIR* dir = opendir(path->getCString());
  if (!dir)
  {
    // you might not have access right.
    *status = FMS_ERROR_ITEM_CAN_NOT_OPEN;
    return false;
  }
  bool ret_p = true;
  struct dirent* ent;
  while ((ent = readdir(dir)) != NULL)
  {
    PathString sub_path(ent->d_name);
    PathString full_path;
    full_path.assignWithAppendingPathComponent(path, &sub_path);
    struct stat s;
    //if (trace_symbolic_link_p)
    //{
    //  stat(full_path, &s);
    //}
    //else
    //{
    lstat(full_path.getCString(), &s);
    //}
    if (S_ISDIR(s.st_mode))
    {
      if ((strcmp(ent->d_name, ".") != 0) &&
	  (strcmp(ent->d_name, "..") != 0))
      {
	if (!FileManager::removeDirectory(&full_path, status))
	{
	  ret_p = false;
	  break;
	}
      }
    }
    else
    {
      if (unlink(full_path.getCString()) != 0)
      {
	*status = FMS_ERROR_ITEM_CAN_NOT_REMOVE;
	ret_p = false;
	break;
      }
    }
  }
  closedir(dir);
  if (!ret_p)
  {
    return false;
  }
  if (rmdir(path->getCString()) != 0)
  {
    // TODO: more detail error detection is needed.
    *status = FMS_ERROR_ITEM_CAN_NOT_REMOVE;
    return false;
  }
  *status = FMS_SUCCESS;
  return true;
#endif /* _WIN32 */
}

#ifdef __APPLE__
bool
FileManager::removeNFCDirectory(const Utf8String* path, Status* status)
{
  bool dir_p = false;
  do
  {
    struct stat info;
    int ret = stat(path->getCString(), &info);
    if (ret != 0)
    {
      *status = FMS_ERROR_ITEM_DOES_NOT_EXIST;
      return false;
    }
    if (S_ISDIR(info.st_mode))
    {
      dir_p = true;
      break;
    }
  }
  while (false);
  if (!dir_p)
  {
    *status = FMS_ERROR_ITEM_IS_NOT_DIRECTORY;
    return false;
  }
  DIR* dir = opendir(path->getCString());
  if (!dir)
  {
    // you might not have access right.
    *status = FMS_ERROR_ITEM_CAN_NOT_OPEN;
    return false;
  }
  bool ret_p = true;
  struct dirent* ent;
  while ((ent = readdir(dir)) != NULL)
  {
    Utf8String full_path(path);
    full_path.add('/');
    full_path.add(ent->d_name);
    struct stat s;
    lstat(full_path.getCString(), &s);
    if (S_ISDIR(s.st_mode))
    {
      if ((strcmp(ent->d_name, ".") != 0) &&
	  (strcmp(ent->d_name, "..") != 0))
      {
	if (!FileManager::removeNFCDirectory(&full_path, status))
	{
	  ret_p = false;
	  break;
	}
      }
    }
    else
    {
      if (unlink(full_path.getCString()) != 0)
      {
	*status = FMS_ERROR_ITEM_CAN_NOT_REMOVE;
	ret_p = false;
	break;
      }
    }
  }
  closedir(dir);
  if (!ret_p)
  {
    return false;
  }
  if (rmdir(path->getCString()) != 0)
  {
    // TODO: more detail error detection is needed.
    *status = FMS_ERROR_ITEM_CAN_NOT_REMOVE;
    return false;
  }
  *status = FMS_SUCCESS;
  return true;
}
#endif /* __APPLE__ */

bool
FileManager::createDirectory(const PathString* path, Status* status)
{
#ifdef _WIN32
#  ifdef CCC_UTF16_PATHSTRING
  DWORD dwAttrib = GetFileAttributesW((wchar_t*)path->getCString());
#  else /* CCC_UTF16_PATHSTRING */
  DWORD dwAttrib = GetFileAttributesA(path->getCString());
#  endif /* CCC_UTF16_PATHSTRING */
  if (dwAttrib != INVALID_FILE_ATTRIBUTES)
  {
    if (dwAttrib & FILE_ATTRIBUTE_DIRECTORY)
    {
      return true;
    }
  }
#  ifdef CCC_UTF16_PATHSTRING
  if (::CreateDirectoryW((wchar_t*)path->getCString(), NULL) == 0)
#  else /* CCC_UTF16_PATHSTRING */
  if (::CreateDirectoryA(path->getCString(), NULL) == 0)
#  endif /* CCC_UTF16_PATHSTRING */
  {
    *status = FMS_ERROR_ITEM_CAN_NOT_CREATE_DIRECTORY;
    return false;
  }
  *status = FMS_SUCCESS;
  return true;
#else /* _WIN32 */
  struct stat info;
  int ret = stat(path->getCString(), &info);
  if (ret == 0)
  {
    if (S_ISDIR(info.st_mode))
    {
      return true;
    }
    *status = FMS_ERROR_ITEM_CAN_NOT_CREATE_DIRECTORY;
    return false;
  }
  mode_t mode = S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IWGRP | S_IXGRP | S_IROTH | S_IXOTH | S_IXOTH;
  if (mkdir(path->getCString(), mode) != 0)
  {
    *status = FMS_ERROR_ITEM_CAN_NOT_CREATE_DIRECTORY;
    return false;
  }
  *status = FMS_SUCCESS;
  return true;
#endif /* _WIN32 */
}

bool
FileManager::createDirectoryWithIntermediateDirectories(const PathString* path, Status* status)
{
  /* check intermediate path */
  const PathChar* x = path->getCString();
  PathString try_path;
  for (;;)
  {
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
    const PathChar* y = (PathChar*)wcschr((wchar_t*)x, directory_separator);
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
    const char* y = strchr(x, directory_separator);
#endif /* CCC_UTF16_PATHSTRING */
    if (y == 0)
    {
      break;
    }
    PathString sub_path(x, y - x);
    try_path.add(sub_path);
#ifdef _WIN32
    if ((try_path.getLength() == 2) &&
	(x[1] == ':'))
    {
      // driver letter. ex. "D:"
      // skip
    }
    else
#endif /* _WIN32 */
    if (try_path.getLength() > 0)
    {
      if (!FileManager::createDirectory(&try_path, status))
      {
	return false;
      }
    }
    try_path.add(directory_separator);
    x = y + 1;
  }
  return FileManager::createDirectory(path, status);
}

bool
FileManager::contentsOfDirectoryAtPath(const PathString* path, ArrayList<PathString>* contents, Status* status)
{
  contents->clear();
#ifdef _WIN32
  PathString pattern(*path);
  pattern.add(directory_separator);
  pattern.add('*');
  WIN32_FIND_DATAW fdata;
  HANDLE hfind = ::FindFirstFileW((wchar_t*)pattern.getCString(), &fdata);
  if (hfind == INVALID_HANDLE_VALUE)
  {
    *status = FMS_ERROR_ITEM_CAN_NOT_OPEN;
    return false;
  }
  bool ret = true;
  do
  {
    if (fdata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
    {
      if ((wcscmp(L".", fdata.cFileName) != 0) &&
	  (wcscmp(L"..", fdata.cFileName) != 0))
      {
	PathString* sub_path = new PathString((PathChar*)fdata.cFileName);
	contents->add(sub_path);
      }
    }
  }
  while (::FindNextFileW(hfind, &fdata));
  ::FindClose(hfind);
  *status = FMS_SUCCESS;
  return ret;
#else /* _WIN32 */
  DIR* dir = opendir(path->getCString());
  if (!dir)
  {
    // you might not have the access right.
    *status = FMS_ERROR_ITEM_CAN_NOT_OPEN;
    return false;
  }
  bool ret_p = true;
  struct dirent* ent;
  while ((ent = readdir(dir)) != NULL)
  {
    if ((strcmp(ent->d_name, ".") != 0) &&
	(strcmp(ent->d_name, "..") != 0))
    {
      PathString* sub_path = new PathString(ent->d_name);
      contents->add(sub_path);
    }
  }
  closedir(dir);
  *status = FMS_SUCCESS;
  return ret_p;
#endif /* _WIN32 */
}

bool
FileManager::moveItem(const PathString* src_path, const PathString* dest_path, Status* status)
{
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  if (_wrename((wchar_t*)src_path->getCString(), (wchar_t*)dest_path->getCString()) == 0)
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
  if (rename(src_path->getCString(), dest_path->getCString()) == 0)
#endif /* CCC_UTF16_PATHSTRING */
  {
    *status = FMS_SUCCESS;
    return true;
  }
  *status = FMS_ERROR_ITEM_CAN_NOT_RENAME;
  return false;
}

void
FileManager::getStatusMessage(Status status, Utf8String* result)
{
  struct pair
  {
    Status status;
    const char* msg;
  };
  static const pair x[FMS_NUMBER_OF_STATUS + 1] =
  {
    { FMS_SUCCESS, "success" },
    { FMS_ERROR_ITEM_DOES_NOT_EXIST, "Does not exist the item." },
    { FMS_ERROR_ITEM_CAN_NOT_OPEN, "Can not open the item." },
    { FMS_ERROR_ITEM_IS_NOT_FILE, "The item is not a file." },
    { FMS_ERROR_ITEM_READ, "The item read error." },
    { FMS_ERROR_ITEM_CLOSE, "The item close error." },
    { FMS_ERROR_ITEM_IS_NOT_DIRECTORY, "The item is not a directory." },
    { FMS_ERROR_ITEM_CAN_NOT_REMOVE, "Can not remove the item." },
    { FMS_ERROR_ITEM_CAN_NOT_GET_STATUS, "Can not get status of the item." },
    { FMS_ERROR_ITEM_CAN_NOT_CREATE_DIRECTORY, "Can not create the directory." },
    { FMS_ERROR_ITEM_CAN_NOT_RENAME, "Can not rename the item." },
    { FMS_ERROR_DEST_CAN_NOT_OPEN, "Can not open the destination item." },
    { FMS_ERROR_DEST_WRITE, "Can not write the destination item." },
    { FMS_ERROR_DEST_CLOSE, "Can not close the destination item." },
    { FMS_ERROR_OTHER, "Operation error." },
    { FMS_NUMBER_OF_STATUS, 0 },
  };
  result->clear();
  int n;
  for (n = 0; n < FMS_NUMBER_OF_STATUS; n++)
  {
    if (x[n].status == status)
    {
      result->assign(x[n].msg);
      break;
    }
  }
}

/*!
 * This is a compare function for directory sort.
 */
static int
compare(const void* v1, const void* v2)
{
  const PathString** s1 = (const PathString**)v1;
  const PathString** s2 = (const PathString**)v2;
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  return wcscmp((wchar_t*)(*s1)->getCString(), (wchar_t*)(*s2)->getCString());
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
  return strcmp((*s1)->getCString(), (*s2)->getCString());
#endif /* CCC_UTF16_PATHSTRING */
}

bool
FileManager::getDirectoryList(const PathString* path, ArrayList<PathString>* list)
{
#ifdef _WIN32
  PathString pattern(*path);
  pattern.add(directory_separator);
  pattern.add('*');
  WIN32_FIND_DATAW fdata;
  HANDLE hfind = ::FindFirstFileW((wchar_t*)pattern.getCString(), &fdata);
  if (hfind == INVALID_HANDLE_VALUE)
  {
    return false;
  }
  do
  {
    PathString* sub_path = new PathString((PathChar*)fdata.cFileName);
    list->add(sub_path);
  }
  while (::FindNextFileW(hfind, &fdata));
  ::FindClose(hfind);
#else /* _WIN32 */
  DIR* dp = opendir(path->getCString());
  if (!dp)
  {
    // error
    return false;
  }
  struct dirent* de;
  while ((de = readdir(dp)) != 0)
  {
    PathString* entry = new PathString(de->d_name);
    list->add(entry);
  }
  closedir(dp);
#endif /* _WIN32 */
  qsort((void*)list->getArrayBuffer(), (size_t)list->number(), sizeof(char**), compare);
  return true;
}

bool
FileManager::ls(const PathString* path, bool ctime_p, Utf8String* result)
{
#ifdef _WIN32
  struct _stat info;
#  ifdef CCC_UTF16_PATHSTRING
  int ret = _wstat((wchar_t*)path->getCString(), &info);
#  else /* CCC_UTF16_PATHSTRING */
  int ret = _stat(path->getCString(), &info);
#  endif /* CCC_UTF16_PATHSTRING */
#else /* _WIN32 */
  struct stat info;
  int ret = stat(path->getCString(), &info);
#endif /* _WIN32 */
  if (ret != 0)
  {
    result->assign("ERROR: can not access the path:");
    result->add(path);
    result->add("\n");
    return false;
  }
  const int mode_len = 10;
  char mode[mode_len + 1];
  int i;
  for (i = 0; i < mode_len; i++)
  {
    mode[i] = '-';
  }
  mode[mode_len] = '\0';
#ifdef _WIN32
  if (info.st_mode & _S_IREAD)
#else /* _WIN32 */
  if (info.st_mode & S_IRUSR)
#endif /* _WIN32 */
  {
    mode[1] = 'r';
  }
#ifdef _WIN32
  if (info.st_mode & S_IWRITE)
#else /* _WIN32 */
  if (info.st_mode & S_IWUSR)
#endif /* _WIN32 */
  {
    mode[2] = 'w';
  }
#ifdef _WIN32
  if (info.st_mode & S_IEXEC)
#else /* _WIN32 */
  if (info.st_mode & S_IXUSR)
#endif /* _WIN32 */
  {
    mode[3] = 'x';
  }
#ifndef _WIN32
  if (info.st_mode & S_IRGRP)
  {
    mode[4] = 'r';
  }
  if (info.st_mode & S_IWGRP)
  {
    mode[5] = 'w';
  }
  if (info.st_mode & S_IXGRP)
  {
    mode[6] = 'x';
  }
  if (info.st_mode & S_IROTH)
  {
    mode[7] = 'r';
  }
  if (info.st_mode & S_IWOTH)
  {
    mode[8] = 'w';
  }
  if (info.st_mode & S_IXOTH)
  {
    mode[9] = 'x';
  }
  if (info.st_mode & S_ISVTX)
  {
    if (mode[9] == 'x')
    {
      mode[9] = 't';
    }
    else
    {
      mode[9] = 'T';
    }
  }
  if (info.st_mode & S_ISUID)
  {
    mode[3] = 's';
  }
  if (info.st_mode & S_ISGID)
  {
    mode[6] = 's';
  }
#endif /* _WIN32 */
#ifdef _WIN32
  if (info.st_mode & _S_IFREG)		// file
#else /* _WIN32 */
  if (S_ISREG(info.st_mode))		// file
#endif /* _WIN32 */
  {
    mode[0] = '-';
  }
#ifdef _WIN32
  else if (info.st_mode & _S_IFDIR)	// directory
#else /* _WIN32 */
  else if (S_ISDIR(info.st_mode))	// directory
#endif /* _WIN32 */
  {
    mode[0] = 'd';
  }
#ifdef _WIN32
  else if (info.st_mode & _S_IFCHR)	// character device
#else /* _WIN32 */
  else if (S_ISCHR(info.st_mode))	// character device
#endif /* _WIN32 */
  {
    mode[0] = 'c';
  }
#ifndef _WIN32
  else if (S_ISBLK(info.st_mode))	// block device
  {
    mode[0] = 'b';
  }
  else if (S_ISFIFO(info.st_mode))	// FIFO
  {
    mode[0] = 'p';
  }
  else if (S_ISLNK(info.st_mode))	// symbolic link
  {
    mode[0] = 'l';
  }
  else if (S_ISSOCK(info.st_mode))	// socket
  {
    mode[0] = 's';
  }
#endif /* _WIN32 */
  Utf8String user_name;
  {
    // get user name of this file.
#if defined(ANDROID) || defined(_WIN32)
    Fmt8 fmt("%d");
    fmt.setSI((int)info.st_uid);
    fmt.writeTo(&user_name);
#else /* ANDROID */
    size_t buffer_len = sysconf(_SC_GETPW_R_SIZE_MAX);
    char* buffer = new char[buffer_len];
    struct passwd pwd;
    struct passwd* result = 0;
    if (getpwuid_r(info.st_uid, &pwd, buffer, buffer_len, &result) == 0)
    {
      // can not get the user name.
      Fmt8 fmt("%d");
      fmt.setSI((int)info.st_uid);
      fmt.writeTo(&user_name);
    }
    else
    {
      user_name.assign(result->pw_name);
    }
    delete[] buffer;
#endif /* ANDROID */
  }
  Utf8String group_name;
  {
    // get group name of this file.
#if defined(ANDROID) || defined(_WIN32)
    Fmt8 fmt("%d");
    fmt.setSI((int)info.st_gid);
    fmt.writeTo(&group_name);
#else /* ANDROID */
    size_t buffer_len = sysconf(_SC_GETGR_R_SIZE_MAX);
    char* buffer = new char[buffer_len];
    struct group grp;
    struct group* result = 0;
    if (getgrgid_r(info.st_gid, &grp, buffer, buffer_len, &result) == 0)
    {
      // can not get the group name.
      Fmt8 fmt("%d");
      fmt.setSI((int)info.st_gid);
      fmt.writeTo(&group_name);
    }
    else
    {
      group_name.assign(result->gr_name);
    }
    delete[] buffer;
#endif /* ANDROID */
  }
  Utf8String timestamp;
  {
    // get time string
    time_t t = ctime_p ? info.st_ctime : info.st_mtime;
    struct tm* ts = localtime(&t);
    char buf[80];
    strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", ts);
    timestamp.assign(buf);
  }
  PathString basename;
  path->getBaseName(&basename);
  // output
  Fmt8 fmt("%s %3d %8s %8s %8u %s %s\n");
  fmt.setTP(mode);
  fmt.setSI((int)info.st_nlink);
  fmt.setTP(user_name.getCString());
  fmt.setTP(group_name.getCString());
  fmt.setUI((unsigned int)info.st_size);
  fmt.setTP(timestamp.getCString());
#ifdef CCC_UTF16_PATHSTRING
  CCC::Utf8String basename8;
  CCC::Iceman::convertToBString(CCC::CEID_UTF16N, CCC::CEID_UTF8N, &basename, &basename8);
  fmt.setTP(basename8.getCString());
#else /* CCC_UTF16_PATHSTRING */
  fmt.setTP(basename.getCString());
#endif /* CCC_UTF16_PATHSTRING */
  fmt.addTo(result);
  return true;
}

bool
FileManager::getTimestamp(const PathString* path, bool ctime_p, Utf8String* result)
{
#ifdef _WIN32
  struct _stat info;
#  ifdef CCC_UTF16_PATHSTRING
  int ret = _wstat((wchar_t*)path->getCString(), &info);
#  else /* CCC_UTF16_PATHSTRING */
  int ret = _stat(path->getCString(), &info);
#  endif /* CCC_UTF16_PATHSTRING */
#else /* _WIN32 */
  struct stat info;
  int ret = stat(path->getCString(), &info);
#endif /* _WIN32 */
  if (ret != 0)
  {
    result->assign("0000-00-00 00:00:00");
    return false;
  }
  // get time string
  time_t t = ctime_p ? info.st_ctime : info.st_mtime;
  struct tm* ts = localtime(&t);
  char buf[80];
  strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", ts);
  result->assign(buf);
  return true;
}

static bool
lsSubDirX(const PathString* path, bool ctime_p, bool top_p, Utf8String* result)
{
#ifdef _WIN32
  struct _stat info;
#  ifdef CCC_UTF16_PATHSTRING
  if (_wstat((wchar_t*)path->getCString(), &info) != 0)
#  else /* CCC_UTF16_PATHSTRING */
  if (_stat(path->getCString(), &info) != 0)
#  endif * CCC_UTF16_PATHSTRING */
#else /* _WIN32 */
  struct stat info;
  if (stat(path->getCString(), &info) != 0)
#endif /* _WIN32 */
  {
    result->assign("ERROR: can not access the path:");
    result->add(path);
    result->add("\n");
    return false;
  }
  bool ret = true;
  if (top_p)
  {
    ret = FileManager::ls(path, ctime_p, result);
  }
#ifdef _WIN32
  if (info.st_mode & _S_IFDIR)	// directory
#else /* _WIN32 */
  if (S_ISDIR(info.st_mode))
#endif /* _WIN32 */
  {
    result->add("\n");
    result->add(path);
    result->add("\n");
    ArrayList<PathString> list;
    if (!FileManager::getDirectoryList(path, &list))
    {
      result->assign("ERROR: can not get a list of directory:");
      result->add(path);
      result->add("\n");
      return false;
    }
    Size num_of_entry = list.number();
    Size n;
    for (n = 0; n < num_of_entry; n++)
    {
      PathString* entry = list.vector(n);
      PathString full_path(path);
      full_path.appendPathComponent(entry);
      if (!FileManager::ls(&full_path, ctime_p, result))
      {
	ret = false;
      }
    }
    for (n = 0; n < num_of_entry; n++)
    {
      PathString* entry = list.vector(n);
      PathString full_path(path);
      full_path.appendPathComponent(entry);
      bool dir_p = false;
      if (FileManager::fileExistsAtPath(&full_path, &dir_p))
      {
	if (dir_p &&
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
	    (wcscmp((wchar_t*)entry->getCString(), L".") != 0) &&
	    (wcscmp((wchar_t*)entry->getCString(), L"..") != 0))
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
	    (strcmp(entry->getCString(), ".") != 0) &&
	    (strcmp(entry->getCString(), "..") != 0))
#endif /* CCC_UTF16_PATHSTRING */
	{
	  if (!lsSubDirX(&full_path, ctime_p, false, result))
	  {
	    ret = false;
	  }
	}
      }
      delete entry;
    }
  }
  return ret;
}

bool
FileManager::lsSubDir(const PathString* path, bool ctime_p, Utf8String* result)
{
  return lsSubDirX(path, ctime_p, true, result);
}

bool
FileManager::getFileSize(const PathString* path, unsigned long long* result, Status* status)
{
  *result = 0l;

#ifdef _WIN32
  struct _stat info;
#  ifdef CCC_UTF16_PATHSTRING
  int ret = _wstat((wchar_t*)path->getCString(), &info);
#  else /* CCC_UTF16_PATHSTRING */
  int ret = _stat(path->getCString(), &info);
#  endif * CCC_UTF16_PATHSTRING */
#else /* _WIN32 */
  struct stat info;
  int ret = stat(path->getCString(), &info);
#endif /* _WIN32 */
  if (ret != 0)
  {
    *status = FMS_ERROR_ITEM_CAN_NOT_OPEN;
    return false;
  }
#ifdef _WIN32
  if (info.st_mode & _S_IFDIR)	// directory
#else /* _WIN32 */
  if (!S_ISREG(info.st_mode))
#endif /* _WIN32 */
  {
    *status = FMS_ERROR_ITEM_IS_NOT_FILE;
    return false;
  }
  *result = (unsigned long long)info.st_size;
  *status = FMS_SUCCESS;
  return true;
}

CCC_NAMESPACE_END(CCC)
