﻿// $Id$

#ifndef INCLUDE_ccc_file_FileManager_h
#define INCLUDE_ccc_file_FileManager_h

#include <ccc/file/PathString.h>
#include <ccc/iceman/Utf8String.h>
#include <ccc/base/ArrayList.h>

CCC_NAMESPACE_START(CCC)

/*!
 * This class manages files or directories.
 */
class FileManager
{
 public:
  /*!
   * status of the operation.
   */
  enum Status
  {
    /*! success */
    FMS_SUCCESS,
    FMS_ERROR_ITEM_DOES_NOT_EXIST,
    FMS_ERROR_ITEM_CAN_NOT_OPEN,
    FMS_ERROR_ITEM_IS_NOT_FILE,
    FMS_ERROR_ITEM_READ,
    FMS_ERROR_ITEM_CLOSE,
    FMS_ERROR_ITEM_IS_NOT_DIRECTORY,
    FMS_ERROR_ITEM_CAN_NOT_REMOVE,
    FMS_ERROR_ITEM_CAN_NOT_GET_STATUS,
    FMS_ERROR_ITEM_CAN_NOT_CREATE_DIRECTORY,
    FMS_ERROR_ITEM_CAN_NOT_RENAME,
    FMS_ERROR_DEST_CAN_NOT_OPEN,
    FMS_ERROR_DEST_WRITE,
    FMS_ERROR_DEST_CLOSE,
    FMS_ERROR_OTHER,
    FMS_NUMBER_OF_STATUS,	/*!< number of statuses */
  };
 public:
  /*!
   * This method gets the status message.
   * \param status status code
   * \param result the result message will be set.
   */
  static void getStatusMessage(Status status, CCC::Utf8String* result);
  /*!
   * This method checks the file or directory existence.
   * \param path the target path.
   * \param dir_p if it is a directory, true will be set.
   * \retval true exist
   * \retval false does not exist
   */
  static bool fileExistsAtPath(const PathString* path, bool* dir_p);
  /*!
   * This method copies a file.
   * \param src source path
   * \param dest destination path
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool copyFile(const PathString* src, const PathString* dest, Status* status);
  /*!
   * This method recursively copies a directory.
   * \param src source path
   * \param dest destination path
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool copyDirectory(const PathString* src, const PathString* dest, Status* status);
  /*!
   * This method removes the file.
   * \param path the target path.
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool removeFile(const PathString* path, Status* status);
  /*!
   * This method removes the directory.
   * \param path the target path.
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool removeDirectory(const PathString* path, Status* status);
#ifdef __APPLE__
  /*!
   * This method removes the directory. Use this method for to remove
   * NFC unicode notatio path files.
   * \param path the target path.
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool removeNFCDirectory(const Utf8String* path, Status* status);
#endif /* __APPLE__ */
  /*!
   * This method creates directory.
   * \parm path the target path.
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool createDirectory(const PathString* path, Status* status);
  /*!
   * This method creates directory and it's intermediate directories.
   * \parm path the target path.
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool createDirectoryWithIntermediateDirectories(const PathString* path, Status* status);
  /*!
   * get the contents of the given directory.
   * \param path the target path.
   * \param contents directory contents will be set.
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool contentsOfDirectoryAtPath(const PathString* path, ArrayList<PathString>* contents, Status* status);
  /*!
   * move a file or directory.
   * \param src_path move source path
   * \param dest_path move destination path
   * \param status operation status will be set.
   * \retval true success
   * \retval false failure
   */
  static bool moveItem(const PathString* src_path, const PathString* dest_path, Status* status);
  /*!
   * This function gets the list of directory.
   * PathString set in this list have to be deleted by caller of this method.
   * \param path a target path
   * \param list result will be set in this array
   * \retval true success
   * \retval false failure
   */
  static bool getDirectoryList(const PathString* path, ArrayList<PathString>* list);
  /*!
   * This method gets the file or the directory list information.
   * \param path path
   * \param r_p recursive flag
   * \param ctime_p true:use ctime, false:use mtime
   * NOTE: The default unix ls command result is mtime.
   * \param result result string
   * \retval true success
   * \retval false failure
   */
  static bool ls(const PathString* path, bool ctime_p, Utf8String* result);
  /*!
   * This method gets the timestamp of the file.
   * \param path path
   * \param ctime_p true:use ctime, false:use mtime
   * NOTE: The default unix ls command result is mtime.
   * \param result result string (YYYY-MM-DD hh:mm:ss)
   */
  static bool getTimestamp(const PathString* path, bool ctime_p, Utf8String* result);
  /*!
   * This method gets the file or the sub directory list information.
   * \param path path
   * \param r_p recursive flag
   * \param ctime_p true:use ctime, false:use mtime
   * NOTE: The default unix ls command result is mtime.
   * \param result result string
   * \retval true success
   * \retval false failure
   */
  static bool lsSubDir(const PathString* path, bool ctime_p, Utf8String* result);
  /*!
   * This method gets file size.
   * \param path path
   * \param result result file size will be stored in it.
   * \param status result status
   * \retval true success
   * \retval false failure
   */
  static bool getFileSize(const PathString* path, unsigned long long* result, Status* status);
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_file_FileManager_h */
