﻿// $Id$

#include <assert.h>
#include <ccc/file/FileOFlow.h>
#include <ccc/file/PathString.h>

CCC_NAMESPACE_START(CCC)

FileOFlow::FileOFlow()
  : OFlow(IOTYPE_ANY)
{
  fp = 0;
  attached_p = false;
}

FileOFlow::~FileOFlow()
{
  if (fp)
  {
    close();
  }
}

bool
FileOFlow::open(const PathString* path, bool append_p, bool binary_p)
{
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  return open((wchar_t*)path->getCString(), append_p, binary_p);
#  else /* _WIN32 */
  assert(false);
  return false;	/* not supported */
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
  return open(path->getCString(), append_p, binary_p);
#endif /* CCC_UTF16_PATHSTRING */
}

bool
FileOFlow::open(const char* path, bool append_p, bool binary_p)
{
  if (append_p)
  {
    fp = fopen(path, binary_p ? "ab" : "a");
  }
  else
  {
    fp = fopen(path, binary_p ? "wb" : "w");
  }
  setTopP(true);
  return !!fp;
}

#ifdef _WIN32
bool
FileOFlow::open(const wchar_t* path, bool append_p, bool binary_p)
{
  if (append_p)
  {
    wchar_t mode_binary[] = { 'a', 'b', 0 };
    wchar_t mode_normal[] = { 'a', 0 };
    fp = _wfopen(path, binary_p ? mode_binary : mode_normal);
  }
  else
  {
    wchar_t mode_binary[] = { 'w', 'b', 0 };
    wchar_t mode_normal[] = { 'w', 0 };
    fp = _wfopen(path, binary_p ? mode_binary : mode_normal);
  }
  setTopP(true);
  return !!fp;
}
#endif /* _WIN32 */

void
FileOFlow::attach(FILE* fp_)
{
  fp = fp_;
  attached_p = true;
  setTopP(false);
}

bool
FileOFlow::close()
{
  if (!openedP())
  {
    return false;
  }
  bool ret = true;
  if (!attached_p)
  {
    ret = !fclose(fp);
  }
  fp = 0;
  return ret;
}

void
FileOFlow::flush() CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  int ret = fflush(fp);
  if (ret != 0)
  {
    throw IOException(__FILE__, __LINE__, IOException::FILE_IO_ERROR);
  }
}

void
FileOFlow::putInt8(Int8 c) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  if (fwrite(&c, sizeof(Int8), 1, fp) != 1) 
  {
    throw IOException(__FILE__, __LINE__, IOException::WRITE_BEYOND_THE_EOF);
  }
  setTopP(false);
}

void
FileOFlow::putUInt16(UInt16 c) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  // native endian
  if (fwrite(&c, sizeof(UInt16), 1, fp) != 1)
  {
    throw IOException(__FILE__, __LINE__, IOException::WRITE_BEYOND_THE_EOF);
  }
  setTopP(false);
}

void
FileOFlow::putUInt32(UInt32 c) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  // native endian
  if (fwrite(&c, sizeof(UInt32), 1, fp) != 1)
  {
    throw IOException(__FILE__, __LINE__, IOException::WRITE_BEYOND_THE_EOF);
  }
  setTopP(false);
}

void
FileOFlow::putInt8Block(Size block_size, const Int8* ptr, Size& put_size) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  put_size = 0;
  size_t w =  fwrite((void*)ptr, sizeof(Int8), block_size, fp);
  if (w != block_size)
  {
    put_size = (Size)w;
    throw IOException(__FILE__, __LINE__, IOException::WRITE_BEYOND_THE_EOF);
  }
  put_size = block_size;
  setTopP(false);
}

void
FileOFlow::putUInt16Block(Size block_size, const UInt16* ptr, Size& put_size) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  put_size = 0;
  size_t w = fwrite((void*)ptr, sizeof(UInt16), block_size, fp);
  if (w != block_size)
  {
    put_size = (Size)w;
    throw IOException(__FILE__, __LINE__, IOException::WRITE_BEYOND_THE_EOF);
  }
  put_size = block_size;
  setTopP(false);
}

void
FileOFlow::putUInt32Block(Size block_size, const UInt32* ptr, Size& put_size) CCC_RAISES(IOException)
{
  if (!openedP())
  {
    throw IOException(__FILE__, __LINE__, IOException::NOT_OPENED);
  }
  assert(fp);
  put_size = 0;
  size_t w = fwrite((void*)ptr, sizeof(UInt32), block_size, fp);
  if (w != block_size)
  {
    put_size = (Size)w;
    throw IOException(__FILE__, __LINE__, IOException::WRITE_BEYOND_THE_EOF);
  }
  put_size = block_size;
  setTopP(false);
}

bool
FileOFlow::rewind()
{
  if (!openedP())
  {
    return false;
  }
  assert(fp);
  int ret = fseek(fp, 0L, SEEK_SET);
  if (ret != 0)
  {
    return false;
  }
  return OFlow::rewind();
}

bool
FileOFlow::openedP()
{
  return fp != 0;
}

CCC_NAMESPACE_END(CCC)
