﻿// $Id$

#include <string.h>
#include <stdio.h>
#ifdef _WIN32
#  include <io.h>
#  include <wchar.h>
#endif /* _WIN32 */
#include <ccc/file/PathString.h>
#include <ccc/iceman/Iceman.h>

#ifdef __APPLE__
#include <CoreFoundation/CoreFoundation.h>
#include <CoreFoundation/CFString.h>
#endif /* __APPLE__ */

CCC_NAMESPACE_START(CCC)

#ifdef __APPLE__
/*!
 * convert NFC UTF8 string to NFD UTF8 string.
 * \param utf8_nfc input UTF8 string
 * \return NFD UTF8 string, need to be deleted by caller.
 */
static char*
convertFromNFCtoNFD(const char* utf8_nfc)
{
  CFStringRef cfStringRef = CFStringCreateWithCString(kCFAllocatorDefault, utf8_nfc, kCFStringEncodingUTF8);
  CFMutableStringRef cfMutable = CFStringCreateMutableCopy(NULL, 0, cfStringRef);
  CFStringNormalize(cfMutable, kCFStringNormalizationFormD);

  char* mem = 0;
  size_t len = strlen(utf8_nfc);
  bool done_flag = true;
  do
  {
    size_t mem_len = len * 2;
    mem = new char[mem_len + 1];
    if (CFStringGetCString(cfMutable, mem, mem_len + 1, kCFStringEncodingUTF8))
    {
      done_flag = true;
    }
    else
    {
      /* need to expand buffer size */
      done_flag = false;
      delete [] mem;
      mem = 0;
    }
    len *= 2;
  }
  while (!done_flag);
  CFRelease(cfStringRef);
  CFRelease(cfMutable);
  return mem;
}
#endif /* __APPLE__ */

void
PathString::normalize()
{
#ifdef __APPLE__
  const char* utf8_c_str = this->getCString();
  char* x = convertFromNFCtoNFD(utf8_c_str);
  //printf("from:[%s] to:[%s]\n", utf8_c_str, x);
  CCC::BString::assign(x);
  delete [] x;
#endif /* __APPLE__ */
}

PathString::PathString()
{
}

PathString::PathString(const PathString& path)
{
  assign(path);
  normalize();
}

PathString::PathString(const PathString* path)
{
  assign(path);
  normalize();
}

PathString::PathString(const PathChar* path)
{
  assign(path);
  normalize();
}

PathString::PathString(const PathChar* path, Size size)
{
  assign(path, size);
  normalize();
}

#ifdef CCC_UTF16_PATHSTRING
PathString::PathString(const char* utf8_path)
{
  assign(utf8_path);
  normalize();
}

PathString::PathString(const char* utf8_path, Size size)
{
  assign(utf8_path, size);
  normalize();
}

PathString::PathString(CeId enc, const char* path)
{
  assign(enc, path);
  normalize();
}

PathString::PathString(CeId enc, const char* path, Size size)
{
  assign(enc, path, size);
  normalize();
}

PathString::PathString(const DOMString* path)
{
  assign(path->getCString());
  normalize();
}

#else /* CCC_UTF16_PATHSTRING */
#ifndef CCC_SINGLE_BYTE_DOM_STRING
PathString::PathString(const DOMString* path)
{
  assign(path);
  normalize();
}
#endif /* CCC_SINGLE_BYTE_DOM_STRING */
#endif /* CCC_UTF16_PATHSTRING */

PathString::~PathString()
{
}

void
PathString::assign(const PathString& path)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::assign(path);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::assign(path);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::assign(const PathString* path)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::assign(path);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::assign(path);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::assign(const PathChar* path)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::assign(path);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::assign(path);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::assign(const PathChar* path, Size size)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::assign(path, size);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::assign(path, size);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

#ifdef CCC_UTF16_PATHSTRING
void
PathString::assign(const char* utf8_path)
{
  CCC::BString x(utf8_path);
  CCC::DString y;
  CCC::Iceman::convertToDString(CEID_UTF8, CEID_UTF16N, &x, &y);
  CCC::DString::assign(&y);
  normalize();
}

void
PathString::assign(const char* utf8_path, Size size)
{
  CCC::BString x(utf8_path, size);
  CCC::DString y;
  CCC::Iceman::convertToDString(CEID_UTF8, CEID_UTF16N, &x, &y);
  CCC::DString::assign(&y);
  normalize();
}

void
PathString::assign(CeId enc, const char* path)
{
  CCC::BString x(path);
  CCC::DString y;
  CCC::Iceman::convertToDString(enc, CEID_UTF16N, &x, &y);
  CCC::DString::assign(&y);
  normalize();
}

void
PathString::assign(CeId enc, const char* path, Size size)
{
  CCC::BString x(path, size);
  CCC::DString y;
  CCC::Iceman::convertToDString(enc, CEID_UTF16N, &x, &y);
  CCC::DString::assign(&y);
  normalize();
}

#else /* CCC_UTF16_PATHSTRING */
#ifndef CCC_SINGLE_BYTE_DOM_STRING
void
PathString::assign(const DOMString* path)
{
  CCC::Iceman::convertToBString(CCC_DOM_CEID, CEID_UTF8N, path, this);
  normalize();
}
#endif /* CCC_SINGLE_BYTE_DOM_STRING */
#endif /* CCC_UTF16_PATHSTRING */

void
PathString::add(const PathString& path)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::add(path);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::add(path);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::add(const PathString* path)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::add(*path);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::add(*path);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::add(const PathChar* path)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::add(path);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::add(path);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::add(const PathChar* path, Size size)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::add(path, size);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::add(path, size);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

void
PathString::add(PathChar c)
{
#ifdef CCC_UTF16_PATHSTRING
  CCC::DString::add(c);
#else /* CCC_UTF16_PATHSTRING */
  CCC::BString::add(c);
#endif /* CCC_UTF16_PATHSTRING */
  normalize();
}

#ifdef CCC_UTF16_PATHSTRING
void
PathString::add(const char* utf8_path)
{
  CCC::BString x(utf8_path);
  CCC::DString y;
  CCC::Iceman::convertToDString(CEID_UTF8, CEID_UTF16N, &x, &y);
  CCC::DString::add(&y);
  normalize();
}

void
PathString::add(const char* utf8_path, Size size)
{
  CCC::BString x(utf8_path, size);
  CCC::DString y;
  CCC::Iceman::convertToDString(CEID_UTF8, CEID_UTF16N, &x, &y);
  CCC::DString::add(&y);
  normalize();
}

void
PathString::add(CeId enc, const char* path)
{
  CCC::BString x(path);
  CCC::DString y;
  CCC::Iceman::convertToDString(enc, CEID_UTF16N, &x, &y);
  CCC::DString::add(&y);
  normalize();
}

void
PathString::add(CeId enc, const char* path, Size size)
{
  CCC::BString x(path, size);
  CCC::DString y;
  CCC::Iceman::convertToDString(enc, CEID_UTF16N, &x, &y);
  CCC::DString::add(&y);
  normalize();
}

void
PathString::add(char c)
{
  unsigned char u = (unsigned char)c;
  add((UInt16)u);
  normalize();
}
#else /* CCC_UTF16_PATHSTRING */
#ifndef CCC_SINGLE_BYTE_DOM_STRING
void
PathString::add(const DOMString* path)
{
  CCC::PathString tmp;
  CCC::Iceman::convertToBString(CCC_DOM_CEID, CEID_UTF8N, path, &tmp);
  add(&tmp);
  normalize();
}
#endif /* CCC_SINGLE_BYTE_DOM_STRING */
#endif /* CCC_UTF16_PATHSTRING */

void
PathString::removeLastDirectorySeparator()
{
  chop(directory_separator);
}

void
PathString::addDirectorySeparator()
{
  removeLastDirectorySeparator();
  add(directory_separator);
}

void
PathString::appendPathComponent(const PathString* sub_path)
{
#ifdef _WIN32
  CCC::PathString parent_pattern("..\\");
  CCC::PathString current_pattern(".\\");
#else /* _WIN32 */
  CCC::PathString parent_pattern("../");
  CCC::PathString current_pattern("./");
#endif /* _WIN32 */
  CCC::PathString sub(sub_path);
  {
    CCC::PathString x;
    x.add(directory_separator);
    if (this->strCmp(x) != 0)
    {
      chop(directory_separator);
    }
  }
  for (;;)
  {
    //printf("this:[%s] sub:[%s]\n", this->getCString(), sub.getCString());
    if (sub.getLength() >= 3)
    {
      PathSubString x(sub, 0, 3);
      if (x.strCmp(parent_pattern) == 0)
      {
	// "../"
	if (this->getLength() > 0)
	{
	  sub.remove(0, 3);
	  if (!deleteLastPathComponentX())
	  {
	    clear();
	  }
	  continue;
	}
	break;
      }
    }
    if (sub.getLength() >= 2)
    {
      PathSubString x(sub, 0, 2);
      if (x.strCmp(current_pattern) == 0)
      {
	// "./"
	sub.remove(0, 2);
	continue;
      }
    }
    break;
  }
  if ((this->getLength() > 0) &&
      (sub.getLength() > 0))
  {
    addDirectorySeparator();
  }
  PathString appending(sub);
  appending.chopHead(directory_separator);
  add(appending);
  normalize();
}

void
PathString::appendPathComponent(const PathChar* sub_path)
{
  PathString tmp(sub_path);
  appendPathComponent(&tmp);
  normalize();
}

void
PathString::appendPathComponent(const PathChar* sub_path, Size len)
{
  PathString tmp(sub_path, len);
  appendPathComponent(&tmp);
  normalize();
}

#ifdef CCC_UTF16_PATHSTRING
void
PathString::appendPathComponent(const char* utf8_sub_path)
{
  PathString tmp(utf8_sub_path);
  appendPathComponent(&tmp);
  normalize();
}
#endif /* CCC_UTF16_PATHSTRING */

void
PathString::assignWithAppendingPathComponent(const PathString* ref, const PathString* sub_path)
{
  assign(ref);
  appendPathComponent(sub_path);
  normalize();
}

void
PathString::assignWithAppendingPathComponent(const PathString* ref, const PathChar* sub_path)
{
  assign(ref);
  appendPathComponent(sub_path);
  normalize();
}

void
PathString::assignWithAppendingPathComponent(const PathString* ref, const PathChar* sub_path, Size len)
{
  assign(ref);
  appendPathComponent(sub_path, len);
  normalize();
}

#ifdef CCC_UTF16_PATHSTRING
void
PathString::assignWithAppendingPathComponent(const PathString* ref, const char* utf8_sub_path)
{
  assign(ref);
  appendPathComponent(utf8_sub_path);
  normalize();
}
#endif /* CCC_UTF16_PATHSTRING */

void
PathString::deleteLastPathComponent()
{
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  PathChar* x = this->getCString();
  PathChar* y = (PathChar*)wcsrchr((wchar_t*)x, (wchar_t)directory_separator);
  if (y)
  {
    remove((y - x), wcslen((wchar_t*)y));
  }
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#endif /* CCC_UTF16_PATHSTRING */
#if defined(CCC_CP932_PATHSTRING) || defined(CCC_UTF8_PATHSTRING)
  char* x = this->getCString();
  char* y = strrchr(x, directory_separator);
  if (y)
  {
    remove((y - x), strlen(y));
  }
#endif
}

bool
PathString::deleteLastPathComponentX()
{
  bool ret = false;
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  PathChar* x = this->getCString();
  PathChar* y = (PathChar*)wcsrchr((wchar_t*)x, (wchar_t)directory_separator);
  if (y)
  {
    remove((y - x), wcslen((wchar_t*)y));
    ret = true;
  }
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#endif /* CCC_UTF16_PATHSTRING */
#if defined(CCC_CP932_PATHSTRING) || defined(CCC_UTF8_PATHSTRING)
  char* x = this->getCString();
  char* y = strrchr(x, directory_separator);
  if (y)
  {
    remove((y - x), strlen(y));
    ret = true;
  }
#endif
  return ret;
}

void
PathString::getLastPathComponent(PathString* result) const
{
  result->clear();
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  PathChar* x = this->getCString();
  PathChar* y = (PathChar*)wcsrchr((wchar_t*)x, (wchar_t)directory_separator);
  if (y)
  {
    result->assign(y + 1);
  }
  else
  {
    result->assign(this);
  }
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#endif /* CCC_UTF16_PATHSTRING */
#if defined(CCC_CP932_PATHSTRING) || defined(CCC_UTF8_PATHSTRING)
  char* x = this->getCString();
  char* y = strrchr(x, directory_separator);
  if (y)
  {
    result->assign(y + 1);
  }
  else
  {
    result->assign(this);
  }
#endif
}

void
PathString::getBaseName(PathString* basename) const
{
  getLastPathComponent(basename);
}

void
PathString::getFileExtention(PathString* ext) const
{
  CCC::PathString basename;
  getLastPathComponent(&basename);
  PathChar* x = basename.getCString();
#ifdef CCC_UTF16_PATHSTRING
#  ifdef _WIN32
  PathChar* y = (PathChar*)wcsrchr((wchar_t*)x, (wchar_t)'.');
#  else /* _WIN32 */
#    error TODO
#  endif /* _WIN32 */
#else /* CCC_UTF16_PATHSTRING */
  PathChar* y = strrchr(x, '.');
#endif /* CCC_UTF16_PATHSTRING */
  if (y == 0)
  {
    ext->clear();
  }
  else
  {
    ext->assign(y + 1);
  }
}

unsigned int
PathString::getPathDepth()
{
  PathChar* x = getCString();
  unsigned int depth = 0;
  while (*x != '\0')
  {
    if (*x++ == directory_separator)
    {
      depth++;
    }
  }
  return depth;
}

#ifdef _WIN32
PathString::PathString(const wchar_t* wstr)
{
  UInt16* s = (UInt16*)wstr;
  assign(s);
  normalize();
}

void
PathString::assign(const wchar_t* wstr)
{
  UInt16* s = (UInt16*)wstr;
  assign(s);
  normalize();
}

void
PathString::add(const wchar_t* wstr)
{
  UInt16* s = (UInt16*)wstr;
  add(s);
  normalize();
}

#endif /* _WIN32 */

CCC_NAMESPACE_END(CCC)

#if 0
// test case of appendPathComponent
// needs -framework Foundation for linking on macOS or iOS.

#include <stdio.h>
#include <ccc/file/PathString.h>

int
main()
{
  struct TestCase 
  {
    const char* a;
    const char* b;
    const char* should_be;
  };
  static testcase table[] =
  {
    { "abc/def",	"ghi",		"abc/def/ghi" },
    { "abc/def/",	"ghi",		"abc/def/ghi" },
    { "abc/def/", 	"ghi/jkl",	"abc/def/ghi/jkl" },
    { "foo/abc/def",	"../bar",	"foo/abc/bar" },
    { "foo/abc/def",	"../bar/baz",	"foo/abc/bar/baz" },
    { "foo/abc/def", 	"../../bar/baz",	"foo/bar/baz" },
    { "foo/abc/def",	"../../../bar/baz",	"bar/baz" },
    { "foo/abc/def",	"../../../../bar/baz",	"../bar/baz" },
    { "foo/abc/def/",	"../bar",		"foo/abc/bar" },
    { "foo/abc/def/",	"../../bar/baz",	"foo/bar/baz" },
    { "foo/abc/def/",	"../../../bar/baz",	"bar/baz" },
    { "foo/abc/def/",	"../../../../bar/baz",	"../bar/baz" },
    { "foo/abc/def",	"./bar",		"foo/abc/def/bar" },
    { "foo/abc/def",	"././bar",		"foo/abc/def/bar" },
    { "foo/abc/def",	"./././bar",		"foo/abc/def/bar" },
    { "foo/abc/def/",	"./bar",		"foo/abc/def/bar" },
    { "foo/abc/def/",	"././bar",		"foo/abc/def/bar" },
    { "foo/abc/def/",	"./././bar",		"foo/abc/def/bar" },
    { "foo/abc/def/",	"./",			"foo/abc/def" },
    { "foo/abc/def",	"",			"foo/abc/def" },
    { "foo/abc/def/",	"",			"foo/abc/def" },
    { "foo", 		"",		"foo" },
    { "foo",		"a",		"foo/a" },
    { "foo",		"./b",		"foo/b" },
    { "",		"",		"" },
    { "",		"a",		"a" },
    { "",		"./b",		"b" },
    { "",		"../",		"../" },
    { "",		"../abc",	"../abc" },
    { "/",		"",		"/" },
    { "/",		"a",		"/a" },
    { "/",		"./b",		"/b" },
    { "/",		"./",		"/" },
    { "/",		"../",		"" },
#ifdef __APPLE__
    // NFC to NFD
    { "\xe3\x83\x94", "", "\xe3\x83\x92\xe3\x82\x9a" },
    { "\xe3\x83\x94", "\xe3\x83\x94", "\xe3\x83\x92\xe3\x82\x9a/\xe3\x83\x92\xe3\x82\x9a" },
    { "\xe3\x83\x94\xe3\x83\x94\xe3\x83\x94\xe3\x83\x94\xe3\x83\x94", "", "\xe3\x83\x92\xe3\x82\x9a\xe3\x83\x92\xe3\x82\x9a\xe3\x83\x92\xe3\x82\x9a\xe3\x83\x92\xe3\x82\x9a\xe3\x83\x92\xe3\x82\x9a" },
#endif /* __APPLE__ */
    { 0, 0, 0 },
  };

  int pass = 0;
  int n = 0;
  while (table[n].a != 0)
  {
    TestCase* t = &table[n];
    CCC::PathString a(t->a);
    CCC::PathString b(t->b);
    a.appendPathComponent(&b);
    CCC::PathString should_be(t->should_be);
    if (should_be.strCmp(a) == 0)
    {
      printf("test case %d PASS: \"%s\" + \"%s\" => \"%s\"  should be \"%s\"\n", n, t->a, t->b, a.getCString(), t->should_be);
      pass++;
    }
    else
    {
      printf("test case %d NOT GOOD: \"%s\" + \"%s\" => \"%s\"  should be \"%s\"\n", n, t->a, t->b, a.getCString(), t->should_be);
    }
    n++;
  }
  printf("\nPASS:%d TOTAL:%d\n", pass, n);
  return (pass != n) ? -1 : 0;
}
#endif

