﻿// $Id$
/* 日本語 */

#include <stdio.h>
#include <ccc/base/base.h>
#include <ccc/base/StringOFlow.h>
#include <ccc/base/StringIFlow.h>
#include <ccc/base/TextReader.h>
#include <ccc/http/HttpHeaderLine.h>
#include <ccc/http/HttpStatusCode.h>

CCC_NAMESPACE_START(CCC)

HttpHeaderLine::HttpHeaderLine(BString* str_)
{
  str = new BString(str_);
  value1 = 0;
  value2 = 0;
  field_type = FT_UNINITIALIZED;
  request_method = RM_UNKNOWN;
  status_code = HttpStatusCode::HSC_UNDEFINDED;
  analyze();
}
    
HttpHeaderLine::HttpHeaderLine(const char* str_)
{
  str = new BString(str_);
  value1 = 0;
  value2 = 0;
  field_type = FT_UNINITIALIZED;
  request_method = RM_UNKNOWN;
  status_code = HttpStatusCode::HSC_UNDEFINDED;
  analyze();
}
    
HttpHeaderLine::HttpHeaderLine(BString* str_, FieldType ft)
{
  str = new BString(str_);
  value1 = 0;
  value2 = 0;
  field_type = FT_UNINITIALIZED;
  request_method = RM_UNKNOWN;
  status_code = HttpStatusCode::HSC_UNDEFINDED;
  analyze(ft);
}

HttpHeaderLine::HttpHeaderLine(const char* str_, FieldType ft)
{
  str = new BString(str_);
  value1 = 0;
  value2 = 0;
  field_type = FT_UNINITIALIZED;
  request_method = RM_UNKNOWN;
  status_code = HttpStatusCode::HSC_UNDEFINDED;
  analyze(ft);
}

HttpHeaderLine::~HttpHeaderLine()
{
  delete str;
  delete value1;
  delete value2;
}

void
HttpHeaderLine::analyze()
{
  /* フィールド名をパース */
  struct Ft
  {
    FieldType t;
    const char* str;
  };
  static Ft fields[] = 
  {
    // general-header
    { FT_CACHE_CONTROL,		"Cache-Control" },
    { FT_CONNECTION,		"Connection" },
    { FT_DATE,			"Date" },
    { FT_PRAGMA,		"Pragma" },
    { FT_TRAILER,		"Trailer" },
    { FT_TRANSFER_ENCODING,	"Transfer-Encoding" },
    { FT_UPGRADE,		"Upgrade" },
    { FT_VIA,			"Via" },
    { FT_WARNING,		"Warning" },

    // request-header 
    { FT_ACCEPT,		"Accept" },
    { FT_ACCEPT_CHARSET,	"Accept-Charset" },
    { FT_ACCEPT_ENCODING,	"Accept-Encoding" },
    { FT_ACCEPT_LANGUAGE,	"Accept-Language" },
    { FT_AUTHORIZATION,		"Authorization" },
    { FT_EXPECT,		"Expect" },
    { FT_FROM,			"From" },
    { FT_HOST,			"Host" },
    { FT_IF_MATCH,		"If-Match" },
    { FT_IF_MODIFIED_SINCE,	"If-Modified-Since" },
    { FT_IF_NONE_MATCH,		"If-None-Match" },
    { FT_IF_RANGE,		"If-Range" },
    { FT_IF_UNMODIFIED_SINCE,	"If-Unmodified-Since" },
    { FT_MAX_FORWARDS,		"Max-Forwards" },
    { FT_PROXY_AUTHORIZATION,	"Proxy-Authorization" },
    { FT_RANGE,			"Range" },
    { FT_REFERER,		"Referer" },
    { FT_TE,			"TE" },
    { FT_USER_AGENT,		"User-Agent" },

    // response-header
    { FT_ACCEPT_RANGES,		"Accept-Ranges" },
    { FT_AGE,			"Age" },
    { FT_ETAG,			"ETag" },
    { FT_LOCATION,		"Location" },
    { FT_PROXY_AUTHENTICATE,	"Proxy-Authenticate" },
    { FT_RETRY_AFTER,		"Retry-After" },
    { FT_SERVER,		"Server" },
    { FT_VARY,			"Vary" },
    { FT_WWW_AUTHENTICATE,	"WWW-Authenticate" },

    // entity-header
    { FT_ALLOW,			"Allow" },
    { FT_CONTENT_ENCODING,	"Content-Encoding" },
    { FT_CONTENT_LANGUAGE,	"Content-Language" },
    { FT_CONTENT_LENGTH,	"Content-Length" },
    { FT_CONTENT_LOCATION,	"Content-Location" },
    { FT_CONTENT_MD5,		"Content-MD5" },
    { FT_CONTENT_RANGE,		"Content-Range" },
    { FT_CONTENT_TYPE,		"Content-Type" },
    { FT_EXPIRES,		"Expires" },
    { FT_LAST_MODIFIED,		"Last-Modified" },

    // cookie RFC6265
    { FT_COOKIE,		"Cookie" },
    { FT_SET_COOKIE,		"Set-Cookie" },

    { FT_UNINITIALIZED,	0 },	// END
  };

  StringIFlow<Int8> flow(str);
  TextReader<Int8> reader(&flow);
  try
  {
    BString* lws = reader.skipCandidates(" \t");
    if (lws && lws->getLength() >= 1)
    {
      field_type = FT_CONTINUE;
      value1 = lws;
    }
    else
    {
      delete lws;
      BString* f = reader.readDelimitedString(":");
      if (f == 0)
      {
	// ERROR
	field_type = FT_INVALID_LINE;
      }
      else
      {
	//printf("#%s\n", f->getCString());
	field_type = FT_UNKNOWN;
	Ft* x = fields;
	while (x->str)
	{
	  if (f->strNCaseCmp(x->str) == 0)
	  {
	    field_type = x->t;
	    break;
	  }
	  x++;
	}
	value1 = f;
      }
    }
    /* フィールドデータをパース */
    lws = reader.skipCandidates(" \t");
    delete lws;
    value2 = reader.readDelimitedString("\r\n");
  }
  catch (IOException ioe)
  {
  }
  //printf("HEADER: type:%d'%s' value:'%s'\n", field_type, value1->getCString(), value2->getCString());
}

void
HttpHeaderLine::analyze(FieldType ft)
{
  switch (ft)
  {
   case FT_REQUEST_LINE:
    analyzeRequestLine();
    break;
    
   case FT_STATUS_LINE:
    analyzeStatusLine();
    break;
    
   default:
#ifdef DEBUG
    fprintf(stderr, "ERROR: unsuppored field type, HttpHeaderLine::analyze(FieldType %d)\n", ft);
#endif /* DEBUG */
    assert(false);
    break;
  }
}

struct Rm
{
  HttpRequestMethod t;
  const char* str;
};
static Rm methods[] = 
{
  { RM_OPTIONS, "OPTIONS" },
  { RM_GET, "GET" },
  { RM_HEAD, "HEAD" },
  { RM_POST, "POST" },
  { RM_PUT, "PUT" },
  { RM_DELETE, "DELETE" },
  { RM_TRACE, "TRACE" },
  { RM_CONNECT, "CONNECT" },
  { RM_UNKNOWN, 0 },
};

void
HttpHeaderLine::analyzeRequestLine()
{
  // Request-Line   = Method SP  Request-URI SP HTTP-Version CRLF
  field_type = FT_REQUEST_LINE;
  //printf("METHOD: %s", str->getCString());
  StringIFlow<Int8> flow(str);
  TextReader<Int8> reader(&flow);
  try
  {
    request_method = RM_UNKNOWN;
    Rm* x = methods;
    while (x->str)
    {
      if (reader.readString(x->str))
      {
	request_method = x->t;
	break;
      }
      x++;
    }
    BString* sp = reader.skipChar(' ');
    delete sp;
    value1 = reader.readDelimitedString(" ");	/* Request-URI */
    sp = reader.skipChar(' ');
    delete sp;
    value2 = reader.readDelimitedString("\r\n");	/* HTTP-Version */
  }
  catch (IOException ioe)
  {
  }
  //printf("Requedt-Line: method:'%s' request-uri:'%s' http-version:'%s'\n",
  //	 methods[request_method].str, value1->getCString(), value2->getCString());
}

const char*
HttpHeaderLine::getRequestMethodString()
{
  Rm* x = methods;
  while (x->str)
  {
    if (request_method == x->t)
    {
      return x->str;
    }
    x++;
  }
  return "";
}

void
HttpHeaderLine::analyzeStatusLine()
{
  field_type = FT_STATUS_LINE;
  StringIFlow<Int8> flow(str);
  TextReader<Int8> reader(&flow);
  // ex. "HTTP/1.1 200 OK\r\n"
  try
  {
    value1 = reader.readDelimitedString(" ");	/* http version */
    BString* code_str = reader.readDelimitedString(" ");
    status_code = HttpStatusCode::getStatusCode(code_str->getCString());
    value2 = reader.readDelimitedString("\r\n");	/* reason */
  }
  catch (IOException ioe)
  {
  }
}

BString*
HttpHeaderLine::getHttpVersion()
{
  if (field_type == FT_REQUEST_LINE)
  {
    return value2;
  }
  if (field_type == FT_STATUS_LINE)
  {
    return value1;
  }
  return 0;
}

const char*
HttpHeaderLine::getFieldTypeString()
{
  struct Ft
  {
    FieldType t;
    const char* str;
  };
  static Ft fields[] = 
  {
    { FT_UNINITIALIZED,		"@UNINITIALIZED@" },
    { FT_INVALID_LINE,		"@INVALID_LINE@" },
    { FT_UNKNOWN,		"@UNKNOWN@" },
    { FT_CONTINUE,		"@CONTINUE@" },

    // Request-Line   = Method SP  Request-URI SP HTTP-Version CRLF
    { FT_REQUEST_LINE,		"@REQUEST_LINE@" },

    // Status-Line = HTTP-Version SP Status-Code SP Reason-Phrase CRLF
    { FT_STATUS_LINE,		"@STATUS_LINE@" },

    // general-header
    { FT_CACHE_CONTROL,		"Cache-Control" },
    { FT_CONNECTION,		"Connection" },
    { FT_DATE,			"Date" },
    { FT_PRAGMA,		"Pragma" },
    { FT_TRAILER,		"Trailer" },
    { FT_TRANSFER_ENCODING,	"Transfer-Encoding" },
    { FT_UPGRADE,		"Upgrade" },
    { FT_VIA,			"Via" },
    { FT_WARNING,		"Warning" },

    // request-header 
    { FT_ACCEPT,		"Accept" },
    { FT_ACCEPT_CHARSET,	"Accept-Charset" },
    { FT_ACCEPT_ENCODING,	"Accept-Encoding" },
    { FT_ACCEPT_LANGUAGE,	"Accept-Language" },
    { FT_AUTHORIZATION,		"Authorization" },
    { FT_EXPECT,		"Expect" },
    { FT_FROM,			"From" },
    { FT_HOST,			"Host" },
    { FT_IF_MATCH,		"If-Match" },
    { FT_IF_MODIFIED_SINCE,	"If-Modified-Since" },
    { FT_IF_NONE_MATCH,		"If-None-Match" },
    { FT_IF_RANGE,		"If-Range" },
    { FT_IF_UNMODIFIED_SINCE,	"If-Unmodified-Since" },
    { FT_MAX_FORWARDS,		"Max-Forwards" },
    { FT_PROXY_AUTHORIZATION,	"Proxy-Authorization" },
    { FT_RANGE,			"Range" },
    { FT_REFERER,		"Referer" },
    { FT_TE,			"TE" },
    { FT_USER_AGENT,		"User-Agent" },

    // response-header
    { FT_ACCEPT_RANGES,		"Accept-Ranges" },
    { FT_AGE,			"Age" },
    { FT_ETAG,			"ETag" },
    { FT_LOCATION,		"Location" },
    { FT_PROXY_AUTHENTICATE,	"Proxy-Authenticate" },
    { FT_RETRY_AFTER,		"Retry-After" },
    { FT_SERVER,		"Server" },
    { FT_VARY,			"Vary" },
    { FT_WWW_AUTHENTICATE,	"WWW-Authenticate" },

    // entity-header
    { FT_ALLOW,			"Allow" },
    { FT_CONTENT_ENCODING,	"Content-Encoding" },
    { FT_CONTENT_LANGUAGE,	"Content-Language" },
    { FT_CONTENT_LENGTH,	"Content-Length" },
    { FT_CONTENT_LOCATION,	"Content-Location" },
    { FT_CONTENT_MD5,		"Content-MD5" },
    { FT_CONTENT_RANGE,		"Content-Range" },
    { FT_CONTENT_TYPE,		"Content-Type" },
    { FT_EXPIRES,		"Expires" },
    { FT_LAST_MODIFIED,		"Last-Modified" },

    // cookie RFC6265
    { FT_COOKIE,		"Cookie" },
    { FT_SET_COOKIE,		"Set-Cookie" },

    { FT_UNINITIALIZED,	0 },	// END
  };

  const char* field_name = "@@";
  if (field_type == FT_REQUEST_LINE)
  {
    Rm* x = methods;
    while (x->str)
    {
      if (x->t == request_method)
      {
	field_name = x->str;
	break;
      }
      x++;
    }
  }
  else
  {
    Ft* x = fields;
    while (x->str)
    {
      if (x->t == field_type)
      {
	field_name = x->str;
	break;
      }
      x++;
    }
  }
  return field_name;
}

void
HttpHeaderLine::dump(TextWriter<Int8>* out)
{
  const char* line_type = "HEADER_LINE";
  switch (field_type)
  {
   case FT_REQUEST_LINE:
    line_type = "REQUEST_LINE";
    break;
    
   case FT_STATUS_LINE:
    line_type = "STATUS_LINE";
    break;
   default:
    break;
  }
  out->printf("%s: type:'%s' value1:'%s' value2:'%s'\n",
	      line_type,
	      getFieldTypeString(),
	      value1 ? value1->getCString() : "",
	      value2 ? value2->getCString() : "");
}

CCC_NAMESPACE_END(CCC)
