﻿// $Id$

#ifndef INCLUDE_ccc_http_HttpHeaderLine_h
#define INCLUDE_ccc_http_HttpHeaderLine_h

#include <ccc/base/base.h>
#include <ccc/base/TString.h>
#include <ccc/base/TextWriter.h>
#include <ccc/http/HttpRequestMethod.h>

CCC_NAMESPACE_START(CCC)

/*! HTTPのヘッダー行を表現します。 */
class HttpHeaderLine
{
 public:
  /*!
   * フィールド種別
   */
  enum FieldType
  {
    FT_UNINITIALIZED,		/*!< 未初期化 */
    FT_INVALID_LINE,		/*!< 不正な行 */
    FT_UNKNOWN,			/*!< 未定義フィールド */
    FT_CONTINUE,		/*!< 前行の継続行 */

    // Request-Line   = Method SP  Request-URI SP HTTP-Version CRLF
    FT_REQUEST_LINE,		/*!< リクエスト行(Method) */

    // Status-Line = HTTP-Version SP Status-Code SP Reason-Phrase CRLF
    FT_STATUS_LINE,		/*!< ステータス行 */

    // general-header
    FT_CACHE_CONTROL,		/*!< Cache-Control */
    FT_CONNECTION,		/*!< Connection */
    FT_DATE,			/*!< Date */
    FT_PRAGMA,			/*!< Pragma */
    FT_TRAILER,			/*!< Trailer */
    FT_TRANSFER_ENCODING,	/*!< Transfer-Encoding */
    FT_UPGRADE,			/*!< Upgrade */
    FT_VIA,			/*!< Via */
    FT_WARNING,			/*!< Warning */

    // request-header 
    FT_ACCEPT,			/*!< Accept */
    FT_ACCEPT_CHARSET,		/*!< Accept-Charset */
    FT_ACCEPT_ENCODING,		/*!< Accept-Encoding */
    FT_ACCEPT_LANGUAGE,		/*!< Accept-Language */
    FT_AUTHORIZATION,		/*!< Authorization */
    FT_EXPECT,			/*!< Expect */
    FT_FROM,			/*!< From */
    FT_HOST,			/*!< Host */
    FT_IF_MATCH,		/*!< If-Match */
    FT_IF_MODIFIED_SINCE,	/*!< If-Modified-Since */
    FT_IF_NONE_MATCH,		/*!< If-None-Match */
    FT_IF_RANGE,		/*!< If-Range */
    FT_IF_UNMODIFIED_SINCE,	/*!< If-Unmodified-Since */
    FT_MAX_FORWARDS,		/*!< Max-Forwards */
    FT_PROXY_AUTHORIZATION,	/*!< Proxy-Authorization */
    FT_RANGE,			/*!< Range */
    FT_REFERER,			/*!< Referer */
    FT_TE,			/*!< TE */
    FT_USER_AGENT,		/*!< User-Agent */

    // response-header
    FT_ACCEPT_RANGES,		/*!< Accept-Ranges */
    FT_AGE,			/*!< Age */
    FT_ETAG,			/*!< ETag */
    FT_LOCATION,		/*!< Location */
    FT_PROXY_AUTHENTICATE,	/*!< Proxy-Authenticate */
    FT_RETRY_AFTER,		/*!< Retry-After */
    FT_SERVER,			/*!< Server */
    FT_VARY,			/*!< Vary */
    FT_WWW_AUTHENTICATE,	/*!< WWW-Authenticate */

    // entity-header
    FT_ALLOW,			/*!< Allow */
    FT_CONTENT_ENCODING,	/*!< Content-Encoding */
    FT_CONTENT_LANGUAGE,	/*!< Content-Language */
    FT_CONTENT_LENGTH,		/*!< Content-Length */
    FT_CONTENT_LOCATION,	/*!< Content-Location */
    FT_CONTENT_MD5,		/*!< Content-MD5 */
    FT_CONTENT_RANGE,		/*!< Content-Range */
    FT_CONTENT_TYPE,		/*!< Content-Type */
    FT_EXPIRES,			/*!< Expires */
    FT_LAST_MODIFIED,		/*!< Last-Modified */

    // cookie RFC6265
    FT_COOKIE,			/*!< Cookie */
    FT_SET_COOKIE,		/*!< Set-Cookie */
  };
 private:
  BString* str;			/*!< 行内容 */
  BString* value1;			/*!< フィールド1 */
  BString* value2;			/*!< フィールド2 */
  FieldType field_type;			/*!< フィールド種別 */
  HttpRequestMethod request_method;	/*!< リクエストメソッドの種別 */
  int status_code;			/*!< ステータスコード番号 */
 public:
  /*!
   * コンストラクタ
   * \param str 行内容(行末のCR+LFを含む)
   */
  HttpHeaderLine(BString* str);
  /*!
   * コンストラクタ
   * \param str 行内容(行末のCR+LFを含む)
   */
  HttpHeaderLine(const char* str);
  /*!
   * フィールドタイプコンストラクタ
   * \param str 行内容(行末のCR+LFを含む)
   * \param ft フィールドタイプ
   */
  HttpHeaderLine(BString* str, FieldType ft);
  /*!
   * フィールドタイプコンストラクタ
   * \param str 行内容(行末のCR+LFを含む)
   * \param ft フィールドタイプ
   */
  HttpHeaderLine(const char* str, FieldType ft);
  /*!
   * デストラクタ
   */
  ~HttpHeaderLine();
  /*!
   * 行内容文字列を返します。
   * \return 行内容文字列
   */
  BString* getHeaderLineString() { return str; }
  /*!
   * ヘッダ行の種別を取得します。
   * \return ヘッダ行の種別
   */
  FieldType getFieldType() { return field_type; }
  /*!
   * リクエストメソッドの種別を取得します。
   * \return リクエストメソッドの種別
   */
  HttpRequestMethod getRequestMethod() { return request_method; }
  /*!
   * リクエストメソッドの種別を示す文字列を取得します。
   * 返り値は変更したり、解放しないでください。
   * \return リクエストメソッドの種別文字列
   */
  const char* getRequestMethodString();
  /*!
   * リクエスト行のリクエストURIを取得します。
   * \return リクエストURI
   */
  BString* getRequestUri() { return field_type == FT_REQUEST_LINE ? value1 : 0; }
  /*!
   * リクエスト行もしくはステータス行のHTTPバージョンを取得します。
   * \return リクエストURI
   */
  BString* getHttpVersion();
  /*!
   * ステータス行のステータスコードを取得します。
   * \return ステータスコード
   */
  int getStatusCode() { return status_code; }
  /*!
   * ステータス行の事由文字列を取得します。
   * \return 事由文字列
   */
  BString* getReasonPhase() { return field_type == FT_STATUS_LINE ? value2 : 0; }
  /*!
   * フィールド名の値を取得します。
   * \return フィールド名の値
   */
  BString* getFieldName() { return value1; }
  /*!
   * フィールド値を取得します。
   * \return フィールド値
   */
  BString* getFieldValue() { return value2; }
  /*!
   * フィールド1の値を取得します。
   * \return フィールド1の値
   */
  BString* getValue1() { return value1; }
  /*!
   * フィールド2の値を取得します。
   * \return フィールド2の値
   */
  BString* getValue2() { return value2; }
  /*!
   * メッセージの内容を指定先に出力します。
   * \param out 出力先
   */
  void dump(TextWriter<Int8>* out);
 private:
  /*!
   * 行の内容を解析します。
   */
  void analyze();
  /*!
   * 特定のフィールドタイプで行の内容を解析します。
   * \param ft フィールドタイプ
   */
  void analyze(FieldType ft);
  /*!
   * リクエスト行として行内容を解析します。
   */
  void analyzeRequestLine();
  /*!
   * リクエスト行として行内容を解析します。
   */
  void analyzeStatusLine();
  /*!
   * フィールドタイプの文字列を取得します。
   * \return フィールド名の文字列
   */
  const char* getFieldTypeString();
};

CCC_NAMESPACE_END(CCC)

#endif /* INCLUDE_ccc_http_HttpHeaderLine_h */
