﻿// $Id$

#include <assert.h>
#include <string.h>
#include <ccc/base/Conveyer.h>
#include <ccc/base/MemOFlow.h>
#include <ccc/iceman/HtmlDetector.h>

CCC_NAMESPACE_START(CCC);

/*!
 * パーサ
 */
class HtmlParser
{
  /*!
   * 解析対象HTMLのイメージ
   */
  char* html;
  /*!
   * 現在の解析位置
   */
  char* current;
  /*!
   * 現在の解析位置がHTMLエレメント内の時true
   */
  bool in_html_p;
  /*!
   * 現在の解析位置がHEADエレメント内の時true
   */
  bool in_head_p;
  /*!
   * 判定結果
   */
  CeId ceid;
 public:
  /*!
   * コンストラクタ
   * \param html 解析対象のイメージ
   */
  HtmlParser(char* html);
  /*!
   * デストラクタ
   */
  ~HtmlParser();
  /*!
   * HTMLを解析しMETAタグのHTTP-EQUIV属性を検知してエンコーディングを判定します。
   * XHTMLの場合には、XML宣言内のencodingも検知してエンコーディングを判定します。
   * \return 判定結果の文字セット・エンコーディング, 判定できない場合にはCEID_NULLが返ります。
   */
  CeId detect();
 private:
  /*!
   * 文字列が解析対象にマッチしているかどうかを調べます。
   * 解析対象が小文字の場合には大文字に変換した上で比較します。
   * \param from 対象
   * \param token 比較対象文字列(大文字で指定します)
   * \retval true マッチ
   * \retval false マッチしていません。
   */
  static bool match(const char* from, const char* token);
  /*!
   * 文字列が複数候補にマッチしているかどうか調べます。
   * \param from 対象
   * \param tbl 終端が0にセットされた文字列の配列
   * \return マッチした1から始まる候補番号,いずれの候補にもマッチしなければ0が返ります。
   */
  static int matchTable(const char* from, const char** tbl);
  /*!
   * トップレベルの解析を行ないます。
   */
  void analizeTopLevel();
  /*!
   * コメントの<!--の以降をコメントの終わりまでパースします。
   */
  void parseComment();
  /*!
   * プロセッサインストラクションの<?以降をプロセッサインストラクション終わりまでパースします。
   */
  void parsePI();
  /*!
   * 実体化記法を;までパースします。
   */
  void parseEntity();
  /*!
   * エレメントをパースします。
   */
  void parseElement();
  /*!
   * エレメントの終了タグをパースします。
   */
  void parseEndElement();
};

HtmlParser::HtmlParser(char* html)
{
  HtmlParser::html = html;
  current = html;
  in_html_p = false;
  in_head_p = false;
  ceid = CEID_NULL;
}

HtmlParser::~HtmlParser()
{
}

CeId
HtmlParser::detect()
{
  analizeTopLevel();
  return ceid;
}

bool
HtmlParser::match(const char* p, const char* token)
{
  const char* t = token;
  for (;;)
  {
    if (*t == '\0')
    {
      return true;
    }
    if (*p == '\0')
    {
      break;
    }
    char c = *p;
    if ((c >= 'a') && (c <= 'z'))
    {
      c -= 'a' - 'A';
    }
    if (c != *t)
    {
      break;
    }
    p++;
    t++;
  }
  return false;
}

int
HtmlParser::matchTable(const char* from, const char** tbl)
{
  int n = 1;
  while (*tbl)
  {
    if (match(from, *tbl))
    {
      return n;
    }
    n++;
    tbl++;
  }
  return 0;
}

void
HtmlParser::analizeTopLevel()
{
  while (*current != '\0')
  {
    static const char* tbl[] = 
    {
      "<?",
      "<!--",
      "</",
      "<",
      "&",
      0,
    };
    int n = matchTable(current, tbl);
    if (n != 0)
    {
      current += strlen(tbl[n - 1]);
    }
    switch (n)
    {
     case 0:
      //put(*current);
      current++;
      break;

     case 1:	// <?
      parsePI();
      break;
      
     case 2:	// <!--
      parseComment();
      break;
      
     case 3:	// </
      //printf("parseEndElement:%s\n", current);
      parseEndElement();
      break;
      
     case 4:	// <
      //printf("parseElement:%s\n", current);
      parseElement();
      break;

     case 5:	// &
      parseEntity();
      break;
      
     default:
      assert(false);
      break;
    }
  }
}

void
HtmlParser::parseComment()
{
  static char s_comment_end[] = "-->";
  while (*current != '\0')
  {
    if (match(current, s_comment_end))
    {
      current += sizeof(s_comment_end) - 1;
      break;
    }
    current++;
  }
}

void
HtmlParser::parsePI()
{
  static char s_pi_end[] = "?>";
  while (*current != '\0')
  {
    if (match(current, s_pi_end))
    {
      current += sizeof(s_pi_end) - 1;
      break;
    }
    current++;
  }
}

void
HtmlParser::parseEntity()
{
  //CCC::BString entity;
  while (*current != '\0')
  {
    if (*current == ';')
    {
      current++;
      break;
    }
    current++;
  }
}

void
HtmlParser::parseElement()
{
  CCC::BString tag;
  bool solo_p = false;
  bool no_attribute_p = false;
  while (*current != '\0')
  {
    char c = *current;
    if ((c == ' ') ||
	(c == '\n') ||
	(c == '\r') ||
	(c == '\r'))
    {
      current++;
      break;
    }
    if (c == '>')
    {
      no_attribute_p = true;
      current++;
      break;
    }
    if ((c == '/') && (current[1] == '>'))
    {
      no_attribute_p = true;
      solo_p = true;
      current += 2;
      break;
    }
    tag.add(*current++);
  }

  BString content_attr;
  bool content_type_p = false;
  if (no_attribute_p == false)
  {
    bool tag_parse_p = true;
    
    while (tag_parse_p && *current != '\0')
    {
      static const char* tbl2[] = 
      {
	"/>",
	">",
	"HTTP-EQUIV=\"CONTENT-TYPE\"",
	"CONTENT=\"",
	0,
      };
      int n = matchTable(current, tbl2);
      switch (n)
      {
       case 0: // doesn't hit
	//printf("%c", *current);
	current++;
	break;

       case 1:	// "/>"
	solo_p = true;
	current += strlen(tbl2[n - 1]);
	tag_parse_p = false;
	break;
	
       case 2:	// ">"
	current += strlen(tbl2[n - 1]);
	tag_parse_p = false;
	break;

       case 3:	// http-equiv="Content-Type"
	/* Content-Typeは大文字小文字の区別無し、詳しくは次のURL
	 * http://questionbox.jp.msn.com/qa1665626.html?StatusCheck=ON
	 */
	current += strlen(tbl2[n - 1]);
	content_type_p = true;
	break;
	
       case 4:	// content="..."
	current += strlen(tbl2[n - 1]);
	while ((*current != '\0') && (*current != '"'))
	{
	  content_attr.add(*current++);
	}
	if (*current == '"')
	{
	  current++;
	}
	break;
      }
    }
  }
  
  static const char* tbl[] = 
  {
    "HTML",
    "HEAD",
    "META",
    0,
  };
  int n = matchTable(tag.getCString(), tbl);
  switch (n)
  {
   case 0:
    break;

   case 1:	// HTML
    if (solo_p == false)
    {
      in_html_p = true;
    }
    break;

   case 2:	// HEAD
    if (solo_p == false)
    {
      in_head_p = true;
    }
    break;
    
   case 3:	// META
    if (in_html_p && in_head_p && content_type_p)
    {
      // 'text/html; charset=shift_jis'
      char* content = content_attr.getCString();
      //printf("content:[%s]\n", content);
      static char text_html[] = "TEXT/HTML;";
      static char charset[] = "CHARSET=";
      if (match(content, text_html))	// RFC2045: Matching of media type and subtype is always case-insensitive.
      {
	content += (sizeof(text_html) - 1);
	while (*content == ' ')
	{
	  content++;
	}
	if (match(content, charset))	// RFC2045: Matching of media type and subtype is always case-insensitive.
	{
	  content += (sizeof(charset) - 1);
	  char* brace = strchr(content, '(');
	  if (brace)			/* "text/plain; charset=us-ascii (Plain text)" のようなparameter付のケース */
	  {
	    BString buf(content, (Size)(brace - content));
	    //printf("charset:[%s]\n", buf.getCString());
	    ceid = Iceman::stringToCeId(buf.getCString());
	  }
	  else
	  {
	    //printf("charset:[%s]\n", content);
	    ceid = Iceman::stringToCeId(content);
	  }
	}
      }
    }
    //printf("<meta:%s>\n", tag.getCString());
    break;
    
   default:
    assert(false);
    break;
  }
}

void
HtmlParser::parseEndElement()
{
  CCC::BString tag;
  while (*current != '\0')
  {
    if (*current == '>')
    {
      current++;
      break;
    }
    tag.add(*current++);
  }
  static const char* tbl[] = 
  {
    "HTML",
    "HEAD",
    0,
  };
  int n = matchTable(tag.getCString(), tbl);
  switch (n)
  {
   case 0:
    break;

   case 1:	// HTML
    in_html_p = false;
    break;

   case 2:	// HEAD
    in_head_p = false;
    break;
    
   default:
    assert(false);
    break;
  }
}

// ------------------------------------------------------------------------
// class HtmlDetector
HtmlDetector::HtmlDetector()
{
  // <META HTTP-EQUIV="Content-Type" CONTENT="text/html; charset=shift_jis">
}

HtmlDetector::~HtmlDetector()
{
}

CeId
HtmlDetector::detect(IFlow* in)
{
  MemOFlow out;
  Conveyer conv(in, &out);
  IOType io_type = in->getIOType();
  if (io_type & IOTYPE_INT8)
  {
    try
    {
      conv.carry(IOTYPE_INT8);
    }
    catch (IOException* /* ioe */)
    {
    }
    BString html_image((char*)out.getMem(), out.getSize());
    //printf("%s\n", html_image.getCString());
    HtmlParser parser(html_image.getCString());
    CeId ceid = parser.detect();
    return ceid;
  }
  return CEID_NULL;
}

bool
HtmlDetector::rewind()
{
  return true;
}

Detector*
HtmlDetector::createDetector()
{
  return new HtmlDetector;
}

CCC_NAMESPACE_END(CCC);

#if 0
// for TEST

#include <ccc/base/base.h>
#include <ccc/base/memflow.h>
#include <ccc/iceman/HtmlDetector.h>
#include <ccc/iceman/Iceman.h>

static char test1[] = 
"<html>\n"
"<head>\n"
"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=shift_jis\">\n"
"<meta http-equiv=\"content-script-type\" content=\"text/javascript\" />\n"
"<title>test</title>\n"
"</head>\n"
"<body>\n"
"<h1>test1</h1>\n"
"</body>\n"
"</html>\n";

static char test2[] = 
"<html>\n"
"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=shift_jis\">\n"		/* METAがHEADにないケース */
"<head>\n"
"<meta http-equiv=\"content-script-type\" content=\"text/javascript\" />\n"
"<title>test</title>\n"
"</head>\n"
"<body>\n"
"<h1>test1</h1>\n"
"</body>\n"
"</html>\n";

static char test3[] = 
"<html>\n"
"<head>\n"
"<meta http-equiv=\"Content-Type\" content=\"text/html;charset=UTF8\">\n"
"<meta http-equiv=\"content-script-type\" content=\"text/javascript\" />\n"
"<title>test</title>\n"
"</head>\n"
"<body>\n"
"<h1>test1</h1>\n"
"</body>\n"
"</html>\n";

static char test4[] = 
"<html>\n"
"<head>\n"
"<meta http-equiv=\"Content-Type\" content=\"text/html;  charset=EUC-JP\">\n"
"<meta http-equiv=\"content-script-type\" content=\"text/javascript\" />\n"
"<title>test</title>\n"
"</head>\n"
"<body>\n"
"<h1>test1</h1>\n"
"</body>\n"
"</html>\n";

static char test5[] = 
"<html>\n"
"<head>\n"
"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=us-ascii(Plain text)\">\n"
"<meta http-equiv=\"content-script-type\" content=\"text/javascript\" />\n"
"<title>test</title>\n"
"</head>\n"
"<body>\n"
"<h1>test1</h1>\n"
"</body>\n"
"</html>\n";

void
doTest(char* test, size_t size)
{
  CCC::HtmlDetector detector;
  CCC::FixedMemIFlow in(test, size);
  CCC::CeId ceid = detector.detect(&in);
  printf("ceid:%d %s\n", ceid, CCC::Iceman::ceIdToString(ceid));
}

int
main()
{
  CCC::Ccc::initialize();

  doTest(test1, sizeof(test1));
  doTest(test2, sizeof(test2));
  doTest(test3, sizeof(test3));
  doTest(test4, sizeof(test4));
  doTest(test5, sizeof(test5));

  CCC::Ccc::unInitialize();
}
#endif
