﻿// @@DISTHDR@@
// $Id$

#include <assert.h>
#include <ccc/base/base.h>
#include <ccc/base/CompositIFilter.h>
#include <ccc/file/FileIFlow.h>
#include <ccc/file/FileOFlow.h>
#include <ccc/base/Conveyer.h>
#include <ccc/base/FixedMemIFlow.h>
#include <ccc/base/FixedMemOFlow.h>
#include <ccc/base/StringIFlow.h>
#include <ccc/base/StringOFlow.h>
#include <ccc/iceman/Iceman.h>
#include <ccc/iceman/CenDictionary.h>
#include <ccc/iceman/IFilterDictionary.h>
#include <ccc/iceman/unitfilter.h> 
#include <ccc/iceman/Detector.h>
#include <ccc/iceman/DetectorDictionary.h>
#include <ccc/iceman/utility.h>

CCC_NAMESPACE_START(CCC);

bool
Iceman::initialize()
{
  CenDictionary::initialize();
  IFilterDictionary::initialize();
  return true;
}

bool
Iceman::unInitialize()
{
  IFilterDictionary::unInitialize();
  CenDictionary::unInitialize();
  return true;
}

static bool
initializeIceman(bool init_p)
{
  if (init_p)
  {
    return Iceman::initialize();
  }
  else
  {
    return Iceman::unInitialize();
  }
}

static int initialize_iceman_dummy = Ccc::setInitializeHandler(CM_ICEMAN, &initializeIceman);

// charset encoding name

CeId
Iceman::stringToCeId(const char* str)
{
  return CenDictionary::searchId(str);
}

const char*
Iceman::ceIdToString(CeId ceid)
{
  return CenDictionary::searchStr(ceid);
}

struct CeIdCeGId
{
  CeId id;
  CeGId gid;
  CeId nid;
};

CeIdCeGId ceidcegid_map[] =
{
#define CCC_ICEMAN_CEID_CEGID_MAP
#include <ccc/iceman/ces.h>  
#undef CCC_ICEMAN_CEID_CEGID_MAP
  { CEID_NULL, CEGID_NOGROUP },
};

CeGId
Iceman::getCeGId(CeId ceid)
{
  CeIdCeGId* p = ceidcegid_map;
  while (p->id != CEID_NULL)
  {
    if (p->id == ceid)
    {
      return p->gid;
    }
    p++;
  }
  return CEGID_NOGROUP;
}

CeId
Iceman::getNativeCeId(CeId ceid)
{
  CeIdCeGId* p = ceidcegid_map;
  while (p->id != CEID_NULL)
  {
    if (p->id == ceid)
    {
      return p->nid;
    }
    p++;
  }
  return CEID_NULL;
}

IOType
Iceman::getIOType(CeId ceid)
{
  IOType dest_io_type;
  CeGId gid = CCC::Iceman::getCeGId(ceid);
  switch (gid)
  {
   case CEGID_8BIT:
    dest_io_type = IOTYPE_INT8;
    break;
   case CEGID_16BIT_NE:
    dest_io_type = IOTYPE_UINT16;
    break;
   case CEGID_32BIT_NE:
    dest_io_type = IOTYPE_UINT32;
    break;
   default:
    dest_io_type = IOTYPE_NULL;
    break;
  }
  return dest_io_type;
}

CharsetEncoding*
Iceman::getCharsetEncodingList()
{
  static CharsetEncoding charset_encoding[] =
  {
#define CCC_ICEMAN_CEN_CHARSET_DEF
#include <ccc/iceman/ces.h>
#undef CCC_ICEMAN_CEN_CHARSET_DEF
    { CEID_NULL, CEGID_NOGROUP, CEID_NULL, 0, 0, 0, 0 },
  };
  return charset_encoding;
}

// converter

void
Iceman::createIFilters(IFilterDictionaryEntry* fde, CompositIFilter* cf)
{
  int len = fde->getLength();
  for (int i = 0; i < len; i++)
  {
    assert((*fde)[i] != -1);
    UnitIFilterEntry* ufe = &unit_filters[(*fde)[i]];
    IFilter* f = (*ufe->create_func)();
    cf->add(f);
  }
}

CompositIFilter*
Iceman::createCompositIFilter(CeId from, CeId to)
{
  // null filter support
  if (from == to)
  {
    CompositIFilter* cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
    IFilter* null_filter = new IFilter(IOTYPE_ANY, IOTYPE_ANY);
    cf->add(null_filter);
    return cf;
  }
  
  IFilterDictionaryEntry* fde = IFilterDictionary::searchId(from, to);
  if (!fde)
  {
    return 0;
  }
  CompositIFilter* cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
  createIFilters(fde, cf);
  return cf;
}

CompositIFilter*
Iceman::createCompositIFilter(char* from, char* to)
{
  CeId ce_from = CenDictionary::searchId(from);
  CeId ce_to = CenDictionary::searchId(to);
  if ((ce_from == CEID_NULL) ||
      (ce_to == CEID_NULL))
  {
    return 0;
  }
  return createCompositIFilter(ce_from, ce_to);
}

CompositIFilter*
Iceman::createCompositIFilter(CeId from, CeId to, Utility* util, UtilPos pos)
{
  CeId util_from = util->getFromCeId();
  CeId util_to = util->getToCeId();
  CompositIFilter* cf = 0;
  if ((from == util_from) && (pos != UTILPOS_ONLY_LAST))
  {
    // util -> filters
    IFilterDictionaryEntry* fde = IFilterDictionary::searchId(util_to, to);
    if (fde || (util_to == to))
    {
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      cf->add(util);
      if (fde)
      {
	createIFilters(fde, cf);
      }
    }
  }
  if ((cf == 0) && (to == util_to) && (pos != UTILPOS_ONLY_FIRST))
  {
    // filters -> util
    IFilterDictionaryEntry* fde = IFilterDictionary::searchId(from, util_from);
    if (fde || (from == util_from))
    {
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      if (fde)
      {
	createIFilters(fde, cf);
      }
      cf->add(util);
    }
  }
  if ((cf == 0) && (pos == UTILPOS_ANYWHERE))
  {
    // pre filters -> util -> post filters
    IFilterDictionaryEntry* fde_pre = IFilterDictionary::searchId(from, util_from);
    IFilterDictionaryEntry* fde_post = IFilterDictionary::searchId(util_to, to);
    if ((fde_pre || (from == util_from)) && 
	(fde_post || (util_to == to)))
    {
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      if (fde_pre)
      {
	createIFilters(fde_pre, cf);
      }
      cf->add(util);
      if (fde_post)
      {
	createIFilters(fde_post, cf);
      }
    }
  }
  return cf;
}

CompositIFilter*
Iceman::createCompositIFilter(CeId from, CeId to, GroupUtility* gutil, UtilPos pos)
{
  CeGId gid = gutil->getFromToCeGId();
  CeGId from_gid = getCeGId(from);
  CeGId to_gid = getCeGId(to);
  CompositIFilter* cf = 0;
  IFilterDictionaryEntry* fde = IFilterDictionary::searchId(from, to);
  if (fde || (from == to))
  {
    if ((from_gid == gid) && (pos != UTILPOS_ONLY_LAST))
    {
      // gutil -> filters
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      cf->add(gutil);
      if (fde)
      {
	createIFilters(fde, cf);
      }
    }
    if ((cf == 0) && (to_gid == gid))
    {
      // filters -> gutil
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      if (fde)
      {
	createIFilters(fde, cf);
      }
      cf->add(gutil);
    }
    if ((cf == 0) && (from_gid == CEGID_NOGROUP) && (pos != UTILPOS_ONLY_LAST))
    {
      // gutil -> native charset -> filters
      CeId from_na = getNativeCeId(from);
      CeGId from_na_gid = getCeGId(from_na);
      if (from_na_gid == gid)
      {
	IFilterDictionaryEntry* fde_na = IFilterDictionary::searchId(from, from_na);
	fde = IFilterDictionary::searchId(from_na, to);
	if ((fde_na || from == from_na) && 
	    (fde || from_na == to))
	{
	  cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
	  if (fde_na)
	  {
	    createIFilters(fde_na, cf);
	  }
	  cf->add(gutil);
	  if (fde)
	  {
	    createIFilters(fde, cf);
	  }
	}
      }
    }
    if ((cf == 0) && (to_gid == CEGID_NOGROUP) && (pos != UTILPOS_ONLY_FIRST))
    {
      // filters -> native charset -> gutil
      CeId to_na = getNativeCeId(to);
      CeGId to_na_gid = getCeGId(to_na);
      if (to_na_gid == gid)
      {
	fde = IFilterDictionary::searchId(from, to_na);
	IFilterDictionaryEntry* fde_na = IFilterDictionary::searchId(to_na, to);
	if ((fde || from == to_na) && 
	    (fde_na || to_na == to))
	{
	  cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
	  if (fde)
	  {
	    createIFilters(fde, cf);
	  }
	  cf->add(gutil);
	  if (fde_na)
	  {
	    createIFilters(fde_na, cf);
	  }
	}
      }
    }
  }
  return cf;
}

CompositIFilter*
Iceman::createCompositIFilter(CeId from, CeId to, Utility* util, GroupUtility* gutil, UtilPos pos)
{
  CompositIFilter* cf_util = createCompositIFilter(from, to, util, pos);

  CeGId gid = gutil->getFromToCeGId();
  CeGId from_gid = getCeGId(from);
  CeGId to_gid = getCeGId(to);
  CompositIFilter* cf = 0;
  if (!!cf_util)
  {
    if ((from_gid == gid) && (pos != UTILPOS_ONLY_LAST))
    {
      // gutil -> cf_util
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      cf->add(gutil);
      cf_util->moveIFilters(cf);
    }
    if ((cf == 0) && (to_gid == gid))
    {
      // cf_util -> gutil
      cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
      cf_util->moveIFilters(cf);
      cf->add(gutil);
    }
    if ((cf == 0) && (from_gid == CEGID_NOGROUP) && (pos != UTILPOS_ONLY_LAST))
    {
      // (from->from_na) -> gutil -> (from_na -> from) -> cf_util
      CeId from_na = getNativeCeId(from);
      CeGId from_na_gid = getCeGId(from_na);
      if (from_na_gid == gid)
      {
	IFilterDictionaryEntry* fde1 = IFilterDictionary::searchId(from, from_na);
	IFilterDictionaryEntry* fde2 = IFilterDictionary::searchId(from_na, from);
	if (!!fde1 && !!fde2)
	{
	  cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
	  createIFilters(fde1, cf);
	  cf->add(gutil);
	  createIFilters(fde2, cf);
	  cf_util->moveIFilters(cf);
	}
      }
    }
    if ((cf == 0) && (to_gid == CEGID_NOGROUP) && (pos != UTILPOS_ONLY_FIRST))
    {
      // cf_util -> (to -> to_na) -> gutil -> (to_na -> to)
      CeId to_na = getNativeCeId(to);
      CeGId to_na_gid = getCeGId(to_na);
      if (to_na_gid == gid)
      {
	IFilterDictionaryEntry* fde1 = IFilterDictionary::searchId(to, to_na);
	IFilterDictionaryEntry* fde2 = IFilterDictionary::searchId(to_na, to);
	if (!!fde1 && !!fde2)
	{
	  cf = new CompositIFilter(IOTYPE_ANY, IOTYPE_ANY);
	  cf_util->moveIFilters(cf);
	  createIFilters(fde1, cf);
	  cf->add(gutil);
	  createIFilters(fde2, cf);
	}
      }
    }
    delete cf_util;
  }
  return cf;
}

CompositIFilter*
Iceman::createCompositIFilter(CeId from_id, CeId to_id, NLConv nlc)
{
  CompositIFilter* cf = 0;
  if (nlc != NLCONV_NONE)
  {
    GroupUtility* gu = createNLConvGroupUtility(nlc, CEGID_8BIT);
    cf = createCompositIFilter(from_id, to_id, gu, UTILPOS_ANYWHERE);
    if (!cf)
    {
      delete gu;
      gu = createNLConvGroupUtility(nlc, CEGID_16BIT_NE);
      cf = createCompositIFilter(from_id, to_id, gu, UTILPOS_ANYWHERE);
    }
    if (!cf)
    {
      delete gu;
      gu = createNLConvGroupUtility(nlc, CEGID_32BIT_NE);
      cf = createCompositIFilter(from_id, to_id, gu, UTILPOS_ANYWHERE);
    }
  }
  else
  {
    cf = createCompositIFilter(from_id, to_id);
  }
  return cf;
}

bool
Iceman::convert(CompositIFilter* cf, FILE* in, FILE* out)
{
  FileIFlow src;
  src.attach(in);
  FileOFlow dest;
  dest.attach(out);
  cf->setSrc(&src);
  Conveyer conv;
  conv.setSrc(cf);
  conv.setDest(&dest);
  try
  {
    conv.carry();
  }
  catch (IOException /* ioe */)
  {
    return false;
  }
  return true;
}

bool
Iceman::convert(CeId from, CeId to, FILE* in, FILE* out)
{
  CompositIFilter* cf = createCompositIFilter(from, to);
  if (!cf)
  {
    return false;
  }

  FileIFlow src;
  src.attach(in);
  FileOFlow dest;
  dest.attach(out);
  cf->setSrc(&src);
  Conveyer conv;
  conv.setSrc(cf);
  conv.setDest(&dest);
  bool ret = true;
  try
  {
    conv.carry();
  }
  catch (IOException /* ioe */)
  {
    ret = false;
  }
  delete cf;
  return ret;
}

bool
Iceman::convert(char* from, char* to, FILE* in, FILE* out)
{
  CeId ce_from = CenDictionary::searchId(from);
  CeId ce_to = CenDictionary::searchId(to);
  if ((ce_from == CEID_NULL) ||
      (ce_to == CEID_NULL))
  {
    return false;
  }
  return convert(ce_from, ce_to, in, out);
}

bool
Iceman::convert(CompositIFilter* cf, void* in, Size in_size, void* out, Size out_size)
{
  FixedMemIFlow src;
  src.set(in, in_size);
  FixedMemOFlow dest;
  dest.set(out, out_size);
  cf->setSrc(&src);
  Conveyer conv;
  conv.setSrc(cf);
  conv.setDest(&dest);
  try
  {
    conv.carry();
  }
  catch (IOException /* ioe */)
  {
    return false;
  }
  return true;
}

bool
Iceman::convert(CeId from, CeId to, void* in, Size in_size, void* out, Size out_size)
{
  CompositIFilter* cf = createCompositIFilter(from, to);
  if (!cf)
  {
    return false;
  }

  FixedMemIFlow src;
  src.set(in, in_size);
  FixedMemOFlow dest;
  dest.set(out, out_size);
  cf->setSrc(&src);
  Conveyer conv;
  conv.setSrc(cf);
  conv.setDest(&dest);
  bool ret = true;
  try
  {
    conv.carry();
  }
  catch (IOException /* ioe */)
  {
    ret = false;
  }
  delete cf;
  return ret;
}

bool
Iceman::convert(char* from, char* to, void* in, Size in_size, void* out, Size out_size)
{
  CeId ce_from = CenDictionary::searchId(from);
  CeId ce_to = CenDictionary::searchId(to);
  if ((ce_from == CEID_NULL) ||
      (ce_to == CEID_NULL))
  {
    return false;
  }
  return convert(ce_from, ce_to, in, in_size, out, out_size);
}

#define CONVERT_TO_CF_STRING(t_from, t_to) \
  dest->clear();			\
  CCC::StringIFlow<t_from> from(src);\
  CCC::StringOFlow<t_to> to(dest);\
  CCC::Conveyer conv;\
  cf->setSrc(&from); \
  conv.setSrc(cf);\
  conv.setDest(&to);\
  try\
  {\
    conv.carry();\
  }\
  catch (CCC::IOException /* ioe */)\
  {\
    assert(false);\
    return false;\
  }\
  return true;

bool
Iceman::convertToBString(IFilter* cf, const BString* src, BString* dest)
{
  CONVERT_TO_CF_STRING(CCC::Int8, CCC::Int8);
}

bool
Iceman::convertToBString(IFilter* cf, const DString* src, BString* dest)
{
  CONVERT_TO_CF_STRING(CCC::UInt16, CCC::Int8);
}

bool
Iceman::convertToBString(IFilter* cf, const WString* src, BString* dest)
{
  CONVERT_TO_CF_STRING(CCC::UInt32, CCC::Int8);
}

bool
Iceman::convertToDString(IFilter* cf, const BString* src, DString* dest)
{
  CONVERT_TO_CF_STRING(CCC::Int8, CCC::UInt16);
}

bool
Iceman::convertToDString(IFilter* cf, const DString* src, DString* dest)
{
  CONVERT_TO_CF_STRING(CCC::UInt16, CCC::UInt16);
}

bool
Iceman::convertToDString(IFilter* cf, const WString* src, DString* dest)
{
  CONVERT_TO_CF_STRING(CCC::UInt32, CCC::UInt16);
}

bool
Iceman::convertToWString(IFilter* cf, const BString* src, WString* dest)
{
  CONVERT_TO_CF_STRING(CCC::Int8, CCC::UInt32);
}

bool
Iceman::convertToWString(IFilter* cf, const DString* src, WString* dest)
{
  CONVERT_TO_CF_STRING(CCC::UInt16, CCC::UInt32);
}

bool
Iceman::convertToWString(IFilter* cf, const WString* src, WString* dest)
{
  CONVERT_TO_CF_STRING(CCC::UInt32, CCC::UInt32);
}

#define CONVERT_TO_STRING(t_from, t_to) \
  dest->clear();			\
  CCC::CompositIFilter* cf = createCompositIFilter(from, to);\
  if (!cf)\
  {\
    return 0;\
  }\
  CCC::StringIFlow<t_from> from_flow(src);\
  CCC::StringOFlow<t_to> to_flow(dest);\
  CCC::Conveyer conv;\
  cf->setSrc(&from_flow); \
  conv.setSrc(cf);\
  conv.setDest(&to_flow);\
  try\
  {\
    conv.carry();\
  }\
  catch (CCC::IOException /* ioe */)\
  {\
    assert(false);\
    return false;\
  }\
  delete cf;\
  return true;

bool
Iceman::convertToBString(CeId from, CeId to, const BString* src, BString* dest)
{
  CONVERT_TO_STRING(CCC::Int8, CCC::Int8);
}

bool
Iceman::convertToBString(CeId from, CeId to, const DString* src, BString* dest)
{
  CONVERT_TO_STRING(CCC::UInt16, CCC::Int8);
}
  
bool
Iceman::convertToBString(CeId from, CeId to, const WString* src, BString* dest)
{
  CONVERT_TO_STRING(CCC::UInt32, CCC::Int8);
}

bool
Iceman::convertToDString(CeId from, CeId to, const BString* src, DString* dest)
{
  CONVERT_TO_STRING(CCC::Int8, CCC::UInt16);
}

bool
Iceman::convertToDString(CeId from, CeId to, const DString* src, DString* dest)
{
  CONVERT_TO_STRING(CCC::UInt16, CCC::UInt16);
}

bool
Iceman::convertToDString(CeId from, CeId to, const WString* src, DString* dest)
{
  CONVERT_TO_STRING(CCC::UInt32, CCC::UInt16);
}

bool
Iceman::convertToWString(CeId from, CeId to, const BString* src, WString* dest)
{
  CONVERT_TO_STRING(CCC::Int8, CCC::UInt32);
}

bool
Iceman::convertToWString(CeId from, CeId to, const DString* src, WString* dest)
{
  CONVERT_TO_STRING(CCC::UInt16, CCC::UInt32);
}

bool
Iceman::convertToWString(CeId from, CeId to, const WString* src, WString* dest)
{
  CONVERT_TO_STRING(CCC::UInt32, CCC::UInt32);
}

// detector

Detector*
Iceman::createDetector(CeId detector)
{
  DetectorEntry* de = DetectorDictionary::search(detector);
  if (!de)
  {
    return 0;
  }
  return (de->func)();
}

void
Iceman::deleteDetector(Detector* detector)
{
  delete detector;
}

CeId
Iceman::detect(Detector* detector, FILE* in)
{
  FileIFlow src;
  src.attach(in);
  CeId id = detector->detect(&src);
  return id;
}

CeId
Iceman::detect(Detector* detector, void* in, Size in_size)
{
  FixedMemIFlow src;
  src.set(in, in_size);
  CeId id = detector->detect(&src);
  return id;
}

CeId
Iceman::detect(CeId detector_id, FILE* in)
{
  FileIFlow src;
  src.attach(in);

  Detector* detector = createDetector(detector_id);
  if (!detector)
  {
    return CEID_NULL;
  }
  
  CeId id = detector->detect(&src);
  delete detector;
  
  return id;
}

CeId
Iceman::detect(CeId detector_id, void* in, Size in_size)
{
  FixedMemIFlow src;
  src.set(in, in_size);

  Detector* detector = createDetector(detector_id);
  if (!detector)
  {
    return CEID_NULL;
  }
  
  CeId id = detector->detect(&src);
  delete detector;
  
  return id;
}

// ------------------------------------------------------------------------
// C interface

bool
Iceman_initialize()
{
  return Iceman::initialize();
}

bool
Iceman_unInitialize()
{
  return Iceman::unInitialize();
}

CeId
Iceman_stringToCeId(const char* str)
{
  return Iceman::stringToCeId(str);
}

const char*
Iceman_ceIdToString(CeId ceid)
{
  return Iceman::ceIdToString(ceid);
}

CeGId
Iceman_getCeGId(CeId ceid)
{
  return Iceman::getCeGId(ceid);
}

CeId
Iceman_getNativeCeId(CeId ceid)
{
  return Iceman::getNativeCeId(ceid);
}

CharsetEncoding*
Iceman_getCharsetEncodingList()
{
  return Iceman::getCharsetEncodingList();
}

CompositIFilter*
Iceman_createCompositIFilter0(CeId from, CeId to)
{
  return Iceman::createCompositIFilter(from, to);
}

CompositIFilter*
Iceman_createCompositIFilter1(char* from, char* to)
{
  return Iceman::createCompositIFilter(from, to);
}

CompositIFilter*
Iceman_createCompositIFilter2(CeId from, CeId to, Utility* util, UtilPos pos)
{
  return Iceman::createCompositIFilter(from, to, util, pos);
}

CompositIFilter*
Iceman_createCompositIFilter3(CeId from, CeId to, GroupUtility* gutil, UtilPos pos)
{
  return Iceman::createCompositIFilter(from, to, gutil, pos);
}

CompositIFilter*
Iceman_createCompositIFilter4(CeId from, CeId to, Utility* util, GroupUtility* gutil, UtilPos pos)
{
  return Iceman::createCompositIFilter(from, to, util, gutil, pos);
}

CompositIFilter*
Iceman_createCompositIFilter5(CeId from, CeId to, NLConv nlc)
{
  return Iceman::createCompositIFilter(from, to, nlc);
}

void
Iceman_deleteCompositIFilter(CompositIFilter* cf)
{
  delete cf;
}

void
Iceman_deleteUtility(Utility* util)
{
  delete util;
}

void
Iceman_deleteGroupUtility(GroupUtility* gutil)
{
  delete gutil;
}

bool
Iceman_convert0(CompositIFilter* cf, FILE* in, FILE* out)
{
  return Iceman::convert(cf, in, out);
}

bool
Iceman_convert1(CeId from, CeId to, FILE* in, FILE* out)
{
  return Iceman::convert(from, to, in, out);
}

bool
Iceman_convert2(char* from, char* to, FILE* in, FILE* out)
{
  return Iceman::convert(from, to, in, out);
}

bool
Iceman_convert3(CompositIFilter* cf, void* in, Size in_size, void* out, Size out_size)
{
  return Iceman::convert(cf, in, in_size, out, out_size);
}

bool
Iceman_convert4(CeId from, CeId to, void* in, Size in_size, void* out, Size out_size)
{
  return Iceman::convert(from, to, in, in_size, out, out_size);
}

bool
Iceman_convert5(char* from, char* to, void* in, Size in_size, void* out, Size out_size)
{
  return Iceman::convert(from, to, in, in_size, out, out_size);
}

Detector*
Iceman_createDetector(CeId detector)
{
  return Iceman::createDetector(detector);
}

void
Iceman_deleteDetector(Detector* detector)
{
#if 1
  Iceman::deleteDetector(detector);
#else  
  return    Iceman::deleteDetector(detector);
#endif
}

CeId
Iceman_detect0(CeId detector, FILE* in)
{
  return Iceman::detect(detector, in);
}

CeId
Iceman_detect1(CeId detector, void* in, Size in_size)
{
  return Iceman::detect(detector, in, in_size);
}

CeId
Iceman_detect2(Detector* detector, FILE* in)
{
  FileIFlow src;
  src.attach(in);
  CeId id = detector->detect(&src);
  return id;
}

CeId
Iceman_detect3(Detector* detector, void* in, Size in_size)
{
  FixedMemIFlow src;
  src.set(in, in_size);
  CeId id = detector->detect(&src);
  return id;
}

CCC_NAMESPACE_END(CCC);
