﻿/* @@DISTHDR@@ */

/* $Id$ */
/*!
 * \file Iceman.h
 * \ja
 * \brief Icemanの基本的なインターフェースはこのファイルで定義されています。
 * \ja_end
 *
 * \en
 * \brief In this file basic interfaces of Iceman are defined.
 * \en_end
 */

#ifndef INCLUDE_ccc_iceman_Iceman_h
#define INCLUDE_ccc_iceman_Iceman_h

#include <stdio.h>
#include <ccc/base/base.h>
#include <ccc/base/Flow.h>
#include <ccc/base/TString.h>
#include <ccc/base/IFilter.h>
#include <ccc/iceman/config.h>

CCC_NAMESPACE_START(CCC);

/*!
 * \ja
 * Icemanのバージョン文字列
 * \ja_end
 * \en
 * an Iceman version string.
 * \en_end
 */
#define ICEMAN_VERSION_STR "2.0"

enum
{
  /*!
   * \ja
   * 複合フィルタのパスの最大長
   * \ja_end
   * \en
   * Maximum length of the composit filter path.
   * \en_end
   */
  ICEMAN_MAX_COMPOSIT_PATH = 8,
};
  
/*!
 * \ja
 * 文字セットエンコーディングのグループID
 * \ja_end
 * \en
 * Charset encoding group id.
 * \en_end
 */
enum CeGId
{
  /*!
   * \ja
   * どのグループにも属しません。
   * \ja_end
   * \en
   * It doesn't belong to any group.
   * \en_end
   */
  CEGID_NOGROUP = 0,
  /*!
   * \ja
   * 8ビットのストリーム
   * \ja_end
   * \en
   * 8 bits stream
   * \en_end
   */
  CEGID_8BIT = 1,
  /*!
   * \ja
   * 16ビットのネイティブエンディアンのストリーム
   * \ja_end
   * \en
   * 16 bits native endian stream.
   * \en_end
   */
  CEGID_16BIT_NE = 2,
  /*!
   * \ja
   * 32ビットのネイティブエンディアンのストリーム
   * \ja_end
   * \en
   * 32 bits native endian stream.
   * \en_end
   */
  CEGID_32BIT_NE = 4,
  /*!
   * \ja
   * ディテクター
   * \ja_end
   * \en
   * Detector
   * \en_end
   */
  CDGID_DETECTOR = 8,
};

#ifndef __cplusplus
typedef enum CeGId CeGId;
#endif /* __cplusplus */
  
/*!
 * \ja
 * 文字セットエンコーディングのID
 * \ja_end
 * \en
 * ID of the charset encoding.
 * \en_end
 */
enum CeId
{
  /*!
   * \ja
   * ヌル、エラー等を示すのに使用します。
   * \ja_end
   * \en
   * Null, which is used for something like error.
   * \en_end
   */
  CEID_NULL = 0,
#define CCC_ICEMAN_CEID_ENUM_DEF
#include <ccc/iceman/ces.h>
#undef CCC_ICEMAN_CEID_ENUM_DEF
};
#ifndef __cplusplus
typedef enum CeId CeId;
#endif /* __cplusplus */

/*!
 * \ja
 * このenumは改行の変換方法を示します。
 * \ja_end
 * \en
 * This enum shows some methods of the new line conversion.
 * \en_end
 */
enum NLConv
{
  /*!
   * \ja
   * 無変換
   * \ja_end
   * \en
   * No conversion.
   * \en_end
   */
  NLCONV_NONE = 0,
  /*!
   * \ja
   * この値はLFをCRに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from LF to CR.
   * \en_end
   */ 
  NLCONV_LF2CR,
  /*!
   * \ja
   * この値はCRをLFに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from CR to LF.
   * \en_end
   */
  NLCONV_CR2LF,
  /*!
   * \ja
   * この値はCR+LFをLFに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from CR+LF to LF.
   * \en_end
   */
  NLCONV_CRLF2LF,
  /*! 
   * \ja
   * この値はCR+LFをCRに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from CR+LF to CR.
   * \en_end
   */
  NLCONV_CRLF2CR,
  /*!
   * \ja
   * この値はLFをCR+LFに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion form LF to CR+LF.
   * \en_end
   */
  NLCONV_LF2CRLF,
  /*!
   * \ja
   * この値はCRをCR+LFに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from CR to CR+LF.
   * \en_end
   */
  NLCONV_CR2CRLF,
  /*!
   * \ja
   * この値はどの改行コードでもCRに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from any new line codes to CR.
   * \en_end
   */
  NLCONV_AUTO2CR,
  /*!
   * \ja
   * この値はどの改行コードでもLFに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from any new line codes to LF.
   * \en_end
   */ 
  NLCONV_AUTO2LF,
  /*!
   * \ja
   * この値はどの改行コードでもCR+LFに変換することを示します。
   * \ja_end
   * \en
   * This value shows a conversion from any new line codes to CR+LF.
   * \en_end
   */
  NLCONV_AUTO2CRLF,
};
#ifndef __cplusplus
typedef enum NLConv NLConv;
#endif /* __cplusplus */

/*!
 * \ja
 * このenumはユーティリティフィルタを入れる位置を示します。
 * \ja_end
 * \en
 * This enum shows the insertion position of the utility filter.
 * \en_end
 */
enum UtilPos
{
  /*!
   * \ja
   * 任意の位置
   * \ja_end
   * \en
   * any where
   * \en_end
   */
  UTILPOS_ANYWHERE = 0,
  /*!
   * \ja
   * 最後のみ
   * \ja_end
   * \en
   * Only last
   * \en_end
   */
  UTILPOS_ONLY_LAST,
  /*!
   * \ja
   * 最初のみ
   * \ja_end
   * \en
   * Only first
   * \en_end
   */
  UTILPOS_ONLY_FIRST,
};
#ifndef __cplusplus
typedef enum UtilPos UtilPos;
#endif /* __cplusplus */

/*!
 * \ja
 * 文字セットエンコーディングの一覧インターフェース用の構造体です。
 * \ja_end
 * \en
 * This is a structure for the charset encoding list interface.
 * \en_end
 */
struct CharsetEncoding
{
  enum
  {
    /*!
     * \ja
     * 文字セットエンコーディング名のエリアスの最大数
     * \ja_end
     * \en
     * Maximum number of the aliases of the charset encoding.
     * \en_end
     */
    ALIASES_MAX = 11,
  };
  /*!
   * \ja
   * 文字セットエンコーディングID
   * \ja_end
   * \en
   * charset encoding id
   * \en_end
   */
  CeId id;
  /*!
   * \ja
   * 文字セットエンコーディンググループID
   * \ja_end
   * \en
   * charset encoding group id
   * \en_end
   */
  CeGId gid;
  /*!
   * \ja
   * 文字セットエンコーディングのネイティブID
   * \ja_end
   * \en
   * charset encoding native id
   * \en_end
   */
  CeId nid;
  /*!
   * \ja
   * 文字セットエンコーディング名
   * \ja_end
   * \en
   * charset encoding names.
   * \en_end
   */
  const char* name[ALIASES_MAX];
};
#ifndef __cpluscplus
typedef struct CharsetEncoding CharsetEncoding;
#endif /* __cplusplus */

#ifdef __cplusplus
// C++ interface

class CompositIFilter;
class Detector;
class Utility;
class GroupUtility;
class IFilterDictionaryEntry;

/*!
 * \ja
 * Icemanクラスは文字セットエンコーディング変換等の機能を提供します。
 * \ja_end
 * \en
 * Iceman class offers some functions like charset encoding conversions.
 * \en_end
 */
class Iceman
{
 public:
  /*!
   * \ja
   * Icemanを初期化します。
   * 複数回このメソッドを呼ばないようにしてください。
   * \ja_end
   * \en
   * This method initialize Iceman class.
   * Don't call this method multipule times.
   * \en_end
   */
  static bool initialize();
  /*!
   * \ja
   * Icemanを終了します。
   * 複数回このメソッドを呼ばないようにしてください。
   * \ja_end
   * \en
   * This method uninitilize Iceman class.
   * Don't call this method multipule times.
   * \en_end
   */
  static bool unInitialize();

  /*!
   * \ja
   * \defgroup ceid Iceman - 文字セットエンコーディング
   * \ja_end
   * \en
   * \defgroup ceid Iceman - charset encoding name
   * \en_end
   * @{
   */
  /*!
   * 文字セットエンコーディングを記述した文字列から該当するCeIdを取得します。
   * 取得できなかった場合には、CEID_NULLが返ります。
   * \param str 文字セットエンコーディング文字列
   * \return CeId
   */
  static CeId stringToCeId(const char* str);
  /*!
   * CeIdから文字セットエンコーディングを記述した文字列を取得します。
   * \param ceid CeId
   * \return 文字セットエンコーディングを記述した文字列
   */
  static const char* ceIdToString(CeId ceid);
  /*!
   * CeIdからその文字セットエンコーディングのCeGIdを取得します。
   * \param ceid CeId
   * \return CeGId
   */
  static CeGId getCeGId(CeId ceid);
  /*!
   * CeIdから該当文字セットエンコーディングのIOTypeを取得します。
   * \param ceid CeId
   * \return IOType, 特定できない場合には、IOTYPE_NULLが返ります。
   */
  static IOType getIOType(CeId ceid);
  /*!
   * CeIdからネイティブCeIdを取得します。
   * \param ceid CeId
   * \return ネイティブCeId
   */
  static CeId getNativeCeId(CeId ceid);
  /*!
   * IcemanでサポートしているCharsetEncodingの一覧が格納された配列を取得します。
   * 配列の最後の要素のCeIdはCEID_NULLになっています。
   * \return CharsetEncodingの配列
   */
  static CharsetEncoding* getCharsetEncodingList();
  /*! @} */

  /*!
   * \ja
   * \defgroup composit_filter Iceman - 複合フィルタ
   * \ja_end
   * \en
   * \defgroup composit_filter Iceman - Composit Filters
   * \en_end
   * @{
   */
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \return 複合フィルタ
   */
  static CompositIFilter* createCompositIFilter(CeId from, CeId to);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディングを記述した文字列
   * \param to 変換先の文字セットエンコーディングを記述した文字列
   * \return 複合フィルタ
   */
  static CompositIFilter* createCompositIFilter(char* from, char* to);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * このメソッドではユーティリティフィルタを複合フィルタの途中に指定するができます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param util ユーティリティフィルタ
   * \param pos ユーティリティフィルタの位置
   * \return 複合フィルタ
   */
  static CompositIFilter* createCompositIFilter(CeId from, CeId to, Utility* util, UtilPos pos);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * このメソッドではグループユーティリティフィルタを複合フィルタの途中に指定するができます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param gutil グループユーティリティフィルタ
   * \param pos グループユーティリティフィルタの位置
   * \return 複合フィルタ
   */
  static CompositIFilter* createCompositIFilter(CeId from, CeId to, GroupUtility* gutil, UtilPos pos);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * このメソッドではユーティリティフィルタとグループユーティリティフィルタ
   * を複合フィルタの途中に指定するができます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param util ユーティリティフィルタ
   * \param gutil グループユーティリティフィルタ
   * \param pos ユーティリティフィルタとグループユーティリティフィルタの位置
   * \return 複合フィルタ
   */
  static CompositIFilter* createCompositIFilter(CeId from, CeId to, Utility* util, GroupUtility* gutil, UtilPos pos);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * このメソッドでは改行コードの変換機能を指定できます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param nlc 改行コードの変換指定
   * \return 複合フィルタ
   */
  static CompositIFilter* createCompositIFilter(CeId from, CeId to, NLConv nlc);
  /*! @} */

  /*!
   * \ja
   * \defgroup converter Iceman - コンバータ
   * \ja_end
   * \en
   * \defgroup converter Iceman - Converters
   * \en_end
   * @{
   */
  /*!
   * 指定した複合フィルタを用いてファイルの内容を変換して別ファイルに出力します。
   * \param cf 複合フィルタ
   * \param in 入力元ファイル
   * \param out 出力先ファイル
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convert(CompositIFilter* cf, FILE* in, FILE* out);
  /*!
   * ファイルの文字セットエンコーディングを変換して別ファイルに出力します。
   * \param from 変換元CeId
   * \param to 変換先CeId
   * \param in 入力元ファイル
   * \param out 出力先ファイル
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convert(CeId from, CeId to, FILE* in, FILE* out);
  /*!
   * ファイルの文字セットエンコーディングを変換して別ファイルに出力します。
   * \param from 変換元文字セットエンコーディング文字列
   * \param to 変換先文字セットエンコーディング文字列
   * \param in 入力元ファイル
   * \param out 出力先ファイル
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convert(char* from, char* to, FILE* in, FILE* out);
  /*!
   * 指定した複合フィルタを用いてメモリの内容を変換して別メモリに出力します。
   * \param cf 複合フィルタ
   * \param in 入力元メモリ
   * \param in_size 入力元メモリサイズ
   * \param out 出力先メモリ
   * \param out_size 出力先メモリサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convert(CompositIFilter* cf, void* in, Size in_size, void* out, Size out_size);
  /*!
   * メモリの内容文字セットエンコーディングを変換して別メモリに出力します。
   * \param from 変換元CeId
   * \param to 変換先CeId
   * \param in 入力元メモリ
   * \param in_size 入力元メモリサイズ
   * \param out 出力先メモリ
   * \param out_size 出力先メモリサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convert(CeId from, CeId to, void* in, Size in_size, void* out, Size out_size);
  /*!
   * メモリの内容文字セットエンコーディングを変換して別メモリに出力します。
   * \param from 変換元文字セットエンコーディング文字列
   * \param to 変換先文字セットエンコーディング文字列
   * \param in 入力元メモリ
   * \param in_size 入力元メモリサイズ
   * \param out 出力先メモリ
   * \param out_size 出力先メモリサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convert(char* from, char* to, void* in, Size in_size, void* out, Size out_size);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToBString(IFilter* cf, const BString* src, BString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToBString(IFilter* cf, const DString* src, BString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToBString(IFilter* cf, const WString* src, BString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToDString(IFilter* cf, const BString* src, DString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToDString(IFilter* cf, const DString* src, DString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToDString(IFilter* cf, const WString* src, DString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToWString(IFilter* cf, const BString* src, WString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToWString(IFilter* cf, const DString* src, WString* dest);
  /*!
   * 文字列を指定のフィルタを使って変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param cf フィルタ
   * \param src 変換元
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToWString(IFilter* cf, const WString* src, WString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング文字列
   * \param to 変換先文字セットエンコーディング文字列
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToBString(CeId from, CeId to, const BString* src, BString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToBString(CeId from, CeId to, const DString* src, BString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToBString(CeId from, CeId to, const WString* src, BString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToDString(CeId from, CeId to, const BString* src, DString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToDString(CeId from, CeId to, const DString* src, DString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToDString(CeId from, CeId to, const WString* src, DString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToWString(CeId from, CeId to, const BString* src, WString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToWString(CeId from, CeId to, const DString* src, WString* dest);
  /*!
   * 文字列を指定の文字セットエンコーディングに変換します。
   * 変換結果の文字列オブジェクトは呼び出し側が破棄してください。
   * \param from 変換元文字セットエンコーディング
   * \param to 変換先文字セットエンコーディング
   * \param src 変換元文字列
   * \param dest 変換先
   * \retval true 成功
   * \retval false 失敗
   */
  static bool convertToWString(CeId from, CeId to, const WString* src, WString* dest);
  /*! @} */

  /*!
   * \ja
   * \defgroup detector Iceman - ディテクタ
   * \ja_end
   * \en
   * \defgroup detector Iceman - Detectors
   * \en_end
   * @{
   */
  /*!
   * 指定した種類のディテクタを生成します。
   * \param detector ディテクタの種類
   * \return 生成されたディテクタ
   */
  static Detector* createDetector(CeId detector);
  /*!
   * ディテクタを破棄します。
   * \param detector 破棄するディテクタ
   */
  static void deleteDetector(Detector* detector);
  /*!
   * 指定ディテクタによってファイルの内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタ
   * \param in ファイル
   * \return 検出結果
   */
  static CeId detect(Detector* detector, FILE* in);
  /*!
   * 指定ディテクタによってメモリ内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタ
   * \param in スキャンするメモリのアドレス
   * \param in_size スキャンする範囲の長さ
   * \return 検出結果
   */
  static CeId detect(Detector* detector, void* in, Size in_size);
  /*!
   * 指定ディテクタによってファイルの内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタのCeId
   * \param in ファイル
   * \return 検出結果
   */
  static CeId detect(CeId detector, FILE* in);
  /*!
   * 指定ディテクタによってメモリ内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタのCeId
   * \param in スキャンするメモリのアドレス
   * \param in_size スキャンする範囲の長さ
   * \return 検出結果
   */
  static CeId detect(CeId detector, void* in, Size in_size);
  /*! @} */

 private:
  static void createIFilters(IFilterDictionaryEntry* fde, CompositIFilter* cf);
};

#else /* __cplusplus */

#endif /* __cplusplus */

#ifdef __cplusplus
/*
 * \ja
 * C言語用のインターフェースです。
 * \ja_end
 * \en
 * Interfaces for the C language.
 * \en_end
 */
extern "C"
{
#endif /* __cplusplus */

  /*!
   * \ja
   * Icemanを初期化します。
   * 複数回このメソッドを呼ばないようにしてください。
   * \ja_end
   * \en
   * This method initialize Iceman class.
   * Don't call this method multipule times.
   * \en_end
   */
  bool Iceman_initialize();
  /*!
   * \ja
   * Icemanを終了します。
   * 複数回このメソッドを呼ばないようにしてください。
   * \ja_end
   * \en
   * This method uninitilize Iceman class.
   * Don't call this method multipule times.
   * \en_end
   */
  bool Iceman_unInitialize();

  /*!
   * \ja
   * \addtogroup ceid
   * \ja_end
   * \en
   * \addtogroup ceid
   * \en_end
   * @{
   */
  /*!
   * 文字セットエンコーディングを記述した文字列から該当するCeIdを取得します。
   * 取得できなかった場合には、CEID_NULLが返ります。
   * \param str 文字セットエンコーディング文字列
   * \return CeId
   */
  CeId Iceman_stringToCeId(const char* str);
  /*!
   * CeIdから文字セットエンコーディングを記述した文字列を取得します。
   * \param ceid CeId
   * \return 文字セットエンコーディングを記述した文字列
   */
  const char* Iceman_ceIdToString(CeId ceid);
  /*!
   * CeIdからその文字セットエンコーディングのCeGIdを取得します。
   * \param ceid CeId
   * \return CeGId
   */
  CeGId Iceman_getCeGId(CeId ceid);
  /*!
   * CeIdからネイティブCeIdを取得します。
   * \param ceid CeId
   * \return ネイティブCeId
   */
  CeId Iceman_getNativeCeId(CeId ceid);
  /*!
   * IcemanでサポートしているCharsetEncodingの一覧が格納された配列を取得します。
   * 配列の最後の要素のCeIdはCEID_NULLになっています。
   * \return CharsetEncodingの配列
   */
  CharsetEncoding* Iceman_getCharsetList();
  /*! @} */

  /*!
   * \ja
   * \addtogroup composit_filter
   * \ja_end
   * \en
   * \addtogroup composit_filter
   * \en_end
   * @{
   */
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \return 複合フィルタ
   */
  CompositIFilter* Iceman_createCompositIFilter0(CeId from, CeId to);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディングを記述した文字列
   * \param to 変換先の文字セットエンコーディングを記述した文字列
   * \return 複合フィルタ
   */
  CompositIFilter* Iceman_createCompositIFilter1(char* from, char* to);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * この関数ではユーティリティフィルタを複合フィルタの途中に指定するができます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param util ユーティリティフィルタ
   * \param pos ユーティリティフィルタの位置
   * \return 複合フィルタ
   */
  CompositIFilter* Iceman_createCompositIFilter2(CeId from, CeId to, Utility* util, UtilPos pos);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * この関数ではグループユーティリティフィルタを複合フィルタの途中に指定するができます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param gutil グループユーティリティフィルタ
   * \param pos グループユーティリティフィルタの位置
   * \return 複合フィルタ
   */
  CompositIFilter* Iceman_createCompositIFilter3(CeId from, CeId to, GroupUtility* gutil, UtilPos pos);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * この関数ではユーティリティフィルタとグループユーティリティフィルタ
   * を複合フィルタの途中に指定するができます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param util ユーティリティフィルタ
   * \param gutil グループユーティリティフィルタ
   * \param pos ユーティリティフィルタとグループユーティリティフィルタの位置
   * \return 複合フィルタ
   */
  CompositIFilter* Iceman_createCompositIFilter4(CeId from, CeId to, Utility* util, GroupUtility* gutil, UtilPos pos);
  /*!
   * 指定した文字セットエンコーディング変換を行なう複合フィルタを生成します。
   * この関数は改行コードの変換機能を指定できます。
   * サポートされていない組合せの場合には0が返ります。
   * \param from 変換元の文字セットエンコーディング
   * \param to 変換先の文字セットエンコーディング
   * \param nlc 改行コードの変換指定
   * \return 複合フィルタ
   */
  CompositIFilter* Iceman_createCompositIFilter5(CeId from, CeId to, NLConv nlc);

  /*!
   * 複合フィルタを破棄します。
   * \param cf 破棄する複合フィルタ
   */
  void Iceman_deleteCompositIFilter(CompositIFilter* cf);
  /*!
   * Utilityを破棄します。
   * \param util 破棄するUtility
   */
  void Iceman_deleteUtility(Utility* util);
  /*!
   * GroupUtilityを破棄します。
   * \param gutil 破棄するGroupUtility
   */
  void Iceman_deleteGroupUtility(GroupUtility* gutil);
  /*! @} */

  /*!
   * \ja
   * \addtogroup converter
   * \ja_end
   * \en
   * \addtogroup converter
   * \en_end
   * @{
   */
  /*!
   * 指定した複合フィルタを用いてファイルの内容を変換して別ファイルに出力します。
   * \param cf 複合フィルタ
   * \param in 入力元ファイル
   * \param out 出力先ファイル
   * \retval true 成功
   * \retval false 失敗
   */
  bool Iceman_convert0(CompositIFilter* cf, FILE* in, FILE* out);
  /*!
   * ファイルの文字セットエンコーディングを変換して別ファイルに出力します。
   * \param from 変換元CeId
   * \param to 変換先CeId
   * \param in 入力元ファイル
   * \param out 出力先ファイル
   * \retval true 成功
   * \retval false 失敗
   */
  bool Iceman_convert1(CeId from, CeId to, FILE* in, FILE* out);
  /*!
   * ファイルの文字セットエンコーディングを変換して別ファイルに出力します。
   * \param from 変換元文字セットエンコーディング文字列
   * \param to 変換先文字セットエンコーディング文字列
   * \param in 入力元ファイル
   * \param out 出力先ファイル
   * \retval true 成功
   * \retval false 失敗
   */
  bool Iceman_convert2(char* from, char* to, FILE* in, FILE* out);
  /*!
   * 指定した複合フィルタを用いてメモリの内容を変換して別メモリに出力します。
   * \param cf 複合フィルタ
   * \param in 入力元メモリ
   * \param in_size 入力元メモリサイズ
   * \param out 出力先メモリ
   * \param out_size 出力先メモリサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  bool Iceman_convert3(CompositIFilter* cf, void* in, Size in_size, void* out, Size out_size);
  /*!
   * メモリの内容文字セットエンコーディングを変換してを変換して別メモリに出力します。
   * \param from 変換元CeId
   * \param to 変換先CeId
   * \param in 入力元メモリ
   * \param in_size 入力元メモリサイズ
   * \param out 出力先メモリ
   * \param out_size 出力先メモリサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  bool Iceman_convert4(CeId from, CeId to, void* in, Size in_size, void* out, Size out_size);
  /*!
   * メモリの内容文字セットエンコーディングを変換してを変換して別メモリに出力します。
   * \param from 変換元文字セットエンコーディング文字列
   * \param to 変換先文字セットエンコーディング文字列
   * \param in 入力元メモリ
   * \param in_size 入力元メモリサイズ
   * \param out 出力先メモリ
   * \param out_size 出力先メモリサイズ
   * \retval true 成功
   * \retval false 失敗
   */
  bool Iceman_convert5(char* from, char* to, void* in, Size in_size, void* out, Size out_size);
  /*! @} */

  /*!
   * \ja
   * \addtogroup detector
   * \ja_end
   * \en
   * \addtogroup detector
   * \en_end
   * @{
   */
  /*!
   * 指定した種類のディテクタを生成します。
   * \param detector ディテクタの種類
   * \return 生成されたディテクタ
   */
  Detector* Iceman_createDetector(CeId detector);
  /*!
   * ディテクタを破棄します。
   * \param detector 破棄するディテクタ
   */
  void Iceman_deleteDetector(Detector* detector);
  /*!
   * 指定ディテクタによってファイルの内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタ
   * \param in ファイル
   * \return 検出結果
   */
  CeId Iceman_detect0(CeId detector, FILE* in);
  /*!
   * 指定ディテクタによってメモリの内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタ
   * \param in メモリのアドレス
   * \param in_size 検出する範囲の長さ
   * \return 検出結果
   */
  CeId Iceman_detect1(CeId detector, void* in, Size in_size);
  /*!
   * 指定ディテクタによってファイルの内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタのCeId
   * \param in ファイル
   * \return 検出結果
   */
  CeId Iceman_detect2(Detector* detector, FILE* in);
  /*!
   * 指定ディテクタによってメモリの内容をスキャンし文字セットエンコーディングを検出します。
   * \param detector 検出に使用するディテクタのCeId
   * \param in メモリのアドレス
   * \param in_size 検出する範囲の長さ
   * \return 検出結果
   */
  CeId Iceman_detect3(Detector* detector, void* in, Size in_size);
  /*! @} */
#ifdef __cplusplus
}
#endif /* __cplusplus */

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_iceman_Iceman_h */
