﻿// @@DISTHDR@@
// $Id$

#include <assert.h>
#include <ccc/iceman/unicodeconv.h>
#include <ccc/iceman/Unicode.h>
#include <ccc/iceman/Utf7.h>
#include <ccc/iceman/Base64.h>

CCC_NAMESPACE_START(CCC);

// ------------------------------------------------------------------------
// class FromUcs2ToUcs2Be, FromUcs4ToUcs4Be, FromUcs2ToUcs2Le, FromUcs4ToUcs4Le
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
FromUcs2ToUcs2Be::FromUcs2ToUcs2Be()
{
}

FromUcs2ToUcs2Be::~FromUcs2ToUcs2Be()
{
}

IFilter*
FromUcs2ToUcs2Be::createIFilter()
{
  return new FromUcs2ToUcs2Be();
}

FromUcs4ToUcs4Be::FromUcs4ToUcs4Be()
{
}

FromUcs4ToUcs4Be::~FromUcs4ToUcs4Be()
{
}

IFilter*
FromUcs4ToUcs4Be::createIFilter()
{
  return new FromUcs4ToUcs4Be();
}

FromUcs2BeToUcs2::FromUcs2BeToUcs2()
{
}

FromUcs2BeToUcs2::~FromUcs2BeToUcs2()
{
}

IFilter*
FromUcs2BeToUcs2::createIFilter()
{
  return new FromUcs2BeToUcs2();
}

FromUcs4BeToUcs4::FromUcs4BeToUcs4()
{
}

FromUcs4BeToUcs4::~FromUcs4BeToUcs4()
{
}

IFilter*
FromUcs4BeToUcs4::createIFilter()
{
  return new FromUcs4BeToUcs4();
}

#elif CCC_ENDIAN == CCC_BIG_ENDIAN
FromUcs2ToUcs2Le::FromUcs2ToUcs2Le()
{
}

FromUcs2ToUcs2Le::~FromUcs2ToUcs2Le()
{
}

IFilter*
FromUcs2ToUcs2Le::createIFilter()
{
  return new FromUcs2ToUcs2Le();
}

FromUcs4ToUcs4Le::FromUcs4ToUcs4Le()
{
}

FromUcs4ToUcs4Le::~FromUcs4ToUcs4Le()
{
}

IFilter*
FromUcs4ToUcs4Le::createIFilter()
{
  return new FromUcs4ToUcs4Le();
}

FromUcs2LeToUcs2::FromUcs2LeToUcs2()
{
}

FromUcs2LeToUcs2::~FromUcs2LeToUcs2()
{
}

IFilter*
FromUcs2LeToUcs2::createIFilter()
{
  return new FromUcs2LeToUcs2();
}

FromUcs4LeToUcs4::FromUcs4LeToUcs4()
{
}

FromUcs4LeToUcs4::~FromUcs4LeToUcs4()
{
}

IFilter*
FromUcs4LeToUcs4::createIFilter()
{
  return new FromUcs4LeToUcs4();
}
#endif /* CCC_ENDIAN */

// ------------------------------------------------------------------------
// class FromUsAsciiToUcs2

FromUsAsciiToUcs2::FromUsAsciiToUcs2()
  : IFilter(IOTYPE_ANY, IOTYPE_ANY)
{
}

FromUsAsciiToUcs2::~FromUsAsciiToUcs2()
{
}

UInt16
FromUsAsciiToUcs2::getUInt16() CCC_RAISES(IOException)
{
  assert(src);
  UInt16 c = (UInt16)((UInt8)src->getInt8());
  return c;
}

IOType
FromUsAsciiToUcs2::getIOType()
{
  return IOTYPE_UINT16;
}

IFilter*
FromUsAsciiToUcs2::createIFilter()
{
  return new FromUsAsciiToUcs2();
}

// ------------------------------------------------------------------------
// class FromUsAsciiToUcs4

FromUsAsciiToUcs4::FromUsAsciiToUcs4()
  : IFilter(IOTYPE_ANY, IOTYPE_ANY)
{
}

FromUsAsciiToUcs4::~FromUsAsciiToUcs4()
{
}

UInt32
FromUsAsciiToUcs4::getUInt32() CCC_RAISES(IOException)
{
  assert(src);
  UInt32 c = (UInt32)((UInt8)src->getInt8());
  return c;
}

IOType
FromUsAsciiToUcs4::getIOType()
{
  return IOTYPE_UINT32;
}

IFilter*
FromUsAsciiToUcs4::createIFilter()
{
  return new FromUsAsciiToUcs4();
}

// ------------------------------------------------------------------------
// class FromUcs4ToUsAscii

FromUcs4ToUsAscii::FromUcs4ToUsAscii()
  : IFilter(IOTYPE_ANY, IOTYPE_ANY)
{
}

FromUcs4ToUsAscii::~FromUcs4ToUsAscii()
{
}

Int8
FromUcs4ToUsAscii::getInt8() CCC_RAISES(IOException)
{
  for (;;)
  {
    UInt32 c = src->getUInt32();
    if (c <= 0x7fu)
    {
      return (Int8)c;
    }
    // skip
  }
}

IOType
FromUcs4ToUsAscii::getIOType()
{
  return IOTYPE_INT8;
}

IFilter*
FromUcs4ToUsAscii::createIFilter()
{
  return new FromUcs4ToUsAscii();
}

// ------------------------------------------------------------------------
// class FromUtf8NToUtf8

FromUtf8NToUtf8::FromUtf8NToUtf8()
{
}

FromUtf8NToUtf8::~FromUtf8NToUtf8()
{
}

Int8
FromUtf8NToUtf8::getInt8() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
    // UTF-8 BOM : EF, BB, BF
    push((Int8)0xbf);
    push((Int8)0xbb);
    return (Int8)0xef;
  }
  if (getQueueLength())
  {
    Int8 ret = pop();
    return ret;
  }
  Int8 c = src->getInt8();
  return c;
}

IFilter*
FromUtf8NToUtf8::createIFilter()
{
  return new FromUtf8NToUtf8();
}

// ------------------------------------------------------------------------
// class FromUtf8ToUtf8N

FromUtf8ToUtf8N::FromUtf8ToUtf8N()
{
}

FromUtf8ToUtf8N::~FromUtf8ToUtf8N()
{
}

Int8
FromUtf8ToUtf8N::getInt8() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
    // UTF-8 BOM : EF, BB, BF
    UInt8 c1 = (UInt8)src->getInt8();
    if (c1 == 0xef)
    {
      UInt8 c2 = (UInt8)src->getInt8();
      if (c2 == 0xbb)
      {
	UInt8 c3 = (UInt8)src->getInt8();
	if (c3 == 0xbf)
	{
	  // BOM
	}
	else
	{
	  // not BOM
	  push((Int8)c3);
	  push((Int8)c2);
	  return (Int8)c1;
	}
      }
      else
      {
	// not BOM
	push((Int8)c2);
	return (Int8)c1;
      }
    }
    else
    {
      // not BOM
      return (Int8)c1;
    }
  }
  if (getQueueLength())
  {
    Int8 ret = pop();
    return ret;
  }
  Int8 c = src->getInt8();
  return c;
}

IFilter*
FromUtf8ToUtf8N::createIFilter()
{
  return new FromUtf8ToUtf8N();
}

// ------------------------------------------------------------------------
// class FromUcs2ToUtf8N

FromUcs2ToUtf8N::FromUcs2ToUtf8N()
 : Int8PushbackIFilter()
{
}

FromUcs2ToUtf8N::~FromUcs2ToUtf8N()
{
}

Int8
FromUcs2ToUtf8N::getInt8() CCC_RAISES(IOException)
{
  if (getQueueLength())
  {
    Int8 ret = pop();
    return ret;
  }
  UInt16 c = src->getUInt16();
  if (c <= 0x7f)
  {
    // 1 byte                    1
    // 0000.0000:0xxx.xxxx => 0xxxxxxx
    return (Int8)c;
  }
  else if ((0x80 <= c) && (c <= 0x7ff))
  {
    // 2 bytes                   1        2
    // 0000.0yyy:yyxx.xxxx => 110yyyyy:10xxxxxx
    UInt8 c1 = ((UInt8)(c >> 6)) | 0xc0u;
    UInt8 c2 = ((UInt8)(c & 0x3fu)) | 0x80u;
    push((Int8)c2);
    return (Int8)c1;
  }
  // else if ((0x800 <= c) && (c <= 0xffff))
  // NOTE: UCS2 doesn't support surrogates pair

  // 3 bytes                   1        2        3
  // zzzz.yyyy:yyxx.xxxx => 1110zzzz:10yyyyyy:10xxxxxx
  UInt8 c1 = ((UInt8)(c >> 12)) | 0xe0u;
  UInt8 c2 = ((UInt8)((c & 0x0fc0u) >> 6)) | 0x80u;
  UInt8 c3 = ((UInt8)(c & 0x3fu)) | 0x80u;
  push((Int8)c3);
  push((Int8)c2);
  return (Int8)c1;
}

IFilter*
FromUcs2ToUtf8N::createIFilter()
{
  return new FromUcs2ToUtf8N();
}

// ------------------------------------------------------------------------
// class FromUtf8NToUcs2

FromUtf8NToUcs2::FromUtf8NToUcs2()
  : IFilter(IOTYPE_ANY, IOTYPE_ANY)
{
}

FromUtf8NToUcs2::~FromUtf8NToUcs2()
{
}

UInt16
FromUtf8NToUcs2::getUInt16() CCC_RAISES(IOException)
{
  UInt16 ret;
  for (;;)
  {
    UInt16 c1 = (UInt16)(src->getInt8());
    if (c1 & 0x0080u)
    {
      UInt16 c2 = (UInt16)(src->getInt8());
      if (c1 & 0x0020u)
      {
	UInt16 c3 = (UInt16)(src->getInt8());
	if (c1 & 0x0010u)
	{
	  // NOTE: UCS2 doesn't support surrogates pair
	  // 4 bytes
	  // UInt16 c4 = (UInt16)(src->getInt8());
	  // UInt32 u = (((UInt32)(c1 & 0x07u)) << 18 |
	  //	      ((UInt32)(c2 & 0x3fu)) << 12 |
	  //	      ((UInt32)(c3 & 0x3fu)) << 6 |
	  //	      ((UInt32)(c4 & 0x3fu)));
	  // skip, surrogates pair
          return 0;
	}
	else
	{
	  // 3 bytes
	  //    c1       c2       c3
	  // 1110zzzz:10yyyyyy:10xxxxxx => zzzz.yyyy:yyxx.xxxx
	  ret = ((c1 & 0x000fu) << 12) | ((c2 & 0x003fu) << 6) | (c3 & 0x003f);
	  return ret;
	}
      }
      else
      {
	// 2 bytes
	//    c1       c2
	// 110yyyyy:10xxxxxx => 0000.0yyy:yyxx.xxxx
	ret = ((c1 & 0x001fu) << 6) | (c2 & 0x003fu);
	return ret;
      }
    }
    else
    {
      // 1 byte
      ret = c1;
      return ret;
    }
  }
}

IOType
FromUtf8NToUcs2::getIOType()
{
  return IOTYPE_UINT16;
}

IFilter*
FromUtf8NToUcs2::createIFilter()
{
  return new FromUtf8NToUcs2();
}

// ------------------------------------------------------------------------
// class FromUtf7ToUcs2

FromUtf7ToUcs2::FromUtf7ToUcs2()
 : UInt16PushbackIFilter()
{
  clear();
}

FromUtf7ToUcs2::~FromUtf7ToUcs2()
{
}

void
FromUtf7ToUcs2::clear()
{
  b64_p = false;
  unencoded = 0;
  has_unencoded_p = false;
}

bool
FromUtf7ToUcs2::rewind()
{
  clear();
  return UInt16PushbackIFilter::rewind();
}

UInt16
FromUtf7ToUcs2::getUInt16() CCC_RAISES(IOException)
{
  for (;;)
  {
    if (getQueueLength())
    {
      UInt16 ret = pop();
      return ret;
    }

    UInt8 c;
    bool c_pending_p = false;
    try
    {
      if (!b64_p)
      {
	c = (UInt8)src->getInt8();
	if (c == '+')
	{
	  c = (UInt8)src->getInt8();
	  if (c == '-')
	  {
	    return (UInt16)'+';	// +-
	  }
	  b64_p = true;
	  has_unencoded_p = false;
	  unencoded = 0;
	  c_pending_p = true;
	}
	if (!b64_p)
	{
	  return (UInt16)c;	// direct
	}
      }
    }
    catch (IOException& s)
    {
      throw s;
    }

    UInt8 in[4];
    in[0] = 0;
    in[1] = 0;
    in[2] = 0;
    in[3] = 0;
    int in_len = 0;
    int i;
    try
    {
      for (i = 0; i <= 3; i++)
      {
	if (c_pending_p)
	{
	  c_pending_p = false;
	}
	else
	{
	  c = (UInt8)src->getInt8();
	}
	if (c == '-')
	{
	  b64_p = false;
	  break;
	}
	if (!Utf7::setBP(c))
	{
	  b64_p = false;
	  push((UInt16)c);	// direct
	  break;
	}
	in[i] = c;
	in_len++;
      }
    }
    catch (IOException& s)
    {
      if (i == 0)
      {
	throw s;
      }
    }
    UInt8 out[3];
    Size out_len = Base64::decodeModified(in, in_len, out);
    UInt16 x;
    if (has_unencoded_p)
    {
      if (out_len >= 2)
      {
	x = (((UInt16)out[1]) << 8) | ((UInt16)out[2]);// MSB octet first
	push(x);
      }
      x = (((UInt16)unencoded) << 8) | ((UInt16)out[0]);// MSB octet first
      has_unencoded_p = false;
      unencoded = 0;
      return x;
    }
    else
    {
      if (out_len > 0)
      {
	x = (((UInt16)out[0]) << 8) | ((UInt16)out[1]);	// MSB octet first
	if (out_len == 3)
	{
	  unencoded = out[2];
	  has_unencoded_p = true;
	}
	return x;
      }
    }
  }
}

IFilter*
FromUtf7ToUcs2::createIFilter()
{
  return new FromUtf7ToUcs2();
}

// ------------------------------------------------------------------------
// class FromUcs2ToUtf7
FromUcs2ToUtf7::FromUcs2ToUtf7()
{
  clear();
}

FromUcs2ToUtf7::~FromUcs2ToUtf7()
{
}

void
FromUcs2ToUtf7::clear()
{
  has_undecoded_p = false;
  undecoded = 0;
  b64_p = false;
}

bool
FromUcs2ToUtf7::rewind()
{
  clear();
  return Int8PushbackIFilter::rewind();
}

Int8
FromUcs2ToUtf7::getInt8() CCC_RAISES(IOException)
{
  if (getQueueLength())
  {
    Int8 ret = pop();
    return ret;
  }

  UInt16 c;
  UInt8 out[4];
  UInt8 in[3];
  try
  {
    c = src->getUInt16();
  }
  catch (IOException& s)
  {
    if (s.errorNum() == IOException::READ_BEYOND_THE_EOF)
    {
      if (has_undecoded_p)
      {
	push('-');		// for end base64 status
	in[0] = undecoded;
	int in_len = 1;
	int len = Base64::encodeModified(in, in_len, out);
	while (len--)
	{
	  push(out[len]);
	}
	has_undecoded_p = false;
	b64_p = false;
	return pop();
      }
      else if (b64_p)
      {
	b64_p = false;
	return (Int8)'-';	// for end base64 status
      }
    }
    throw s;
  }
  if (Utf7::directP(c))	// SET-D and SET-O
  {
    // direct output
    push((UInt8)c);
    if (b64_p && (Utf7::setBP(c) || (c == (UInt16)'-')))
    {
      push((UInt8)'-');		// for end base64 status
    }
    if (has_undecoded_p)
    {
      in[0] = undecoded;
      int in_len = 1;
      Size len = Base64::encodeModified(in, in_len, out);
      while (len--)
      {
	push(out[len]);
      }
      has_undecoded_p = false;
    }
    b64_p = false;
  }
  else if ((b64_p == false) && (c == (UInt16)'+'))
  {
    push((UInt8)'-');
    push((UInt8)'+');
  }
  else
  {
    // modified b64 encode
    bool prev_b64_p = b64_p;
    int in_len = 3;
    try
    {
      if (has_undecoded_p)
      {
	in[0] = undecoded;
	in[1] = (UInt8)(c >> 8);
	in[2] = (UInt8)(c & 0xff);
	has_undecoded_p = false;
	b64_p = true;
      }
      else
      {
	b64_p = true;
	in[0] = (UInt8)(c >> 8);
	in[1] = (UInt8)(c & 0xff);
	UInt16 c2 = src->getUInt16();
	if (Utf7::directP(c2))
	{
	  // direct output
	  in[2] = 0;
	  in_len = 2;
	  push((UInt8)c2);
	  // If c2 is the Set-B - base64 charset -, put '-' end of the base64 character.
	  if (Utf7::setBP(c2) || (c2 == (UInt16)'-'))
	  {
	    push((UInt8)'-');
	  }
	  b64_p = false;
	}
	else
	{
	  // modified b64 encode
	  in[2] = (UInt8)(c2 >> 8);
	  undecoded = (UInt8)(c2 & 0xff);
	  has_undecoded_p = true;
	}
      }
    }
    catch (IOException& s)
    {
      if (s.errorNum() == IOException::READ_BEYOND_THE_EOF)
      {
	in[2] = 0;
	in_len = 2;
	push((UInt8)'-');		// for end base64 status
	b64_p = false;
      }
      else
      {
	throw s;
      }
    }
    Size len = Base64::encodeModified(in, in_len, out);
    while (len--)
    {
      push(out[len]);
    }
    if (!prev_b64_p)
    {
      push('+');
    }
  }
  return pop();
}

IFilter*
FromUcs2ToUtf7::createIFilter()
{
  return new FromUcs2ToUtf7();
}

// ------------------------------------------------------------------------
// class FromUcs2ToUcs4

FromUcs2ToUcs4::FromUcs2ToUcs4()
{
}

FromUcs2ToUcs4::~FromUcs2ToUcs4()
{
}

UInt32
FromUcs2ToUcs4::getUInt32() CCC_RAISES(IOException)
{
  if (getQueueLength())
  {
    UInt32 ret = pop();
    return ret;
  }
  // UCS2 dosen't handle surrogate pair, so conversion is very simple.
  UInt32 c = (UInt32)src->getUInt16();
  return c;
}

IFilter*
FromUcs2ToUcs4::createIFilter()
{
  return new FromUcs2ToUcs4();
}

// ------------------------------------------------------------------------
// class FromUcs4ToUcs2
FromUcs4ToUcs2::FromUcs4ToUcs2()
{
}

FromUcs4ToUcs2::~FromUcs4ToUcs2()
{
}

UInt16
FromUcs4ToUcs2::getUInt16() CCC_RAISES(IOException)
{
  if (getQueueLength())
  {
    UInt16 ret = pop();
    return ret;
  }
  // UCS2 only handle BMP.
  UInt32 c = src->getUInt32();
  while (c > 0x0000ffff)
  {
    // skip not BMP chars
    c = src->getUInt32();
  }
  return (UInt16)c;
}

IFilter*
FromUcs4ToUcs2::createIFilter()
{
  return new FromUcs4ToUcs2();
}

// ------------------------------------------------------------------------
// class FromUtf16NToUcs4

FromUtf16NToUcs4::FromUtf16NToUcs4()
{
}

FromUtf16NToUcs4::~FromUtf16NToUcs4()
{
}

UInt32
FromUtf16NToUcs4::getUInt32() CCC_RAISES(IOException)
{
  UInt32 ucs4;
  for (;;)
  {
    UInt16 c = src->getUInt16();
    if (Unicode::highSurrogateP(c))
    {
      UInt16 low = src->getUInt16();
      if (!Unicode::lowSurrogateP(low))
      {
	// ERROR: low is not surrogate
      }
      else
      {
	ucs4 = Unicode::convertSurrogatePairToUcs4(c, low);
	return ucs4;
      }
    }
    else
    {
      ucs4 = (UInt32)c;
      return ucs4;
    }
  }
}

IFilter*
FromUtf16NToUcs4::createIFilter()
{
  return new FromUtf16NToUcs4();
}
  
// ------------------------------------------------------------------------
// class FromUcs4ToUtf16N

FromUcs4ToUtf16N::FromUcs4ToUtf16N()
{
}

FromUcs4ToUtf16N::~FromUcs4ToUtf16N()
{
}

UInt16
FromUcs4ToUtf16N::getUInt16() CCC_RAISES(IOException)
{
 BOM:
  bool top_p = topP();
  if (getQueueLength())
  {
    UInt16 ret = pop();
    if (top_p && (ret == ICEMAN_UNICODE_ZWNBSP))
    {
      setTopP(false);
      goto BOM;
    }
    return ret;
  }
  for (;;)
  {
    UInt32 ucs4 = src->getUInt32();
    setTopP(false);
    if (top_p && (ucs4 == ICEMAN_UNICODE_ZWNBSP))
    {
      // BOM: skip
    }
    else if (ucs4 <= 0xffff)
    {
      return (UInt16)ucs4;
    }
    else if (ucs4 <= 0x10ffff)
    {
      // surrogate pair
      UInt16 hi, lo;
      bool p = Unicode::convertUcs4ToSurrogatePair(ucs4, &hi, &lo);
      assert(p);
      push(lo);
      return hi;
    }
    else
    {
      // can't convert to Utf16, skip
    }
  }
}

IFilter*
FromUcs4ToUtf16N::createIFilter()
{
  return new FromUcs4ToUtf16N();
}

// ------------------------------------------------------------------------
// class FromUtf16NToUtf16Be

FromUtf16NToUtf16Be::FromUtf16NToUtf16Be()
{
}

FromUtf16NToUtf16Be::~FromUtf16NToUtf16Be()
{
}

UInt16
FromUtf16NToUtf16Be::getUInt16() CCC_RAISES(IOException)
{
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt16 ret = UInt16SwapEndianIFilter::getUInt16();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt16 ret = UInt16PushbackIFilter::getUInt16();
#endif /* CCC_ENDIAN */
  return ret;
}

IFilter*
FromUtf16NToUtf16Be::createIFilter()
{
  return new FromUtf16NToUtf16Be();
}
  
// ------------------------------------------------------------------------
// class FromUtf16NToUtf16Le

FromUtf16NToUtf16Le::FromUtf16NToUtf16Le()
{
}

FromUtf16NToUtf16Le::~FromUtf16NToUtf16Le()
{
}

UInt16
FromUtf16NToUtf16Le::getUInt16() CCC_RAISES(IOException)
{
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt16 ret = UInt16PushbackIFilter::getUInt16();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt16 ret = UInt16SwapEndianIFilter::getUInt16();
#endif /* CCC_ENDIAN */
  return ret;
}

IFilter*
FromUtf16NToUtf16Le::createIFilter()
{
  return new FromUtf16NToUtf16Le();
}

// ------------------------------------------------------------------------
// class FromUtf16BeToUtf16N

FromUtf16BeToUtf16N::FromUtf16BeToUtf16N()
{
}

FromUtf16BeToUtf16N::~FromUtf16BeToUtf16N()
{
}

UInt16
FromUtf16BeToUtf16N::getUInt16() CCC_RAISES(IOException)
{
  bool top_p = topP();
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt16 ret = UInt16SwapEndianIFilter::getUInt16();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt16 ret = UInt16PushbackIFilter::getUInt16();
#endif /* CCC_ENDIAN */
  if (top_p && (ret == ICEMAN_UNICODE_ZWNBSP))
  {
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    ret = UInt16SwapEndianIFilter::getUInt16();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    ret = UInt16PushbackIFilter::getUInt16();
#endif /* CCC_ENDIAN */
  }
  setTopP(false);
  return ret;
}

IFilter*
FromUtf16BeToUtf16N::createIFilter()
{
  return new FromUtf16BeToUtf16N();
}

// ------------------------------------------------------------------------
// class FromUtf16LeToUtf16N

FromUtf16LeToUtf16N::FromUtf16LeToUtf16N()
{
}

FromUtf16LeToUtf16N::~FromUtf16LeToUtf16N()
{
}

UInt16
FromUtf16LeToUtf16N::getUInt16() CCC_RAISES(IOException)
{
  bool top_p = topP();
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt16 ret = UInt16PushbackIFilter::getUInt16();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt16 ret = UInt16SwapEndianIFilter::getUInt16();
#endif /* CCC_ENDIAN */
  if (top_p && (ret == ICEMAN_UNICODE_ZWNBSP))
  {
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    ret = UInt16PushbackIFilter::getUInt16();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    ret = UInt16SwapEndianIFilter::getUInt16();
#endif /* CCC_ENDIAN */
  }
  setTopP(false);
  return ret;
}

IFilter*
FromUtf16LeToUtf16N::createIFilter()
{
  return new FromUtf16LeToUtf16N();
}

// ------------------------------------------------------------------------
// class FromUtf32NToUcs4

FromUtf32NToUcs4::FromUtf32NToUcs4()
{
}

FromUtf32NToUcs4::~FromUtf32NToUcs4()
{
}

UInt32
FromUtf32NToUcs4::getUInt32() CCC_RAISES(IOException)
{
  UInt32 c = src->getUInt32();
  return c;
}

IFilter*
FromUtf32NToUcs4::createIFilter()
{
  return new FromUtf32NToUcs4();
}

// ------------------------------------------------------------------------
// class FromUcs4ToUtf32N

FromUcs4ToUtf32N::FromUcs4ToUtf32N()
{
}

FromUcs4ToUtf32N::~FromUcs4ToUtf32N()
{
}

UInt32
FromUcs4ToUtf32N::getUInt32() CCC_RAISES(IOException)
{
  for (;;)
  {
    UInt32 c = src->getUInt32();
    if (c <= ICEMAN_UNICODE_UTF32_MAX)
    {
      return c;
    }
  }
}

IFilter*
FromUcs4ToUtf32N::createIFilter()
{
  return new FromUcs4ToUtf32N();
}

// ------------------------------------------------------------------------
// class FromUtf32NToUtf32Be

FromUtf32NToUtf32Be::FromUtf32NToUtf32Be()
{
}

FromUtf32NToUtf32Be::~FromUtf32NToUtf32Be()
{
}

UInt32
FromUtf32NToUtf32Be::getUInt32() CCC_RAISES(IOException)
{
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt32 ret = UInt32SwapEndianIFilter::getUInt32();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt32 ret = UInt32PushbackIFilter::getUInt32();
#endif /* CCC_ENDIAN */
  return ret;
}

IFilter*
FromUtf32NToUtf32Be::createIFilter()
{
  return new FromUtf32NToUtf32Be();
}
  
// ------------------------------------------------------------------------
// class FromUtf32NToUtf32Le

FromUtf32NToUtf32Le::FromUtf32NToUtf32Le()
{
}

FromUtf32NToUtf32Le::~FromUtf32NToUtf32Le()
{
}

UInt32
FromUtf32NToUtf32Le::getUInt32() CCC_RAISES(IOException)
{
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt32 ret = UInt32PushbackIFilter::getUInt32();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt32 ret = UInt32SwapEndianIFilter::getUInt32();
#endif /* CCC_ENDIAN */
  return ret;
}

IFilter*
FromUtf32NToUtf32Le::createIFilter()
{
  return new FromUtf32NToUtf32Le();
}

// ------------------------------------------------------------------------
// class FromUtf32BeToUtf32N

FromUtf32BeToUtf32N::FromUtf32BeToUtf32N()
{
}

FromUtf32BeToUtf32N::~FromUtf32BeToUtf32N()
{
}

UInt32
FromUtf32BeToUtf32N::getUInt32() CCC_RAISES(IOException)
{
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt32 ret = UInt32SwapEndianIFilter::getUInt32();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt32 ret = UInt32PushbackIFilter::getUInt32();
#endif /* CCC_ENDIAN */
  return ret;
}

IFilter*
FromUtf32BeToUtf32N::createIFilter()
{
  return new FromUtf32BeToUtf32N();
}

// ------------------------------------------------------------------------
// class FromUtf32LeToUtf32N

FromUtf32LeToUtf32N::FromUtf32LeToUtf32N()
{
}

FromUtf32LeToUtf32N::~FromUtf32LeToUtf32N()
{
}

UInt32
FromUtf32LeToUtf32N::getUInt32() CCC_RAISES(IOException)
{
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
  UInt32 ret = UInt32PushbackIFilter::getUInt32();
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
  UInt32 ret = UInt32SwapEndianIFilter::getUInt32();
#endif /* CCC_ENDIAN */
  return ret;
}

IFilter*
FromUtf32LeToUtf32N::createIFilter()
{
  return new FromUtf32LeToUtf32N();
}

// ------------------------------------------------------------------------
// class FromUtf32NToUtf16N

FromUtf32NToUtf16N::FromUtf32NToUtf16N()
{
}

FromUtf32NToUtf16N::~FromUtf32NToUtf16N()
{
}

UInt16
FromUtf32NToUtf16N::getUInt16() CCC_RAISES(IOException)
{
  if (getQueueLength())
  {
    UInt16 ret = pop();
    return ret;
  }
  for (;;)
  {
    UInt32 c = src->getUInt32();
    if (c <= ICEMAN_UNICODE_SNC)
    {
      UInt16 c2 = (UInt16)c;
      return c2;
    }
    if (c <= ICEMAN_UNICODE_UTF16_MAX)
    {
      UInt16 hi, lo;
      Unicode::convertUcs4ToSurrogatePair(c, &hi, &lo);
      push(lo);
      return hi;
    }
    // skip c, it's a illegal char.
  }
}

IFilter*
FromUtf32NToUtf16N::createIFilter()
{
  return new FromUtf32NToUtf16N();
}

// ------------------------------------------------------------------------
// class FromUtf16NToUtf32N

FromUtf16NToUtf32N::FromUtf16NToUtf32N()
{
}

FromUtf16NToUtf32N::~FromUtf16NToUtf32N()
{
}

UInt32
FromUtf16NToUtf32N::getUInt32() CCC_RAISES(IOException)
{
  for (;;)
  {
    UInt16 c = src->getUInt16();
    if (Unicode::highSurrogateP(c))
    {
      UInt16 c2 = src->getUInt16();
      if (Unicode::lowSurrogateP(c2))
      {
	UInt32 ucs4 = Unicode::convertSurrogatePairToUcs4(c, c2);
	return ucs4;
      }
      // skip surrogate pair error.
    }
    else
    {
      UInt32 c2 = (UInt32)c;
      return c2;
    }
  }
}

IFilter*
FromUtf16NToUtf32N::createIFilter()
{
  return new FromUtf16NToUtf32N();
}

// ------------------------------------------------------------------------
// class FromUtf16NToUtf8

FromUtf16NToUtf8::FromUtf16NToUtf8()
{
}

FromUtf16NToUtf8::~FromUtf16NToUtf8()
{
}

Int8
FromUtf16NToUtf8::getInt8() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
    // UTF-8 BOM : EF, BB, BF
    push((Int8)0xbf);
    push((Int8)0xbb);
    return (Int8)0xef;
  }
  if (getQueueLength())
  {
    Int8 ret = pop();
    return ret;
  }
  for (;;)
  {
    UInt16 c = src->getUInt16();
    if (c <= 0x7f)
    {
      // 1 byte                    1
      // 0000.0000:0xxx.xxxx => 0xxxxxxx
      return (Int8)c;
    }
    else if ((0x80 <= c) && (c <= 0x7ff))
    {
      // 2 bytes                   1        2
      // 0000.0yyy:yyxx.xxxx => 110yyyyy:10xxxxxx
      UInt8 c1 = ((UInt8)(c >> 6)) | 0xc0u;
      UInt8 c2 = ((UInt8)(c & 0x3fu)) | 0x80u;
      push((Int8)c2);
      return (Int8)c1;
    }
    else if (!Unicode::highSurrogateP(c))
    {
      // 3 bytes                   1        2        3
      // zzzz.yyyy:yyxx.xxxx => 1110zzzz:10yyyyyy:10xxxxxx
      UInt8 c1 = ((UInt8)(c >> 12)) | 0xe0u;
      UInt8 c2 = ((UInt8)((c & 0x0fc0u) >> 6)) | 0x80u;
      UInt8 c3 = ((UInt8)(c & 0x3fu)) | 0x80u;
      push((Int8)c3);
      push((Int8)c2);
      return (Int8)c1;
    }
    else
    {
      UInt16 lo = src->getUInt16();
      if (Unicode::lowSurrogateP(lo))
      {
	// surrogate pair
	// 4 bytes
	UInt32 u = Unicode::convertSurrogatePairToUcs4(c, lo);
	// uuuuu,zzzzyyyy,yyxxxxxx =>
	//    1        2       3       4
	// 11110uuu:10uuzzzz:10yyyyy:10xxxxxx
	UInt8 c1 = ((UInt8)(u >> 18)) | 0xf0u;
	UInt8 c2 = (((UInt8)(u >> 12)) & 0x3fu) | 0x80u;
	UInt8 c3 = (((UInt8)(u >> 6)) & 0x3fu) | 0x80u;
	UInt8 c4 = (((UInt8)u) & 0x3fu) | 0x80u;
	push((Int8)c4);
	push((Int8)c3);
	push((Int8)c2);
	return (Int8)c1;
      }
      // skip surrogate error
    }
  }
}

IFilter*
FromUtf16NToUtf8::createIFilter()
{
  return new FromUtf16NToUtf8();
}

// ------------------------------------------------------------------------
// class FromUtf16NToUtf8N

FromUtf16NToUtf8N::FromUtf16NToUtf8N()
{
}

FromUtf16NToUtf8N::~FromUtf16NToUtf8N()
{
}

Int8
FromUtf16NToUtf8N::getInt8() CCC_RAISES(IOException)
{
  if (getQueueLength())
  {
    Int8 ret = pop();
    return ret;
  }
  for (;;)
  {
    UInt16 c = src->getUInt16();
    if (c <= 0x7f)
    {
      // 1 byte                    1
      // 0000.0000:0xxx.xxxx => 0xxxxxxx
      return (Int8)c;
    }
    else if ((0x80 <= c) && (c <= 0x7ff))
    {
      // 2 bytes                   1        2
      // 0000.0yyy:yyxx.xxxx => 110yyyyy:10xxxxxx
      UInt8 c1 = ((UInt8)(c >> 6)) | 0xc0u;
      UInt8 c2 = ((UInt8)(c & 0x3fu)) | 0x80u;
      push((Int8)c2);
      return (Int8)c1;
    }
    else if (!Unicode::highSurrogateP(c))
    {
      // 3 bytes                   1        2        3
      // zzzz.yyyy:yyxx.xxxx => 1110zzzz:10yyyyyy:10xxxxxx
      UInt8 c1 = ((UInt8)(c >> 12)) | 0xe0u;
      UInt8 c2 = ((UInt8)((c & 0x0fc0u) >> 6)) | 0x80u;
      UInt8 c3 = ((UInt8)(c & 0x3fu)) | 0x80u;
      push((Int8)c3);
      push((Int8)c2);
      return (Int8)c1;
    }
    else
    {
      UInt16 lo = src->getUInt16();
      if (Unicode::lowSurrogateP(lo))
      {
	// surrogate pair
	// 4 bytes
	UInt32 u = Unicode::convertSurrogatePairToUcs4(c, lo);
	// uuuuu,zzzzyyyy,yyxxxxxx =>
	//    1        2       3       4
	// 11110uuu:10uuzzzz:10yyyyy:10xxxxxx
	UInt8 c1 = ((UInt8)(u >> 18)) | 0xf0u;
	UInt8 c2 = (((UInt8)(u >> 12)) & 0x3fu) | 0x80u;
	UInt8 c3 = (((UInt8)(u >> 6)) & 0x3fu) | 0x80u;
	UInt8 c4 = (((UInt8)u) & 0x3fu) | 0x80u;
	push((Int8)c4);
	push((Int8)c3);
	push((Int8)c2);
	return (Int8)c1;
      }
      // skip surrogate error
    }
  }
}

IFilter*
FromUtf16NToUtf8N::createIFilter()
{
  return new FromUtf16NToUtf8N();
}

// ------------------------------------------------------------------------
// class FromUtf8ToUtf16N

FromUtf8ToUtf16N::FromUtf8ToUtf16N()
{
}

FromUtf8ToUtf16N::~FromUtf8ToUtf16N()
{
}

UInt16
FromUtf8ToUtf16N::getUInt16() CCC_RAISES(IOException)
{
 BOM:
  UInt16 ret;
  bool top_p = topP();
  if (getQueueLength())
  {
    ret = pop();
    setTopP(false);
    if (top_p && (ret == ICEMAN_UNICODE_ZWNBSP))
    {
      goto BOM;
    }
    return ret;
  }
  for (;;)
  {
    UInt16 c1 = (UInt16)(src->getInt8());
    setTopP(false);
    if (c1 & 0x0080u)
    {
      // more than 2 bytes
      UInt16 c2 = (UInt16)(src->getInt8());
      if (c1 & 0x0020u)
      {
	// more than 3 bytes
	UInt16 c3 = (UInt16)(src->getInt8());
	if (c1 & 0x0010u)
	{
	  // 4 bytes
	  UInt16 c4 = (UInt16)(src->getInt8());
	  UInt32 u = (((UInt32)(c1 & 0x07u)) << 18 |
		      ((UInt32)(c2 & 0x3fu)) << 12 |
		      ((UInt32)(c3 & 0x3fu)) << 6 |
		      ((UInt32)(c4 & 0x3fu)));
	  UInt16 hi, lo;
	  if (Unicode::convertUcs4ToSurrogatePair(u, &hi, &lo))
	  {
	    push(lo);
	    return hi;
	  }
	}
	else
	{
	  // 3 bytes
	  //    c1       c2       c3
	  // 1110zzzz:10yyyyyy:10xxxxxx => zzzz.yyyy:yyxx.xxxx
	  ret = ((c1 & 0x000fu) << 12) | ((c2 & 0x003fu) << 6) | (c3 & 0x003f);
	  if (top_p && (ret == ICEMAN_UNICODE_ZWNBSP))
	  {
	    // remove BOM, skip
	  }
	  else
	  {
	    return ret;
	  }
	}
      }
      else
      {
	// 2 bytes
	//    c1       c2
	// 110yyyyy:10xxxxxx => 0000.0yyy:yyxx.xxxx
	ret = ((c1 & 0x001fu) << 6) | (c2 & 0x003fu);
	return ret;
      }
    }
    else
    {
      // 1 byte
      ret = c1;
      return ret;
    }
  }
}

IFilter*
FromUtf8ToUtf16N::createIFilter()
{
  return new FromUtf8ToUtf16N();
}

// ------------------------------------------------------------------------
// class FromUtf16NToUtf16
FromUtf16NToUtf16::FromUtf16NToUtf16()
{
}

FromUtf16NToUtf16::~FromUtf16NToUtf16()
{
}

UInt16
FromUtf16NToUtf16::getUInt16() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
    return ICEMAN_UNICODE_ZWNBSP;
  }
  UInt16 c = src->getUInt16();
  return c;
}

IFilter*
FromUtf16NToUtf16::createIFilter()
{
  return new FromUtf16NToUtf16();
}

// ------------------------------------------------------------------------
// class FromUtf16ToUtf16N
FromUtf16ToUtf16N::FromUtf16ToUtf16N()
{
  swap_p = false;
}

FromUtf16ToUtf16N::~FromUtf16ToUtf16N()
{
}

UInt16
FromUtf16ToUtf16N::getUInt16() CCC_RAISES(IOException)
{
  bool top_p = topP();
  UInt16 c = src->getUInt16();
  setTopP(false);
  if (top_p)
  {
    if (c == ICEMAN_UNICODE_ZWNBSP)
    {
      swap_p = false;
      c = src->getUInt16();
    }
    else if (c == ICEMAN_UNICODE_I_ZWNBSP)
    {
      swap_p = true;
      c = src->getUInt16();
    }
  }
  if (swap_p)
  {
    c = swap_16(c);
  }
  return c;
}

IFilter*
FromUtf16ToUtf16N::createIFilter()
{
  return new FromUtf16ToUtf16N();
}
  
// ------------------------------------------------------------------------
// class FromUtf32NToUtf32
FromUtf32NToUtf32::FromUtf32NToUtf32()
{
}

FromUtf32NToUtf32::~FromUtf32NToUtf32()
{
}

UInt32
FromUtf32NToUtf32::getUInt32() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
    return ICEMAN_UNICODE_ZWNBSP;
  }
  UInt32 c = src->getUInt32();
  return c;
}

IFilter*
FromUtf32NToUtf32::createIFilter()
{
  return new FromUtf32NToUtf32();
}

// ------------------------------------------------------------------------
// class FromUtf32ToUtf32N
FromUtf32ToUtf32N::FromUtf32ToUtf32N()
{
  swap_p = false;
}

FromUtf32ToUtf32N::~FromUtf32ToUtf32N()
{
}

UInt32
FromUtf32ToUtf32N::getUInt32() CCC_RAISES(IOException)
{
  UInt32 c = src->getUInt32();
  if (topP())
  {
    setTopP(false);
    if (c == ICEMAN_UNICODE_ZWNBSP)
    {
      swap_p = false;
      c = src->getUInt32();
    }
    else if (c == ICEMAN_UNICODE_I_ZWNBSP_32)
    {
      swap_p = true;
      c = src->getUInt32();
    }
  }
  if (swap_p)
  {
    c = swap_32(c);
  }
  return c;
}

IFilter*
FromUtf32ToUtf32N::createIFilter()
{
  return new FromUtf32ToUtf32N();
}

// ------------------------------------------------------------------------
// class FromUcs2ToUtf16N
FromUcs2ToUtf16N::FromUcs2ToUtf16N()
{
}

FromUcs2ToUtf16N::~FromUcs2ToUtf16N()
{
}

UInt16
FromUcs2ToUtf16N::getUInt16() CCC_RAISES(IOException)
{
  bool top_p = topP();
  UInt16 c = src->getUInt16();
  if (top_p && (c == ICEMAN_UNICODE_ZWNBSP))
  {
    c = src->getUInt16();
  }
  setTopP(false);
  return c;
}

IFilter*
FromUcs2ToUtf16N::createIFilter()
{
  return new FromUcs2ToUtf16N();
}

// ------------------------------------------------------------------------
// class FromUtf16BeToUtf16BeBom
FromUtf16BeToUtf16BeBom::FromUtf16BeToUtf16BeBom()
{
}

FromUtf16BeToUtf16BeBom::~FromUtf16BeToUtf16BeBom()
{
}

UInt16
FromUtf16BeToUtf16BeBom::getUInt16() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    return ICEMAN_UNICODE_I_ZWNBSP;
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    return ICEMAN_UNICODE_ZWNBSP;
#endif /* CCC_ENDIAN */
  }
  UInt16 c = src->getUInt16();
  return c;
}

IFilter*
FromUtf16BeToUtf16BeBom::createIFilter()
{
  return new FromUtf16BeToUtf16BeBom();
}

// ------------------------------------------------------------------------
// class FromUtf16LeToUtf16LeBom
FromUtf16LeToUtf16LeBom::FromUtf16LeToUtf16LeBom()
{
}

FromUtf16LeToUtf16LeBom::~FromUtf16LeToUtf16LeBom()
{
}

UInt16
FromUtf16LeToUtf16LeBom::getUInt16() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    return ICEMAN_UNICODE_ZWNBSP;
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    return ICEMAN_UNICODE_I_ZWNBSP;
#endif /* CCC_ENDIAN */
  }
  UInt16 c = src->getUInt16();
  return c;
}

IFilter*
FromUtf16LeToUtf16LeBom::createIFilter()
{
  return new FromUtf16LeToUtf16LeBom();
}

// ------------------------------------------------------------------------
// class FromUtf16BeBomToUtf16Be
FromUtf16BeBomToUtf16Be::FromUtf16BeBomToUtf16Be()
{
}

FromUtf16BeBomToUtf16Be::~FromUtf16BeBomToUtf16Be()
{
}

UInt16
FromUtf16BeBomToUtf16Be::getUInt16() CCC_RAISES(IOException)
{
  UInt16 c = src->getUInt16();
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    if (c == ICEMAN_UNICODE_I_ZWNBSP)
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    if (c == ICEMAN_UNICODE_ZWNBSP)
#endif /* CCC_ENDIAN */
    {
      c = src->getUInt16();
    }
  }
  return c;
}

IFilter*
FromUtf16BeBomToUtf16Be::createIFilter()
{
  return new FromUtf16BeBomToUtf16Be();
}

// ------------------------------------------------------------------------
// class FromUtf16LeBomToUtf16Le
FromUtf16LeBomToUtf16Le::FromUtf16LeBomToUtf16Le()
{
}

FromUtf16LeBomToUtf16Le::~FromUtf16LeBomToUtf16Le()
{
}

UInt16
FromUtf16LeBomToUtf16Le::getUInt16() CCC_RAISES(IOException)
{
  UInt16 c = src->getUInt16();
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    if (c == ICEMAN_UNICODE_ZWNBSP)
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    if (c == ICEMAN_UNICODE_I_ZWNBSP)
#endif /* CCC_ENDIAN */
    {
      c = src->getUInt16();
    }
  }
  return c;
}

IFilter*
FromUtf16LeBomToUtf16Le::createIFilter()
{
  return new FromUtf16LeBomToUtf16Le();
}

// ------------------------------------------------------------------------
// class FromUtf32BeToUtf32BeBom
FromUtf32BeToUtf32BeBom::FromUtf32BeToUtf32BeBom()
{
}

FromUtf32BeToUtf32BeBom::~FromUtf32BeToUtf32BeBom()
{
}

UInt32
FromUtf32BeToUtf32BeBom::getUInt32() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    return ICEMAN_UNICODE_I_ZWNBSP_32;
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    return ICEMAN_UNICODE_ZWNBSP;
#endif /* CCC_ENDIAN */
  }
  UInt32 c = src->getUInt32();
  return c;
}

IFilter*
FromUtf32BeToUtf32BeBom::createIFilter()
{
  return new FromUtf32BeToUtf32BeBom();
}

// ------------------------------------------------------------------------
// class FromUtf32LeToUtf32LeBom
FromUtf32LeToUtf32LeBom::FromUtf32LeToUtf32LeBom()
{
}

FromUtf32LeToUtf32LeBom::~FromUtf32LeToUtf32LeBom()
{
}

UInt32
FromUtf32LeToUtf32LeBom::getUInt32() CCC_RAISES(IOException)
{
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    return ICEMAN_UNICODE_ZWNBSP;
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    return ICEMAN_UNICODE_I_ZWNBSP_32;
#endif /* CCC_ENDIAN */
  }
  UInt32 c = src->getUInt32();
  return c;
}

IFilter*
FromUtf32LeToUtf32LeBom::createIFilter()
{
  return new FromUtf32LeToUtf32LeBom();
}

// ------------------------------------------------------------------------
// class FromUtf32BeBomToUtf32Be
FromUtf32BeBomToUtf32Be::FromUtf32BeBomToUtf32Be()
{
}

FromUtf32BeBomToUtf32Be::~FromUtf32BeBomToUtf32Be()
{
}

UInt32
FromUtf32BeBomToUtf32Be::getUInt32() CCC_RAISES(IOException)
{
  UInt32 c = src->getUInt32();
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    if (c == ICEMAN_UNICODE_I_ZWNBSP_32)
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    if (c == ICEMAN_UNICODE_ZWNBSP)
#endif /* CCC_ENDIAN */
    {
      c = src->getUInt32();
    }
  }
  return c;
}

IFilter*
FromUtf32BeBomToUtf32Be::createIFilter()
{
  return new FromUtf32BeBomToUtf32Be();
}

// ------------------------------------------------------------------------
// class FromUtf32LeBomToUtf32Le
FromUtf32LeBomToUtf32Le::FromUtf32LeBomToUtf32Le()
{
}

FromUtf32LeBomToUtf32Le::~FromUtf32LeBomToUtf32Le()
{
}

UInt32
FromUtf32LeBomToUtf32Le::getUInt32() CCC_RAISES(IOException)
{
  UInt32 c = src->getUInt32();
  if (topP())
  {
    setTopP(false);
#if CCC_ENDIAN == CCC_LITTLE_ENDIAN
    if (c == ICEMAN_UNICODE_ZWNBSP)
#elif CCC_ENDIAN == CCC_BIG_ENDIAN
    if (c == ICEMAN_UNICODE_I_ZWNBSP_32)
#endif /* CCC_ENDIAN */
    {
      c = src->getUInt32();
    }
  }
  return c;
}

IFilter*
FromUtf32LeBomToUtf32Le::createIFilter()
{
  return new FromUtf32LeBomToUtf32Le();
}

CCC_NAMESPACE_END(CCC);

