﻿// $Id$

#ifndef INCLUDE_ccc_xml_CharacterData_h
#define INCLUDE_ccc_xml_CharacterData_h

#include <ccc/base/base.h>
#include <ccc/xml/config.h>

CCC_NAMESPACE_START(CCC);

/*!
 * The CharacterData interface extends Node with a set of attributes and
 * methods for accessing character data in the DOM. For clarity this set is
 * defined here rather than on each object that uses these attributes and
 * methods. No DOM objects correspond directly to CharacterData, though Text
 * and others do inherit the interface from it. All offsets in this interface
 * start from 0.
 *
 * As explained in the DOMString interface, text strings in the DOM are
 * represented in UTF-16, i.e. as a sequence of 16-bit units. In the
 * following, the term 16-bit units is used whenever necessary to indicate
 * that indexing on CharacterData is done in 16-bit units.
 */
class CharacterData
  : public Node
{
 public:
  /*!
   * (CODEX original)<br>
   * コンストラクタ
   * \param document Document
   * \param parent 親ノード
   * \param node_type ノード形式
   */
  CharacterData(Document* document, Node* parent, unsigned short node_type);
  /*!
   * (CODEX original)<br>
   * デストラクタ
   */
  ~CharacterData();
  /*!
   * The character data of the node that implements this interface. The
   * DOM implementation may not put arbitrary limits on the amount of
   * data that may be stored in a CharacterData node. However,
   * implementation limits may mean that the entirety of a node's data
   * may not fit into a single DOMString. In such cases, the user may
   * call substringData to retrieve the data in appropriately sized pieces.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised when the node is readonly.
   */
  void setData(const DOMString* arg) CCC_RAISES(DOMException);
  void setData(const DOMString& arg) CCC_RAISES(DOMException);
  /*!
   * \exception DOMSTRING_SIZE_ERR Raised when it would return more
   *	characters than fit in a DOMString variable on
   *	the implementation platform.
   */
  const DOMString* getData() const CCC_RAISES(DOMException);
  /*!
   * The number of 16-bit units that are available through data and the
   * substringData method below. This may have the value zero, i.e.,
   * CharacterData nodes may be empty.
   */
  Size getLength() const;
  /*!
   * Extracts a range of data from the node.
   *
   * \param offset Start offset of substring to extract.
   * \param count The number of 16-bit units to extract.
   *
   * \return The specified substring. If the sum of offset and count
   *	exceeds the length, then all 16-bit units to the end of the data
   *	are returned.
   *
   * \exception INDEX_SIZE_ERR Raised if the specified offset is
   *	negative or greater than the number of 16-bit units in data,
   *	or if the specified count is negative.
   * \exception DOMSTRING_SIZE_ERR Raised if the specified range of text
   *	does not fit into a DOMString.
   */
  DOMString* substringData(CCC_IN unsigned long offset, CCC_IN unsigned long count) CCC_RAISES(DOMException);
  /*!
   * Append the string to the end of the character data of the node.
   * Upon success, data provides access to the concatenation of data
   * and the DOMString specified.
   *
   * \param arg The DOMString to append.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   */
  void appendData(CCC_IN const DOMString* arg) CCC_RAISES(DOMException);
  void appendData(CCC_IN const DOMString& arg) CCC_RAISES(DOMException);
  /*!
   * Insert a string at the specified 16-bit unit offset.
   *
   * \param offset The character offset at which to insert.
   * \param arg The DOMString to insert.
   *
   * \exception INDEX_SIZE_ERR Raised if the specified offset is negative
   *	or greater than the number of 16-bit units in data.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   */
  void insertData(CCC_IN unsigned long offset, CCC_IN const DOMString* arg) CCC_RAISES(DOMException);
  void insertData(CCC_IN unsigned long offset, CCC_IN const DOMString& arg) CCC_RAISES(DOMException);
  /*!
   * Remove a range of 16-bit units from the node. Upon success, data
   * and length reflect the change.
   *
   * \param offset The offset from which to start removing.
   * \param count The number of 16-bit units to delete. If the sum of
   *	offset and count exceeds length then all 16-bit units from
   *	offset to the end of the data are deleted.
   *
   * \exception INDEX_SIZE_ERR Raised if the specified offset is
   *	negative or greater than the number of 16-bit units in data,
   *	or if the specified count is negative.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
  */
  void deleteData(CCC_IN unsigned long offset, CCC_IN unsigned long count) CCC_RAISES(DOMException);
  /*!
   * Replace the characters starting at the specified 16-bit unit
   * offset with the specified string.
   *
   * \param offset The offset from which to start replacing.
   * \param count The number of 16-bit units to replace. If the sum of
   *	offset and count exceeds length, then all 16-bit units to the
   *	end of the data are replaced; (i.e., the effect is the same as a
   *	remove method call with the same range, followed by an append
   *	method invocation).
   * \param arg The DOMString with which the range must be replaced.
   *
   * \exception INDEX_SIZE_ERR Raised if the specified offset is
   *	negative or greater than the number of 16-bit units in data,
   *	or if the specified count is negative.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   */
  void replaceData(CCC_IN unsigned long offset, CCC_IN unsigned long count, CCC_IN const DOMString* arg) CCC_RAISES(DOMException);
  void replaceData(CCC_IN unsigned long offset, CCC_IN unsigned long count, CCC_IN const DOMString& arg) CCC_RAISES(DOMException);
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_CharacterData_h */
