﻿// $Id$

#ifndef INCLUDE_ccc_xml_Document_h
#define INCLUDE_ccc_xml_Document_h

#include <ccc/base/base.h>
#include <ccc/xml/config.h>

CCC_NAMESPACE_START(CCC);

/*!
 * \brief XML Document class
 *
 * The Document interface represents the entire HTML or XML document.
 * Conceptually, it is the root of the document tree, and provides the
 * primary access to the document's data.
 *
 * Since elements, text nodes, comments, processing instructions, etc.
 * cannot exist outside the context of a Document, the Document interface
 * also contains the factory methods needed to create these objects. The
 * Node objects created have a ownerDocument attribute which associates them
 * with the Document within whose context they were created.
 */
class Document
  : public Node
{
  /*!
   * The DOMImplementation object that handles this document. A DOM
   * application may use objects from multiple implementations.
   */
  DOMImplementation* dom_implementation;
  /*!
   * The Document Type Declaration (see DocumentType) associated with
   * this document. For HTML documents as well as XML documents
   * without a document type declaration this returns null. The DOM
   * Level 2 does not support editing the Document Type Declaration.
   * docType cannot be altered in any way, including through the use
   * of methods inherited from the Node interface, such as insertNode
   * or removeNode.
   */
  DocumentType* document_type;
 public:
  /*!
   * DocumentTypeを取得します。
   * \return DocumentType
   */
  DocumentType* getDoctype() const;
  /*!
   * DOMImplementationを取得します。
   * \return DOMImplemenation
   */
  DOMImplementation* getImplementation() const { return dom_implementation; }
  /*!
   * This is a convenience attribute that allows direct access to the
   * child node that is the root element of the document. For HTML
   * documents, this is the element with the tagName "HTML".
   */
  Element* getDocumentElement() const;
  // creater
  /*!
   * Creates an element of the type specified. Note that the instance
   * returned implements the Element interface, so attributes can be
   * specified directly on the returned object.
   * In addition, if there are known attributes with default values,
   * Attr nodes representing them are automatically created and
   * attached to the element.
   * To create an element with a qualified name and namespace URI, use
   * the createElementNS method.
   *
   * \param tag_name
   *	The name of the element type to instantiate. For XML, this is
   *	case-sensitive. For HTML, the tag_name parameter may be
   *	provided in any case, but it must be mapped to the canonical
   *	uppercase form by the DOM implementation.
   *
   * \return A new Element object with the nodeName attribute set to tag_name,
   *	and localName, prefix, and namespaceURI set to null.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified name contains an illegal character.
   */
  Element* createElement(CCC_IN const DOMString* tag_name) CCC_RAISES(DOMException);
  Element* createElement(CCC_IN const DOMString& tag_name) CCC_RAISES(DOMException);
  Element* createElement(CCC_IN const DOMChar* tag_name) CCC_RAISES(DOMException);
  /*!
   * Creates an empty DocumentFragment object.
   * \return A new DocumentFragment.
   */
  DocumentFragment* createDocumentFragment();
  /*!
   * Creates a Text node given the specified string.
   *
   * \param data The data for the node.
   *
   * \return The new Text object.
   */
  Text* createTextNode(CCC_IN const DOMString* data);
  Text* createTextNode(CCC_IN const DOMString& data);
  /*!
   * Creates a Comment node given the specified string.
   *
   * \param data The data for the node.
   *
   * \return The new Comment object.
   */
  Comment* createComment(CCC_IN const DOMString* data);
  Comment* createComment(CCC_IN const DOMString& data);
  /*!
   * Creates a CDATASection node whose value is the specified string.
   *
   * \param data The data for the CDATASection contents.
   *
   * \return The new CDATASection object.
   *
   * \exception NOT_SUPPORTED_ERR Raised if this document is an HTML document.
   */
  CDATASection* createCDATASection(CCC_IN const DOMString* data) CCC_RAISES(DOMException);
  CDATASection* createCDATASection(CCC_IN const DOMString& data) CCC_RAISES(DOMException);
  /*!
   * Creates a ProcessingInstruction node given the specified name and
   * data strings.
   *
   * \param target The target part of the processing instruction.
   * \param data The data for the node.
   *
   * \return The new ProcessingInstruction object.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified target contains an illegal character.
   * \exception NOT_SUPPORTED_ERR
   *	Raised if this document is an HTML document.
   */
  ProcessingInstruction* createProcessingInstruction(CCC_IN const DOMString* target, CCC_IN const DOMString* data) CCC_RAISES(DOMException);
  ProcessingInstruction* createProcessingInstruction(CCC_IN const DOMString& target, CCC_IN const DOMString& data) CCC_RAISES(DOMException);
  /*!
   * Creates an Attr of the given name. Note that the Attr instance
   * can then be set on an Element using the setAttributeNode method.
   * To create an attribute with a qualified name and namespace URI,
   * use the createAttributeNS method.
   *
   * \param name The name of the attribute.
   *
   * \return A new Attr object with the nodeName attribute set to name,
   *	and localName, prefix, and namespaceURI set to null.
   *	The value of the attribute is the empty string.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified name contains an illegal character.
   */
  Attr* createAttribute(CCC_IN const DOMString* name) CCC_RAISES(DOMException);
  Attr* createAttribute(CCC_IN const DOMString& name) CCC_RAISES(DOMException);
  Attr* createAttribute(CCC_IN const DOMChar* name) CCC_RAISES(DOMException);
  /*!
   * Creates an EntityReference object. In addition, if the referenced
   * entity is known, the child list of the EntityReference node is
   * made the same as that of the corresponding Entity node.
   *
   * Note: If any descendant of the Entity node has an unbound
   * namespace prefix, the corresponding descendant of the created
   * EntityReference node is also unbound; (its namespaceURI is null).
   * The DOM Level 2 does not support any mechanism to resolve
   * namespace prefixes.
   *
   * \param name The name of the entity to reference.
   *
   * \return The new EntityReference object.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified name contains an illegal character.
   *
   * \exception NOT_SUPPORTED_ERR
   *	Raised if this document is an HTML document.
   */
  EntityReference* createEntityReference(CCC_IN const DOMString* name) CCC_RAISES(DOMException);
  EntityReference* createEntityReference(CCC_IN const DOMString& name) CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:
   *
   * Creates an element of the given qualified name and namespace URI.
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param namespace_uri The namespace URI of the element to create.
   * \param qualified_name
   *	The qualified name of the element type to instantiate.
   *
   * \retval Node.nodeName qualifiedName
   * \retval Node.namespaceURI namespaceURI
   * \retval Node.prefix prefix, extracted from qualifiedName,
   *	or null if there is no prefix
   * \retval Node.localName local name, extracted from qualifiedName
   * \retval Element.tagName qualifiedName
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified qualified name contains an illegal character.
   *
   * \exception NAMESPACE_ERR
   *	Raised if the qualifiedName is malformed, if the qualifiedName has a
   *	prefix and the namespaceURI is null, or if the qualifiedName has a
   *	prefix that is "xml" and the namespaceURI is different from
   *	"http://www.w3.org/XML/1998/namespace" [Namespaces].
   */
  Element* createElementNS(CCC_IN const DOMString* namespace_uri,
			   CCC_IN const DOMString* qualified_name) CCC_RAISES(DOMException);
  Element* createElementNS(CCC_IN const DOMString& namespace_uri,
			   CCC_IN const DOMString& qualified_name) CCC_RAISES(DOMException);
  Element* createElementNS(CCC_IN const DOMChar* namespace_uri,
			   CCC_IN const DOMChar* qualified_name) CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:
   *
   * Creates an attribute of the given qualified name and namespace URI.
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param namespaceURI The namespace URI of the attribute to create.
   * \param qualifiedName The qualified name of the attribute to instantiate.
   *
   * \retval Node.nodeName qualifiedName
   * \retval Node.namespaceURI namespaceURI
   * \retval Node.prefix prefix, extracted from qualifiedName,
   *	or null if there is no prefix
   * \retval Node.localName local name, extracted from qualifiedName
   * \retval Attr.name qualifiedName
   * \retval Node.nodeValue the empty string
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified qualified name contains an illegal character.
   * \exception NAMESPACE_ERR
   *	Raised if the qualifiedName is malformed, if the qualifiedName has a
   *	prefix and the namespaceURI is null, if the qualifiedName has a prefix
   *	that is "xml" and the namespaceURI is different from
   *	"http://www.w3.org/XML/1998/namespace", or if the qualifiedName is
   *	"xmlns" and the namespaceURI is different from
   *	"http://www.w3.org/2000/xmlns/".
   */
  Attr* createAttributeNS(CCC_IN const DOMString* namespaceURI,
			  CCC_IN const DOMString* qualifiedName) CCC_RAISES(DOMException);
  Attr* createAttributeNS(CCC_IN const DOMString& namespaceURI,
			  CCC_IN const DOMString& qualifiedName) CCC_RAISES(DOMException);
  Attr* createAttributeNS(CCC_IN const DOMChar* namespaceURI,
			  CCC_IN const DOMChar* qualifiedName) CCC_RAISES(DOMException);
  /*!
   * Returns a NodeList of all the Elements with a given tag name in the order
   * in which they are encountered in a preorder traversal of the Document
   * tree.
   *
   * \param tagname The name of the tag to match on. The special value "*"
   *	matches all tags.
   *
   * \return A new NodeList object containing all the matched Elements.
   */
  NodeList* getElementsByTagName(CCC_IN const DOMString* tagname) const;
  NodeList* getElementsByTagName(CCC_IN const DOMString& tagname) const;
  /*!
   * Introduced in DOM Level 2:
   *
   * Returns a NodeList of all the Elements with a given local name
   * and namespace URI in the order in which they are encountered in a
   * preorder traversal of the Document tree.
   *
   * \param namespaceURI The namespace URI of the elements to match on.
   *	The special value "*" matches all namespaces.
   * \param localName The local name of the elements to match on.
   * The special value "*" matches all local names.
   *
   * \return A new NodeList object containing all the matched Elements.
   */
  NodeList* getElementsByTagNameNS(CCC_IN const DOMString* namespaceURI, CCC_IN const DOMString* localName);
  NodeList* getElementsByTagNameNS(CCC_IN const DOMString& namespaceURI, CCC_IN const DOMString& localName);
  /*!
   * Introduced in DOM Level 2:
   *
   * Imports a node from another document to this document. The returned node
   * has no parent; (parentNode is null). The source node is not altered or
   * removed from the original document; this method creates a new copy of the
   * source node.
   * For all nodes, importing a node creates a node object owned by the
   * importing document, with attribute values identical to the source node's
   * nodeName and nodeType, plus the attributes related to namespaces (prefix,
   * localName, and namespaceURI). As in the cloneNode operation on a Node,
   * the source node is not altered.
   * Additional information is copied as appropriate to the nodeType,
   * attempting to mirror the behavior expected if a fragment of XML or HTML
   * source was copied from one document to another, recognizing that the two
   * documents may have different DTDs in the XML case.
   * The following list describes the specifics for each type of node.
   *
   * - ATTRIBUTE_NODE<br>
   *	The ownerElement attribute is set to null and the specified
   *	flag is set to true on the generated Attr. The descendants of
   *	the source Attr are recursively imported and the resulting
   *	nodes reassembled to form the corresponding subtree.
   *	Note that the deep parameter has no effect on Attr nodes;
   *	they always carry their children with them when imported.
   * - DOCUMENT_FRAGMENT_NODE<br>
   *	If the deep option was set to true, the descendants of the
   *	source element are recursively imported and the resulting
   *	nodes reassembled to form the corresponding subtree.
   *	Otherwise, this simply generates an empty DocumentFragment.
   * - DOCUMENT_NODE<br>
   *	Document nodes cannot be imported.
   * - DOCUMENT_TYPE_NODE<br>
   *	DocumentType nodes cannot be imported.
   * - ELEMENT_NODE<br>
   *	Specified attribute nodes of the source element are imported,
   *	and the generated Attr nodes are attached to the generated
   *	Element. Default attributes are not copied, though if the
   *	document being imported into defines default attributes for
   *	this element name, those are assigned. If the importNode deep
   *	parameter was set to true, the descendants of the source
   *	element are recursively imported and the resulting nodes
   *	reassembled to form the corresponding subtree.
   * - ENTITY_NODE<br>
   *	Entity nodes can be imported, however in the current release
   *	of the DOM the DocumentType is readonly. Ability to add these
   *	imported nodes to a DocumentType will be considered for
   *	addition to a future release of the DOM.
   *	On import, the publicId, systemId, and notationName
   *	attributes are copied. If a deep import is requested, the
   *	descendants of the the source Entity are recursively imported
   *	and the resulting nodes reassembled to form the corresponding
   *	subtree.
   * - ENTITY_REFERENCE_NODE<br>
   *	Only the EntityReference itself is copied, even if a deep
   *	import is requested, since the source and destination
   *	documents might have defined the entity differently. If the
   *	document being imported into provides a definition for this
   *	entity name, its value is assigned.
   * - NOTATION_NODE<br>
   *	Notation nodes can be imported, however in the current
   *	release of the DOM the DocumentType is readonly. Ability to
   *	add these imported nodes to a DocumentType will be considered
   *	for addition to a future release of the DOM.
   *	On import, the publicId and systemId attributes are copied.
   *	Note that the deep parameter has no effect on Notation nodes
   *	since they never have any children.
   * - PROCESSING_INSTRUCTION_NODE<br>
   *	The imported node copies its target and data values from
   *	those of the source node.
   * - TEXT_NODE, CDATA_SECTION_NODE, COMMENT_NODE<br>
   *	These three types of nodes inheriting from CharacterData copy
   *	their data and length attributes from those of the source
   *	node.
   *
   * \param imported_node The node to import.
   * \param deep If true, recursively import the subtree under the specified
   *	node; if false, import only the node itself, as explained above.
   *	This has no effect on Attr, EntityReference, and Notation nodes.
   *
   * \return The imported node that belongs to this Document.
   *
   * \exception NOT_SUPPORTED_ERR
   *	Raised if the type of node being imported is not supported.
   */
  Node* importNode(CCC_IN Node* imported_node, bool deep) CCC_RAISES(DOMException);
   /*!
    * Introduced in DOM Level 2:
    *
    * Returns the Element whose ID is given by elementId. If no such element
    * exists, returns null. Behavior is not defined if more than one element
    * has this ID.
    *
    * Note: The DOM implementation must have information that says which
    * attributes are of type ID. Attributes with the name "ID" are not of type
    * ID unless so defined. Implementations that do not know whether attributes
    * are of type ID or not are expected to return null.
    *
    * \param elementId The unique id value for an element.
    *
    * \return The matching element.
    */
  Element* getElementById(CCC_IN DOMString* elementId);
  Element* getElementById(CCC_IN DOMString& elementId);
  /*!
   * (CODEX original)<br>
   * ノードを複製します。
   * \param deep deep copyするかどうかのフラグ
   * \return 複製されたノード
   */
  virtual Node* cloneNode(CCC_IN bool deep) const CCC_RAISES(DOMException);
 private:
  /*!
   * (CODEX original)<br>
   * Modified in DOM Level 2:
   * ノード名,局所名のアトムを管理するためのテーブルです。
   */
  AtomTable atom_table;
  /*!
   * (CODEX original)<br>
   * Introduced in DOM Level 2:
   * 名前空間URIのアトムを管理するためのテーブルです。
   */
  AtomTable namespace_uri_atom_table;
  /*!
   * (CODEX original)<br>
   * Introduced in DOM Level 2:
   * プレフィックスのアトムを管理するためのテーブルです。
   */
  AtomTable prefix_atom_table;
  /*!
   * XMLDecl information.
   */
  XMLDecl* xmldecl;
 public:
  /*!
   * (CODEX original)<br>
   * コンストラクタ
   * \param dom_implementation DOMImplementation
   */
  Document(DOMImplementation* dom_implementation);
  /*!
   * (CODEX original)<br>
   * コピーコンストラクタ
   * \param document コピーするDocument
   */
  Document(const Document& document);
  /*!
   * (CODEX original)<br>
   * デストラクタ
   */
  ~Document();
  /*!
   * (CODEX original)<br>
   * ノード名,局所名のアトムを管理するテーブルからアトム化した文字列を取り出します。
   * \param node_name ノード名もしくは局所名文字列
   * \return アトム化した文字列
   */
  DOMString* getAtom(const DOMString* node_name);
  DOMString* getAtom(const DOMString& node_name);
  DOMString* getAtom(const DOMChar* node_name);
  /*!
   * (CODEX original)<br>
   * 名前空間URIのアトムを管理するテーブルからアトム化した名前空間URI文字列を取り出します。
   * \param namespace_uri 名前空間URI文字列
   * \return アトム化した名前空間URI文字列
   */
  DOMString* getNamespaceUriAtom(const DOMString* namespace_uri);
  DOMString* getNamespaceUriAtom(const DOMString& namespace_uri);
  DOMString* getNamespaceUriAtom(const DOMChar* namespace_uri);
  /*!
   * (CODEX original)<br>
   * 名前空間URIのアトムを管理するテーブルからアトム化した名前空間URI文字列を取り出します。
   * 存在しない場合には0を返します。
   * \param namespace_uri 名前空間URI文字列
   * \return アトム化した名前空間URI文字列
   */
  DOMString* findNamespaceUriAtom(const DOMString* namespace_uri);
  /*!
   * (CODEX original)<br>
   * プレフィックスのアトムを管理するテーブルからアトム化した
   * プレフィックス文字列を取り出します。
   * \param prefix プレフィックス文字列
   * \return アトム化したプレフィックス文字列
   */
  DOMString* getPrefixAtom(const DOMString* prefix);
  DOMString* getPrefixAtom(const DOMString& prefix);
  DOMString* getPrefixAtom(const DOMChar* prefix);
  /*!
   * (CODEX original)<br>
   * 修飾名からアトム化したプリフィックスを得ます。
   * \param qualified_name 修飾名
   * \return アトム化したプリフィックス
   */
  DOMString* getPrefixAtomFromQualifiedName(CCC_IN const DOMString* qualified_name);
  DOMString* getPrefixAtomFromQualifiedName(CCC_IN const DOMString& qualified_name)
  {
    return getPrefixAtomFromQualifiedName(&qualified_name);
  }
  DOMString* getPrefixAtomFromQualifiedName(CCC_IN const DOMChar* qualified_name);
  /*!
   * (CODEX original)<br>
   * 修飾名からアトム化した局所名を得ます。
   * \param qualified_name 修飾名
   * \return アトム化した局所名
   */
  DOMString* getLocalNameFromQualifiedName(CCC_IN const DOMString* qualified_name);
  DOMString* getLocalNameFromQualifiedName(CCC_IN const DOMString& qualified_name)
  {
    return getLocalNameFromQualifiedName(&qualified_name);
  }
  DOMString* getLocalNameFromQualifiedName(CCC_IN const DOMChar* qualified_name);
  /* ! read XML
   * (CODEX original)<br>
   * An obsolete method.<br>
   * This method uses a MiniXMLParser class.
   */
  //static Document* readXML(InStream* is) CCC_RAISES(DOMException);
  /*!
   * (CODEX original)<br>
   * XMLをストリームに書き出します。
   * \param xo 出力先
   */
  virtual void writeXml(XMLOutputter* xo) const CCC_RAISES(IOException);
  /*!
   * (CODEX original)<br>
   * XMLをストリームに書き出します。
   * \param of 出力先
   */
  void writeXML(OFlow* of) const CCC_RAISES(DOMException);
  /*!
   * get document type
   */
  const DocumentType* getDocumentType() const { return document_type; }
  /*!
   * get nodemap of xmldecl
   */
  const NamedNodeMap* getXMLDeclNodeMap() const;
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_Document_h */
