﻿// $Id$

#ifndef INCLUDE_ccc_xml_Element_h
#define INCLUDE_ccc_xml_Element_h

#include <ccc/base/base.h>
#include <ccc/xml/config.h>

CCC_NAMESPACE_START(CCC);

/*!
 * The Element interface represents an element in an HTML or XML document.
 * Elements may have attributes associated with them; since the Element
 * interface inherits from Node, the generic Node interface attribute
 * attributes may be used to retrieve the set of all attributes for an
 * element. There are methods on the Element interface to retrieve either an
 * Attr object by name or an attribute value by name. In XML, where an
 * attribute value may contain entity references, an Attr object should be
 * retrieved to examine the possibly fairly complex sub-tree representing the
 * attribute value. On the other hand, in HTML, where all attributes have
 * simple string values, methods to directly access an attribute value can
 * safely be used as a convenience.
 *
 * Note: In DOM Level 2, the method normalize is inherited from the Node
 * interface where it was moved.
 */
class Element
  : public Node
{
 public:
  /*!
   * (CODEX original)<br>
   * コンストラクタ<br>
   * DOM Level 1のメソッドにより利用されます。
   * \param document Document
   * \param parent 親ノード
   * \param node_name ノード名
   */
  Element(Document* document, Node* parent, CCC_IN const DOMString* node_name);
  Element(Document* document, Node* parent, CCC_IN const DOMString& node_name);
  /*!
   * (CODEX original)<br>
   * コンストラクタ<br>
   * DOM Level 2のメソッドにより利用されます。
   * \param document Document
   * \param parent 親ノード
   * \param namespace_uri 名前空間URI
   * \param qualified_name 修飾名
   */
  Element(Document* document, Node* parent, CCC_IN const DOMString* namespace_uri, CCC_IN const DOMString* qualified_name);
  Element(Document* document, Node* parent, CCC_IN const DOMString& namespace_uri, CCC_IN const DOMString& qualified_name);
  /*!
   * (CODEX original)<br>
   * コピーコンストラクタ<br>
   * コピーコンストラクタはコピー元から子ノードや属性ノードなどのツリーはコピーしません。
   * これらをコピーする必要のある時には、コピー後にNode::deepCopy()のメソッドを呼び出してください。
   * \param element コピー元
   */
  Element(const Element& element);
  /*!
   * (CODEX original)<br>
   * デストラクタ
   */
  ~Element();
  /*!
   * The name of the element. For example, in:
   * \code
   * <elementExample id="demo">
   *            ...
   * </elementExample> ,
   * \endcode
   *
   * tagName has the value "elementExample". Note that this is
   * case-preserving in XML, as are all of the operations of the DOM.
   * The HTML DOM returns the tagName of an HTML element in the
   * canonical uppercase form, regardless of the case in the source
   * HTML document.
   */
  const DOMString* getTagName() const { return node_name; }
  /*!
   * Retrieves an attribute value by name.
   *
   * \param name The name of the attribute to retrieve.
   *
   * \return The Attr value as a string, or the empty string if that
   *	attribute does not have a specified or default value.
   */
  const DOMString* getAttribute(CCC_IN const DOMString* name) const;
  const DOMString* getAttribute(CCC_IN const DOMString& name) const;
  const DOMString* getAttribute(CCC_IN const DOMChar* name) const;
  /*!
   * Adds a new attribute. If an attribute with that name is already
   * present in the element, its value is changed to be that of the
   * value parameter. This value is a simple string; it is not parsed
   * as it is being set. So any markup (such as syntax to be recognized
   * as an entity reference) is treated as literal text, and needs to
   * be appropriately escaped by the implementation when it is written
   * out. In order to assign an attribute value that contains entity
   * references, the user must create an Attr node plus any Text and
   * EntityReference nodes, build the appropriate subtree, and use
   * setAttributeNode to assign it as the value of an attribute.
   * To set an attribute with a qualified name and namespace URI, use
   * the setAttributeNS method.
   *
   * \param name The name of the attribute to create or alter.
   * \param value Value to set in string form.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified name contains an illegal character.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   */
  void setAttribute(CCC_IN const DOMString* name, CCC_IN const DOMString* value) CCC_RAISES(DOMException);
  void setAttribute(CCC_IN const DOMString& name, CCC_IN const DOMString& value) CCC_RAISES(DOMException);
  void setAttribute(CCC_IN const DOMChar* name, CCC_IN const DOMChar* value) CCC_RAISES(DOMException);
  /*!
   * Removes an attribute by name. If the removed attribute is known to
   * have a default value, an attribute immediately appears containing
   * the default value as well as the corresponding namespace URI,
   * local name, and prefix when applicable.<br>
   * To remove an attribute by local name and namespace URI, use the
   * removeAttributeNS method.
   *
   * \param name The name of the attribute to remove.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   */
  void removeAttribute(CCC_IN const DOMString* name) CCC_RAISES(DOMException);
  void removeAttribute(CCC_IN const DOMString& name) CCC_RAISES(DOMException);
  /*!
   * Retrieves an attribute node by name.<br>
   * To retrieve an attribute node by qualified name and namespace URI,
   * use the getAttributeNodeNS method.
   *
   * \param name The name (nodeName) of the attribute to retrieve.
   *
   * \return The Attr node with the specified name (nodeName) or null if
   *	there is no such attribute.
   */
  Attr* getAttributeNode(CCC_IN const DOMString* name) const;
  Attr* getAttributeNode(CCC_IN const DOMString& name) const;
  /*!
   * Adds a new attribute node. If an attribute with that name
   * (nodeName) is already present in the element, it is replaced by
   * the new one.<br>
   * To add a new attribute node with a qualified name and namespace
   * URI, use the setAttributeNodeNS method.
   *
   * \param new_attr The Attr node to add to the attribute list.
   *
   * \return If the new_attr attribute replaces an existing attribute, the
   *	replaced Attr node is returned, otherwise null is returned.
   *
   * \exception WRONG_DOCUMENT_ERR Raised if new_attr was created from a
   *	different document than the one that created the element.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   * \exception INUSE_ATTRIBUTE_ERR Raised if new_attr is already an
   *	attribute of another Element object. The DOM user must
   *	explicitly clone Attr nodes to re-use them in other elements.
   */
  Attr* setAttributeNode(CCC_IN Attr* new_attr) CCC_RAISES(DOMException);
  /*!
   * Removes the specified attribute node. If the removed Attr has a
   * default value it is immediately replaced. The replacing attribute
   * has the same namespace URI and local name, as well as the original
   * prefix, when applicable.
   *
   * \param old_attr The Attr node to remove from the attribute list.
   *
   * \return The Attr node that was removed.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   * \exception NOT_FOUND_ERR Raised if oldAttr is not an attribute
   *	of the element.
   */
  Attr* removeAttributeNode(CCC_IN Attr* old_attr) CCC_RAISES(DOMException);
  /*!
   * Returns a NodeList of all descendant Elements with a given tag
   * name, in the order in which they are encountered in a preorder
   * traversal of this Element tree.
   *
   * \param name The name of the tag to match on.
   *	The special value "*" matches all tags.
   *
   * \return A list of matching Element nodes.
   */
  NodeList* getElementsByTagName(CCC_IN const DOMString* name) const;
  NodeList* getElementsByTagName(CCC_IN const DOMString& name) const
  {
    return getElementsByTagName(&name);
  }
  NodeList* getElementsByTagName(CCC_IN const DOMChar* name) const;
  /*!
   * Introduced in DOM Level 2:
   *
   * Returns a NodeList of all the descendant Elements with a given
   * local name and namespace URI in the order in which they are
   * encountered in a preorder traversal of this Element tree.
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param namespace_uri The namespace URI of the elements to match on.
   *	The special value "*" matches all namespaces.
   * \param local_name The local name of the elements to match on.
   *	The special value "*" matches all local names.
   *
   * \return A new NodeList object containing all the matched Elements.
   */
  NodeList* getElementsByTagNameNS(CCC_IN const DOMString* namespace_uri, CCC_IN const DOMString* local_name) const;
  NodeList* getElementsByTagNameNS(CCC_IN const DOMString& namespace_uri, CCC_IN const DOMString& local_name) const
  {
    return getElementsByTagNameNS(&namespace_uri, &local_name);
  }
  NodeList* getElementsByTagNameNS(CCC_IN const DOMChar* namespace_uri, CCC_IN const DOMChar* local_name) const;
  /*!
   * (CODEX original)<br>
   * ノードを複製します。
   * \param deep deep copyするかどうかのフラグ
   * \return 複製されたノード
   */
  virtual Node* cloneNode(CCC_IN bool deep) const CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:
   *
   * Retrieves an attribute value by local name and namespace URI.
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param namespace_uri The namespace URI of the attribute to retrieve.
   * \param local_name The local name of the attribute to retrieve.
   *
   * \return The Attr value as a string, or the empty string if that
   *	attribute does not have a specified or default value.
   */
  const DOMString* getAttributeNS(CCC_IN const DOMString* namespace_uri,
				  CCC_IN const DOMString* local_name);
  const DOMString* getAttributeNS(CCC_IN const DOMString& namespace_uri,
				  CCC_IN const DOMString& local_name)
  {
    return getAttributeNS(&namespace_uri, &local_name);
  }
  const DOMString* getAttributeNS(CCC_IN const DOMChar* namespace_uri,
				  CCC_IN const DOMChar* local_name);
  /*!
   * Introduced in DOM Level 2:
   *
   * Adds a new attribute. If an attribute with that local name and
   * that namespace URI is already present in the element, it is
   * replaced by the new one.<br>
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param new_attr The Attr node to add to the attribute list.
   *
   * \return If the newAttr attribute replaces an existing attribute with
   *	the same local name and namespace URI, the replaced Attr node
   *	is returned, otherwise null is returned.
   *
   * \exception WRONG_DOCUMENT_ERR Raised if newAttr was created from
   *	a different document than the one that created the element.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   * \exception INUSE_ATTRIBUTE_ERR Raised if newAttr is already an
   *	attribute of another Element object.  The DOM user must
   *	explicitly clone Attr nodes to re-use them in other elements.
   */
  Attr* setAttributeNodeNS(CCC_IN Attr* new_attr) CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:
   *
   * Adds a new attribute. If an attribute with the same local name and
   * namespace URI is already present on the element, its prefix is
   * changed to be the prefix part of the qualifiedName, and its value
   * is changed to be the value parameter. This value is a simple
   * string; it is not parsed as it is being set. So any markup (such
   * as syntax to be recognized as an entity reference) is treated as
   * literal text, and needs to be appropriately escaped by the
   * implementation when it is written out. In order to assign an
   * attribute value that contains entity references, the user must
   * create an Attr node plus any Text and EntityReference nodes, build
   * the appropriate subtree, and use setAttributeNodeNS or
   * setAttributeNode to assign it as the value of an attribute.<br>
   * HTML-only DOM implementations do not need to implement this
   * method.
   *
   * \param namespace_uri
   *	The namespace URI of the attribute to create or alter.
   * \param qualified_name
   *	The qualified name of the attribute to create or alter.
   * \param value The value to set in string form.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified qualified name contains an illegal character.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   * \exception NAMESPACE_ERR
   *	Raised if the qualifiedName is malformed, if the qualifiedName
   *	has a prefix and the namespaceURI is null, if the qualifiedName
   *	has a prefix that is "xml" and the namespaceURI is different
   *	from "http://www.w3.org/XML/1998/namespace", or if the
   *	qualifiedName is "xmlns" and the namespaceURI is different from
   *	"http://www.w3.org/2000/xmlns/".
   */
  void setAttributeNS(CCC_IN const DOMString* namespace_uri,
		      CCC_IN const DOMString* qualified_name,
		      CCC_IN const DOMString* value) CCC_RAISES(DOMException);
  void setAttributeNS(CCC_IN const DOMString& namespace_uri,
		      CCC_IN const DOMString& qualified_name,
		      CCC_IN const DOMString& value) CCC_RAISES(DOMException)
  {
    setAttributeNS(&namespace_uri, &qualified_name, &value);
  }
  /*!
   * Introduced in DOM Level 2:
   *
   * Removes an attribute by local name and namespace URI. If the
   * removed attribute has a default value it is immediately replaced.
   * The replacing attribute has the same namespace URI and local name,
   * as well as the original prefix.<br>
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param namespace_uri The namespace URI of the attribute to remove.
   * \param local_name The local name of the attribute to remove.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   */
  void removeAttributeNS(CCC_IN const DOMString* namespace_uri,
			 CCC_IN const DOMString* local_name) CCC_RAISES(DOMException);
  void removeAttributeNS(CCC_IN const DOMString& namespace_uri,
			 CCC_IN const DOMString& local_name) CCC_RAISES(DOMException)
  {
    removeAttributeNS(&namespace_uri, &local_name);
  }
  /*!
   * Introduced in DOM Level 2:
   *
   * Retrieves an Attr node by local name and namespace URI. HTML-only
   * DOM implementations do not need to implement this method.
   *
   * \param namespace_uri
   *	The namespace URI of the attribute to retrieve.
   * \param local_name
   *	The local name of the attribute to retrieve.
   *
   * \return The Attr node with the specified attribute local name and
   *	namespace URI or null if there is no such attribute.
   */
  Attr* getAttributeNodeNS(CCC_IN const DOMString* namespace_uri,
			   CCC_IN const DOMString* local_name) const;
  Attr* getAttributeNodeNS(CCC_IN const DOMString& namespace_uri,
			   CCC_IN const DOMString& local_name) const
  {
    return getAttributeNodeNS(&namespace_uri, &local_name);
  }
  Attr* getAttributeNodeNS(CCC_IN const DOMChar* namespace_uri,
			   CCC_IN const DOMChar* local_name) const;
  /*!
   * Introduced in DOM Level 2:
   *
   * Returns true when an attribute with a given name is specified on
   * this element or has a default value, false otherwise.
   *
   * \param name The name of the attribute to look for.
   *
   * \return True if an attribute with the given name is specified on
   *	this element or has a default value, false otherwise.
   */
  bool hasAttribute(CCC_IN const DOMString* name) const;
  bool hasAttribute(CCC_IN const DOMString& name) const
  {
    return hasAttribute(&name);
  }
  /*!
   * Introduced in DOM Level 2:
   *
   * Returns true when an attribute with a given local name and
   * namespace URI is specified on this element or has a default value,
   * false otherwise. HTML-only DOM implementations do not need to
   * implement this method.
   *
   * \param namespace_uri The namespace URI of the attribute to look for.
   * \param local_name The local name of the attribute to look for.
   *
   * \return True if an attribute with the given local name and
   *	namespace URI is specified or has a default value on this
   *	element, false otherwise.
   */
  bool hasAttributeNS(CCC_IN const DOMString* namespace_uri,
		      CCC_IN const DOMString* local_name) const;
  bool hasAttributeNS(CCC_IN const DOMString& namespace_uri,
		      CCC_IN const DOMString& local_name) const
  {
    return hasAttributeNS(&namespace_uri, &local_name);
  }
  /*!
   * (CODEX original)<br>
   * XMLをストリームに書き出します。
   * \param xo 出力先
   */
  virtual void writeXml(XMLOutputter* xo) const CCC_RAISES(IOException);
  /*!
   * (CODEX original)<br>
   * Textの子ノードの文字列値を取得します。複数のText子ノードが存在する場合には、
   * 合成文字列が返ります。孫ノード以下のテキスト文字列は無視されます。
   * 返された文字列オブジェクトは呼び出し側が破棄するようにしてください。
   * \return 合成文字列
   */
  DOMString* getChildTextValue() const;
   /*!
    * (CODEX original)<br>
    * Returns the Element whose ID is given by elementId. If no such element
    * exists, returns null. Behavior is not defined if more than one element
    * has this ID.
    *
    * Note: The DOM implementation must have information that says which
    * attributes are of type ID. Attributes with the name "ID" are not of type
    * ID unless so defined. Implementations that do not know whether attributes
    * are of type ID or not are expected to return null.
    *
    * \param elementId The unique id value for an element.
    *
    * \return The matching element.
    */
  Element* getElementById(CCC_IN DOMString* elementId) const;
  Element* getElementById(CCC_IN DOMString& elementId) const
  {
    return getElementById(& elementId);
  }
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_Element_h */
