﻿// $Id$
// mini XML parser

#ifndef INCLUDE_ccc_xml_MiniXMLParser_h
#define INCLUDE_ccc_xml_MiniXMLParser_h

#include <ccc/base/IFlow.h>
#include <ccc/base/PushbackIFilter.h>
#include <ccc/xml/config.h>
#include <ccc/xml/dom.h>

CCC_NAMESPACE_START(CCC);

/*!
 * \ja
 * 簡易版のXMLパーサの機能を提供するクラスです。<br>
 * XMLを読み込み元ストリームから読み込み、DOMを使ってXMLツリーを作成します。
 * XMLの厳密な定義に従ったパーサではありません。DTDの検証機能もありません。
 * 互換性維持のために用意されているクラスです。<br>
 * V2 CODEXではDocumentTypeの構造が大幅に変化したため、MiniXMLParserでは
 * DTD情報の読み込みが行えません。<br>
 * 厳密なパーサが必要な場合にはXMLParserを使用してください。
 * \ja_end
 */
class MiniXMLParser
{
  /*!
   * \ja
   * 読み込み元ストリーム
   * \ja_end
   */
  IFlow* in;
  /*!
   * \ja
   * ungetを実現するために、inを直接使わずに、PushbackIFilter
   * を使用して読み込みます。
   * \ja_end
   */
  PushbackIFilter<DOMChar>* reader;
  /*!
   * \ja
   * 構築中のドキュメント
   * \ja_end
   */
  Document* document;
 public:
  /*!
   * \ja
   * コンストラクタ
   * \ja_end
   * \en
   * Constructor
   * \en_end
   */
  MiniXMLParser();
  /*!
   * \ja
   * デストラクタ
   * \ja_end
   * \en
   * Destructor
   * \en_end
   */
  ~MiniXMLParser();
  /*!
   * \ja
   * XMLを読み込み元ストリームから読み込み、DOMを使ってXMLツリーを作成します。
   * \param dom_implementation DOMImplementation
   * \param in 読み込み元ストリーム
   * \return 読み込んだ結果生成されたDocument
   * \ja_end
   * \en
   * Read a XML from the input stream, and generate XML tree with DOM.
   * \param dom_implementation DOMImplementation
   * \param in input flow
   * \return A generated document.
   * \en_end
   */
  Document* read(DOMImplementation* dom_implementation, IFlow* in);
 private:
  /*!
   * \ja
   * 空白文字を読み込みます。
   * \ja_end
   * \en
   * Raed white spaces.
   * \en_end
   */
  void readSp() CCC_RAISES(IOException);
  /*!
   * \ja
   * 改行文字を読み込みます。
   * \ja_end
   * \en
   * Raed new lines.
   * \en_end
   */
  void readNl() CCC_RAISES(IOException);
  /*!
   * \ja
   * '<?xml .... ?>'を読み込みます。
   * \ja_end
   * \en
   * Read '<?xml .... ?>'
   * \en_end
   */
  void readS1() CCC_RAISES(IOException);
  /*!
   * \ja
   * '<?xml ... ?>'の属性を読み込みます。
   * \ja_end
   * \en
   * Read attributes of '<?xml ... ?>'
   * \en_end
   */
  void readS11() CCC_RAISES(IOException);
  /*!
   * \ja
   * '<!DOCTYPE decl [ markup_decl ] >'を読み込みます。
   * \ja_end
   * \en
   * Read '<!DOCTYPE decl [ markup_decl ] >'
   * \en_end
   */
  void readS2() CCC_RAISES(IOException);
  /*!
   * \ja
   * マークアップ宣言を読み込みます。
   * \ja_end
   * \en
   * Read markup_decl
   * \en_end
   */
  void readS21() CCC_RAISES(IOException);
  /*!
   * \ja
   * '<element> .... </element>'もしくは'<element/>'を読み込みます。
   * \param param 親ノード
   * \ja_end
   * \en
   * Read '<element> .... </element>' or '<element/>'
   * \param param parent node
   * \en_end
   */
  void readS3(Node* parent) CCC_RAISES(IOException);
  /*!
   * \ja
   * エレメントおん子を読み込みます。
   * \param param 親ノード
   * \ja_end
   * \en
   * Read children of an element.
   * \param param parent node
   * \en_end
   */
  void readS4(Node* parent) CCC_RAISES(IOException);
  /*!
   * \ja
   * '</element>'を読み込みます。
   * \ja_end
   * \en
   * Read '</element>'
   * \en_end
   */
  void readS5() CCC_RAISES(IOException);
  /*!
   * \ja
   * 指定した文字がXMLの空白文字かどうかを調べます。
   * \param c 文字
   * \return 結果
   * \ja_end
   * \en
   * Check whether a specified character is a XML white space or not.
   * \param c a character
   * \return result
   * \en_end
   */
  static bool whiteSpP(DOMChar c);
  /*!
   * \ja
   * 指定した文字列と終端の複数空白文字列を読み込みます。
   * \param str 指定文字列
   * \return 読み込めた場合に真、読み込めなかった場合に偽
   * \ja_end
   */
  bool readStr(DOMChar* str) CCC_RAISES(IOException);
  /*!
   * \ja
   * 改行文字かどうかを調べます。
   * \return 結果
   * \ja_end
   * \en
   * check new line
   * \return result
   * \en_end
   */
  static bool newLineP(DOMChar c);
  /*!
   * \ja
   * 空白文字が来るまで文字を読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \en
   * read until white space
   * \return read string
   * \en_end
   */
  DOMString* getString() CCC_RAISES(IOException);
  /*!
   * \ja
   * いずれかのデリミタ文字が来るまで文字を読み込みます。
   * \param デリミタ文字候補文字列
   * \return 読み込んだ文字列
   * \ja_end
   * \en
   * read until delimiter chars
   * \param delimiters candidates of the delimiters
   * \return read string
   * \en_end
   */
  DOMString* getString2(DOMChar* delimters) CCC_RAISES(IOException);
  /*!
   * \ja
   * デリミタ文字列が来るまで文字を読み込みます。
   * \param デリミタ文字列
   * \return 読み込んだ文字列
   * \ja_end
   * \en
   * Read until delimiter string.
   * \param delimiter chars
   * \return read string
   * \en_end
   */
  DOMString* getString3(DOMChar* delimter) CCC_RAISES(IOException);
  /*!
   * \ja
   * '<'の文字が来るまで文字を読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \en
   * get string until '<'
   * \return read string
   * \en_end
   */
  DOMString* getText() CCC_RAISES(IOException);
  /*!
   * \ja
   * ストリームから一字読み込みます。
   * \return 読み込んだ文字
   * \ja_end
   * \en
   * Read a character from the stream.
   * \return read character.
   * \en_end
   */
  DOMChar getChar();
  /*!
   * \ja
   * ストリームから実体参照形式を展開後の文字を読み込みます。
   * \param escaped_p 実体参照形式だった場合にはtrueがセットされます。
   * \return 読み込んだ文字
   * \ja_end
   * \en
   * Read characters which are entity reference expanded from the stream.
   * \param escaped_p if read character is an entity reference, true is set.
   * \return A read character.
   * \en_end
   */
  DOMChar getUnEscapedChar(bool& escaped_p);
  /*!
   * \ja
   * ストリームに一字戻します。
   * \param c 戻す文字
   * \ja_end
   * \en
   * Push back a one character to the stream.
   * \param c A push backed character.
   * \en_end
   */
  void ungetChar(DOMChar c);
  /*!
   * \ja
   * ストリームに文字列を戻します。
   * \param s 戻す文字列
   * \ja_end
   * \en
   * Push back the contents of the string to the stream.
   * \param s A push backed string.
   * \en_end
   */
  void ungetString(DOMString* s);
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_MiniXMLParser_h */
