﻿// $Id$

#ifndef INCLUDE_ccc_xml_NamedNodeMap_h
#define INCLUDE_ccc_xml_NamedNodeMap_h

#include <ccc/base/base.h>
#include <ccc/xml/config.h>

CCC_NAMESPACE_START(CCC);

/*!
 * Objects implementing the NamedNodeMap interface are used to represent
 * collections of nodes that can be accessed by name. Note that NamedNodeMap
 * does not inherit from NodeList; NamedNodeMaps are not maintained in any
 * particular order. Objects contained in an object implementing NamedNodeMap
 * may also be accessed by an ordinal index, but this is simply to allow
 * convenient enumeration of the contents of a NamedNodeMap, and does not
 * imply that the DOM specifies an order to these Nodes.
 *
 * NamedNodeMap objects in the DOM are live.
 */
class NamedNodeMap
#ifdef CCC_NUSE_AVL_NAMEDNODEMAP
  : private LinkDict<Node, DOMString>
#else /* CCC_NUSE_AVL_NAMEDNODEMAP */
  : private AVLDict<Node, DOMString>
#endif /* CCC_NUSE_AVL_NAMEDNODEMAP */
{
  /*!
   * NamedNodeMapが所属するノード
   */
  Node* owner_node;
 protected:
  /*!
   * 登録要素から文字列のキーを返します。
   * \param element ノード登録要素
   * \return 文字列のキー
   */
  virtual const DOMString* getKey(const Node* element) const;
  /*!
   * 2つのキーを比較します。
   * \param key1 キー1文字列
   * \param key2 キー2文字列
   * \retval 0 同一
   * \retval マイナス値 キー1の方が小さい
   * \retval プラス値 キー1の方が大きい
   */
  virtual int compare(const DOMString* key1, const DOMString* key2) const;
 public:
  /*!
   * (CODEX original)<br>
   * コンストラクタ
   * \param owner 所属ノード
   */
  NamedNodeMap(Node* owner);
  /*!
   * (CODEX original)<br>
   * デストラクタ<br>
   * デストラクタの呼び出しにより、このオブジェクトの管理下にある
   * Attrオブジェクトも破棄されます。
   */
  virtual ~NamedNodeMap();
  /*!
   * Retrieves a node specified by name.
   *
   * \param name The nodeName of a node to retrieve.
   *
   * \return A Node (of any type) with the specified nodeName,
   *	or null if it does not identify any node in this map.
   */
  Node* getNamedItem(CCC_IN const DOMString* name) const;
  Node* getNamedItem(CCC_IN const DOMString& name) const;
  Node* getNamedItem(CCC_IN const DOMChar* name) const;
  /*!
   * Adds a node using its nodeName attribute. If a node with that name
   * is already present in this map, it is replaced by the new one.
   * As the nodeName attribute is used to derive the name which the
   * node must be stored under, multiple nodes of certain types (those
   * that have a "special" string value) cannot be stored as the names
   * would clash. This is seen as preferable to allowing nodes to be
   * aliased.
   *
   * \param arg A node to store in this map. The node will later be
   *	accessible using the value of its nodeName attribute.
   *
   * \return If the new Node replaces an existing node the replaced Node
   *	is returned, otherwise null is returned.
   *
   * \exception WRONG_DOCUMENT_ERR Raised if arg was created from a
   *	different document than the one that created this map.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this map is readonly.
   * \exception INUSE_ATTRIBUTE_ERR Raised if arg is an Attr that is
   *	already an attribute of another Element object. The DOM user must
   *	explicitly clone Attr nodes to re-use them in other elements.
   */
  Node* setNamedItem(CCC_IN Node* arg) CCC_RAISES(DOMException);
  /*!
   * Removes a node specified by name. When this map contains the
   * attributes attached to an element, if the removed attribute is
   * known to have a default value, an attribute immediately appears
   * containing the default value as well as the corresponding
   * namespace URI, local name, and prefix when applicable.
   *
   * \param name The nodeName of the node to remove.
   *
   * \return The node removed from this map if a node with such a name
   *	exists.
   *
   * \exception NOT_FOUND_ERR
   *	Raised if there is no node named name in this map.
   * \exception NO_MODIFICATION_ALLOWED_ERR
   *	Raised if this map is readonly.
   */
  Node* removeNamedItem(CCC_IN const DOMString* name) CCC_RAISES(DOMException);
  Node* removeNamedItem(CCC_IN const DOMString& name) CCC_RAISES(DOMException)
  {
    return removeNamedItem(&name);
  }
  Node* removeNamedItem(CCC_IN const DOMChar* name) CCC_RAISES(DOMException);
  /*!
   * Returns the indexth item in the map. If index is greater than or
   * equal to the number of nodes in this map, this returns null.
   *
   * \param index Index into this map.
   *
   * \return The node at the indexth position in the map, or null if that
   *	is not a valid index.
   */
  Node* item(CCC_IN Size index) const;
  /*!
   * The number of nodes in this map. The range of valid child node
   * indices is 0 to length-1 inclusive.
   */
  Size getLength() const;
  /*!
   * Introduced in DOM Level 2:
   *
   * Retrieves a node specified by local name and namespace URI.
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param namespace_uri The namespace URI of the node to retrieve.
   * \param local_name The local name of the node to retrieve.
   *
   * \return A Node (of any type) with the specified local name and
   *	namespace URI, or null if they do not identify any node
   *	in this map.
   */
  Node* getNamedItemNS(CCC_IN const DOMString* namespace_uri,
		       CCC_IN const DOMString* local_name);
  Node* getNamedItemNS(CCC_IN const DOMString& namespace_uri,
		       CCC_IN const DOMString& local_name)
  {
    return getNamedItemNS(&namespace_uri, &local_name);
  }
  Node* getNamedItemNS(CCC_IN const DOMChar* namespace_uri,
		       CCC_IN const DOMChar* local_name);
  /*!
   * (CODEX original)
   */
  Node* getNamedItemNSWithAtom(const DOMString* namespace_uri_atom,
			       const DOMString* local_name_atom);
  /*!
   * Introduced in DOM Level 2:
   *
   * Adds a node using its namespaceURI and localName. If a node with
   * that namespace URI and that local name is already present in this
   * map, it is replaced by the new one.<br>
   * HTML-only DOM implementations do not need to implement this method.
   *
   * \param arg A node to store in this map. The node will later be
   *	accessible using the value of its namespaceURI and localName
   *	attributes.
   *
   * \return If the new Node replaces an existing node the replaced Node
   *	is returned, otherwise null is returned.
   *
   * \exception WRONG_DOCUMENT_ERR Raised if arg was created from a
   *	different document than the one that created this map.
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this map is readonly.
   * \exception INUSE_ATTRIBUTE_ERR Raised if arg is an Attr that is
   *	already an attribute of another Element object. The DOM user
   *	must explicitly clone Attr nodes to re-use them in other elements.
   */
  Node* setNamedItemNS(CCC_IN Node* arg) CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:
   *
   * Removes a node specified by local name and namespace URI. A
   * removed attribute may be known to have a default value when this
   * map contains the attributes attached to an element, as returned by
   * the attributes attribute of the Node interface. If so, an
   * attribute immediately appears containing the default value as well
   * as the corresponding namespace URI, local name, and prefix when
   * applicable.<br>
   * HTML-only DOM implementations do not need to implement this
   * method.
   *
   * \param namespace_uri The namespace URI of the node to remove.
   * \param local_name The local name of the node to remove.
   *
   * \return The node removed from this map if a node with such a local
   *	name and namespace URI exists.
   *
   * \exception NOT_FOUND_ERR Raised if there is no node with
   *	the specified namespaceURI and localName in this map.
   * \exception NO_MODIFICATION_ALLOWED_ERR
   *	Raised if this map is readonly.
   */
  Node* removeNamedItemNS(CCC_IN const DOMString* namespace_uri,
			  CCC_IN const DOMString* local_name) CCC_RAISES(DOMException);
  Node* removeNamedItemNS(CCC_IN const DOMString& namespace_uri,
			  CCC_IN const DOMString& local_name) CCC_RAISES(DOMException)
  {
    return removeNamedItemNS(&namespace_uri, &local_name);
  }
  /*!
   * (CODEX original)<br>
   * 指定したノードを辞書から外します。
   * \param node 外すノード
   * \return 外されたノード
   */
  Node* removeNode(Node* node);
#ifdef CCC_NUSE_AVL_NAMEDNODEMAP
  using LinkDict<Node, DOMString>::createIterator;
  using LinkDict<Node, DOMString>::emptyP;
  using LinkDict<Node, DOMString>::clear;
#else /* CCC_NUSE_AVL_NAMEDNODEMAP */
  using AVLDict<Node, DOMString>::createIterator;
  using AVLDict<Node, DOMString>::emptyP;
  using AVLDict<Node, DOMString>::clear;
#endif /* CCC_NUSE_AVL_NAMEDNODEMAP */
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_NamedNodeMap_h */
