﻿// $Id$

#ifndef INCLUDE_ccc_xml_Node_h
#define INCLUDE_ccc_xml_Node_h

#include <ccc/base/base.h>
#include <ccc/xml/config.h>

CCC_NAMESPACE_START(CCC);

/*!
 * The Node interface is the primary datatype for the entire Document Object
 * Model. It represents a single node in the document tree. While all objects
 * implementing the Node interface expose methods for dealing with children,
 * not all objects implementing the Node interface may have children.
 * For example, Text nodes may not have children, and adding children to such
 * nodes results in a DOMException being raised.
 *
 * The attributes nodeName, nodeValue and attributes are included as a
 * mechanism to get at node information without casting down to the specific
 * derived interface. In cases where there is no obvious mapping of these
 * attributes for a specific nodeType (e.g., nodeValue for an Element or
 * attributes for a Comment), this returns null. Note that the specialized
 * interfaces may contain additional and more convenient mechanisms to get
 * and set the relevant information.
 *
 * - About NodeType (node_type)<br>
 * <br>
 * An integer indicating which type of node this is.<br>
 * <br>
 * Note: Numeric codes up to 200 are reserved to W3C for possible future
 * use.<br>
 * <br>
 * The values of nodeName, nodeValue, and attributes vary according to
 * the node type as follows:
 * <br>
 * <table>
 * <tr><td align="center">Interface	<td align="center">nodeName
 *	<td align="center">nodeValue		<td align="center">attributes
 * <tr><td>Attr				<td>name of attribute
 *	<td>value of attribute			<td>null
 * <tr><td>CDATASection			<td>\#cdata-section
 *	<td>content of the CDATA Section	<td>null
 * <tr><td>Comment			<td>\#comment
 *	<td>content of the comment		<td>null
 * <tr><td>Document			<td>\#document
 *	<td>null				<td>null
 * <tr><td>DocumentFragment		<td>\#document-fragment
 *	<td>null				<td>null
 * <tr><td>DocumentType			<td>document type name
 *	<td>null				<td>null
 * <tr><td>Element			<td>tag name
 *	<td>null				<td>NamedNodeMap
 * <tr><td>Entity			<td>entity name
 *	<td>null				<td>null
 * <tr><td>EntityReference		<td>name of entity referenced
 *	<td>null				<td>null
 * <tr><td>Notation			<td>notation name
 *	<td>null				<td>null
 * <tr><td>ProcessingInstruction	<td>target
 *	<td>entire content excluding the target	<td>null
 * <tr><td>Text				<td>\#text
 *	<td>content of the text node		<td>null
 * </table>
 */
class Node
  : public NDLinkListNode<Node>
{
  friend class XMLParser;
 public:
  /*
   * About NodeType
   * 200までの数値コードが潜在的な将来の使用のためにW3Cに対して予約
   * されています。
   * (CODEX original)
   * CODEXでは1000番以降の数値を使います。
   */
#ifndef DONT_SUPPORT_CONST_VALUE_MEMBER
  //! The node is an Element.
  static const unsigned short ELEMENT_NODE                   = 1;
  //! The node is an Attr.
  static const unsigned short ATTRIBUTE_NODE                 = 2;
  //! The node is a Text node.
  static const unsigned short TEXT_NODE                      = 3;
  //! The node is a CDATASection.
  static const unsigned short CDATA_SECTION_NODE             = 4;
  //! The node is an EntityReference.
  static const unsigned short ENTITY_REFERENCE_NODE          = 5;
  //! The node is an Entity.
  static const unsigned short ENTITY_NODE                    = 6;
  //! The node is a ProcessingInstruction.
  static const unsigned short PROCESSING_INSTRUCTION_NODE    = 7;
  //! The node is a Comment.
  static const unsigned short COMMENT_NODE                   = 8;
  //! The node is a Document.
  static const unsigned short DOCUMENT_NODE                  = 9;
  //! The node is a DocumentType.
  static const unsigned short DOCUMENT_TYPE_NODE             = 10;
  //! The node is a DocumentFragment.
  static const unsigned short DOCUMENT_FRAGMENT_NODE         = 11;
  //! The node is a Notation.
  static const unsigned short NOTATION_NODE                  = 12;
  //! (CODEX original)
  static const unsigned short XMLDECL_NODE                   = 1000;
  //! (CODEX original)
  static const unsigned short DTD_ELEMENT_NODE               = 1001;
  //! (CODEX original)
  static const unsigned short DTD_ATTLIST_NODE               = 1002;
  //! (CODEX original)
  static const unsigned short DTD_PENTITY_NODE               = 1003;
  //! (CODEX original)
  static const unsigned short DTD_ENTITY_NODE                = ENTITY_NODE;
  //! (CODEX original)
  static const unsigned short DTD_NOTATION_NODE              = NOTATION_NODE;
#else /* DONT_SUPPORT_CONST_VALUE_MEMBER */
  /*
   * static const定数メンバをサポートしていないVC6などの処理系用のenumです。
   * 各enum値は上記の定義に従います。
   */
  enum
  {
    ELEMENT_NODE                   = 1,
    ATTRIBUTE_NODE                 = 2,
    TEXT_NODE                      = 3,
    CDATA_SECTION_NODE             = 4,
    ENTITY_REFERENCE_NODE          = 5,
    ENTITY_NODE                    = 6,
    PROCESSING_INSTRUCTION_NODE    = 7,
    COMMENT_NODE                   = 8,
    DOCUMENT_NODE                  = 9,
    DOCUMENT_TYPE_NODE             = 10,
    DOCUMENT_FRAGMENT_NODE         = 11,
    NOTATION_NODE                  = 12,
    // (CODEX original)
    XMLDECL_NODE                   = 1000,
    DTD_ELEMENT_NODE               = 1001,
    DTD_ATTLIST_NODE               = 1002,
    DTD_PENTITY_NODE               = 1003,
    DTD_ENTITY_NODE                = ENTITY_NODE,
    DTD_NOTATION_NODE              = NOTATION_NODE,
  };
#endif /* DONT_SUPPORT_CONST_VALUE_MEMBER */
 private:
  /*!
   * Modified in DOM Level 2:<br>
   * The Document object associated with this node. This is also the Document
   * object used to create new nodes. When this node is a Document or
   * a DocumentType which is not used with any Document yet, this is null.
   */
  Document* document;
  /*!
   * The parent of this node. All nodes, except Attr, Document,
   * DocumentFragment, Entity, and Notation may have a parent.  However,
   * if a node has just been created and not yet added to the tree, or
   * if it has been removed from the tree, this is null.
   */
  Node* parent;
  /*!
   * A code representing the type of the underlying object, as defined above.
   */
  unsigned short node_type;
  /*!
   * A NodeList that contains all children of this node. If there are no
   * children, this is a NodeList containing no nodes.
   */
  NNodeList children;
  /*!
   * A NamedNodeMap containing the attributes of this node (if it is an
   * Element) or null otherwise.
   */
  NamedNodeMap* attributes;
#ifndef CODEX_PURE_DOM
  /*!
   * (CODEX original)
   * BufferPointerのリスト
   */
  BPList bp_list;
  /*!
   * (CODEX original)
   * ノードの追加要素
   */
  NodeExtension* node_extention;
  /*!
   * 子要素が空の要素を出力する際に<A></A>のように冗長表現で
   * 出力します。
   */
  bool redundant_empty_tag_expression_p;
#endif /* CODEX_PURE_DOM */
 protected:
  /*!
   * The name of this node, depending on its type; see the table above.
   */
  DOMString* node_name;
  /*!
   * Introduced in DOM Level 2:
   * The namespace prefix of this node, or null if it is unspecified.
   */
  DOMString* prefix;
  /*!
   * Introduced in DOM Level 2:<br>
   * the local part of the qualified name of this node.
   */
  DOMString* local_name;
  /*!
   * Introduced in DOM Level 2:<br>
   * The namespace URI of this node, or null if it is unspecified.
   */
  DOMString* namespace_uri;
  /*!
   * The value of this node, depending on its type; see the table above.
   * When it is defined to be null, setting it has no effect.
   */
  DOMString node_value;
  /*!
   * (CODEX original)
   */
  static DOMString* null_string;
 public:
  /*!
   * for initialize class
   * \param p true:initialize, false:uninitialize
   */
  static void initialize(bool p);
  /*!
   * get node type.
   * \return node type.
   */
  virtual unsigned short getNodeType() const { return node_type; }
  /*!
   * get parent node.
   * \return parent node.
   */
  virtual Node* getParentNode() const { return parent; }
  /*!
   * set parent node.
   * \param new_parent new parent node.
   */
  void setParentNode(Node* new_parent) { parent = new_parent; }
  /*!
   * get node name.
   * \return node name.
   */
  virtual const DOMString* getNodeName() const { return node_name; }
  /*
   * get node value.
   * \return node value.
   */
  virtual const DOMString* getNodeValue() const CCC_RAISES(DOMException);
  /*!
   * set node value
   * \param new_value string
   */
  virtual void setNodeValue(const DOMString* new_value) CCC_RAISES(DOMException);
  virtual void setNodeValue(const DOMString& new_value) CCC_RAISES(DOMException);
  virtual void setNodeValue(const DOMChar* new_value) CCC_RAISES(DOMException);
  /*!
   * set node value
   * \param new_value string
   * \param length length of string 
   */
  virtual void setNodeValue(const DOMChar* new_value, Size length) CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:
   *
   * The namespace prefix of this node, or null if it is unspecified.
   * Note that setting this attribute, when permitted, changes the
   * nodeName attribute, which holds the qualified name, as well as the
   * tagName and name attributes of the Element and Attr interfaces, when
   *	applicable.
   * Note also that changing the prefix of an attribute that is known to
   * have a default value, does not make a new attribute with the default
   * value and the original prefix appear, since the namespaceURI and
   * localName do not change.
   * For nodes of any type other than ELEMENT_NODE and ATTRIBUTE_NODE and
   * nodes created with a DOM Level 1 method, such as createElement from
   * the Document interface, this is always null.
   *
   * \exception INVALID_CHARACTER_ERR
   *	Raised if the specified prefix contains an illegal character.
   * \exception NO_MODIFICATION_ALLOWED_ERR
   *	Raised if this node is readonly.
   * \exception NAMESPACE_ERR
   *	Raised if the specified prefix is malformed, if the namespaceURI of
   *	this node is null, if the specified prefix is "xml" and the
   *	namespaceURI of this node is different from
   *	"http://www.w3.org/XML/1998/namespace", if this node is an attribute
   *	and the specified prefix is "xmlns" and the namespaceURI of this node
   *	is different from "http://www.w3.org/2000/xmlns/", or if this node is
   *	an attribute and the qualifiedName of this node is "xmlns"
   *	[Namespaces].
   */
  virtual const DOMString* getPrefix() const { return prefix; }
  /*!
   * Introduced in DOM Level 2:
   * Set the namespace prefix of this node.
   * \param new_prefix the namespace prefix of this node.
   */
  virtual void setPrefix(const DOMString* new_prefix) CCC_RAISES(DOMException);  
  virtual void setPrefix(const DOMString& new_prefix) CCC_RAISES(DOMException);  
  virtual void setPrefix(const DOMChar* new_prefix) CCC_RAISES(DOMException);
  /*!
   * Introduced in DOM Level 2:<br>
   * Returns the local part of the qualified name of this node.
   * For nodes of any type other than ELEMENT_NODE and ATTRIBUTE_NODE
   * and nodes created with a DOM Level 1 method, such as createElement
   * from the Document interface, this is always null.
   */
  virtual const DOMString* getLocalName() const { return local_name; }
  /*!
   * Introduced in DOM Level 2:<br>
   * The namespace URI of this node, or null if it is unspecified.
   * This is not a computed value that is the result of a namespace
   * lookup based on an examination of the namespace declarations in
   * scope. It is merely the namespace URI given at creation time.
   * For nodes of any type other than ELEMENT_NODE and ATTRIBUTE_NODE
   * and nodes created with a DOM Level 1 method, such as createElement
   * from the Document interface, this is always null.
   *
   * Note: Per the Namespaces in XML Specification [Namespaces] an
   * attribute does not inherit its namespace from the element it is
   * attached to. If an attribute is not explicitly given a namespace,
   * it simply has no namespace.
   */
  virtual const DOMString* getNamespaceURI() const { return namespace_uri; }
  /*!
   * get ChildNode.
   */
  virtual NodeList* getChildNodes() const { return (NodeList*)&children; }
  /*!
   * The first child of this node. If there is no such node, this
   * returns null.
   */
  virtual Node* getFirstChild() const;
  /*!
   * The last child of this node. If there is no such node, this
   * returns null.
   */
  virtual Node* getLastChild() const;
  /*!
   * The node immediately preceding this node. If there is no such
   * node, this returns null.
   */
  virtual Node* getPreviousSibling() const;
  /*!
   * The node immediately following this node. If there is no such
   * node, this returns null.
   */
  virtual Node* getNextSibling() const;
  /*!
   * get attributes.
   * \return NamedNodeMap.
   */
  virtual NamedNodeMap* getAttributes() const { return attributes; }
  /*!
   * Introduced in DOM Level 2:<br>
   * get owner document.
   * \return owner document.
   */
  virtual Document* getOwnerDocument() const { return document; }
  /*!
   * Inserts the node newChild before the existing child node refChild.
   * If refChild is null, insert newChild at the end of the list of
   * children.
   * If newChild is a DocumentFragment object, all of its children are
   * inserted, in the same order, before refChild. If the newChild is
   * already in the tree, it is first removed.
   *
   * \param newChild The node to insert.
   * \param refChild
   *	The reference node, i.e., the node before which the new node
   *	must be inserted.
   *
   * \return The node being inserted.
   *
   * \exception HIERARCHY_REQUEST_ERR
   *	Raised if this node is of a type that does not allow children of
   *	the type of the newChild node, or if the node to insert is one
   *	of this node's ancestors.
   * \exception WRONG_DOCUMENT_ERR
   *	Raised if newChild was created from a different document than
   *	the one that created this node.
   * \exception NO_MODIFICATION_ALLOWED_ERR
   *	Raised if this node is readonly or if the parent of the node
   *	being inserted is readonly.
   * \exception NOT_FOUND_ERR
   *	Raised if refChild is not a child of this node.
   */
  virtual Node* insertBefore(CCC_IN Node* newChild, CCC_IN Node* refChild) CCC_RAISES(DOMException);
  /*!
   * (CODEX original)
   * Inserts the node newChild after the existing child node refChild. 
   * \param newChild The node to insert.
   * \param refChild
   *	The reference node, i.e., the node after which the new node
   *	must be inserted.
   *
   * \return The node being inserted.
   */
  virtual Node* insertAfter(CCC_IN Node* newChild, CCC_IN Node* refChild) CCC_RAISES(DOMException);
  /*!
   * Replaces the child node oldChild with newChild in the list of
   * children, and returns the oldChild node.
   * If newChild is a DocumentFragment object, oldChild is replaced by
   * all of the DocumentFragment children, which are inserted in the
   * same order. If the newChild is already in the tree, it is first
   * removed.
   *
   * \param newChild The new node to put in the child list.
   * \param oldChild The node being replaced in the list.
   *
   * \return The node replaced.
   *
   * \exception HIERARCHY_REQUEST_ERR
   *	Raised if this node is of a type that does not allow children of
   *	the type of the newChild node, or if the node to put in is one
   *	of this node's ancestors.
   * \exception WRONG_DOCUMENT_ERR
   *	Raised if newChild was created from a different document than
   *	the one that created this node.
   * \exception NO_MODIFICATION_ALLOWED_ERR
   *	Raised if this node or the parent of the new node is readonly.
   * \exception NOT_FOUND_ERR
   *	Raised if oldChild is not a child of this node.
   */
  virtual Node* replaceChild(CCC_IN Node* newChild, CCC_IN Node* oldChild) CCC_RAISES(DOMException);
  /*!
   * Removes the child node indicated by oldChild from the list of
   * children, and returns it.
   *
   * \param oldChild The node being removed.
   *
   * \return The node removed.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR Raised if this node is readonly.
   * \exception NOT_FOUND_ERR Raised if oldChild is not a child of this node.
   */
  virtual Node* removeChild(CCC_IN Node* oldChild) CCC_RAISES(DOMException);
  /*!
   * Adds the node newChild to the end of the list of children of this
   * node. If the newChild is already in the tree, it is first removed.
   *
   * \param newChild
   *	The node to add.<br>
   * 	If it is a DocumentFragment object, the entire contents of the
   * 	document fragment are moved into the child list of this node
   *
   * \return The node added.
   *
   * \exception HIERARCHY_REQUEST_ERR
   *	Raised if this node is of a type that does not allow children
   *	of the type of the newChild node, or if the node to append is
   *	one of this node's ancestors.
   *
   * \exception WRONG_DOCUMENT_ERR
   *	Raised if newChild was created from a different document
   *	than the one that created this node.
   *
   * \exception NO_MODIFICATION_ALLOWED_ERR
   *	Raised if this node is readonly.
   */
  virtual Node* appendChild(CCC_IN Node* newChild) CCC_RAISES(DOMException);
  /*!
   * Returns whether this node has any children.
   *
   * \return True if this node has any children, false otherwise.
   */
  virtual bool hasChildNodes() const;
  /*!
   * Returns a duplicate of this node, i.e., serves as a generic copy
   * constructor for nodes. The duplicate node has no parent;
   * (parentNode is null.).
   * Cloning an Element copies all attributes and their values,
   * including those generated by the XML processor to represent
   * defaulted attributes, but this method does not copy any text it
   * contains unless it is a deep clone, since the text is contained in
   * a child Text node. Cloning an Attribute directly, as opposed to be
   * cloned as part of an Element cloning operation, returns a
   * specified attribute (specified is true). Cloning any other type of
   * node simply returns a copy of this node.
   * Note that cloning an immutable subtree results in a mutable copy,
   * but the children of an EntityReference clone are readonly. In
   * addition, clones of unspecified Attr nodes are specified. And,
   * cloning Document, DocumentType, Entity, and Notation nodes is
   * implementation dependent.
   *
   * \param deep
   *	If true, recursively clone the subtree under the specified
   *	node; if false, clone only the node itself (and its
   *	attributes, if it is an Element).
   *
   * \return The duplicate node.
   */
  virtual Node* cloneNode(CCC_IN bool deep) const CCC_RAISES(DOMException) = 0;
  /*!
   * Modified in DOM Level 2:<br>
   * Puts all Text nodes in the full depth of the sub-tree underneath
   * this Node, including attribute nodes, into a "normal" form where
   * only structure (e.g., elements, comments, processing instructions,
   * CDATA sections, and entity references) separates Text nodes, i.e.,
   * there are neither adjacent Text nodes nor empty Text nodes. This
   * can be used to ensure that the DOM view of a document is the same
   * as if it were saved and re-loaded, and is useful when operations
   * (such as XPointer [XPointer] lookups) that depend on a particular
   * document tree structure are to be used.
   *
   * Note: In cases where the document contains CDATASections, the
   * normalize operation alone may not be sufficient, since XPointers
   * do not differentiate between Text nodes and CDATASection nodes.
   */
  void normalize();
  /*!
   * Introduced in DOM Level 2:<br>
   * Returns whether this node (if it is an element) has any attributes.
   * \return True if this node has any attributes, false otherwise.
   */
  bool hasAttributes();
  /*!
   * Introduced in DOM Level 2:<br>
   * Tests whether the DOM implementation implements a specific feature
   * and that feature is supported by this node.
   *
   * \param feature
   *	The name of the feature to test. This is the same name which
   *	can be passed to the method hasFeature on DOMImplementation.
   * \param version
   *	This is the version number of the feature to test. In Level 2,
   *	version 1, this is the string "2.0". If the version is not
   *	specified, supporting any version of the feature will cause
   *	the method to return true.
   *
   * \return Returns true if the specified feature is supported on this
   *	node, false otherwise.
   */
  bool isSupported(CCC_IN DOMString& feature, CCC_IN DOMString& version);
 protected:
  /*!
   * (CODEX original)<br>
   * コンストラクタ
   * \param document Document
   * \param parent 親ノード
   * \param node_type 種類
   */
  Node(Document* document, Node* parent, unsigned short node_type);
  /*!
   * (CODEX original)<br>
   * コピーコンストラクタ<br>
   * コピーコンストラクタはコピー元から子ノードや属性ノードなどのツリーはコピーしません。
   * \param node コピーするノード
   */
  Node(const Node& node);
  /*!
   * (CODEX original)<br>
   * コンストラクタ
   * \param node_type ノード形式
   */
  Node(unsigned short node_type);
  /*
   * (CODEX original)<br>
   * ノードの種類をセットします。
   * \param t ノードの種類
   */
  void setNodeType(unsigned short t) { node_type = t; }
 public:
  /*!
   * (CODEX original)<br>
   * ノード名を設定します。
   * プレフィックス,局所名,名前空間URIはDOM Level 2の要請より0に設定されます。
   * DOMではノード名はリードオンリーと定義されています。
   * このメソッドは継承されたクラスのDOM Level 1のメソッドから呼び出される
   * ことを前提としています。
   * \param new_node_name 新しいノード名
   */
  void setNodeName(const DOMString* new_node_name);
  void setNodeName(const DOMString& new_node_name);
  void setNodeName(const DOMChar* new_node_name);
  /*!
   * (CODEX original)<br>
   * 名前空間URIと修飾名、ノード名を設定します。
   * このメソッドは継承されたクラスのDOM Level 2のメソッドやXMLPerserから
   * 呼び出されることを前提としています。
   * \param namespace_uri 名前空間URI
   * \param qualified_name 修飾名
   */
  void setNodeNameNS(CCC_IN const DOMString* namespace_uri, CCC_IN const DOMString* qualified_name);
  void setNodeNameNS(CCC_IN const DOMString& namespace_uri, CCC_IN const DOMString& qualified_name);
  void setNodeNameNS(CCC_IN const DOMChar* namespace_uri, CCC_IN const DOMChar* qualified_name);
  /*!
   * (CODEX original)<br>
   * デストラクタ
   */
  virtual ~Node();
  /*!
   * (CODEX original)<br>
   * 属性ノードと子孫ノードを'to_node'に複製します。
   * \param to_node コピー先のノード
   */
  void deepCopy(Node* to_node) const;
  /*!
   * (CODEX original)<br>
   * XMLをストリームに書き出します。
   * \param xo 出力先
   */
  virtual void writeXml(XMLOutputter* xo) const CCC_RAISES(IOException);
  /*!
   * (CODEX original)<br>
   * 引数で指定したアトムのノード名を持つ祖先のノードを取得します。
   * \param atom_node_name アトム化したノード名
   * \return 見つかった祖先のノード、0が返った場合は該当するノード
   * が存在しなかったことを示します。
   */
  Node* getAncestor(const DOMString* atom_node_name);
#ifndef CODEX_PURE_DOM
  /*!
   * (CODEX original)<br>
   * このノードに登録されているBufferPointerに対してclear()
   * メソッドを呼び出します。
   */
  void clearBps();
  /*!
   * (CODEX original)<br>
   * BufferPointerをノードに登録します。
   * \param bp 登録するBufferPointer
   */
  void addBp(BufferPointer* bp);
  /*!
   * (CODEX original)<br>
   * BufferPointerをノードから登録解除します。
   * \param bp 登録解除するBufferPointer
   */
  void removeBp(BufferPointer* bp);
  /*!
   * (CODEX original)<br>
   * 引数で指定されたBufferPointerのリストにあるBufferPointer
   * をこのノードから登録解除します。
   */
  void removeBps(BPList* bps);
  /*!
   * (CODEX original)<br>
   * BufferPointerのリストを取得します。
   * \return BufferPointerのリスト
   */
  BPList* getBPList() { return &bp_list; }
  /*!
   * (CODEX original)
   * ノードの追加要素を設定します。
   * Nodeの破棄時に追加要素のデストラクタが呼ばれます。
   * \param ext 追加要素
   */
  void setNodeExtension(NodeExtension* ext) { node_extention = ext; }
  /*!
   * (CODEX original)
   * ノードの追加要素を得ます。
   * \return ノードの追加要素
   */
  NodeExtension* getNodeExtension() { return node_extention; }
  /*!
   * 子要素が空の要素を出力する際に<A></A>のように冗長表現で
   * 出力するかどうかを取得します。
   * \retval true 冗長表現で出力する
   * \retval false 冗長表現で出力しない
   */
  bool getRedundantEmptyTagExpressionP() const { return redundant_empty_tag_expression_p; }
  /*!
   * 子要素が空の要素を出力する際に<A></A>のように冗長表現で
   * 出力するかどうかを指定します。
   * 注意:
   * この指定はXMLOutputterクラスで、USE_REDUNDANT_EMPTY_TAG_EXPRESSION_CONTROL
   * オプションを指定した時に限り有効になります。
   * \param p 冗長表現で出力するかどうか
   */
  void setRedundantEmptyTagExpressionP(bool p) { redundant_empty_tag_expression_p = p; }
#endif /* CODEX_PURE_DOM */
  /*!
   * (CODEX original)<br>
   * 指定したタグ名の最初の子エレメントを取得します。
   * \param name タグ名
   * \return 見つかったエレメント、見つからない場合には0が返ります。
   */
  Element* getFirstChildElementByTagName(const DOMString* name) const;
  Element* getFirstChildElementByTagName(const DOMString& name) const
  {
    return getFirstChildElementByTagName(&name);
  }
  Element* getFirstChildElementByTagName(const DOMChar* name) const;
  /*!
   * (CODEX original)<br>
   * 指定した名前空間とローカル名の最初の子エレメントを取得します。
   * \param namespace_uri_ 名前空間
   * \param local_name ローカル名
   * \return 見つかったエレメント、見つからない場合には0が返ります。
   */
  Element* getFirstChildElementByTagNameNS(const DOMString* namespace_uri_, const DOMString* local_name) const;
  Element* getFirstChildElementByTagNameNS(const DOMString& namespace_uri_, const DOMString& local_name) const
  {
    return getFirstChildElementByTagNameNS(&namespace_uri_, &local_name);
  }
  
  /*!
   * (CODEX original)<br>
   * 指定した名前空間とローカル名の最初の子エレメントを取得します。
   * \param namespace_uri_ 名前空間
   * \param local_name ローカル名
   * \return 見つかったエレメント、見つからない場合には0が返ります。
   */
  Element* getFirstChildElementByTagNameNS(const DOMChar* namespace_uri_, const DOMChar* local_name) const;
  /*!
   * (CODEX original)<br>
   * 子ノードから空や改行のみなどのテキストノードを削除します。
   */
  void removeStealthTextNode();
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_Node_h */
