﻿// $Id$
/*!
 * \file
 * XML Parser
 */

#ifndef INCLUDE_CCC_XML_XMLPARSER_H
#define INCLUDE_CCC_XML_XMLPARSER_H

#include <ccc/base/LinkList.h>
#include <ccc/base/TextWriter.h>
#include <ccc/file/PathString.h>
#include <ccc/xml/dom.h>
#include <ccc/xml/Catalog.h>
#include <ccc/xml/XMLReader.h>
#include <ccc/xml/XMLParserSrc.h>
#include <ccc/xml/XmlCharClass.h>
#include <ccc/xml/XMLParseError.h>
#include <ccc/xml/NamespaceUnit.h>

CCC_NAMESPACE_START(CCC);

/*!
 * \ja
 * \brief XML LL(1)パーサクラス 
 * \ja_end
 * \en
 * \brief XML LL(1) Parser class
 * \en_end
 * \nosubgrouping
 */
class XMLParser
{
 public:
  /*!
   * \ja
   * \brief XMLパーサのオプション
   *
   * オプションにはこれらのenum値をORした値を指定します。
   * \ja_end
   */
  enum Option
  {
    /*!
     * \ja
     * オプション無し
     * \ja_end
     */
    NONE = 0,
    /*!
     * \ja
     * 妥当性をチェックします。
     * \ja_end
     * \en
     * check validity
     * \en_end
     */
    VALIDITY_CHECK = 0x01,
    /*!
     * \ja
     * 名前空間をサポートしないDOM Level 1の方法で処理を行ないます。
     * \ja_end
     */
    DO_NOT_SUPPORT_NAMESPACE = 0x02,
    /*!
     * \ja
     * 実体参照(\&foo;)の定義が見つからない場合に無視して処理を継続します。
     * \ja_end
     */
    IGNORE_UNRESOLVED_ENTITY_REFERENCE = 0x04,
    /*!
     * \ja
     * standaloneがYESの時も外部ソースを読み込みます。
     * \ja_end
     */
    READ_EXTERNAL_IN_CASE_OF_STANDALONE = 0x10,
    /*!
     * \ja
     * 外部ソースをCatalogを元に読み込みます。
     * \ja_end
     */
    READ_EXTERNAL_WITH_CATALOG = 0x20,
    /*!
     * \ja
     * 外部ソースをURLから読み込みます。
     * \ja_end
     */
    READ_EXTERNAL_FROM_URL = 0x40,
    /*!
     * \ja
     * エンティティを展開しません。
     * \ja_end
     */
    PRIVENT_ENTITY_EXTRACTION = 0x100,
    KEEP_ENTITY_REFERENCE = PRIVENT_ENTITY_EXTRACTION,
    /*!
     * \ja
     * エンティティを独自形式で置き換え(&を0x01, ;を0x02)、テキスト中で表現可能な形式にします。
     * \ja_end
     */
    ENTITY_REPLACING = 0x200,
    /*!
     * \ja
     * ENTITY_REPLACINGのオプションを指定すると、&lt;&gt;&amp;&apos;&quot;も
     * 独自形式への置き換え対象となりますが、このオプションは、
     * &lt;&gt;&amp;&apos;&quot;を通常通りにエンティティの展開を行うオプションです。
     * \ja_end
     */
    ENTITY_REPLACING_EXCEPT_BASIC_ENTITY = 0x400,
    /*!
     * 属性値に<が入っていてもエラーにせず<の文字が記載されているものとして
     * 処理を行います。
     */
    HANDLE_LT_IN_THE_ATTVALUE = 0x800,
#ifndef CODEX_PURE_DOM
    /*!
     * 空子要素の要素が冗長表現で記述されていた場合に、この表現形式を
     * 保持します。
     */
    KEEP_REDUNDANT_EMPTY_TAG_EXPRESSION = 0x1000,
#endif /* CODEX_PURE_DOM */
    /*!
     * DTDの定義内で、未定義の実態参照を無視します。
     */
    IGNORE_DTD_UNRESOLVED_ENTITY_REFERENCE = 0x2000,
    /*!
     * \ja
     * デフォルトのオプション
     * \ja_end
     */
    DEFAULT_OPTION = (READ_EXTERNAL_WITH_CATALOG |
		      READ_EXTERNAL_FROM_URL |
		      IGNORE_UNRESOLVED_ENTITY_REFERENCE),
  };
 private:
  /*!
   * DOMImplementation
   */
  DOMImplementation* dom_imp;
  /*!
   * XMLReader
   */
  XMLReader* reader;
  /*!
   * \ja
   * ログの出力先
   * \ja_end
   */
  TextWriter<DOMChar>* log;
  /*!
   * \ja
   * XML パーサの解析元
   * \ja_end
   * \en
   * XML parser source
   * \en_end
   */
  XMLParserSrc* src;
  /*!
   * \ja
   * XML Document
   * \ja_end
   * \en
   * XML Document
   * \en_end
   */
  Document* document;
  /*!
   * \ja
   * XML宣言<br>
   * '<?xml version="1.0" encoding="UTF-16" standalone="yes" ?>'の属性
   * \ja_end
   * \en
   * XML declaration.<br>
   * attributes of '<?xml version="1.0" encoding="UTF-16" standalone="yes" ?>'
   * \en_end
   */
  NamedNodeMap* xml_decl_node_map;
  /*!
   * 名前空間を処理するためのスタックです。
   */
  LinkList<NamespaceUnit> namespace_stack;
  /*!
   * \ja
   * DTD
   * \ja_end
   * \en
   * DTD
   * \en_end
   */
  DocumentType* document_type;
  /*!
   * \ja
   * オプション
   * \ja_end
   * \en
   * Option
   * \en_end
   */
  Option option;
  /*!
   * A flag of external subset processing.
   */
  bool ext_subset_processing_p;
  /*!
   * \ja
   * 真に設定された場合、名前空間をサポートしないDOM Level 1
   * の方法でパーサの処理を行ないます。
   * \ja_end
   */
  bool do_not_support_namespace_p;
  /* ------------------------------------------------------------ */
  /*!
   * 指定した名前空間に入ります。
   * \param prefix プリフィックス
   * \param namespace_uri 名前空間URI
   * \param element 定義されたエレメント
   */
  void enterNamespace(DOMString* prefix, DOMString* namespace_uri, Element* element);
  /*!
   * 引数で指定したエレメントから脱出します。
   * このエレメントで定義された名前空間があれば無効にします。
   * \param element 脱出するエレメント
   */
  void exitNamespace(Element* element);
  /*!
   * 指定したプレフィックスが定義されているかを調べます。
   * 定義されていれば該当する名前空間URIのアトムを返します。
   * \param prefix プレフィックス
   * \return 名前空間URIのアトム,0が返った場合には該当する
   * 名前空間が定義されていないことを示します。
   */
  const DOMString* findNamespaceUri(const DOMString* prefix);
  /*!
   * 空プレフィックスが定義されているかを調べます。
   * 定義されていれば該当する名前空間URIのアトムを返します。
   * \return 名前空間URIのアトム,0が返った場合には空プレフィックス
   * に対する名前空間が定義されていないことを示します。
   */
  const DOMString* findNonamePrefixNamespaceUri();
 public:
  /*!
   * 指定したURIからコンテンツを読み込み、パースしてDocumentを生成します。
   * \param uri URI
   * \param dom_imp DOMImplementation, 0を指定するとデフォルトのDOMImplementationが内部で生成され使用されます。
   * \param reader XMLReader, 0を指定するとデフォルトのXMLReaderが内部で生成され使用されます。
   * \param log ログ, 0を指定するとログは出力されません。
   * \param option パーサオプション
   * \return 生成したDocument, エラー時には0が返ります。
   */
  static Document* readXML(DOMString* uri, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION);
  /*!
   * IFlowからコンテンツを読み込み、パースしてDocumentを生成します。
   * \param in コンテンツを読み込むIFlow
   * \param name 名前, 0を指定すると"unknown"として扱われます。
   * \param dom_imp DOMImplementation, 0を指定するとデフォルトのDOMImplementationが内部で生成され使用されます。
   * \param reader XMLReader, 0を指定するとデフォルトのXMLReaderが内部で生成され使用されます。
   * \param log ログ, 0を指定するとログは出力されません。
   * \param option パーサオプション
   * \return 生成したDocument, エラー時には0が返ります。
   */
  static Document* readXML(IFlow* in, const DOMString* name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION);
  static Document* readXML(IFlow* in, const DOMString& name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION)
  {
    return readXML(in, &name, dom_imp, reader, log, option);
  }
  /*!
   * ファイルからコンテンツを読み込み、パースしてDocumentを生成します。
   * \param filename コンテンツを読み込むファイルのパス
   * \param name 名前, 0を指定すると"unknown"として扱われます。
   * \param dom_imp DOMImplementation, 0を指定するとデフォルトのDOMImplementationが内部で生成され使用されます。
   * \param reader XMLReader, 0を指定するとデフォルトのXMLReaderが内部で生成され使用されます。
   * \param log ログ, 0を指定するとログは出力されません。
   * \param option パーサオプション
   * \return 生成したDocument, エラー時には0が返ります。
   */
  static Document* readXML(const char* filename, const DOMString* name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION);
  static Document* readXML(const char* filename, const DOMString& name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION)
  {
    return readXML(filename, &name, dom_imp, reader, log, option);
  }
#ifdef _WIN32
  /*!
   * ファイルからコンテンツを読み込み、パースしてDocumentを生成します。
   * \param filename コンテンツを読み込むファイルのパス
   * \param name 名前, 0を指定すると"unknown"として扱われます。
   * \param dom_imp DOMImplementation, 0を指定するとデフォルトのDOMImplementationが内部で生成され使用されます。
   * \param reader XMLReader, 0を指定するとデフォルトのXMLReaderが内部で生成され使用されます。
   * \param log ログ, 0を指定するとログは出力されません。
   * \param option パーサオプション
   * \return 生成したDocument, エラー時には0が返ります。
   */
  static Document* readXML(const wchar_t* filename, const DOMString* name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION);
  static Document* readXML(const wchar_t* filename, const DOMString& name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION)
  {
    return readXML(filename, &name, dom_imp, reader, log, option);
  }
#endif /* _WIN32 */
  /*!
   * ファイルからコンテンツを読み込み、パースしてDocumentを生成します。
   * \param path コンテンツを読み込むファイルのパス
   * \param name 名前, 0を指定すると"unknown"として扱われます。
   * \param dom_imp DOMImplementation, 0を指定するとデフォルトのDOMImplementationが内部で生成され使用されます。
   * \param reader XMLReader, 0を指定するとデフォルトのXMLReaderが内部で生成され使用されます。
   * \param log ログ, 0を指定するとログは出力されません。
   * \param option パーサオプション
   * \return 生成したDocument, エラー時には0が返ります。
   */
  static Document* readXML(const PathString* path, const DOMString* name, DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option = DEFAULT_OPTION);
  /*!
   * ファイルからコンテンツを読み込み、パースしてDocumentを生成します。
   * \param path コンテンツを読み込むファイルのパス
   * \param option パーサオプション
   * \return 生成したDocument, エラー時には0が返ります。
   */
  static Document* readXML(const PathString* path, Option option = DEFAULT_OPTION)
  {
    return readXML(path, 0, 0, 0, 0, option);
  }
 protected:
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * コンストラクタ
   * \param dom_imp DOMImplementation
   * \param reader XMLReader
   * \param log ログ
   * \param option パーサオプション
   * \ja_end
   * \en
   * Constructor
   * \param src XML Parser source
   * \en_end
   */
  XMLParser(DOMImplementation* dom_imp, XMLReader* reader, TextWriter<DOMChar>* log, Option option);
  /*!
   * \ja
   * デストラクタ
   * \ja_end
   * \en
   * Destructor
   * \en_end
   */
  ~XMLParser();
  /*!
   * \ja
   * XMLParserSrcを得ます。
   * \return XMLParserSrc
   * \ja_end
   */
  XMLParserSrc* getXMLParserSrc() { return src; }
  /*!
   * \ja
   * XMLReaderを得ます。
   * \ja_end
   */
  XMLReader* getXMLReader() { return reader; }
  /*!
   * \ja
   * DOMImplementationを得ます。
   * \ja_end
   */
  DOMImplementation* getDOMImplementation() { return dom_imp; }
  /*!
   * \ja
   * ログ用のTextWriter\<DOMChar\>フローを得ます。
   * \param ログ
   * \ja_end
   */
  TextWriter<DOMChar>* getLogTextWriter() { return log; }
  /*!
   * \ja
   * Documentを得ます。
   * \return Document
   * \ja_end
   * \en
   * get Document
   * \return Document
   * \en_end
   */
  Document* getDocument() { return document; }
  /*!
   * \ja
   * URIからトップレベルのデータを読み込みます。
   * \ja_end
   */
  bool readFromUri(DOMString* uri);
  /*!
   * \ja
   * 入力フローからトップレベルのデータを読み込みます。
   * \param name フローの名称
   * \param in 入力フロー
   * \ja_end
   */
  bool readFromIFlow(const DOMString* name, IFlow* in);
  /*!
   * \ja
   * ファイルからトップレベルのデータを読み込みます。
   * \param name ファイル名
   * \param file ファイルのパス
   * \ja_end
   */
  bool readFromFile(const DOMString* name, const char* file);
#ifdef _WIN32
  /*!
   * \ja
   * ファイルからトップレベルのデータを読み込みます。
   * \param name ファイル名
   * \param file ファイルのパス
   * \ja_end
   */
  bool readFromFile(const DOMString* name, const wchar_t* file);
#endif /* _WIN32 */
  /*!
   * \ja
   * XMLをパースします。
   * \return パース結果
   * \ja_end
   * \en
   * parse XML
   * \return result
   * \en_end
   */
  bool parse();
  /*!
   * \ja
   * 外部サブセットをパースします。
   * \ja_end
   * \en
   * parse External Subset
   * \en_end
   */
  void parseExtSubset();

 private:
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * \name デバッグ用のメソッド
   * \ja_end
   * \en
   * \name Methods for debugging.
   * \en_end
   *@{
   */
  /*!
   * \ja
   * 残りの入力内容を読み込み表示します。
   * \ja_end
   */
  void showRest();
  /*!
   * \ja
   * 引数で指定した文字列を表示します。
   * \param msg 最初に表示する文字列
   * \param str 次に表示する文字列
   * \ja_end
   */
  void show(char* msg, DOMString* str);
  //@}
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * \name ソースストリームの扱い用のメソッド
   * \ja_end
   * \en
   * \name Methods for source stream handling.
   * \en_end
   * @{
   */
  /*!
   * \ja
   * ソースストリームの残りサイズを取得します。
   * \return 
   * \ja_end
   * \en
   * This method gets the size of a rest source stream.
   * \return The size of a rest source stream.
   * \en_end
   */
  Size restSize();
  /*!
   * \ja
   * 現在の入力位置のイメージと与えられた文字列を比較します。
   * \param str 比較対象の文字列
   * \param len 比較対象の文字列の長さ
   * \return 結果
   * \ja_end
   * \en
   * This method compares an image of current input position and a given string.
   * \param str A compared string.
   * \param len Length of the compared string.
   * \return result
   * \en_end
   */
  bool match(DOMChar* str, Size len);
  /*!
   * \ja
   * 引数で与えた文字列が現在の入力位置のイメージと一致したら、入力イメージを受理し
   * 入力位置を文字列の長さ分進めます。
   * \param str 比較対象の文字列
   * \param len 比較対象の文字列の長さ
   * \return 結果
   * \ja_end
   * \en
   * If a given stirng is match with an current input position image, This method accepts
   * the current image and forward the input position.
   * \param str A checked string.
   * \param len len Length of the checked string.
   * \return result
   * \en_end
   */
  bool acceptCheck(DOMChar* str, Size len);
  /*!
   * \ja
   * 文字が候補文字列に含まれているかどうかを調べます。
   * \param c 調べる文字
   * \param candidates 候補文字列
   * \param candidates_len 候補文字列の長さ
   * \return 結果
   * \ja_end
   * \en
   * This method checks whether a character is a member of the candidates.
   * \param c A checked character.
   * \param candidates A set of candidates
   * \param candidates_len Length of the candidates string.
   * \return result
   * \en_end
   */
  static bool matchClass(DOMChar c, DOMChar* candidates, Size candidates_len);
  /*!
   * \ja
   * 終端文字列までの文字列を入力から取得します。
   * \param delimiter_str 終端文字列
   * \param delimiter_str_len 終端文字列の長さ
   * \return 取得した文字列
   * \ja_end
   * \en
   * The character string to immediately before the delimiter string is acquired.
   * \param delimiter_str delimiter string
   * \param delimiter_str_len delimiter string length
   * \return Acquired string
   * \en_end
   */
  DOMString* getStringDS(DOMChar* delimiter_str, Size delimiter_str_len);
  /*!
   * \ja
   * 終端文字列候補までの文字列を入力から取得します。
   * \param delimiter_strs 終端文字列の配列へのポインタ
   * \param delimiter_number 終端文字列の数
   * \return 取得した文字列
   * \ja_end
   * \en
   * The character string to immediately before the delimiter strings is acquired.
   * \param delimiter_strs pointer to delimiter strings array
   * \param delimiter_number delimiter strings array number
   * \return Acquired string
   * \en_end
   */
  DOMString* getStringDSs(DOMChar** delimiter_strs, Size delimiter_number);
  /*!
   * \ja
   * 終端文字候補までの文字列を入力から取得します。
   * \param delimiters 終端文字候補
   * \param delimiters_len 終端文字候補の長さ
   * \retval 0 取得できません
   * \retval other 取得した文字列
   * \ja_end
   * \en
   * The character string to immediately before the delimiter character class is acquired.
   * \param delimiters delimiter characters
   * \param delimiters_len delimiter characters length
   * \retval 0 can't get string
   * \retval other acquired string
   * \en_end
   */
  DOMString* getStringDC(DOMChar* delimiters, Size delimiters_len);
  /*!
   * \ja
   * 特定の文字クラスの文字列を入力から取得します。
   * \param cc 文字クラス
   * \param str 読み込み先
   * \return false 読み込めません
   * \return true 読み込めました
   * \ja_end
   */
  bool getClassString(XmlCharClass cc, DOMString* str);
  /*!
   * \ja
   * 特定の入力候補の文字列を入力から取得します。
   * \param candidates 入力候補
   * \param candidates_len 入力候補の長さ
   * \param str 読み込み先
   * \return false 読み込めません
   * \return true 読み込めました
   * \ja_end
   */
  bool getClassString(DOMChar* candidates, Size candidates_len, DOMString* str);
  /*!
   * \ja
   * Nameを読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \code
   * [5]
   * Name ::= ( Letter | '_' | ':' ) ( NameChar )*
   * NameChar ::=  Letter | Digit | '.' | '-' | '_' | ':' | CombiningChar | Extender
   * Letter ::= BaseChar | Ideographic
   * \endcode
   */
  DOMString* getName();
  /*!
   * \ja
   * Nmtokenを読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \code
   * [7]
   * Nmtoken ::= ( NameChar )+
   * \endcode
   */
  DOMString* getNmtoken();
  /*! @} */
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * \name リテラル
   * \ja_end
   * \en
   * \name Literals
   * \en_end
   *@{
   */
  /*!
   * \ja
   * AttValueを読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \code
   * [10]
   * AttValue ::= '"' ( [^<&"] | Reference )* '"'
   *            | "'" ( [^<&'] | Reference )* "'"
   * \endcode
   */
  DOMString* getAttValue();
  /*!
   * \ja
   * EntityValueを読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \code
   * [9]
   * EntityValue ::= '"' ( [^%&"] | PEReference | Reference )* '"'
   *               | "'" ( [^%&'] | PEReference | Reference )* "'"
   * \endcode
   */
  DOMString* getEntityValue();
  /*!
   * \ja
   * SystemLiteralを読み込みます。
   * \param permit_not_fount_p trueに指定するとSystemLiteralが無い場合にエラーになりません。
   * \return 読み込んだ文字列
   * \retval 0 見つかりません(permit_not_fount_pがtrueの時のみ)
   * \ja_end
   * \en
   * \param permit_not_fount_p The case which is not found is permitted.
   * \return system literal.
   * \retval 0 not found. In case of permit_not_found_p is true only.
   * \en_end
   * \code
   * [11]
   * SystemLiteral ::= ( '"' [^"]* '"') | ( "'" [^']* "'")
   * \endcode
   */
  DOMString* getSystemLiteral(bool permit_not_found_p = false);
  /*!
   * \ja
   * PubidLiteralを読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \code
   * [12]
   * PubidLiteral ::= '"' PubidChar * '"'
   *                | "'" ( PubidChar - "'" )* "'"
   * [13]
   * PubidChar ::= #x20 | #xD | #xA | [a-zA-Z0-9] | [-'()+,./:=?;!*#@$_%]
   * \endcode
   */
  DOMString* getPubidLiteral();
  /*!
   * \ja
   * Ignoreを読み込みます。
   * \return 読み込んだ文字列
   * \ja_end
   * \code
   * [65]
   * Ignore ::= Char* - ( Char* ('<![' | ']]>' ) Char* )
   * \endcode
   */
  DOMString* getIgnore();
  /*! @} */

  /*!
   * \ja
   * \name パーサメソッド
   * paserXxx. Xxxはそれぞれのルールに当たります。
   * l1_pのboolの引数を持つメソッドはそれぞれL1の受理ステータスを返します。
   * \ja_end
   * \en
   * \name Parser methods
   * paserXxx. Xxx is a each rule.
   * Every method which has a bool l1_p argument, returns the L1 parser acceptance status.
   * \en_end
   *@{
   */
  /*!
   * \ja
   * Documentをパースします。
   * \ja_end
   * \code
   * [1]
   * document ::= prolog element Misc*
   * \endcode
   */
  void parseDocument();
  /*!
   * \ja
   * Prologをパースします。
   * \ja_end
   * \code
   * [22]
   * prolog ::= XMLDecl? Misc* ( doctypedecl Misc*)?
   * \endcode
   */
  void parseProlog();
  /*!
   * \ja
   * 様々なエピローグ(documentのMisc*)をパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   */
  bool parseMiscEpilog(bool l1_p);
  /*!
   * \ja
   * XMLDeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [23]
   * XMLDecl ::= '<?xml' VersionInfo EncodingDecl? SDDecl? S? '?>'
   * LL(1) =>
   * XMLDecl ::= '<?xml' VersionInfo (S EncodingDecl)? (S SDDecl)? S? '?>'
   * \endcode
   */
  bool parseXMLDecl(bool l1_p);
  /*!
   * \ja
   * XMLDecl直後のMiscをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [27]
   * Misc ::= Comment | PI | S
   * \endcode
   */
  bool parsePrologMisc1(bool l1_p);
  /*!
   * \ja
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   */
  bool parsePrologMisc2(bool l1_p);
  /*!
   * \ja
   * doctypedecl直後のMiscをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   */
  bool parseVersionInfo(bool l1_p);
  /*!
   * \ja
   * EncodingDeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [80] 
   * EncodingDecl ::= S 'encoding' Eq ( '"' EncName '"' | "'" EncName "'" )
   * LL(1) =>
   * EncodingDecl ::= 'encoding' Eq ( '"' EncName '"' | "'" EncName "'" )
   * \endcode
   */
  bool parseEncodingDecl(bool l1_p);
  /*!
   * \ja
   * SDDeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [32] 
   * SDDecl ::= S 'standalone' Eq (( "'" ( 'yes' | 'no' ) "'" ) | ( '"' ( 'yes' | 'no' ) '"' ))
   * LL(1) =>
   * SDDecl ::= 'standalone' Eq (( "'" ( 'yes' | 'no' ) "'" ) | ( '"' ( 'yes' | 'no' ) '"' ))
   * \endcode
   */
  bool parseSDDecl(bool l1_p);
  /*!
   * \ja
   * 空白文字をパースします。
   * 親ノードが指定された場合には、読み込んだ空白文字を持つテキストノード
   * を生成し親ノードの子ノードとして追加します。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \param parent 親ノード
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [3]
   * S ::= ( #x20 | #x9 | #xD | #xA )+
   * \endcode
   */
  bool parseS(bool l1_p, Node* parent = 0);
  /*!
   * \ja
   * 空白文字をパースします。
   * \ja_end
   */
  void parseS();
  /*!
   * \ja
   * Eqをパースします。
   * \ja_end
   * \code
   * [25]
   * Eq ::= S? '=' S?
   * \endcode
   */
  void parseEq();
  /*!
   * \ja
   * EncNameをパースします。
   * \return エンコーディング名文字列
   * \ja_end
   * \code
   * [81]
   * EncName ::= [A-Za-z] ( [A-Za-z0-9._] | '-' )*
   * \endcode
   */
  DOMString* parseEncName();
  /*!
   * \ja
   * VersionNumをパースします。
   * \return バージョン番号文字列
   * \ja_end
   * \code
   * [26]
   * VersionNum ::= ( [a-zA-Z0-9_.:] | '-' )+
   * \endcode
   */
  DOMString* parseVersionNum();
  /*!
   * \ja
   * Commentをパースします。
   * 親ノードの子ノードとしてコメントノードを追加します。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \param parent 親ノード
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [15]
   * Comment ::= '<!--' (( Char - '-') | ('-' ( Char - '-')))* '-->'
   * \endcode
   */
  bool parseComment(bool l1_p, Node* parent);
  /*!
   * \ja
   * PIをパースします。
   * 親ノードの子ノードとしてPIノードを追加します。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \parent parent 親ノード
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [16]
   * PI ::= '<?' PITarget ( S ( Char* - ( Char* '?>' Char*)))? '?>'
   * \endcode
   */
  bool parsePI(bool l1_p, Node* parent);
  /*!
   * \ja
   * PITargetをパースします。
   * \return PITarget名を返します。
   * \ja_end
   * \code
   * [17]
   * PITarget ::= Name - (( 'X' | 'x' ) ( 'M' | 'm' ) ( 'L' | 'l' ))
   * \endcode
   */
  DOMString* parsePITarget();
  /*!
   * \ja
   * Referenceをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \param s l1_pがfalseの時にReferenceの内容をセットします。
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [67] 
   * Reference ::= EntityRef | CharRef
   * [66] 
   * CharRef ::= '&#' [0-9]+ ';'
   *           | '&#x' [0-9a-fA-F]+ ';'
   * [68] 
   * EntityRef ::= '&' Name ';'
   * =>
   * Reference ::= '&' ( '#' ( [0-9]+ | 'x' [0-9a-fA-F]+ ) ';' )
   *             | '&' Name ';'
   * \endcode
   */
  bool parseReference(bool l1_p, DOMString* s);
  /*!
   * \ja
   * PEReferenceをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [69] 
   * PEReference ::= '%' Name ';'
   * \endcode
   */
  bool parsePEReference(bool l1_p);
  /*! @} */
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * \name DTDパーサメソッド
   * \ja_end
   * \en
   * \name DTD Parser methods
   * \en_end
   * @{
   */
  /*!
   * \ja
   * doctypedeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [28] 
   * doctypedecl ::= '<!DOCTYPE' S Name ( S ExternalID )? S? ( '[' ( markupdecl | DeclSep )* ']' S? )? '>'
   * =>
   * doctypedecl ::= '<!DOCTYPE' S Name ( S ( ExternalID1 | ExternalID2 ) )? S? ( '['
   *   ( markupdecl | DeclSep )* ']' S? )? '>'
   * =>
   * [29]
   * markupdecl ::= elementdecl | AttlistDecl | EntityDecl | NotationDecl | PI | Comment
   * [28a]
   * DeclSep ::= PEReference | S
   * doctypedecl ::= '<!DOCTYPE' S Name ( S ( ExternalID1 | ExternalID2 ) )? S? ( '['
   *   ( elementdecl | AttlistDecl | EntityDecl | NotationDecl | PI | Comment | PEReference | S )*
   *   ']' S? )? '>'
   * \endcode
   */
  bool parseDoctypedecl(bool l1_p);
  /*!
   * \ja
   * ExternalID1をパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [75] 
   * ExternalID ::= 'SYSTEM' S SystemLiteral
   *              | 'PUBLIC' S PubidLiteral S SystemLiteral
   * =>
   * ExternalID1 ::= 'SYSTEM' S SystemLiteral
   * ExternalID2 ::= 'PUBLIC' S PubidLiteral S SystemLiteral
   * \endcode
   */
  bool parseExternalID1(bool l1_p, External* ext);
  /*!
   * \ja
   * ExternalID2をパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [75] 
   * ExternalID ::= 'SYSTEM' S SystemLiteral
   *              | 'PUBLIC' S PubidLiteral S SystemLiteral
   * =>
   * ExternalID1 ::= 'SYSTEM' S SystemLiteral
   * ExternalID2 ::= 'PUBLIC' S PubidLiteral S SystemLiteral
   * \endcode
   */
  bool parseExternalID2(bool l1_p, External* ext);
  /*!
   * \ja
   * elementdeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [45] 
   * elementdecl  ::= '<!ELEMENT' S Name S contentspec S? '>'
   * [46]
   * contentspec  ::= 'EMPTY' | 'ANY' | Mixed | children
   * [51] 
   * Mixed  ::= '(' S? '#PCDATA' ( S? '|' S? Name )* S? ')*'
   *          | '(' S? '#PCDATA' S? ')'
   * [47] 
   * children ::= ( choice | seq ) ( '?' | '*' | '+' )?
   * [49]
   * choice  ::= '(' S? cp ( S? '|' S? cp )+ S? ')'
   * [50] 
   * seq  ::= '(' S? cp ( S? ',' S? cp )* S? ')'
   * [48]
   * cp  ::= ( Name | choice | seq ) ( '?' | '*' | '+' )?
   * =>
   * contentspec  ::= 'EMPTY'
   *                | 'ANY'
   *                | '(' S? '#PCDATA' ( S? '|' S? Name )* S? ')*'
   *                | '(' S? '#PCDATA' S? ')'
   *                | '(' S? cp S? ( ( '|' S? cp S? )+ | ( ',' S? cp S? )* ) S? ')' ( '?' | '*' | '+' )?
   * cp  ::= ( Name | '(' S? cp S? ( ( '|' S? cp S? )+ | ( ',' S? cp S? )* ) S? ')' ( '?' | '*' | '+' )?
   * \endcode
   */
  bool parseElementdecl(bool l1_p);
  /*!
   * \ja
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   */
  //bool parseContentspec(bool l1_p);
  /*!
   * \ja
   * cpをパースします。
   * \ja_end
   * \code
   * [48]
   * cp  ::= ( Name | choice | seq ) ( '?' | '*' | '+' )?
   * [49]
   * choice  ::= '(' S? cp ( S? '|' S? cp )+ S? ')'
   * [50] 
   * seq  ::= '(' S? cp ( S? ',' S? cp )* S? ')'
   * cp  ::= ( Name | '(' S? cp S? ( ( '|' S? cp S? )+ | ( ',' S? cp S? )* ) S? ')' ) ( '?' | '*' | '+' )?
   * \endcode
   */
  DtdECUnit* parseCp();
  /*!
   * \ja
   * AttlistDeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [52] 
   * AttlistDecl ::= '<!ATTLIST' S Name AttDef* S? '>'
   * [53]
   * AttDef  ::= S Name S AttType S DefaultDecl
   * =>
   * AttlistDecl ::= '<!ATTLIST' S Name (S Name S AttType S DefaultDecl)* S? '>'
   * \endcode
   */
  bool parseAttlistDecl(bool l1_p);
  /*!
   * \ja
   * AttTypeをパースします。
   * \param attlist DTD属性リスト
   * \ja_end
   * \code
   * [54] 
   * AttType ::= StringType | TokenizedType | EnumeratedType
   * [55] 
   * StringType ::= 'CDATA'
   * [56] 
   * TokenizedType ::= 'ID'
   *                | 'IDREF'
   *                | 'IDREFS'
   *                | 'ENTITY'
   *                | 'ENTITIES'
   *                | 'NMTOKEN'
   *                | 'NMTOKENS'
   * [57] 
   * EnumeratedType ::= NotationType | Enumeration
   * [58] 
   * NotationType ::= 'NOTATION' S '(' S? Name ( S? '|' S? Name )* S? ')'
   * [59] 
   * Enumeration ::= '(' S? Nmtoken ( S? '|' S? Nmtoken )* S? ')'
   *=>
   * AttType ::= 'CDATA'
   *           | 'ID'
   *           | 'IDREF'
   *           | 'IDREFS'
   *           | 'ENTITY'
   *           | 'ENTITIES'
   *           | 'NMTOKEN'
   *           | 'NMTOKENS'
   *           | 'NOTATION' S '(' S? Name S? ( '|' S? Name S? )* ')'
   *           | '(' S? Nmtoken S? ( '|' S? Nmtoken S? )* ')'
   * \endcode
   */
  void parseAttType(DtdAttlist* attlist);
  /*!
   * \ja
   * DefaultDeclをパースします。
   * \param attlist DTD属性リスト
   * \ja_end
   * \code
   * [60]
   * DefaultDecl ::= '#REQUIRED'
   *               | '#IMPLIED'
   *               | (('#FIXED' S )? AttValue )
   *=>
   * DefaultDecl ::= '#REQUIRED'
   *               | '#IMPLIED'
   *               | '#FIXED' S AttValue
   *               |  AttValue
   * \endcode
   */
  void parseDefaultDecl(DtdAttlist* attlist);
  /*!
   * \ja
   * EntityDeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [70] 
   * EntityDecl  ::= GEDecl | PEDecl
   * [71] 
   * GEDecl  ::= '<!ENTITY' S Name S EntityDef S? '>'
   * [72] 
   * PEDecl  ::= '<!ENTITY' S '%' S Name S PEDef S? '>'
   * [73]
   * EntityDef ::= EntityValue | ( ExternalID NDataDecl? )
   * [74] 
   * PEDef ::= EntityValue | ExternalID
   * [76] 
   * NDataDecl ::= S 'NDATA' S Name
   *=>
   * EntityDecl  ::= '<!ENTITY' S Name S ( EntityValue | ( ExternalID NDataDecl? ) ) S? '>'
   *               | '<!ENTITY' S '%' S Name S ( EntityValue | ExternalID ) S? '>'
   *=>
   * EntityDecl  ::= '<!ENTITY' S '%' S Name S ( EntityValue | ExternalID1 | ExternalID2 ) S? '>'
   *               | '<!ENTITY' S Name S ( EntityValue | ( ( ExternalID1 | ExternalID2 ) ( S 'NDATA' S Name )? ) ) S? '>'
   * \endcode
   */
  bool parseEntityDecl(bool l1_p);
  /*!
   * \ja
   * NotationDeclをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [82] 
   * NotationDecl ::= '<!NOTATION' S Name S ( ExternalID | PublicID ) S? '>'
   *=>
   * NotationDecl ::= '<!NOTATION' S Name S ( ExternalID1 | ExternalID2 | PublicID ) S? '>'
   * ExternalID1 ::= 'SYSTEM' S SystemLiteral
   * ExternalID2 ::= 'PUBLIC' S PubidLiteral S SystemLiteral
   * PublicID ::= 'PUBLIC' S PubidLiteral
   *=>
   * NotationDecl ::= '<!NOTATION' S Name S ( ExternalID1 | PublicID ) S? '>'
   * ExternalID1 ::= 'SYSTEM' S SystemLiteral
   * PublicID ::= 'PUBLIC' S PubidLiteral (S SystemLiteral)?
   * \endcode
   */
  bool parseNotationDecl(bool l1_p);
  /*!
   * \ja
   * PublicIDをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [83] 
   * PublicID ::= 'PUBLIC' S PubidLiteral
   *=>
   * PublicID ::= 'PUBLIC' S PubidLiteral (S SystemLiteral)?
   * \endcode
   * see bool XMLParser::parseNotationDecl(bool l1_p).
   */
  bool parsePublicID(bool l1_p, Notation* notation);
  //@}
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * \name Elementパーサメソッド
   * \ja_end
   * \en
   * \name Element Parser methods
   * \en_end
   * @{
   */
  /*!
   * \ja
   * elementをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \param parent 親ノード
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [39]
   * element ::= EmptyElemTag
   *           | STag content ETag
   * [40] 
   * STag  ::= '<' Name ( S Attribute )* S? '>'
   * [44] 
   * EmptyElemTag  ::= '<' Name ( S Attribute )* S? '/>'
   * [43]
   * content ::= CharData? ( ( element | Reference | CDSect | PI | Comment ) CharData? )*
   * [14]
   * CharData ::= [^<&]* - ( [^<&]* ']]>' [^<&]* )
   * =>
   * element ::= '<' Name ( S Attribute )* S? ( '/>' | ( '>' content ETag ) )
   * content ::= ( ( element | Reference | CDSect | PI | Comment | CharData )*
   * CharData ::= [^<&]* - ( [^<&]* ']]>' [^<&]* )
   * \endcode
   */
  bool parseElement(bool l1_p, Node* parent);
  /*!
   * \ja
   * Attributeをパースします。
   * \ja_end
   * \code
   * [41]
   * Attribute ::= Name Eq AttValue
   * \endcode
   */
  void parseAttribute(Element* element);
  /*!
   * \ja
   * 属性の名前空間の処理を行ないます。<br>
   * Attributeのパースと同時に名前空間の処理を行なわないのは、
   * 同じエレメント内で定義された名前空間の宣言を用いたプレフィックス
   * の使用が定義に先んじて出現する可能性があるためです。
   * \param element 処理を行なうエレメント
   * \ja_end
   */
  void parseAttributeNS(Element* element);
  /*!
   * \ja
   * ETagをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [42] 
   * ETag  ::= '</' Name S? '>'
   * \endcode
   */
  bool parseETag(bool l1_p);
  /*!
   * \ja
   * CDSectをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \param parent 親ノード
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [18]
   * CDSect ::= CDStart CData CDEnd
   * [19]
   * CDStart ::= '<![CDATA['
   * [20]
   * CData ::= ( Char* - ( Char* ']]>' Char*))
   * [21]
   * CDEnd ::= ']]>'
   * =>
   * CDSect ::= '<![CDATA[' CData ']]>'
   * \endcode
   */
  bool parseCDSect(bool l1_p, Node* parent);
  /*! @} */
  /* ------------------------------------------------------------ */
  /*!
   * \ja
   * \name External subsetパーサメソッド
   * \ja_end
   * \en
   * \name External subset Parser methods
   * \en_end
   * @{
   */
  /*!
   * \ja
   * extSubsetをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [30]
   * extSubset ::= TextDecl? extSubsetDecl
   * \endcode
   */
  bool parseTextDecl(bool l1_p);
  /*!
   * \ja
   * extSubsetDeclをパースします。
   * \ja_end
   * \code
   * [31]
   * extSubsetDecl ::= ( markupdecl | conditionalSect | DeclSep )*
   * [28a]
   * DeclSep ::= PEReference | S
   * [29]
   * markupdecl ::= elementdecl | AttlistDecl | EntityDecl | NotationDecl | PI | Comment
   * =>
   * extSubsetDecl ::= ( elementdecl | AttlistDecl | EntityDecl | NotationDecl | PI | Comment | conditionalSect | PEReference | S )*
   * \endcode
   */
  void parseExtSubsetDecl();
  /*!
   * \ja
   * conditionalSectをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [61] 
   * conditionalSect ::= includeSect | ignoreSect
   * [62] 
   * includeSect  ::= '<![' S? 'INCLUDE' S? '[' extSubsetDecl ']]>'
   * [63] 
   * ignoreSect  ::= '<![' S? 'IGNORE' S? '[' ignoreSectContents* ']]>'
   * =>
   * conditionalSect ::= '<![' S? ( ( 'INCLUDE' S? '[' extSubsetDecl ']]>' ) | 'IGNORE' S? '[' ignoreSectContents* ']]>' )
   * \endcode
   */
  bool parseConditionalSect(bool l1_p);
  /*!
   * \ja
   * ignoreSectContentsをパースします。
   * \param l1_p 先読み動作指定時にtrueを指定
   * \return L1受理ステータス
   * \ja_end
   * \code
   * [64] 
   * ignoreSectContents ::= Ignore ( '<![' ignoreSectContents ']]>' Ignore )*
   * \endcode
   */
  bool parseIgnoreSectContents(bool l1_p);
  /*! @} */
};

/*!
 * \ja
 * 実体記述の置き換えを表現するための構造体
 * \ja_end
 */
struct EntityReplacement
{
  DOMChar* esc;	/*!< エスケープした表現 */
  DOMChar src;	/*!< 元表現 */
};

/*!
 * \ja
 * XMLのトークン
 * \ja_end
 * \en
 * XML tokens
 * \en_end
 */
struct XMLToken
{
  static DOMChar str_xmldecl_start[]; //!< "<?xml"
  static DOMChar str_xmldecl_end[]; //!< "?>"
  static DOMChar str_quote[]; //!< "'"
  static DOMChar str_dquote[]; //!< "\""
  static DOMChar str_eq[]; //!< "="
  static DOMChar str_open[]; //!< "<"
  static DOMChar str_close[]; //!< ">"
  static DOMChar str_openbr[]; //!< "("
  static DOMChar str_closebr[]; //!< ")"
  static DOMChar str_question[]; //!< "?"
  static DOMChar str_aster[]; //!< "*"
  static DOMChar str_plus[]; //!< "+"
  static DOMChar str_or[]; //!< "|"
  static DOMChar str_seq[]; //!< ","
  static DOMChar str_percent[]; //!< "%"
  static DOMChar str_semicol[]; //!< ";"
  static DOMChar str_markupdecl_start[]; //!< "["
  static DOMChar str_markupdecl_end[]; //!< "]"
  static DOMChar str_cond_start[]; //!< "<!["
  static DOMChar str_cond_end[]; //!< "]]>"
  static DOMChar str_and[]; //!< "&"
  static DOMChar str_sharp[]; //!< "#"
  static DOMChar str_x[]; //!< "x"
  static DOMChar str_version[]; //!< "version"
  static DOMChar str_encoding[]; //!< "encoding"
  static DOMChar str_standalone[]; //!< "standalone"
  static DOMChar str_yes[]; //!< "yes"
  static DOMChar str_no[]; //!< "no"
  static DOMChar str_comment_start[]; //!< "<!--"
  static DOMChar str_comment_end[]; //!< "-->"
  static DOMChar str_pi_start[]; //!< "<?"
  static DOMChar str_pi_end[]; //!< "?>"
  static DOMChar str_doctype_start[]; //!< "<!DOCTYPE"
  static DOMChar str_externalidsystem[]; //!< "SYSTEM"
  static DOMChar str_externalidpublic[]; //!< "PUBLIC"
  static DOMChar str_element_start[]; //!< "<!ELEMENT"
  static DOMChar str_empty[]; //!< "EMPTY"
  static DOMChar str_any[]; //!< "ANY"
  static DOMChar str_pcdata[]; //!< "#PCDATA"
  static DOMChar str_attlist_start[]; //!< "<!ATTLIST"
  static DOMChar str_cdata[]; //!< "CDATA"
  static DOMChar str_id[]; //!< "ID"
  static DOMChar str_idref[]; //!< "IDREF"
  static DOMChar str_idrefs[]; //!< "IDREFS"
  static DOMChar str_entity[]; //!< "ENTITY"
  static DOMChar str_entities[]; //!< "ENTITIES"
  static DOMChar str_nmtoken[]; //!< "NMTOKEN"
  static DOMChar str_nmtokens[]; //!< "NMTOKENS"
  static DOMChar str_notation[]; //!< "NOTATION"
  static DOMChar str_required[]; //!< "#REQUIRED"
  static DOMChar str_implied[]; //!< "#IMPLIED"
  static DOMChar str_fixed[]; //!< "#FIXED"
  static DOMChar str_entity_start[]; //!< "<!ENTITY"
  static DOMChar str_ndata[]; //!< "NDATA"
  static DOMChar str_notation_start[]; //!< "<!NOTATION"
  static DOMChar str_empclose[]; //!< "/>"
  static DOMChar str_etag[]; //!< "</"
  static DOMChar str_cdata_start[]; //!< "<![CDATA["
  static DOMChar str_cdata_end[]; //!< "]]>"
  static DOMChar str_include[]; //!< "INCLUDE"
  static DOMChar str_ignore[]; //!< "IGNORE"
  // constnts below is used only when writing.
  static DOMChar str_null[];	// ""
  static DOMChar str_sp[]; //!< " "
  // entity
  static DOMChar str_lt[]; //!< lt
  static DOMChar str_gt[]; //!< gt
  static DOMChar str_amp[]; //!< amp
  static DOMChar str_apos[]; //!< apos
  static DOMChar str_quot[]; //!< quot
  // namespace
  static DOMChar str_xml[]; //!< xml
  static DOMChar str_xml_uri[]; //!< http://www.w3.org/XML/1998/namespace
  static DOMChar str_xmlns[]; //!< xmlns
  static DOMChar str_xmlns_uri[]; //!< http://www.w3.org/2000/xmlns/
  // XSLT
  static DOMChar str_xsl[]; //!< xsl
  static DOMChar str_xmlns_xsl[]; //!< xmlns:xsl
  static DOMChar str_xmlns_xsl_uri[]; //!< http://www.w3.org/1999/XSL/Transform

  // for entity replacement (CODEX original rule)
  static DOMChar str_entity_rep_and[];	//! 0x01: replacement of '&'
  static DOMChar str_entity_rep_semicol[];	//! 0x02: replacement of ';'

  static EntityReplacement basic_entity[];

  // for entity extraction
  static EntityReplacement basic_entity_src[];
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_CCC_XML_XMLPARSER_H */
