﻿// $Id$

#include <string.h>
#ifdef _WIN32
// TODO:
#else /* _WIN32 */
#  include <sys/stat.h>
#  include <fcntl.h>
#  include <sys/types.h>
#ifndef _WIN32_WCE
#  include <sys/uio.h>
#  include <unistd.h>
#endif /* _WIN32_WCE */
#endif /* _WIN32 */
#include <ccc/base/Spooler.h>
#include <ccc/base/FixedMemIFlow.h>
#include <ccc/file/FileIFlow.h>
#include <ccc/base/Conveyer.h>
#include <ccc/xml/XMLReader.h>
#include <ccc/iceman/Iceman.h>
#include <ccc/iceman/Detector.h>
#include <ccc/fetch/Uri.h>
#include <ccc/fetch/HttpFetch.h>

CCC_NAMESPACE_START(CCC);

// ------------------------------------------------------------------------
// class XMLReader

XMLReader::XMLReader(Catalog* catalog, TextWriter<DOMChar>* log)
{
  XMLReader::catalog = catalog;
  XMLReader::log = log;
  detector = Iceman::createDetector(CEID_XMLDETECTOR);
  file_system_ceid = CEID_USASCII;
  system_root = 0;
  cat_ref_root = 0;
}

XMLReader::~XMLReader()
{
  delete detector;
  delete system_root;
  delete cat_ref_root;
}

void
XMLReader::setCatalogRefRootPath(char* path)
{
  delete cat_ref_root;
  size_t len = strlen(path);
  cat_ref_root = new char[len + 1];
  strcpy(cat_ref_root, path);
#ifdef _WIN32
  if (len > 0 && cat_ref_root[len - 1] == '\\')
#else /* _WIN32 */
  if (len > 0 && cat_ref_root[len - 1] == '/')
#endif /* _WIN32 */
  {
    cat_ref_root[len - 1] = '\0';
  }
}

void
XMLReader::setSystemRoot(const char* path)
{
  delete system_root;
  size_t len = strlen(path);
  system_root = new char[len + 1];
  strcpy(system_root, path);
#ifdef _WIN32
  if (len > 0 && system_root[len - 1] == '\\')
#else /* _WIN32 */
  if (len > 0 && system_root[len - 1] == '/')
#endif /* _WIN32 */
  {
    system_root[len - 1] = '\0';
  }
}

char*
XMLReader::catPath(const char* path1, const char* path2)
{
  char* ret;
  size_t l2 = strlen(path2);
  if (path1)
  {
    size_t l1 = strlen(path1);
    ret = new char[l1 + 1 + l2 + 1];
    strcpy(ret, path1);
#ifdef _WIN32
    strcat(ret, "\\");
#else /* _WIN32 */
    strcat(ret, "/");
#endif /* _WIN32 */
    strcat(ret, path2);
  }
  else
  {
    ret = new char[l2 + 1];
    strcpy(ret, path2);
  }
  return ret;
}

char*
XMLReader::makeSystemFilePath(const char* path)
{
  return catPath(system_root, path);
}

char*
XMLReader::makeCatalogRefFilePath(const char* path)
{
  return catPath(cat_ref_root, path);
}

void
XMLReader::setDetector(Detector* d)
{
  delete detector;
  detector = d;
}

bool
XMLReader::convertToDomCeid(Allocator* from, Allocator* to)
{
  CompositIFilter* cf = 0;
  {
    FixedMemIFlow from_iflow(from);
    CeId ceid = detector->detect(&from_iflow);
    if (ceid == CEID_NULL)
    {
      return false;
    }
    cf = Iceman::createCompositIFilter(ceid, CCC_DOM_CEID);
  }
  if (!cf)
  {
    // detection error
    return false;
  }
  {
    FixedMemIFlow from_iflow(from);
    MemOFlow to_oflow(to);
    cf->setSrc(&from_iflow);
    Conveyer conv(cf, &to_oflow);
    conv.carry();
  }
  delete cf;
  return true;
}

XMLParserSrcUnit*
XMLReader::readFromIFlow(const DOMString* name, IFlow* in) CCC_RAISES(IOException)
{
  /* データをメモリに読み込みます。*/
  Allocator src_image;		/* ソースイメージを格納したメモリ */
  {
    MemOFlow o(&src_image);
    Conveyer conv(in, &o);
    conv.carry();
  }
  /* ソースイメージをDOMCharの文字セットエンコーディングにコンバートします。 */
  Allocator parser_src;		/* DOMCharにコンバートした結果 */
  if (!convertToDomCeid(&src_image, &parser_src))
  {
    return 0;
  }
  XMLParserSrcUnit* unit = new XMLParserSrcUnit(name,
						(DOMChar*)parser_src.getMem(),
						parser_src.getUsedSize() / sizeof(DOMChar));
  return unit;
}

XMLParserSrcUnit*
XMLReader::readFromFile(const DOMString* name, const char* file) CCC_RAISES(IOException)
{
#ifdef _WIN32
  FileIFlow iflow;
  if (!iflow.open(file))
  {
    return 0;
  }
  return readFromIFlow(name, &iflow);
#else /* _WIN32 */
  struct stat buf;
  if (stat(file, &buf) != 0)
  {
    // ERROR
    return 0;
  }
  int fd = open(file, O_RDONLY);
  if (fd < 0)
  {
    // ERROR
    return 0;
  }
  size_t file_size = buf.st_size;
  Allocator src_image(file_size);
  void* x = src_image.getMem();
  size_t rest_file_size = file_size;
  while (rest_file_size > 0)
  {
    const size_t read_unit = 1024 * 100;
    size_t read_size = (read_unit < rest_file_size) ? read_unit : rest_file_size;
    read_size = read(fd, x, read_size);
    rest_file_size -= read_size;
    x = (void*)((char*)x + read_size);
  }
  src_image.setUsedSize((Size)file_size);
  close(fd);
  Allocator parser_src(file_size * 2);		/* DOMCharにコンバートした結果 */
  if (!convertToDomCeid(&src_image, &parser_src))
  {
    return 0;
  }
  XMLParserSrcUnit* unit = new XMLParserSrcUnit(name,
						(DOMChar*)parser_src.getMem(),
						parser_src.getUsedSize() / sizeof(DOMChar));
  return unit;
#endif /* _WIN32 */
}

#ifdef _WIN32
XMLParserSrcUnit*
XMLReader::readFromFile(const DOMString* name, const wchar_t* file) CCC_RAISES(IOException)
{
  FileIFlow iflow;
  if (!iflow.open(file))
  {
    return 0;
  }
  return readFromIFlow(name, &iflow);
}
#endif /* _WIN32 */

XMLParserSrcUnit*
XMLReader::readFromUri(DOMString* uri)
{
  BString b_uri;
  CCC::Iceman::convertToBString(CCC_DOM_CEID, file_system_ceid, uri, &b_uri);
  Uri x_uri(b_uri.getCString());
  Uri::Scheme scheme = x_uri.getScheme();
  XMLParserSrcUnit* ret = 0;
  Allocator* mem = 0;
  switch (scheme)
  {
   case Uri::SCHEME_HTTP:	// http://host/path
#if 1
    {
      CCC::HttpRequest http_req;
      {
	CCC::BString buf;
	{
	  CCC::Fmt8 fmt("GET %s HTTP/1.1\r\n");
	  fmt.setTP(b_uri.getCString());
	  fmt.writeTo(&buf);
	}
	http_req.addMethod(&buf);
	buf.assign("User-Agent: ccc\r\n");
	http_req.addHeader(&buf);
	buf.assign("Connection: close\r\n");
	http_req.addHeader(&buf);
      }
      CCC::HttpFetch http_fetch;
      CCC::HttpFetch::FetchStatus status;
      CCC::HttpResponse* http_res = http_fetch.fetch(&x_uri, &http_req, status);
      if (http_res == 0)
      {
	return 0;
      }
      Allocator parser_src;		/* DOMCharにコンバートした結果 */
      if (!convertToDomCeid(http_res->getBody(), &parser_src))
      {
	return 0;
      }
      ret = new XMLParserSrcUnit(uri,
				 (DOMChar*)parser_src.getMem(),
				 parser_src.getUsedSize() / sizeof(DOMChar));
    }
#else
    {
      mem = new Allocator();
      HttpFetch http_fetch;
      OFlow* dest = log->getDest();	// NOTE: dest is DOMChar base straem
      Spooler spool(IOTYPE_INT8, IOTYPE_UINT16, dest);
      CompositIFilter* cf = Iceman::createCompositIFilter(CEID_USASCII, CCC_DOM_CEID);
      spool.add(cf);
      bool status = http_fetch.fetch(&x_uri, HttpFetch::METHOD_GET, 0, 0, 0, 0, mem, &spool);
      spool.flush();
      delete cf;
      if (!status)
      {
	delete mem;
	mem = 0;
      }
      Allocator parser_src;		/* DOMCharにコンバートした結果 */
      if (!convertToDomCeid(mem, &parser_src))
      {
	return 0;
      }
      ret = new XMLParserSrcUnit(uri,
				 (DOMChar*)parser_src.getMem(),
				 parser_src.getUsedSize() / sizeof(DOMChar));
    }
#endif
    break;
    
   case Uri::SCHEME_FILE:	// file://path
    {
      const char* path = x_uri.getPath();
      if (path)
      {
	FileIFlow file_in;
	if (file_in.open(path))
	{
	  try
	  {
	    ret = readFromIFlow(uri, &file_in);
	  }
	  catch (IOException /* ioe */)
	  {
	    ret = 0;
	  }
	}
      }
    }
    break;

   case Uri::SCHEME_UNKNOWN:	// path
    {
      const char* path = x_uri.getPath();
      if (path)
      {
	const char* x_path = makeSystemFilePath(path);
	FileIFlow file_in;
	if (file_in.open(x_path))
	{
	  try
	  {
	    ret = readFromIFlow(uri, &file_in);
	  }
	  catch (IOException /* ioe */)
	  {
	    ret = 0;
	  }
	}
#ifdef _WIN32_WCE
	delete (void *) x_path;
#else /*_WIN32_WCE */
	delete x_path;
#endif /*_WIN32_WCE */
      }
    }
    break;
    
    // NOT YET SUPPORTED
   default:
   case Uri::SCHEME_FTP:
   case Uri::SCHEME_NNTP:
   case Uri::SCHEME_HTTPS:
   case Uri::SCHEME_WAIS:
   case Uri::SCHEME_CID:
   case Uri::SCHEME_MID:
    break;

    // should be error
   case Uri::SCHEME_DEBUG:
   case Uri::SCHEME_GOPHER:
   case Uri::SCHEME_MAILTO:
   case Uri::SCHEME_NEWS:
   case Uri::SCHEME_TELNET:
   case Uri::SCHEME_PROSPERO:
    ret = 0;
    break;
  }

  delete mem;
  return ret;
}

XMLParserSrcUnit*
XMLReader::readFromPublicIDWithCatalog(DOMString* id)
{
  BString b_id;
  CCC::Iceman::convertToBString(CCC_DOM_CEID, file_system_ceid, id, &b_id);
  const char* path = catalog->getPublicIdPath(b_id.getCString());
  if (!path)
  {
    return 0;
  }
  const char* x_path = makeCatalogRefFilePath(path);
  FileIFlow file_in;
  XMLParserSrcUnit* ret = 0;
  if (file_in.open(x_path))
  {
    try
    {
      ret = readFromIFlow(id, &file_in);
    }
    catch (IOException /* ioe */)
    {
      ret = 0;
    }
  }
#ifdef _WIN32_WCE
  delete (void *) x_path;
#else /*_WIN32_WCE */
  delete x_path;
#endif /*_WIN32_WCE */
  return ret;
}

XMLParserSrcUnit*
XMLReader::readFromSystemIDWithCatalog(DOMString* id)
{
  BString b_id;
  CCC::Iceman::convertToBString(CCC_DOM_CEID, file_system_ceid, id, &b_id);
  const char* path = catalog->getSystemIdPath(b_id.getCString());
  if (!path)
  {
    return 0;
  }
  const char* x_path = makeCatalogRefFilePath(path);
  FileIFlow file_in;
  XMLParserSrcUnit* ret = 0;
  if (file_in.open(x_path))
  {
    try
    {
      ret = readFromIFlow(id, &file_in);
    }
    catch (IOException /* ioe */)
    {
      ret = 0;
    }
  }
#ifdef _WIN32_WCE
  delete (void *) x_path;
#else /*_WIN32_WCE */
  delete x_path;
#endif /*_WIN32_WCE */
  return ret;
}

XMLParserSrcUnit*
XMLReader::readFromPublicID(DOMString* id, DOMString* uri)
{
  XMLParserSrcUnit* ret = 0;
  if (id)
  {
    ret = XMLReader::readFromPublicIDWithCatalog(id);
  }
  if (!ret)
  {
    ret = XMLReader::readFromUri(id);
  }
  return ret;
}

XMLParserSrcUnit*
XMLReader::readFromSystemID(DOMString* id)
{
  XMLParserSrcUnit* ret = 0;
  ret = readFromSystemIDWithCatalog(id);
  if (!ret)
  {
    ret = XMLReader::readFromUri(id);
  }
  return ret;
}

CCC_NAMESPACE_END(CCC);
