﻿// $Id$

#ifndef INCLUDE_ccc_xml_XMLReader_h
#define INCLUDE_ccc_xml_XMLReader_h

#include <ccc/base/base.h>
#include <ccc/base/TextWriter.h>
#include <ccc/iceman/Iceman.h>
#include <ccc/iceman/Detector.h>
#include <ccc/fetch/HttpFetch.h>
#include <ccc/xml/XMLParserSrc.h>
#include <ccc/xml/Catalog.h>

CCC_NAMESPACE_START(CCC);

/*!
 * XMLParserはXMLReaderによって、指定されたデータの読み込み元からデータを取得し、
 * 文字コードを判定し、DOMの内部文字コード(デフォルトUTF-16)に変換し、
 * XMLParserSrcにXMLParserSrcUnitとして読み込みます。
 * パーサは、こうして得られたXMLParserSrcに格納されたデータを読み込みながらパース
 * します。また、パース途中で、外部データを読み込む必要が出てきた場合には、同様に、
 * XMLReaderでデータを取得し、XMLParserSrcにXMLParserSrcUnitとして読み込みます。
 * 別の手段で外部データを読み込むようにカスタマイズする場合には、XMLReaderクラスを
 * 継承し動作を変更するようにします。
 */
class XMLReader
  : public HttpFetch
{
  /*!
   * カタログ
   */
  Catalog* catalog;
  /*!
   * ログ
   */
  TextWriter<DOMChar>* log;
  /*!
   * ディテクタ
   * デフォルトはCEID_XMLDETECTORのディテクタです。
   */
  Detector* detector;
  /*!
   * システム指定ファイルのルートディレクトリ
   * デフォルト値は空文字列です。
   */
  char* system_root;
  /*!
   * ファイルシステムの文字セットエンコーディング
   */
  CeId file_system_ceid;
  /*!
   * カタログによって参照されるパスのルートディレクトリ
   */
  char* cat_ref_root;
 public:
  /*!
   * コンストラクタ
   * \param catalog カタログ
   * \param log ログ、0を指定するとログはどこにも記録されません。
   */
  XMLReader(Catalog* catalog, TextWriter<DOMChar>* log = 0);
  /*!
   * デストラクタ
   */
  ~XMLReader();
  /*!
   * 文字セットエンコーディングの判定に用いるディテクタを指定します。
   * ディテクタが指定されない場合には、CEID_XMLDETECTORのディテクタが
   * 使われます。
   * \param detector ディテクタ
   */
  void setDetector(Detector* detector);
  /*!
   * カタログによって参照されるパスのルートディレクトリを設定します。
   * \param path DTDのルートパス
   */
  void setCatalogRefRootPath(char* path);
  /*!
   * システム指定ファイルのルートディレクトリを設定します。
   * パスのデータをコピーしたものがこのオブジェクト内部で保持されます。
   * \param path パス
   */
  void setSystemRoot(const char* path);
  /*!
   * ファイルシステムの文字セットエンコーディングを設定します。
   * \param ceid ファイルシステムの文字セットエンコーディング
   */
  void setFileSystemCeId(CeId ceid) { file_system_ceid = ceid; }
 protected:
  /*!
   * CCC_DOM_CEIDで指定されているDOMCharの文字セットエンコーディングに
   * 指定されているディテクタを利用してコンバートします。
   * \param from 変換元のイメージ
   * \param to 変換先のイメージ
   * \return 変換できない場合にfalseを返します。
   */
  bool convertToDomCeid(Allocator* from, Allocator* to);
  /*!
   * パスを連結します。
   * \param path1 最初のパス、文字列の最後にはパスの区切り文字がついていないものとします。0を渡すこともできます。
   * \param path2 後につなげるパス
   * \return 連結されたパス
   */
  char* catPath(const char* path1, const char* path2);
  /*!
   * システム指定ファイルのパスにルートディレクトリを加えた文字列を作成します。
   * \param path システム指定ファイルのパス
   * \return パス文字列
   */
  char* makeSystemFilePath(const char* path);
  /*!
   * カタログによって参照されるパスにルートディレクトリを加えた文字列を作成します。
   * \param path カタログによって参照されるパス
   * \return パス文字列
   */
  char* makeCatalogRefFilePath(const char* path);
  /*!
   * Public IDをCatalog経由でデータを読み込みます。
   * \param id Public ID文字列
   */
  virtual XMLParserSrcUnit* readFromPublicIDWithCatalog(DOMString* id);
  /*!
   * SYSTEM IDをCatalog経由でデータを読み込みます。
   * \param id System ID文字列
   */
  virtual XMLParserSrcUnit* readFromSystemIDWithCatalog(DOMString* id);
 public:
  /*!
   * ストリームからXMLデータを読み込みます。
   * \param name フローのURIやパス名などの名称、ログの生成目的で使われます。
   * \param in 読み込み元フロー
   */
  virtual XMLParserSrcUnit* readFromIFlow(const DOMString* name, IFlow* in) CCC_RAISES(IOException);
  /*!
   * ファイルからXMLデータを読み込みます。
   * \param name 名称、ログの生成目的で使われます。
   * \param file 読み込み元ファイルのパス
   */
  virtual XMLParserSrcUnit* readFromFile(const DOMString* name, const char* file) CCC_RAISES(IOException);
#ifdef _WIN32
  /*!
   * ファイルからXMLデータを読み込みます。
   * \param name 名称、ログの生成目的で使われます。
   * \param file 読み込み元ファイルのパス
   */
  virtual XMLParserSrcUnit* readFromFile(const DOMString* name, const wchar_t* file) CCC_RAISES(IOException);
#endif /* _WIN32 */
  /*!
   * URIからデータを読み込みます。
   * \param uri URI
   */
  virtual XMLParserSrcUnit* readFromUri(DOMString* uri);
  /*!
   * Public IDからデータを読み込みます。
   * \param id Public ID文字列, 0が指定された場合にはCatalog経由で読み込まれません。
   * \param uri URI, 0が指定された場合にはURIからデータを取得しません。
   */
  virtual XMLParserSrcUnit* readFromPublicID(DOMString* id, DOMString* uri);
  /*!
   * System IDからデータを読み込みます。
   * System ID文字列にURI以外の相対パスが指定された場合にはsystem_rootに対する相対
   * ファイルが指定されたものとみなします。
   * \param id System ID文字列
   */
  virtual XMLParserSrcUnit* readFromSystemID(DOMString* id);
};

CCC_NAMESPACE_END(CCC);

#endif /* INCLUDE_ccc_xml_XMLReader_h */
