﻿// $Id$

#include <ccc/xml/sgmlentity.h>
#include <ccc/base/cstring.h>
#ifdef TEST_MAIN
#  include <stdio.h>
#  include "ccc/iceman/iceman.h"
#endif /* TEST_MAIN */

CCC_NAMESPACE_START(CCC);

static CCC::UInt16 str_lt_src[] = { 0x6c /* l */, 0x74 /* t */, 0x00, };
static CCC::UInt16 str_lt_dest[] = { 0x3c /* < */, 0x00 };
 
static CCC::UInt16 str_gt_src[] = { 0x67 /* g */, 0x74 /* t */, 0x00, };
static CCC::UInt16 str_gt_dest[] = { 0x3e /* > */, 0x00, };

static CCC::UInt16 str_amp_src[] = { 0x61 /* a */, 0x6d /* m */, 0x70 /* p */, 0x00, };
static CCC::UInt16 str_amp_dest[] = { 0x26 /* & */, 0x00, };

static CCC::UInt16 str_apos_src[] = { 0x61 /* a */, 0x70 /* p */, 0x6f /* o */, 0x73 /* s */, 0x00, };
static CCC::UInt16 str_apos_dest[] = { 0x27 /* \' */, 0x00, };

static CCC::UInt16 str_quot_src[] = { 0x71 /* q */, 0x75 /* u */, 0x6f /* o */, 0x74 /* t */, 0x00, };
static CCC::UInt16 str_quot_dest[] = { 0x22 /* " */, 0x00, };

EntityRef basic_entities[] =
{
  { str_lt_src, str_lt_dest },
  { str_gt_src, str_gt_dest },
  { str_amp_src, str_amp_dest },
  { str_apos_src, str_apos_dest },
  { str_quot_src, str_quot_dest },
  { 0, 0 },
};

// ------------------------------------------------------------------------
// class DereferSgmlEntity

/*!
 * \param src 比較元文字列
 * \param candidate 実体参照パターン
 * \retval true 合致
 * \retval false 合致しません
 */
static bool
matchEntity(CCC::UInt16* src, CCC::UInt16* candidate)
{
  CCC::Size len = CCC::strLen(candidate);
  while (len > 0)
  {
    if (*src++ != *candidate++)
    {
      return false;
    }
    len--;
  }
  return (*src == ';');
}

/*!
 * 10進数でエンコードされた数値をデコードします。
 * エラーの場合には0xffffが返ります。
 * \param src エンコードされた文字列数値(&#の直後の文字列)
 * \return デコードされた値
 */
static CCC::UInt16
decodeDecimal(CCC::UInt16* src)
{
  CCC::UInt16 value = 0;
  while (*src != ';')
  {
    CCC::UInt16 c = *src;
    switch (c)
    {
     case '0':
     case '1':
     case '2':
     case '3':
     case '4':
     case '5':
     case '6':
     case '7':
     case '8':
     case '9':
      value = value * 10 + (CCC::UInt16)(c - '0');
      break;

     default:
      // ERROR
      return 0xffff;
    }
    src++;
  }
  return value;
}

#ifdef TEST_MAIN
static int total_fail = 0;
static int total_test = 0;

#define CHECK_EQ(a, b) \
{\
  total_test++;\
  if ((a) != (b))\
  {\
    total_fail++;\
    fprintf(stderr, "%s:%d CHECK ERROR! a:0x%x b:0x%x\n", __FILE__, __LINE__, a, b); \
  }\
}

static void
decodeDecimal_Test()
{
  struct TestPattern
  {
    CCC::UInt16* pattern;
    CCC::UInt16 expected_value;
  };
  static CCC::UInt16 tests_str_0[] = { '0', ';' , 0 };
  static CCC::UInt16 tests_str_1[] = { '1', ';', 'x', 'y', 'z', 0 };
  static CCC::UInt16 tests_str_2[] = { '2', '2', '9', ';', 0 };
  static CCC::UInt16 tests_str_3[] = { '1', '0', '4', '8', ';', 0 };
  static CCC::UInt16 tests_str_4[] = { '6', '5', ';', 'B', 'C', 0 };
  // error case
  static CCC::UInt16 tests_str_5[] = { '0', '1', 0 };
  static CCC::UInt16 tests_str_6[] = { 'A', 'B', 'C', 0 };
  static CCC::UInt16 tests_str_7[] = { '6', 'A', ';', 'B', 'C', 0 };
  static TestPattern tests[] =
  {
    { tests_str_0, 0 },
    { tests_str_1, 1 },
    { tests_str_2, 229 },
    { tests_str_3, 1048 },
    { tests_str_4, 'A' },
    // ERROR case
    { tests_str_5, 0xffff },
    { tests_str_6, 0xffff },
    { tests_str_7, 0xffff },
    { 0, 0 },
  };
  TestPattern* x = tests;
  while (x->pattern)
  {
    CCC::UInt16 c = decodeDecimal(x->pattern);
    CHECK_EQ(c, x->expected_value)
    x++;
  }
}
#endif /* TEST_MAIN */

/*!
 * 16進数でエンコードされた数値をデコードします。
 * エラーの場合には0xffffが返ります。
 * \param src エンコードされた文字列数値(&#xもしくは&#X直後の文字列)
 * \return デコードされた値
 */
static CCC::UInt16
decodeHex(CCC::UInt16* src)
{
  CCC::UInt16 value = 0;
  while (*src != ';')
  {
    CCC::UInt16 c = *src;
    switch (c)
    {
     case '0':
     case '1':
     case '2':
     case '3':
     case '4':
     case '5':
     case '6':
     case '7':
     case '8':
     case '9':
      value = value * 16 + (CCC::UInt16)(c - '0');
      break;

     case 'A':
     case 'B':
     case 'C':
     case 'D':
     case 'E':
     case 'F':
      value = value * 16 + (CCC::UInt16)(c - 'A' + 10);
      break;

     case 'a':
     case 'b':
     case 'c':
     case 'd':
     case 'e':
     case 'f':
      value = value * 16 + (CCC::UInt16)(c - 'a' + 10);
      break;

     default:
      // ERROR
      return 0xffff;
    }
    src++;
  }
  return value;
}

#ifdef TEST_MAIN
static void
decodeHex_Test()
{
  struct TestPattern
  {
    CCC::UInt16* pattern;
    CCC::UInt16 expected_value;
  };
  static CCC::UInt16 tests_str_0[] = { '0', ';' , 0 };
  static CCC::UInt16 tests_str_1[] = { '1', ';', 'x', 'y', 'z', 0 };
  static CCC::UInt16 tests_str_2[] = { '2', '2', '9', ';', 0 };
  static CCC::UInt16 tests_str_3[] = { '1', 'A', 'B', 'C', ';', 0 };
  static CCC::UInt16 tests_str_4[] = { '1', 'D', 'E', 'F', ';', 0 };
  static CCC::UInt16 tests_str_5[] = { '1', 'a', 'b', 'c', ';', 0 };
  static CCC::UInt16 tests_str_6[] = { '1', 'd', 'e', 'f', ';', 0 };
  static CCC::UInt16 tests_str_7[] = { '4', '1', ';', 'B', 'C', 0 };
  // ERROR case
  static CCC::UInt16 tests_str_10[] = { '0', '1', 0 };
  static CCC::UInt16 tests_str_11[] = { 'G', 'H', 'I', 0 };
  static CCC::UInt16 tests_str_12[] = { '6', '.', ';', 'B', 'C', 0 };
  static TestPattern tests[] =
  {
    { tests_str_0, 0 },
    { tests_str_1, 1 },
    { tests_str_2, 0x229 },
    { tests_str_3, 0x1abc },
    { tests_str_4, 0x1def },
    { tests_str_5, 0x1abc },
    { tests_str_6, 0x1def },
    { tests_str_7, 'A' },
    // ERROR case
    { tests_str_10, 0xffff },
    { tests_str_11, 0xffff },
    { tests_str_12, 0xffff },
    { 0, 0 },
  };
  TestPattern* x = tests;
  while (x->pattern)
  {
    CCC::UInt16 c = decodeHex(x->pattern);
    CHECK_EQ(c, x->expected_value)
    x++;
  }
}
#endif /* TEST_MAIN */

void 
DereferSgmlEntity::convert(const CCC::DString* from, CCC::DString* to, EntityRef* additional_refs)
{
  CCC::UInt16* s = from->getCString();
  CCC::UInt16 c;
  while ((c = *s) != '\0')
  {
    if (c == '&')
    {
      if (*(s + 1) == '#')
      {
	// Numbrer
	if ((*(s + 2) == 'x') ||
	    (*(s + 2) == 'X'))
	{
	  // HEX:
	  // The syntax "&#xH;" or "&#XH;", where H is a hexadecimal number, refers 
	  // to the ISO 10646 hexadecimal character number H. Hexadecimal numbers in
	  // numeric character references are case-insensitive.
	  c = decodeHex(s + 3);
	  if (c != 0xffff)
	  {
	    to->add(c);
	    while (*s != ';')
	    {
	      s++;
	    }
	    s++;
	  }
	  else
	  {
	    // ERROR
	    s++;
	  }
	}
	else
	{
	  // Decimal:
	  // The syntax "&#D;", where D is a decimal number, refers to the ISO 10646
	  // decimal character number D.
	  c = decodeDecimal(s + 2);
	  if (c != 0xffff)
	  {
	    to->add(c);
	    while (*s != ';')
	    {
	      s++;
	    }
	    s++;
	  }
	  else
	  {
	    // ERROR
	    s++;
	  }
	}
      }
      else
      {
	int i;
	for (i = 0; i < 3; i++)
	{
	  if (i == 2)
	  {
	    // can't find entity.
	    s++;
	    break;
	  }
	  EntityRef* x = i == 0 ? basic_entities : additional_refs;
	  if (x)
	  {
	    while (x->ref != 0)
	    {
	      if (matchEntity(s + 1, x->ref))
	      {
		CCC::Size len = CCC::strLen(x->ref);
		s += len + 2;
		to->add(x->entity);
		break;
	      }
	      x++;
	    }
	    if (x->ref)
	    {
	      // find
	      break;
	    }
	  }
	}
      }
    }
    else
    {
      to->add(c);
      s++;
    }
  }
}

CCC_NAMESPACE_END(CCC);

#ifdef TEST_MAIN
class DereferSgmlEntityTest
{
 public:
  /*!
   * DereferSgmlEntity::convertの単体テスト
   */
  static void convertTest()
  {
    static CCC::UInt16 test_str_x[] = { 'F', 'O', 'O', 0 };
    static CCC::UInt16 test_str_0[] = { 'A', 'B', '&', 'a', 'm', 'p', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_0a[] = { 'A', 'B', '&', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_1[] = { 'A', 'B', '&', 'a', 'm', 'p', ';', 0 };
    static CCC::UInt16 test_str_1a[] = { 'A', 'B', '&', 0 };
    static CCC::UInt16 test_str_2[] = { '&', 'a', 'm', 'p', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_2a[] = { '&', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_3[] = { 'A', '&', 'l', 't', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_3a[] = { 'A', '<', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_4[] = { 'A', '&', 'g', 't', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_4a[] = { 'A', '>', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_5[] = { 'A', '&', 'a', 'p', 'o', 's', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_5a[] = { 'A', '\'', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_6[] = { 'A', '&', 'q', 'u', 'o', 't', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_6a[] = { 'A', '"', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_7[] = { 'A', '&', '#', '6', '5', ';', '.', 0 };
    static CCC::UInt16 test_str_7a[] = { 'A', 'A', '.', 0 };
    static CCC::UInt16 test_str_8[] = { 'A', '&', '#', 'x', '4', '1', ';', '.', 0 };
    static CCC::UInt16 test_str_8a[] = { 'A', 'A', '.', 0 };
    static CCC::UInt16 test_str_9[] = { 'A', '&', '#', 'X', '4', '1', ';', '.', 0 };
    static CCC::UInt16 test_str_9a[] = { 'A', 'A', '.', 0 };
    // ERROR case
    static CCC::UInt16 test_str_100[] = { 'A', '&', 'a', 'b', 'c', 'd', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_100a[] = { 'A', 'a', 'b', 'c', 'd', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_101[] = { 'A', '&', 'l', 't', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_101a[] = { 'A', 'l', 't', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_102[] = { 'A', '&', '#', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_102a[] = { 'A', '#', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_103[] = { 'A', '&', '#', 'x', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_103a[] = { 'A', '#', 'x', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };

    struct TestPat
    {
      /*! 変換元 */
      CCC::UInt16* from;
      /*! 変換後として期待される文字列 */
      CCC::UInt16* to;
    };
    static TestPat patterns[] =
    {
      { test_str_x, test_str_x },
      { test_str_0, test_str_0a },
      { test_str_1, test_str_1a },
      { test_str_2, test_str_2a },
      { test_str_3, test_str_3a },
      { test_str_4, test_str_4a },
      { test_str_5, test_str_5a },
      { test_str_6, test_str_6a },
      { test_str_7, test_str_7a },
      { test_str_8, test_str_8a },
      { test_str_9, test_str_9a },
      // ERROR case
      { test_str_100, test_str_100a },
      { test_str_101, test_str_101a },
      { test_str_102, test_str_102a },
      { test_str_103, test_str_103a },
      { 0, 0 },
    };

    TestPat* x = patterns;
    while (x->from != 0)
    {
      CCC::DString from(x->from);
      CCC::DString to_a(x->to);
      CCC::DString to;
      DereferSgmlEntity::convert(&from, &to, 0);
      {
	total_test++;
	if (to_a.strCmp(to) != 0)
	{
	  total_fail++;
	  CCC::BString* a = CCC::Iceman::convertToBString(CCC::CEID_UTF16N, CCC::CEID_USASCII, &to);
	  CCC::BString* b = CCC::Iceman::convertToBString(CCC::CEID_UTF16N, CCC::CEID_USASCII, &to_a);
	  fprintf(stderr, "%s:%d CHECK ERROR! value:%s expected:%s\n", __FILE__, __LINE__, a->getCString(), b->getCString());
	  delete a;
	  delete b;
	}
      }
      x++;
    }
  }
  /*!
   * DereferSgmlEntity::convertの単体テスト
   * 追加の実体参照テーブル付きのテスト
   */
  static void convertTest_2()
  {
    static CCC::UInt16 str_abcd_src[] = { 'a', 'b', 'c', 'd', 0x00, };
    static CCC::UInt16 str_abcd_dest[] = { 0x3000, 'A', '-', 'A', 0x00 };
    static CCC::UInt16 str_efgh_src[] = { 'E', 'f', 'g', 'h', 0x00, };
    static CCC::UInt16 str_efgh_dest[] = { '(', '^', '_', '^', ')', 0x00, };
    EntityRef additional_entities[] =
    {
      { str_abcd_src, str_abcd_dest },
      { str_efgh_src, str_efgh_dest },
      { 0, 0 },
    };

    static CCC::UInt16 test_str_x[] = { 'F', 'O', 'O', 0 };
    static CCC::UInt16 test_str_0[] = { 'A', 'B', '&', 'a', 'm', 'p', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_0a[] = { 'A', 'B', '&', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_1[] = { 'A', 'B', '&', 'a', 'm', 'p', ';', 0 };
    static CCC::UInt16 test_str_1a[] = { 'A', 'B', '&', 0 };
    static CCC::UInt16 test_str_2[] = { '&', 'a', 'm', 'p', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_2a[] = { '&', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_3[] = { 'A', '&', 'l', 't', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_3a[] = { 'A', '<', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_4[] = { 'A', '&', 'g', 't', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_4a[] = { 'A', '>', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_5[] = { 'A', '&', 'a', 'p', 'o', 's', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_5a[] = { 'A', '\'', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_6[] = { 'A', '&', 'q', 'u', 'o', 't', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_6a[] = { 'A', '"', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_7[] = { 'A', '&', '#', '6', '5', ';', '.', 0 };
    static CCC::UInt16 test_str_7a[] = { 'A', 'A', '.', 0 };
    static CCC::UInt16 test_str_8[] = { 'A', '&', '#', 'x', '4', '1', ';', '.', 0 };
    static CCC::UInt16 test_str_8a[] = { 'A', 'A', '.', 0 };
    static CCC::UInt16 test_str_9[] = { 'A', '&', '#', 'X', '4', '1', ';', '.', 0 };
    static CCC::UInt16 test_str_9a[] = { 'A', 'A', '.', 0 };
    static CCC::UInt16 test_str_10[] = { 'A', '&', 'a', 'b', 'c', 'd', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_10a[] = { 'A', 0x3000, 'A', '-', 'A', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_11[] = { 'A', '&', 'E', 'f', 'g', 'h', ';', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_11a[] = { 'A', '(', '^', '_', '^', ')', '0', '1', '.', 0 };
    // ERROR case
    static CCC::UInt16 test_str_101[] = { 'A', '&', 'l', 't', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_101a[] = { 'A', 'l', 't', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_102[] = { 'A', '&', '#', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_102a[] = { 'A', '#', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_103[] = { 'A', '&', '#', 'x', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };
    static CCC::UInt16 test_str_103a[] = { 'A', '#', 'x', 'Y', ';', 'c', 'd', '0', '1', '.', 0 };

    struct TestPat
    {
      /*! 変換元 */
      CCC::UInt16* from;
      /*! 変換後として期待される文字列 */
      CCC::UInt16* to;
    };
    static TestPat patterns[] =
    {
      { test_str_x, test_str_x },
      { test_str_0, test_str_0a },
      { test_str_1, test_str_1a },
      { test_str_2, test_str_2a },
      { test_str_3, test_str_3a },
      { test_str_4, test_str_4a },
      { test_str_5, test_str_5a },
      { test_str_6, test_str_6a },
      { test_str_7, test_str_7a },
      { test_str_8, test_str_8a },
      { test_str_9, test_str_9a },
      { test_str_10, test_str_10a },
      { test_str_11, test_str_11a },
      // ERROR case
      { test_str_101, test_str_101a },
      { test_str_102, test_str_102a },
      { test_str_103, test_str_103a },
      { 0, 0 },
    };

    TestPat* x = patterns;
    while (x->from != 0)
    {
      CCC::DString from(x->from);
      CCC::DString to_a(x->to);
      CCC::DString to;
      DereferSgmlEntity::convert(&from, &to, additional_entities);
      {
	total_test++;
	if (to_a.strCmp(to) != 0)
	{
	  total_fail++;
	  CCC::BString* a = CCC::Iceman::convertToBString(CCC::CEID_UTF16N, CCC::CEID_USASCII, &to);
	  CCC::BString* b = CCC::Iceman::convertToBString(CCC::CEID_UTF16N, CCC::CEID_USASCII, &to_a);
	  fprintf(stderr, "%s:%d CHECK ERROR! value:%s expected:%s\n", __FILE__, __LINE__, a->getCString(), b->getCString());
	  delete a;
	  delete b;
	}
      }
      x++;
    }
  }
 
  static void test()
  {
    convertTest();
    convertTest_2();
  }
};
#endif /* TEST_MAIN */

// ------------------------------------------------------------------------
// class ReferSgmlEntity

//void
//ReferSgmlEntity::convert(const CCC::DString* from, CCC::DString* to, EntityRef* additional_refs)
//{
//}

#ifdef TEST_MAIN
int
main()
{
  CCC::Ccc::initialize();
  decodeDecimal_Test();
  decodeHex_Test();
  DereferSgmlEntityTest::test();

  printf("total fail:%d\n", total_fail);
  printf("total tasts:%d\n", total_test);
  CCC::Ccc::unInitialize();
  return (total_fail == 0) ? 0 : -1;
}
#endif /* TEST_MAIN */
