﻿#line 1 "MecabIf.upp"
/* -*- coding: utf-8 -*- */

#include <iostream>
#include <memory>
#include <stdio.h>
#include "MecabIf.h"

struct TextSegment
{
  CCC::WString oyamoji;
  CCC::WString ruby;
  bool hiragana_p;
};

struct ReplaceType
{
  const char* src;
  const char* dst;
};

MecabIf::MecabIf()
{
  tagger = 0;
  wakachi_p = true;
  jyoshi_wakachi_p = false;
  mitumura_p = false;
  ruby_out_p = true;
}

bool
MecabIf::initMecab(const char* mecabrc_path)
{
  if (mecabrc_path)
  {
    const char* mecab_option[] =
      {
        "--rcfile",
        mecabrc_path,
        nullptr
      };
    tagger = MeCab::createTagger(2, (char**) mecab_option);
  }
  else
  {
    tagger = MeCab::createTagger("");
  }
  return (tagger != nullptr);
}

MecabIf::~MecabIf()
{
  delete tagger;
}

void
MecabIf::convertUtf8FromKatakanaToHiragana(const CCC::WString* katakana, CCC::WString* hiragana)
{
  struct Pair
  {
    CCC::UInt32 from;
    CCC::UInt32 to;
  };
  const int tbl_size = 86;
  static Pair tbl[tbl_size] =
    {
      {0x30a1, 0x3041},        // ァ
      {0x30a2, 0x3042},        // ア
      {0x30a3, 0x3043},        // ィ
      {0x30a4, 0x3044},        // イ
      {0x30a5, 0x3045},        // ゥ
      {0x30a6, 0x3046},        // ウ
      {0x30a7, 0x3047},        // ェ
      {0x30a8, 0x3048},        // エ
      {0x30a9, 0x3049},        // ォ
      {0x30aa, 0x304a},        // オ
      {0x30ab, 0x304b},        // カ
      {0x30ac, 0x304c},        // ガ
      {0x30ad, 0x304d},        // キ
      {0x30ae, 0x304e},        // ギ
      {0x30af, 0x304f},        // ク
      {0x30b0, 0x3050},        // グ
      {0x30b1, 0x3051},        // ケ
      {0x30b2, 0x3052},        // ゲ
      {0x30b3, 0x3053},        // コ
      {0x30b4, 0x3054},        // ゴ
      {0x30b5, 0x3055},        // サ
      {0x30b6, 0x3056},        // ザ
      {0x30b7, 0x3057},        // シ
      {0x30b8, 0x3058},        // ジ
      {0x30b9, 0x3059},        // ス
      {0x30ba, 0x305a},        // ズ
      {0x30bb, 0x305b},        // セ
      {0x30bc, 0x305c},        // ゼ
      {0x30bd, 0x305d},        // ソ
      {0x30be, 0x305e},        // ゾ
      {0x30bf, 0x305f},        // タ
      {0x30c0, 0x3060},        // ダ
      {0x30c1, 0x3061},        // チ
      {0x30c2, 0x3062},        // ヂ
      {0x30c3, 0x3063},        // ッ
      {0x30c4, 0x3064},        // ツ
      {0x30c5, 0x3065},        // ヅ
      {0x30c6, 0x3066},        // テ
      {0x30c7, 0x3067},        // デ
      {0x30c8, 0x3068},        // ト
      {0x30c9, 0x3069},        // ド
      {0x30ca, 0x306a},        // ナ
      {0x30cb, 0x306b},        // ニ
      {0x30cc, 0x306c},        // ヌ
      {0x30cd, 0x306d},        // ネ
      {0x30ce, 0x306e},        // ノ
      {0x30cf, 0x306f},        // ハ
      {0x30d0, 0x3070},        // バ
      {0x30d1, 0x3071},        // パ
      {0x30d2, 0x3072},        // ヒ
      {0x30d3, 0x3073},        // ビ
      {0x30d4, 0x3074},        // ピ
      {0x30d5, 0x3075},        // フ
      {0x30d6, 0x3076},        // ブ
      {0x30d7, 0x3077},        // プ
      {0x30d8, 0x3078},        // ヘ
      {0x30d9, 0x3079},        // ベ
      {0x30da, 0x307a},        // ペ
      {0x30db, 0x307b},        // ホ
      {0x30dc, 0x307c},        // ボ
      {0x30dd, 0x307d},        // ポ
      {0x30de, 0x307e},        // マ
      {0x30df, 0x307f},        // ミ
      {0x30e0, 0x3080},        // ム
      {0x30e1, 0x3081},        // メ
      {0x30e2, 0x3082},        // モ
      {0x30e3, 0x3083},        // ャ
      {0x30e4, 0x3084},        // ヤ
      {0x30e5, 0x3085},        // ュ
      {0x30e6, 0x3086},        // ユ
      {0x30e7, 0x3087},        // ョ
      {0x30e8, 0x3088},        // ヨ
      {0x30e9, 0x3089},        // ラ
      {0x30ea, 0x308a},        // リ
      {0x30eb, 0x308b},        // ル
      {0x30ec, 0x308c},        // レ
      {0x30ed, 0x308d},        // ロ
      {0x30ee, 0x308e},        // ヮ
      {0x30ef, 0x308f},        // ワ
      {0x30f0, 0x3090},        // ヰ
      {0x30f1, 0x3091},        // ヱ
      {0x30f2, 0x3092},        // ヲ
      {0x30f3, 0x3093},        // ン
      {0x30f4, 0x3094},        // ヴ
      {0x30f5, 0x3095},        // ヵ
      {0x30f6, 0x3096},        // ヶ
      // { 0x30f7, 0x },	// ヷ : 平仮名は未定義なので変換しない
      // { 0x30f8, 0x },	// ヸ : 平仮名は未定義なので変換しない
      // { 0x30f9, 0x },	// ヹ : 平仮名は未定義なので変換しない
      // { 0x30fa, 0x },	// ヺ : 平仮名は未定義なので変換しない
    };
  CCC::UInt32* x = katakana->getCString();
  CCC::UInt32 c;
  while ((c = *x++) != '\0')
  {
    int n;
    for (n = 0; n < tbl_size; n++)
    {
      if (tbl[n].from == c)
      {
        c = tbl[n].to;
        break;
      }
    }
    hiragana->add(c);
  }
}

void
MecabIf::convertUtf8FromKatakanaToHiragana(const CCC::BString* katakana, CCC::BString* hiragana)
{
  CCC::WString ucs4_katakana;
  CCC::Iceman::convertToWString(CCC::CEID_UTF8N, CCC::CEID_UCS4, katakana, &ucs4_katakana);
  CCC::WString ucs4_hiragana;
  convertUtf8FromKatakanaToHiragana(&ucs4_katakana, &ucs4_hiragana);
  CCC::Iceman::convertToBString(CCC::CEID_UCS4, CCC::CEID_UTF8N, &ucs4_hiragana, hiragana);
}

void
MecabIf::getItemFromMecabFeature(const char* feature, int column_number, CCC::BString* value)
{
  // feature 例:
  // 形容詞,自立,*,*,形容詞・アウオ段,基本形,良い,ヨイ,ヨイ
  //                                              ^^^^
  // 名詞,一般,*,*,*,*,木の実,コノミ,コノミ
  //                          ^^^^^^
  // 名詞,副詞可能,*,*,*,*,今日,キョウ,キョー
  //                            ^^^^^^
  value->clear();
  const char* x = feature;
  char c;
  int column = 0;
  while ((c = *x++) != '\0')
  {
    if (c == ',')
    {
      column++;
      if (column >= column_number + 1)
      {
        break;
      }
    }
    else
    {
      if (column == column_number)
      {
        value->add(c);
      }
    }
  }
}

void
MecabIf::getHinshiFromMecabFeature(const char* feature, CCC::BString* hinshi)
{
  // feature 例:
  // 形容詞,自立,*,*,形容詞・アウオ段,基本形,良い,ヨイ,ヨイ
  // ^^^^^^
  // 名詞,一般,*,*,*,*,木の実,コノミ,コノミ
  // ^^^^
  // 名詞,副詞可能,*,*,*,*,今日,キョウ,キョー
  // ^^^^
  getItemFromMecabFeature(feature, 0, hinshi);
}

void
MecabIf::getYomiFromMecabFeature(const char* feature, CCC::BString* yomi)
{
  // feature 例:
  // 形容詞,自立,*,*,形容詞・アウオ段,基本形,良い,ヨイ,ヨイ
  //                                              ^^^^
  // 名詞,一般,*,*,*,*,木の実,コノミ,コノミ
  //                          ^^^^^^
  // 名詞,副詞可能,*,*,*,*,今日,キョウ,キョー
  //                            ^^^^^^
  getItemFromMecabFeature(feature, 7, yomi);
}

// UTF-8の単語が漢字を含むかどうかを調べます。
bool
MecabIf::hasKanji(const CCC::BString* word)
{
  CCC::WString ucs4;
  CCC::Iceman::convertToWString(CCC::CEID_UTF8N, CCC::CEID_UCS4, word, &ucs4);
  CCC::UInt32 c;
  CCC::UInt32* x = ucs4.getCString();
  while ((c = *x++) != '\0')
  {
    // Unicodeの漢字の範囲: 以下のURL参照
    // https://tama-san.com/kanji-regex/
    // 4E00..9FFF CJK統合漢字
    if ((c >= 0x4e00) && (c <= 0x9fff))
    {
      return true;
    }
    // 3005 々（漢字の踊り字）
    // 3007 〇（漢数字のゼロ）
    // 303B 〻（漢字の踊り字）
    if ((c == 0x3005) ||
      (c == 0x3007) ||
      (c == 0x303b))
    {
      return true;
    }
    // 3400..4DBF CJK統合漢字拡張A
    if ((c >= 0x3400) && (c <= 0x4dbf))
    {
      return true;
    }
    // 2E80..2FDF CJK部首補助＋康熙部首
    if ((c >= 0x2e80) && (c <= 0x2fdf))
    {
      return true;
    }
    // F900..FAFF CJK互換漢字
    if ((c >= 0xf900) && (c <= 0xfaff))
    {
      return true;
    }
    // 20000..2FFFF CJK統合漢字拡張B〜F＋CJK互換漢字追加＋念のためU+2FFFFまで
    if ((c >= 20000) && (c <= 0x2ffff))
    {
      return true;
    }
  }
  return false;
}

#if 0
inline bool isHiragana(CCC::UInt32 c)
{
  if ((c >= 0x3041) &&
      (c <= 0x3096))
  {
    return true;
  }
  return false;
}

struct TextSegment
{
  CCC::WString oyamoji;
  CCC::WString ruby;
  bool hiragana_p;
};
#endif

void
MecabIf::dumpTextSegmentListItems(CCC::ArrayList<TextSegment>* list)
{
  CCC::Size num = list->number();
  CCC::Size n;
  for (n = 0; n < num; n++)
  {
    TextSegment* ts = list->vector(n);
    CCC::Utf8String oyamoji;
    CCC::Utf8String ruby;
    CCC::Iceman::convertToBString(CCC::CEID_UCS4, CCC::CEID_UTF8N, &ts->oyamoji, &oyamoji);
    CCC::Iceman::convertToBString(CCC::CEID_UCS4, CCC::CEID_UTF8N, &ts->ruby, &ruby);
    printf("oyamoji:[%s] ruby[%s] hiraganga_p:%s\n",
           oyamoji.getCString(),
           ruby.getCString(),
           ts->hiragana_p ? "true" : "false");
  }
}

void
MecabIf::outputTextSegmentListItems(CCC::ArrayList<TextSegment>* list, CCC::BString* out)
{
  CCC::Size num = list->number();
  CCC::Size n;
  for (n = 0; n < num; n++)
  {
    TextSegment* ts = list->vector(n);
    CCC::Utf8String oyamoji;
    CCC::Utf8String ruby;
    CCC::Iceman::convertToBString(CCC::CEID_UCS4, CCC::CEID_UTF8N, &ts->oyamoji, &oyamoji);
    CCC::Iceman::convertToBString(CCC::CEID_UCS4, CCC::CEID_UTF8N, &ts->ruby, &ruby);
    //printf("oyamoji:[%s] ruby[%s] hiraganga_p:%s\n",
    //	   oyamoji.getCString(),
    //	   ruby.getCString(),
    //	   ts->hiragana_p ? "true" : "false");
    if ((!ruby_out_p) || ts->hiragana_p)
    {
      out->add(oyamoji);
    }
    else
    {
      out->add("\xef" "\xbd" "\x9c" "");
      out->add(oyamoji);
      out->add("\xe3" "\x80" "\x8a" "");
      out->add(ruby);
      out->add("\xe3" "\x80" "\x8b" "");
    }
  }
}

void
MecabIf::createTextSegmentListItems(const CCC::BString* oyamoji, CCC::ArrayList<TextSegment>* list)
{
  // listを作成
  CCC::WString ucs4_oyamoji;
  CCC::Iceman::convertToWString(CCC::CEID_UTF8N, CCC::CEID_UCS4, oyamoji, &ucs4_oyamoji);
  CCC::UInt32* x = ucs4_oyamoji.getCString();
  CCC::UInt32 c;
  TextSegment* ts = nullptr;
  while ((c = *x++) != '\0')
  {
    bool hiragana_p = isHiragana(c);
    if ((ts == 0) || (ts->hiragana_p != hiragana_p))
    {
      ts = new TextSegment();
      list->add(ts);
      ts->oyamoji.add(c);
      ts->hiragana_p = hiragana_p;
    }
    else
    {
      ts->oyamoji.add(c);
    }
  }
}

void
MecabIf::setRubyToTestSegmentListItems(CCC::ArrayList<TextSegment>* list, CCC::BString* ruby)
{
  CCC::WString ucs4_ruby;
  CCC::Iceman::convertToWString(CCC::CEID_UTF8N, CCC::CEID_UCS4, ruby, &ucs4_ruby);
  CCC::Size start_pos = 0;
  CCC::Size end_pos = (list->number() > 0) ? (list->number() - 1) : 0;
  // listの頭が平仮名の場合、rubyから該当部分を削除
  TextSegment* head = list->head();
  //head->ruby.assign(ucs4_ruby);
  if (head->hiragana_p)
  {
    start_pos++;
    CCC::WSubString head_ruby(ucs4_ruby, 0, head->oyamoji.getLength());
    //CCC::WString head_ruby(ucs4_ruby.getCString(), head->oyamoji.getLength());
    if (head->oyamoji.strCmp(head_ruby) == 0)
    {
      ucs4_ruby.remove(0, head->oyamoji.getLength());
      head->ruby.assign(head->oyamoji);
    }
    else
    {
      // 親文字に平仮名が入っているが、ルビの読みと合わない例外ケース
      // 単にスキップ
      ;
    }
  }
  // listの末が平仮名の場合、rubyから該当部分を削除
  TextSegment* tail = list->tail();
  if (tail->hiragana_p)
  {
    end_pos--;
    CCC::WSubString tail_ruby(ucs4_ruby,
                              ucs4_ruby.getLength() - tail->oyamoji.getLength());
    if (tail->oyamoji.strCmp(tail_ruby) == 0)
    {
      ucs4_ruby.remove(ucs4_ruby.getLength() - tail->oyamoji.getLength(),
                       tail->oyamoji.getLength());
      tail->ruby.assign(tail->oyamoji);
    }
    else
    {
      // 親文字に平仮名が入っているが、ルビの読みと合わない例外ケース
      // 単にスキップ
      ;
    }
  }
  if (start_pos == end_pos)
  {
    // セグメントが一つなので、ルビをすべてこのセグメントに設定
    TextSegment* x = list->vector(start_pos);
    x->ruby.assign(ucs4_ruby);
  }
  else
  {
    // 残りのルビをリストに割り振ります。
    CCC::Size n = start_pos;
    for (n = start_pos; n <= end_pos; n++)
    {
      TextSegment* ts = list->vector(n);
      //printf("n:%lu hiragana_p:%s\n", n, ts->hiragana_p ? "true" : "false");
      if (ts->hiragana_p)
      {
        // ucs4_rubyから親文字と合致した長さを削除し、ルビに割り当てます。
        ucs4_ruby.remove(0, ts->oyamoji.getLength());
        ts->ruby.assign(ts->oyamoji);
      }
      else
      {
        // 次の平仮名のTextSegmentを取り出します。
        if (n + 1 <= end_pos)
        {
          TextSegment* next_ts = list->vector(n + 1);
          // 存在する場合は、ucs4_rubyを検索し、その直前までの文字列をこのセグメントに割り当てます。
          assert(next_ts->hiragana_p == true);
          if (next_ts->hiragana_p == true)
          {
            CCC::Size pos = ucs4_ruby.simpleMatch(next_ts->oyamoji);
            if (pos != 0)
            {
              ts->ruby.assign(ucs4_ruby.getCString(), pos - 1);
              ucs4_ruby.remove(0, pos - 1);
            }
          }
        }
        else
        {
          // 存在しないので、残りの全てのucs4_rubyをこのセグメントに割り当てます。
          ts->ruby.assign(ucs4_ruby);
          ucs4_ruby.clear();
        }
      }
    }
  }
}

void
MecabIf::outputToRuby(CCC::BString* oyamoji, CCC::BString* ruby, CCC::BString* out)
{
  CCC::BString hiragana_ruby;
  convertUtf8FromKatakanaToHiragana(ruby, &hiragana_ruby);
#if 1
  CCC::ArrayList<TextSegment> list;
  createTextSegmentListItems(oyamoji, &list);
  setRubyToTestSegmentListItems(&list, &hiragana_ruby);
  outputTextSegmentListItems(&list, out);
#else
  // ひとまず"お金", "木の実", "怒っ" の問題は後回し
  out->add("\xef" "\xbd" "\x9c" "");
  out->add(oyamoji);
  out->add("\xe3" "\x80" "\x8a" "");
  out->add(hiragana_ruby);
  out->add("\xe3" "\x80" "\x8b" "");
#endif
}

void
MecabIf::analizeByMecab(const CCC::BString* line, CCC::BString* out)
{
  const MeCab::Node* node = tagger->parseToNode(line->getCString());
  while (node != nullptr)
  {
    if (mitumura_p)
    {
      wakachiMitumura(out, node);
    }
    else
    {
      wakachiNormal(out, node);
    }
    CCC::BString x = rubyOut(out, node);
#if 0
    printf("[%s]\t%s\n", x.getCString(), node->feature);
    //std::cout << "id        : " << node->id        << std::endl;
    //std::cout << "surface   : " << node->surface   << std::endl;
    //std::cout << "feature   : " << node->feature   << std::endl;
    //std::cout << "length    : " << node->length    << std::endl;
#endif
    node = node->next;
  }
  if (mitumura_p)
  {
    lastFix(out);
  }
}

CCC::BString MecabIf::rubyOut(CCC::BString* out, const MeCab::Node* node)
{
  CCC::BString x(node->surface, node->length);
  // printf("%s:%s:%d\n", x.getCString(), node->surface, node->length);
  if (hasKanji(&x))
  {
    // ルビ付き出力
    CCC::BString yomi;
    getYomiFromMecabFeature(node->feature, &yomi);
    outputToRuby(&x, &yomi, out);
  }
  else
  {
    out->add(x);
  }
  return x;
}

void MecabIf::wakachiNormal(CCC::BString* out, const MeCab::Node* node)
{
  CCC::BString prev_hinshi;
  const char* prev_feature = nullptr;
  CCC::BString hinshi;
  getHinshiFromMecabFeature(node->feature, &hinshi);
  if (node->prev)
  {
    prev_feature = node->prev->feature;
    getHinshiFromMecabFeature(prev_feature, &prev_hinshi);
  }
  if (prev_feature != nullptr)
  {
    if ((prev_hinshi.strCmp("BOS/EOS") == 0) ||
      (hinshi.strCmp("BOS/EOS") == 0) ||
      (hinshi.strCmp("\xe8" "\xa8" "\x98" "\xe5" "\x8f" "\xb7" "") == 0) ||
      // 複合名詞は分かち書きせずにつなげる
        ((prev_hinshi.strCmp("\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" "") == 0) &&
          (hinshi.strCmp("\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" "") == 0)))
    { ; // skip
    }
    else if ((hinshi.strCmp("\xe5" "\x8a" "\xa9" "\xe8" "\xa9" "\x9e" "") == 0) ||
      (hinshi.strCmp("\xe5" "\x8a" "\xa9" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "") == 0))
    {
      if (jyoshi_wakachi_p)
      {
        out->add("~");
      }
    }
    else
    {
      if (wakachi_p)
      {
        out->add("_");
      }
    }
  }
}

void MecabIf::wakachiMitumura(CCC::BString* out, const MeCab::Node* node)
{
  bool is_jyoshi;

  bool wakachi_cut = surfaceCheck(node, is_jyoshi);
  if (wakachi_cut && wakachi_p)
  {
    if (jyoshi_wakachi_p)
    {
      out->add("~");
    }
    else
    {
      out->add("_");
    }
  }
  else if (is_jyoshi)
  {
    if (jyoshi_wakachi_p)
    {
      out->add("~");
    }
  }
}

void
MecabIf::analizeLine(const CCC::BString* line, CCC::BString* out)
{
  unsigned char* x = (unsigned char*) line->getCString();
  unsigned char c;
  CCC::BString buf;
  while ((c = *x++) != '\0')
  {
    if (c < 0x80)
    {
      if (buf.getLength() > 0)
      {
        analizeByMecab(&buf, out);
        buf.clear();
      }
      out->add(c);
    }
    else
    {
      buf.add(c);
    }
  }
  if (buf.getLength() > 0)
  {
    analizeByMecab(&buf, out);
  }
}

CCC::ArrayList<CCC::BString>*
MecabIf::stringSplit(const char* feature, const char split_char)
{
  CCC::ArrayList<CCC::BString>* split = new CCC::ArrayList<CCC::BString>();
  const char* fp = nullptr;
  for (const char* p = feature; p != nullptr;)
  {
    fp = strchr(p, split_char);
    if (fp == nullptr)
    {
      split->add(new CCC::BString(p));
      p = nullptr;
    }
    else
    {
      split->add(new CCC::BString(p, fp - p));
      p = fp + 1;
    }
  }
  return split;
}

bool MecabIf::typeEqual(const CCC::ArrayList<CCC::BString>* feature, const char* hinshi)
{
  if (feature == nullptr)
  {
    return false;
  }
  return strcmp((*feature)[0]->getCString(), hinshi) == 0;
}

bool MecabIf::featureContains(const CCC::ArrayList<CCC::BString>* feature, const char* type_str)
{
  if (feature == nullptr)
  {
    return false;
  }
  for (int n = 0; n <= 5; n++)
  {
    if (strcmp((*feature)[n]->getCString(), type_str) == 0)
    {
      return true;
    }
  }
  return false;
}

bool MecabIf::featureOriginal(const CCC::ArrayList<CCC::BString>* feature, const char* type_str)
{
  if (feature == nullptr)
  {
    return false;
  }
  return strcmp((*feature)[6]->getCString(), type_str) == 0;
}

bool MecabIf::featureOriginalList(const CCC::ArrayList<CCC::BString>* feature, const char** type_strings)
{
  if (feature == nullptr)
  {
    return false;
  }
  return inString((*feature)[6]->getCString(), type_strings);
}

bool MecabIf::inString(const char* s1_string, const char** s2_strings)
{
  while (*s2_strings)
  {
    if (strcmp(*s2_strings, s1_string) == 0)
    {
      return true;
    }
    s2_strings++;
  }
  return false;
}

bool MecabIf::checkNodeSurface(const MeCab::Node* node, const char* str)
{
  CCC::BString surface(node->surface, node->length);
  return surface.strCmp(str) == 0;
}

bool MecabIf::checkNodeSurfaceList(const MeCab::Node* node, const char** string_list)
{
  CCC::BString surface(node->surface, node->length);
  while (*string_list)
  {
    if (surface.strCmp(*string_list) == 0)
    {
      return true;
    }
    string_list++;
  }
  return false;
}

const char* _mecabif_mark_strings[] =
  {
    "\xe3" "\x80" "\x81" "", "\xe3" "\x80" "\x82" "", "\xe3" "\x80" "\x8c" "", "\xe3" "\x80" "\x8e" "", nullptr
  };

bool
MecabIf::surfaceCheck(const MeCab::Node* node, bool& is_jyoshi)
{
  std::unique_ptr<CCC::ArrayList<CCC::BString> > current_features(stringSplit(node->feature, ','));
  const char* prev_feature_string = (node->prev == nullptr) ? "" : node->prev->feature;
  std::unique_ptr<CCC::ArrayList<CCC::BString> > prev_features(stringSplit(prev_feature_string, ','));

  is_jyoshi = typeEqual(current_features.get(), "\xe5" "\x8a" "\xa9" "\xe8" "\xa9" "\x9e" "") || typeEqual(current_features.get(), "\xe5" "\x8a" "\xa9" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "");
  if ((node->char_type > 0) && !(typeEqual(prev_features.get(), "BOS/EOS")) &&
    !(typeEqual(current_features.get(), "BOS/EOS")) &&
    !(typeEqual(current_features.get(), "\xe8" "\xa8" "\x98" "\xe5" "\x8f" "\xb7" "")))
  {
    if (checkNodeSurfaceList(node->prev, _mecabif_mark_strings))
    {
      return false;
    }
    const char* feature_hinsi = (*current_features)[0]->getCString();
    if (strcmp("\xe5" "\x8a" "\xa9" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return false;
    }
    if (strcmp("\xe8" "\xa8" "\x98" "\xe5" "\x8f" "\xb7" "", feature_hinsi) == 0)
    {
      return false;
    }
    if (strcmp("\xe5" "\x8a" "\xa9" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return procJyodousi(*current_features, *prev_features, node);
    }
    if (strcmp("\xe5" "\x89" "\xaf" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return true;
    }
    if (strcmp("\xe9" "\x80" "\xa3" "\xe4" "\xbd" "\x93" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return true;
    }
    if (strcmp("\xe6" "\x8e" "\xa5" "\xe9" "\xa0" "\xad" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return true;
    }
    if (strcmp("\xe6" "\x8e" "\xa5" "\xe7" "\xb6" "\x9a" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return true;
    }
    if (strcmp("\xe6" "\x84" "\x9f" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return procKandousi(*current_features, *prev_features, node);
    }
    if (strcmp("\xe5" "\xbd" "\xa2" "\xe5" "\xae" "\xb9" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return procKeiyousi(*current_features, *prev_features, node);
    }
    if (strcmp("\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return procMeisi(*current_features, *prev_features, node);
    }
    if (strcmp("\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "", feature_hinsi) == 0)
    {
      return procDousi(*current_features, *prev_features, node);
    }
  }
  return false;
}

bool
MecabIf::procJyodousi(CCC::ArrayList<CCC::BString>& current_features,
                      CCC::ArrayList<CCC::BString>& prev_features,
                      const MeCab::Node* node)
{
  if (featureContains(&current_features, "\xe7" "\x89" "\xb9" "\xe6" "\xae" "\x8a" "\xe3" "\x83" "\xbb" "\xe3" "\x83" "\x8a" "\xe3" "\x82" "\xa4" ""))
  {
    if (checkNodeSurface(node->prev, "\xe3" "\x81" "\x97" ""))
    {
      return false;
    }
    else if (typeEqual(&prev_features, "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" ""))
    {
      return false;
    }
    return true;
  }
  return false;
}

bool
MecabIf::procKeiyousi(CCC::ArrayList<CCC::BString>& current_features,
                      CCC::ArrayList<CCC::BString>& prev_features,
                      const MeCab::Node* node)
{
  if (featureContains(&current_features, "\xe9" "\x9d" "\x9e" "\xe8" "\x87" "\xaa" "\xe7" "\xab" "\x8b" ""))
  {
    return false;
  }
  else if (featureOriginal(&current_features, "\xe3" "\x81" "\x84" "\xe3" "\x81" "\x84" ""))
  {
    if (typeEqual(&prev_features, "\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" ""))
    {
      return false;
    }
  }
  return true;
}

bool
MecabIf::procMeisi(CCC::ArrayList<CCC::BString>& current_features,
                   CCC::ArrayList<CCC::BString>& prev_features,
                   const MeCab::Node* node)
{
  if (featureContains(&current_features, "\xe6" "\x8e" "\xa5" "\xe5" "\xb0" "\xbe" ""))
  {
    return false;
  }
  if (featureContains(&current_features, "\xe9" "\x9d" "\x9e" "\xe8" "\x87" "\xaa" "\xe7" "\xab" "\x8b" ""))
  {
    if (checkNodeSurface(node, "\xe3" "\x81" "\xae" "") ||
      checkNodeSurface(node, "\xe3" "\x82" "\x93" "") ||
      featureContains(&current_features, "\xe5" "\x8a" "\xa9" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "\xe8" "\xaa" "\x9e" "\xe5" "\xb9" "\xb9" "") ||
      featureContains(&current_features, "\xe5" "\xbd" "\xa2" "\xe5" "\xae" "\xb9" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" "\xe8" "\xaa" "\x9e" "\xe5" "\xb9" "\xb9" ""))
    {
      return false;
    }
  }
  if (typeEqual(&prev_features, "\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" ""))
  {
    if (checkNodeSurface(node->prev, "\xe3" "\x81" "\x98" "\xe3" "\x82" "\x85" "\xe3" "\x81" "\x86" "") || checkNodeSurface(node, "\xe3" "\x81" "\xba" "\xe3" "\x81" "\x9f" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xa8" ""))
    {
      return true; // 名詞扱いされているのは辞書のバグ？副詞では？
    }
    else if (featureContains(&prev_features, "\xe6" "\x8e" "\xa5" "\xe5" "\xb0" "\xbe" "") &&
      featureContains(&prev_features, "\xe5" "\x8a" "\xa9" "\xe6" "\x95" "\xb0" "\xe8" "\xa9" "\x9e" ""))
    {
      return true; // 名詞扱いされているのは辞書のバグ？副詞では？
    }
    return false;
  }
  if (typeEqual(&prev_features, "\xe6" "\x8e" "\xa5" "\xe9" "\xa0" "\xad" "\xe8" "\xa9" "\x9e" ""))
  {
    return false;
  }
  if (typeEqual(&prev_features, "\xe5" "\x89" "\xaf" "\xe8" "\xa9" "\x9e" "") && checkNodeSurface(node->prev, "\xe3" "\x81" "\xa8" "\xe3" "\x81" "\x8f" "\xe3" "\x81" "\xb9" "\xe3" "\x81" "\xa4" ""))
  {
    return false;
  }
  if (typeEqual(&prev_features, "\xe9" "\x80" "\xa3" "\xe4" "\xbd" "\x93" "\xe8" "\xa9" "\x9e" "") && checkNodeSurface(node->prev, "\xe3" "\x81" "\x86" "\xe3" "\x81" "\xa1" ""))
  {
    return false;
  }
  return true;
}

bool
MecabIf::procDousi(CCC::ArrayList<CCC::BString>& current_features,
                   CCC::ArrayList<CCC::BString>& prev_features,
                   const MeCab::Node* node)
{
  static const char* dousi_sp_1[] =
  {"\xe3" "\x81" "\x8f" "\xe3" "\x82" "\x8c" "\xe3" "\x82" "\x8b" "", "\xe3" "\x81" "\x84" "\xe3" "\x82" "\x8b" "", "\xe3" "\x81" "\x84" "\xe3" "\x81" "\x8f" "", "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xbe" "\xe3" "\x81" "\x86" "", "\xe3" "\x81" "\x8f" "\xe3" "\x82" "\x8b" "", "\xe3" "\x82" "\x84" "\xe3" "\x82" "\x8b" "", "\xe3" "\x81" "\xbf" "\xe3" "\x82" "\x8b" "", "\xe3" "\x82" "\x82" "\xe3" "\x82" "\x89" "\xe3" "\x81" "\x88" "\xe3" "\x82" "\x8b" "", "\xe3" "\x82" "\x86" "\xe3" "\x81" "\x8f" "", "\xe3" "\x81" "\x8f" "\xe3" "\x81" "\xa0" "\xe3" "\x81" "\x95" "\xe3" "\x82" "\x8b" "", nullptr};

  static const char* dousi_sp_2[] = {"\xe3" "\x81" "\xaa" "\xe3" "\x82" "\x8b" "", "\xe5" "\x8a" "\xa9" "\xe8" "\xa9" "\x9e" "", nullptr}; // TODO: これはおかしいと思う

  static const char* dousi_sp_3[] = {"\xe4" "\xbb" "\xae" "\xe5" "\xae" "\x9a" "\xe7" "\xb8" "\xae" "\xe7" "\xb4" "\x84" "", "\xe4" "\xbb" "\xae" "\xe5" "\xae" "\x9a" "\xe7" "\xb8" "\xae" "\xe7" "\xb4" "\x84" "\xef" "\xbc" "\x91" "", "\xe4" "\xbb" "\xae" "\xe5" "\xae" "\x9a" "\xe7" "\xb8" "\xae" "\xe7" "\xb4" "\x84" "\xef" "\xbc" "\x92" "", nullptr};

  if (typeEqual(&prev_features, "\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" ""))
  {
    if (checkNodeSurface(node->prev, "\xe3" "\x81" "\x84" "\xe3" "\x81" "\xa4" ""))
    {
      return false;
    }
    else if (checkNodeSurface(node->prev, "\xe3" "\x81" "\x98" "\xe3" "\x82" "\x85" "\xe3" "\x81" "\x86" ""))
    {
      return true;
    }
  }
  if (featureContains(&current_features, "\xe6" "\x8e" "\xa5" "\xe5" "\xb0" "\xbe" ""))
  {
    return false;
  }
  if (featureContains(&current_features, "\xe9" "\x9d" "\x9e" "\xe8" "\x87" "\xaa" "\xe7" "\xab" "\x8b" ""))
  {
    if (featureOriginalList(&current_features, dousi_sp_1))
    {
      return true;
    }
    if (featureOriginalList(&current_features, dousi_sp_2) &&
      checkNodeSurface(node->prev, "\xe3" "\x81" "\xb0" ""))
    {
      return true;
    }
    return false;
  }
  if (featureContains(&current_features, "\xe6" "\x9c" "\xaa" "\xe7" "\x84" "\xb6" "\xe3" "\x83" "\xac" "\xe3" "\x83" "\xab" "\xe6" "\x8e" "\xa5" "\xe7" "\xb6" "\x9a" ""))
  {
    return false;
  }
  if (featureContains(&current_features, "\xe3" "\x82" "\xb5" "\xe5" "\xa4" "\x89" "\xe3" "\x83" "\xbb" "\xe3" "\x82" "\xb9" "\xe3" "\x83" "\xab" ""))
  {
    if (typeEqual(&prev_features, "\xe5" "\x90" "\x8d" "\xe8" "\xa9" "\x9e" ""))
    {
      if (checkNodeSurface(node->prev, "\xe4" "\xbd" "\x95" ""))
      {
        return true;
      }
      return false;
    }
    if (typeEqual(&prev_features, "\xe5" "\x89" "\xaf" "\xe8" "\xa9" "\x9e" ""))
    {
      return true;
    }
  }
  if (typeEqual(&prev_features, "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" ""))
  {
    if (featureOriginalList(&prev_features, dousi_sp_3))
    {
      return true;
    }
    return false;
  }
  if (typeEqual(&prev_features, "\xe6" "\x8e" "\xa5" "\xe7" "\xb6" "\x9a" "\xe5" "\x8a" "\xa9" "\xe8" "\xa9" "\x9e" "") || typeEqual(&prev_features, "\xe6" "\x8e" "\xa5" "\xe9" "\xa0" "\xad" "\xe8" "\xa9" "\x9e" ""))
  {
    return false;
  }
  return true;
}

bool MecabIf::procKandousi(CCC::ArrayList<CCC::BString>& current_features,
                           CCC::ArrayList<CCC::BString>& prev_features,
                           const MeCab::Node* node)
{
  return !(typeEqual(&prev_features, "\xe6" "\x84" "\x9f" "\xe5" "\x8b" "\x95" "\xe8" "\xa9" "\x9e" ""));
}

bool MecabIf::lastFix(CCC::BString* re_assembled)
{
  static ReplaceType replace_table[] = {
    { "\xe3" "\x80" "\x81" "_", "\xe3" "\x80" "\x81" "" },
    { "\xe3" "\x80" "\x82" "_", "\xe3" "\x80" "\x82" "" },
    { "\xef" "\xbc" "\x8c" "_", "\xef" "\xbc" "\x8c" "" },
    { "\xef" "\xbc" "\x8e" "_", "\xef" "\xbc" "\x8e" "" },
    { "\xe3" "\x80" "\x8c" "_", "\xe3" "\x80" "\x8c" "" },
    // { "｜（《（》_", "｜（《（》" },  ---- bug
    { "\xe3" "\x81" "\xab" "\xef" "\xbd" "\x9c" "\xe5" "\xaf" "\xbe" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x9f" "\xe3" "\x81" "\x84" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xa6" "", "\xe3" "\x81" "\xab" "_\xef" "\xbd" "\x9c" "\xe5" "\xaf" "\xbe" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x9f" "\xe3" "\x81" "\x84" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xa6" "" },
    { "\xe3" "\x81" "\xa8" "\xe3" "\x81" "\x84" "\xe3" "\x81" "\x86" "", "\xe3" "\x81" "\xa8" "_\xe3" "\x81" "\x84" "\xe3" "\x81" "\x86" "" },
    { "\xe3" "\x82" "\x84" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa6" "\xef" "\xbd" "\x9c" "\xe6" "\x9d" "\xa5" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8d" "\xe3" "\x80" "\x8b" "", "\xe3" "\x82" "\x84" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa6" "_\xef" "\xbd" "\x9c" "\xe6" "\x9d" "\xa5" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8d" "\xe3" "\x80" "\x8b" "" },
    { "\xe3" "\x81" "\x9d" "\xe3" "\x81" "\xae" "\xef" "\xbd" "\x9c" "\xe5" "\xbe" "\x8c" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x94" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\xab" "", "\xe3" "\x81" "\x9d" "\xe3" "\x81" "\xae" "_\xef" "\xbd" "\x9c" "\xe5" "\xbe" "\x8c" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x94" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\xab" "" },
    { "\xe3" "\x81" "\x9a" "_\xe3" "\x81" "\x86" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa8" "", "\xe3" "\x81" "\x9a" "\xe3" "\x81" "\x86" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa8" "" },
    { "\xef" "\xbd" "\x9c" "\xe5" "\x9b" "\x9b" "\xe3" "\x80" "\x8a" "\xe3" "\x82" "\x88" "\xe3" "\x82" "\x93" "\xe3" "\x80" "\x8b" "\xef" "\xbd" "\x9c" "\xe5" "\x88" "\x86" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\xb5" "\xe3" "\x82" "\x93" "\xe3" "\x80" "\x8b" "_\xef" "\xbd" "\x9c" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8a" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xb7" "\xe3" "\x80" "\x8b" "\xef" "\xbd" "\x9c" "\xe4" "\xb8" "\x80" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x84" "\xe3" "\x81" "\xa1" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\x93" "", "\xef" "\xbd" "\x9c" "\xe5" "\x9b" "\x9b" "\xe5" "\x88" "\x86" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xb6" "\xe3" "\x81" "\x8a" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xb7" "\xe3" "\x80" "\x8b" "_\xef" "\xbd" "\x9c" "\xe4" "\xb8" "\x80" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x84" "\xe3" "\x81" "\xa3" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\x93" "" },
    { "\xef" "\xbd" "\x9c" "\xe5" "\x9b" "\x9b" "\xe3" "\x80" "\x8a" "\xe3" "\x82" "\x88" "\xe3" "\x82" "\x93" "\xe3" "\x80" "\x8b" "\xef" "\xbd" "\x9c" "\xe5" "\x88" "\x86" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\xb5" "\xe3" "\x82" "\x93" "\xe3" "\x80" "\x8b" "_\xef" "\xbd" "\x9c" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8a" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xb7" "\xe3" "\x80" "\x8b" "", "\xef" "\xbd" "\x9c" "\xe5" "\x9b" "\x9b" "\xe5" "\x88" "\x86" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xb6" "\xe3" "\x81" "\x8a" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xb7" "\xe3" "\x80" "\x8b" "" },
    { "\xef" "\xbd" "\x9c" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8a" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xb7" "\xe3" "\x80" "\x8b" "\xef" "\xbd" "\x9c" "\xe4" "\xb8" "\x80" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x84" "\xe3" "\x81" "\xa1" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\x93" "", "\xef" "\xbd" "\x9c" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8a" "\xe3" "\x82" "\x93" "\xe3" "\x81" "\xb7" "\xe3" "\x80" "\x8b" "_\xef" "\xbd" "\x9c" "\xe4" "\xb8" "\x80" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x84" "\xe3" "\x81" "\xa3" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\x93" "" },
    { "\xe3" "\x81" "\xaf" "\xe3" "\x81" "\x98" "\xe3" "\x81" "\xa3" "_\xe3" "\x81" "\x93" "", "\xe3" "\x81" "\xaf" "\xe3" "\x81" "\x98" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\x93" "" },
    { "\xef" "\xbd" "\x9c" "\xe6" "\x9c" "\xa8" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x93" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\xae" "\xef" "\xbd" "\x9c" "\xe4" "\xb8" "\x8b" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\x9f" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\xa7" "", "\xef" "\xbd" "\x9c" "\xe6" "\x9c" "\xa8" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x8d" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\xae" "\xef" "\xbd" "\x9c" "\xe4" "\xb8" "\x8b" "\xe3" "\x80" "\x8a" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\x9f" "\xe3" "\x80" "\x8b" "\xe3" "\x81" "\xa7" "" },
    { nullptr, nullptr }
  };
  static ReplaceType no_ruby_replace_table[] = {
    { "\xe3" "\x80" "\x81" "_", "\xe3" "\x80" "\x81" "" },
    { "\xe3" "\x80" "\x82" "_", "\xe3" "\x80" "\x82" "" },
    { "\xef" "\xbc" "\x8c" "_", "\xef" "\xbc" "\x8c" "" },
    { "\xef" "\xbc" "\x8e" "_", "\xef" "\xbc" "\x8e" "" },
    { "\xe3" "\x80" "\x8c" "_", "\xe3" "\x80" "\x8c" "" },
    { "\xef" "\xbc" "\x88" "_", "\xef" "\xbc" "\x88" "" },
    { "\xe3" "\x81" "\xab" "\xe5" "\xaf" "\xbe" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xa6" "", "\xe3" "\x81" "\xab" "_\xe5" "\xaf" "\xbe" "\xe3" "\x81" "\x97" "\xe3" "\x81" "\xa6" "" },
    { "\xe3" "\x81" "\xa8" "\xe3" "\x81" "\x84" "\xe3" "\x81" "\x86" "", "\xe3" "\x81" "\xa8" "_\xe3" "\x81" "\x84" "\xe3" "\x81" "\x86" "" },
    { "\xe3" "\x82" "\x84" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa6" "\xe6" "\x9d" "\xa5" "", "\xe3" "\x82" "\x84" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa6" "_\xe6" "\x9d" "\xa5" "" },
    { "\xe3" "\x81" "\x9d" "\xe3" "\x81" "\xae" "\xe5" "\xbe" "\x8c" "\xe3" "\x81" "\xab" "", "\xe3" "\x81" "\x9d" "\xe3" "\x81" "\xae" "_\xe5" "\xbe" "\x8c" "\xe3" "\x81" "\xab" "" },
    { "\xe3" "\x81" "\x9a" "_\xe3" "\x81" "\x86" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa8" "", "\xe3" "\x81" "\x9a" "\xe3" "\x81" "\x86" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\xa8" "" },
    { "\xe5" "\x9b" "\x9b" "\xe5" "\x88" "\x86" "_\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "", "\xe5" "\x9b" "\x9b" "\xe5" "\x88" "\x86" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "" },
    { "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe4" "\xb8" "\x80" "\xe3" "\x81" "\x93" "", "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "_\xe4" "\xb8" "\x80" "\xe3" "\x81" "\x93" "" },
    { "\xe3" "\x81" "\xaf" "\xe3" "\x81" "\x98" "\xe3" "\x81" "\xa3" "_\xe3" "\x81" "\x93" "", "\xe3" "\x81" "\xaf" "\xe3" "\x81" "\x98" "\xe3" "\x81" "\xa3" "\xe3" "\x81" "\x93" "" },
    { "\xe6" "\x9c" "\xa8" "\xe3" "\x81" "\xae" "\xe4" "\xb8" "\x8b" "", "\xe6" "\x9c" "\xa8" "\xe3" "\x81" "\xae" "_\xe4" "\xb8" "\x8b" ""},
    { nullptr, nullptr }
  };
  ReplaceType* t = ruby_out_p ? replace_table : no_ruby_replace_table;
  while (t->src != nullptr)
  {
    re_assembled->replaceAll(t->src, t->dst);
    t++;
  }
  if (jyoshi_wakachi_p)
  {
    ReplaceType* t = ruby_out_p ? replace_table : no_ruby_replace_table;
    while (t->src != nullptr)
    {
      CCC::BString src(t->src);
      CCC::BString dst(t->dst);
      src.replaceAll("_", "~");
      dst.replaceAll("_", "~");
      re_assembled->replaceAll(src.getCString(), dst.getCString());
      t++;
    }
  }
  return true;
}

#ifdef TEST_MAIN
int
main(int argc, char** argv)
{
  CCC::Ccc::initialize();
  MecabIf mif;
  mif.setMitumuraP(true);
  mif.setRubyP(true);
  mif.initMecab();
  CCC::BString line("\xe5" "\x9b" "\x9b" "\xe5" "\x88" "\x86" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe4" "\xb8" "\x80" "\xe3" "\x81" "\xa4" "\xe3" "\x81" "\xa7" "\xe5" "\x9b" "\x9b" "\xe5" "\x88" "\x86" "\xe9" "\x9f" "\xb3" "\xe7" "\xac" "\xa6" "\xe4" "\xb8" "\x80" "\xe3" "\x81" "\x93" "");
  CCC::BString re_assembled;
  mif.analizeLine(&line, &re_assembled);
  printf("    SOURCE:[%s]\nREASSEMBED:[%s]\n\n", line.getCString(), re_assembled.getCString());
  CCC::Ccc::unInitialize();
}
#endif /* TEST_MAIN */
