﻿#ifndef INCLUDE_MecabIf_h
#define INCLUDE_MecabIf_h

#include <mecab.h>
#include <ccc/base/Ccc.h>
#include <ccc/iceman/Utf16String.h>
#include <ccc/iceman/Utf8String.h>
#include <ccc/base/TextReader.h>
#include <ccc/file/FileIFlow.h>
#include <ccc/iceman/Iceman.h>
#include <ccc/base/ArrayList.h>

struct TextSegment;

class MecabIf
{
  MeCab::Tagger* tagger;         /*!< Mecab instance */
  bool wakachi_p;                /*!< 分かち書き出力フラグ(デフォルトON) */
  bool jyoshi_wakachi_p;         /*!< 助詞・助動詞前の分かち書き出力フラグ(デフォルトOFF) */
  bool mitumura_p;               /*!< 光村形式(デフォルトOFF) */
  bool ruby_out_p;               /*!< ルビ出力フラグ(デフォルトON) */
public:
  /*!
   * コンストラクタ
   */
  MecabIf();
  /*!
   * デストラクタ
   */
  ~MecabIf();
  /*!
   * MeCabを初期化します。
   * \param mecabrc_path Mecabのリソースファイルのパス、0の場合は辞書のパスを指定せずに初期化します。
   * \retval true 成功
   * \retval false 失敗
   */
  bool initMecab(const char* mecabrc_path = 0);
  /*!
   * 分かち書き出力フラグを設定します。
   * \param p フラグ
   */
  void setWakachiP(bool p)
  {
    wakachi_p = p;
  }
  /*!
   * ルビ出力フラグを設定します。(デフォルトON)
   * \param p フラグ
   */
  void setRubyP(bool p)
  {
    ruby_out_p = p;
  }
  /*!
   * 助詞・助動詞前の分かち書き出力フラグを設定します。
   * \param p フラグ
   */
  void setJyoshiWakachiP(bool p)
  {
    jyoshi_wakachi_p = p;
  }
  /*!
   * 光村形式フラグを設定します。
   * \param p フラグ
   */
  void setMitumuraP(bool p)
  {
    mitumura_p = p;
  }
  /*!
   * Mecabの形態素解析エンジンを使って文字列のルビ振り、分かち書きの処理を行います。
   * \param in 入力文字列(UTF-8)
   * \param out 出力文字列(UTF-8)
   */
  void analizeLine(const CCC::BString* text_in, CCC::BString* text_out);
private:
  /*!
   * 文字列を指定した文字で分割してリストにします。
   * \param feature 元文字列
   * \param split_char 分割する文字列
   * \return 文字列のリスト (メモリ解放は利用者が行うこと)
   */
  static CCC::ArrayList<CCC::BString>* stringSplit(const char* feature, const char split_char);
  /*!
   * ruby部分を出力します。
   * \param oyamoji 親文字
   * \param ruby ルビ(カタカナ)
   * \param out 分かち書き出力バッファ
   */
  void outputToRuby(CCC::BString* oyamoji, CCC::BString* ruby, CCC::BString* out);
  /*!
   * 分割されたTextSegmentのリストから分かち書き出力に変換します。
   * \param list 分かち書き単位に分割されたリスト
   * \param out 分かち書き文字列が出力されるバッファー
   */
  void outputTextSegmentListItems(CCC::ArrayList<TextSegment>* list, CCC::BString* out);
  /*!
   * 文字列を親文字とひらがなに分割してTextSegmentに追加します。 TODO:
   * \param oyamoji 入力文字列
   * \param list 追加する TextSegmentのリスト
   */
  void createTextSegmentListItems(const CCC::BString* oyamoji, CCC::ArrayList<TextSegment>* list);
  /*!
   * ルビの情報をTextSegmentに追加します。
   * \param list 追加する TextSegmentのリスト
   * \param ruby 追加するルビ
   */
  void setRubyToTestSegmentListItems(CCC::ArrayList<TextSegment>* list, CCC::BString* ruby);
  /*!
   * UCS4の文字列中のカタカナをひらがなに変換します。
   * \param hiragana 入力のカタカナ文字列
   * \param katakana 出力のひらがな文字列
   * \return 無し
   */
  void convertUtf8FromKatakanaToHiragana(const CCC::WString* katakana, CCC::WString* hiragana);
  /*!
   * UTF8の文字列中のカタカナをひらがなに変換します。
   * \param hiragana 入力のカタカナ文字列
   * \param katakana 出力のひらがな文字列
   * \return 無し
   */
  void convertUtf8FromKatakanaToHiragana(const CCC::BString* katakana, CCC::BString* hiragana);
  /*!
   * Mecabのfeatureから指定された項目の文字列を取り出します。
   * \param feature 取り出す元のfeature
   * \param column_number 取り出す位置 0ベース
   * \param value 取りだした値
   * \return 無し
   */
  void getItemFromMecabFeature(const char* feature, int column_number, CCC::BString* value);
  /*!
   * Mecabのfeatureから品詞の文字列を取り出します。
   * \param feature 取り出すfeature
   * \param hinshi 取りだした品詞を入れるバッファー
   * \return 無し
   */
  void getHinshiFromMecabFeature(const char* feature, CCC::BString* hinshi);
  /*!
   * Mecabのfeatureから読みの文字列を取り出します。
   * \param feature 取り出すfeature
   * \param hinshi 取りだした読みを取り出すバッファー
   */
  void getYomiFromMecabFeature(const char* feature, CCC::BString* yomi);
  /*!
   * UTF-8の単語が漢字を含むかどうかを調べます。
   * \param word 単語
   * \retval true 漢字を含む
   * \retval false 漢字を含まない
   */
  bool hasKanji(const CCC::BString* word);
  /*!
   * 平仮名かどうかを調べます。
   * \param c UCS2の文字
   * \retval true 平仮名
   * \retval false 平仮名ではない
   */
  inline bool isHiragana(CCC::UInt32 c)
  {
    if ((c >= 0x3041u) &&
      (c <= 0x3096u))
    {
      return true;
    }
    return false;
  }
  /*!
   * MeCabを使った分かち書きとルビのための読み情報を付加した文字列を作成します。
   * \param line 入力文字列
   * \param out
   */
  void analizeByMecab(const CCC::BString* line, CCC::BString* out);
  /*!
   * MeCabを使った分かち書きの読み情報を付加した文字列を作成します。
   * \param line 入力文字列
   * \param out
   */
  void wakachiNormal(CCC::BString* out, const MeCab::Node* node);
  /*!
   * 光村分かち書をします。
   * \param out 出力する文字列バッファ
   * \param node 追加すべきMeCabのNode
   */
  void wakachiMitumura(CCC::BString* out, const MeCab::Node* node);
  /*!
   * ルビの出力をします。
   * \param out 出力する文字列バッファ
   * \param node 追加すべきMeCabのNode
  */
  CCC::BString rubyOut(CCC::BString* out, const MeCab::Node* node);
  /*!
   * 分解したFeatureの品詞が指定された文字列と同一か判断します。
   * \param feature 分愛したnode->feature
   * \param hinshi 比較する品詞文字列
   * \return 同一時 True
   */
  static bool typeEqual(const CCC::ArrayList<CCC::BString>* feature, const char* hinshi);
  /*!
   * 分解したFeatureの品詞、活用形から指定したtype_strを探し見つかった場合はTrueを返します。
   * \param feature 分愛したnode->feature
   * \param type_str 比較する文字列
   * \return 見つかった場合True
   */
  static bool featureContains(const CCC::ArrayList<CCC::BString>* feature, const char* type_str);
  /*!
   * 分解したFeatureの原型[6] とtype_strと比較して一致した場合はTrueを返します。
   * \param feature 分愛したnode->feature
   * \param type_str 比較する文字列
   * \return 同じ場合True
   */
  static bool featureOriginal(const CCC::ArrayList<CCC::BString>* feature, const char* type_str);
  /*!
   * 文字列とFeatureの原型[6]と比較し見つかったらTrueを返します。
   * \param feature
   * \param type_strings
   * \return 見つかった場合True
   */
  bool checkNodeSurface(const MeCab::Node* node, const char* str);
  /*!
   * 複数の文字列とFeatureの原型[6]と比較し見つかったらTrueを返します。
   * \param node 調べるNode
   * \param string_list　検索する複数の文字列
   * \return 一致する桃が1つでもあればTrue
   */
  static bool checkNodeSurfaceList(const MeCab::Node* node, const char** string_list);
  /*!
   * featureの原型[6]を複数の文字列と比較します。
   * \param feature 原型を取り出すfeature
   * \param type_strings 複数の文字列 nullptrで終端
   * \return 見つかった場合はTrue
   */
  static bool featureOriginalList(const CCC::ArrayList<CCC::BString>* feature, const char** type_strings);
  /*!
   * s1_stringがs2_stringsに含まれるか判定します。
   * \param s1_string 探す文字列
   * \param s2_strings 文字列のリスト
   * \return 完全に一致する文字列があった場合True
   */
  static bool inString(const char* s1_string, const char** s2_strings);
  /*!
   * 光村分かち書きで指定されたnodeが分かち書き位置か判定します。
   * \param node 判定するNode
   * \param is_jyoshi 現在のNodeが助詞、助動詞の場合にTrue
   * \return 判定した結果が分かち書きする位置の場合True
   */
  bool surfaceCheck(const MeCab::Node* node, bool& is_jyoshi);
  /*!
   * 光村図書分かち書きの助動詞の判定します。
   * \param current_features 現在のNode Mecab Features
   * \param prev_features 1つ前のNode Mecab Features
   * \param node 現在のNode
   * \return このNodeで分かち書きする場合はTrue
   */
  bool procJyodousi(CCC::ArrayList<CCC::BString>& current_features,
                    CCC::ArrayList<CCC::BString>& prev_features,
                    const MeCab::Node* node);
  /*!
   * 光村図書分かち書き感動詞の判定します。
   * \param current_features 現在のNode Mecab Features
   * \param prev_features 1つ前のNode Mecab Features
   * \param node 現在のNode
   * \return このNodeで分かち書きする場合はTrue
   */
  bool procKandousi(CCC::ArrayList<CCC::BString>& current_features,
                    CCC::ArrayList<CCC::BString>& prev_features,
                    const MeCab::Node* node);
  /*!
   * 光村図書分かち書き形容詞の判定をします。
   * \param current_features 現在のNode Mecab Features
   * \param prev_features 1つ前のNode Mecab Features
   * \param node 現在のNode
   * \return このNodeで分かち書きする場合はTrue
   */
  bool procKeiyousi(CCC::ArrayList<CCC::BString>& current_features,
                    CCC::ArrayList<CCC::BString>& prev_features,
                    const MeCab::Node* node);
  /*!
   * 光村図書分かち書き名詞の判定を行う
   * \param current_features 現在のNode Mecab Features
   * \param prev_features 1つ前のNode Mecab Features
   * \param node 現在のNode
   * \return このNodeで分かち書きする場合はTrue
   */
  bool procMeisi(CCC::ArrayList<CCC::BString>& current_features,
                 CCC::ArrayList<CCC::BString>& prev_features,
                 const MeCab::Node* node);
  /*!
   * 光村図書分かち書き動詞の判定をします。
   * \param current_features 現在のNode Mecab Features
   * \param prev_features 1つ前のNode Mecab Features
   * \param node 現在のNode
   * \return このNodeで分かち書きする場合はTrue
   */
  bool procDousi(CCC::ArrayList<CCC::BString>& current_features,
                 CCC::ArrayList<CCC::BString>& prev_features,
                 const MeCab::Node* node);
  /*!
   * 光村図書分かちの最終修正をします。
   * \param re_assembled 最終修正をする文字列。修正されます。
   * \return 無し
   */
  bool lastFix(CCC::BString* re_assembled);
  /*!
   * TextSegmentのリストをデバッグ表示をします。
   * \param list 表示するTextSegment
   */
  void dumpTextSegmentListItems(CCC::ArrayList<TextSegment>* list);
};

#endif /* INCLUDE_MecabIf_h */
