﻿#include <iostream>
#include "MecabIf.h"
#include <memory>
#include "ccc/file/PathString.h"
#include "ccc/app/Application.h"
#include "version.h"
#include "wakach.h"
#include "ccc/file/FileOFlow.h"
#include "ccc/base/TextWriter.h"

#ifdef _WIN32
/*!
 * PJAのレジストリからIPADICの辞書のパスを取得します。
 */
static bool
getMecabrcPath(CCC::PathString* mecabrc_path)
{
  HKEY hkey;
#define REGISTORY_TOP "Software\\CYPAC\\PJA"
  if (::RegOpenKeyExA(HKEY_LOCAL_MACHINE, REGISTORY_TOP, 0, KEY_QUERY_VALUE, &hkey) != ERROR_SUCCESS)
  {
    return false;
  }
  DWORD data_type;
  DWORD buf_len = 0;
  if (::RegQueryValueExA(hkey, "InstDir", NULL, &data_type, 0, &buf_len) != ERROR_SUCCESS)
  {
    return false;
  }
  std::shared_ptr<unsigned char> buf(new unsigned char[buf_len + 1]);
  if (::RegQueryValueExA(hkey, "InstDir", NULL, &data_type, buf.get(), &buf_len) != ERROR_SUCCESS)
  {
    return false;
  }
  mecabrc_path->assign((char*)buf.get());
  ::RegCloseKey(hkey);
  mecabrc_path->add("mecabrc");
  return true;
}
#endif /* _WIN32 */

bool
Wakati::initialize()
{
  bool argument_ok_p = parseArgument();
  return argument_ok_p;
}

int Wakati::unInitialize(int ret)
{
  return ret;
}


void Wakati::exit(int ret)
{
  throw static_cast<int>(ret);
}

Wakati::Wakati()
  : CCC::Application()
{
  this->run_mode = OPT_NORMAL;
}

Wakati::Wakati(int _argc, char** _argv)
  : CCC::Application(_argc, _argv)
{
  rubi_flag = false;
  mitumura_flag = true;
  joshi_wakachi = false;
}

Wakati::~Wakati()
{
}

CCC::AppArg*
Wakati::getAppArg()
{
  static CCC::AppArg args[] =
  {
    { "--version", OPT_MODE_VERSION },
    { "-V", OPT_MODE_VERSION },
    { "-h", OPT_MODE_HELP },
    { "-help", OPT_MODE_HELP },
    { "--help", OPT_MODE_HELP },
    { "-r", OPT_RUBI },
    { "--ruby", OPT_RUBI },
    { "-m-", OPT_PRE_MITUMURA_WAKACHI },
    { "--mitumura-", OPT_PRE_MITUMURA_WAKACHI },
    { "-j", OPT_PRE_JYOSHI_WAKACHI },
    { "--jyoshi", OPT_PRE_JYOSHI_WAKACHI },
    { 0, 0  },
  };
  return args;
}

void
Wakati::showHelp()
{
  puts("mitumurawakati.exe [options] source_path destination_path");
  puts("OPTION:");
  puts("-r, --ruby         Output ruby information");
  puts("-m-, --mitumura-   Don't use mitumura wakati rule");
  puts("-j, --jyoshi       Output jyoshi wakati");
}

void
Wakati::showVersion()
{
  puts("mitumurawakakati Copyright (C) 2019 CYPAC version " PJA_VERSION);
}

void Wakati::wakachi_convert_stream(CCC::BTextReader& reader, CCC::BTextWriter& writer)
{
#ifdef WITH_W32_PJA
  CCC::PathString mecabrc_path;
  if (!getMecabrcPath(&mecabrc_path))
  {
    fprintf(stderr, "ERROR: Can't get mecabrc path.\n");
    throw 3;
  }
  CCC::Utf8String mecabrc_path8;
  CCC::Iceman::convertToBString(CCC_PATH_ENCODING, CCC::CEID_CP932, &mecabrc_path, &mecabrc_path8);
#endif /* WITH_W32_PJA */
  MecabIf mif;
  mif.setMitumuraP(mitumura_flag);
  mif.setRubyP(rubi_flag);
  mif.setJyoshiWakachiP(joshi_wakachi);
#ifdef WITH_W32_PJA
  mif.initMecab(mecabrc_path8.getCString());
#else /* WITH_W32_PJA */
  mif.initMecab();
#endif /* WITH_W32_PJA */
  for (;;)
  {
    CCC::BString* line = reader.readLine();
    if (line == 0)
    {
      break;
    }
    line->chop('\n');
    line->chop('\r');
    CCC::BString re_assembled;
    mif.analizeLine(line, &re_assembled);
    re_assembled.replaceAll("_",  "\xe3\x80\x80" /* "　" */ );
#ifdef _WIN32
    writer.printf("%s\r\n", re_assembled.getCString());
#else /* _WIN32 */
    writer.printf("%s\n", re_assembled.getCString());
#endif /* _WIN32 */
    delete line;
  }
}

int
Wakati::wakati_convert_file()
{
  CCC::FileIFlow file_source_flow;
  if (! file_source_flow.open(&source_path))
  {
    throw 2;
  }
  CCC::BTextReader reader(&file_source_flow);
  CCC::FileOFlow file_output_flow;
  if (! file_output_flow.open(&dist_path))
  {
    throw 3;
  }
  CCC::BTextWriter writer(&file_output_flow);

  wakachi_convert_stream(reader, writer);
  return 0;
}

int
Wakati::wakati_convert_file_input()
{
  CCC::FileIFlow file_source_flow;
  if (! file_source_flow.open(&source_path))
  {
    throw 2;
  }
  CCC::BTextReader reader(&file_source_flow);
  CCC::BTextWriter writer(CCC::stdout_flow);
  wakachi_convert_stream(reader, writer);
  return 0;
}

int
Wakati::wakati_convert_filter()
{
  CCC::BTextWriter writer(CCC::stdout_flow);
  CCC::BTextReader reader(CCC::stdin_flow);
  wakachi_convert_stream(reader, writer);
  return 0;
}

bool
Wakati::checkArgument(int n, int& i, char* eq_str)
{
  bool argument_ok_p = true;
  switch (n)
  {
   case OPT_MODE_VERSION:
    run_mode = OPT_MODE_VERSION;
    break;

   case OPT_MODE_HELP:
    run_mode = OPT_MODE_HELP;
    break;

   case OPT_PRE_JYOSHI_WAKACHI:
    joshi_wakachi = true;
    break;

   case OPT_PRE_MITUMURA_WAKACHI:
    mitumura_flag = false;
    break;

   case OPT_RUBI:
    rubi_flag = true;
    break;

   default:
    if (source_path.getLength() == 0)
    {
      source_path.assign(argv[i]);
    }
    else if (dist_path.getLength() == 0)
    {
      dist_path.assign(argv[i]);
    }
    else
    {
      fprintf(stderr, "ERROR: unknown option:%s\n", argv[i]);
      argument_ok_p = false;
    }
    break;
  }
  return argument_ok_p;
}

int
Wakati::run()
{
  try
  {
    // main process
    switch (run_mode)
    {
     case OPT_MODE_VERSION:
      showVersion();
      break;

     case OPT_MODE_HELP:
      showHelp();
      break;


     default:
      if (source_path.getLength() == 0)
      {
        return wakati_convert_filter();
      }
      else
      {
        if (dist_path.getLength() == 0)
        {
          return wakati_convert_file_input();
        }
        else
        {
          return wakati_convert_file();
        }
      }
      break;
    }
  }
  catch (int i)
  {
    // caught exit
    return i;
  }
  return 0;
}

int
main(int argc, char **argv)
{
  std::shared_ptr<Wakati> app(new Wakati(argc, argv));
  bool ini_p;
  try
  {
    ini_p = app->initialize();
  }
  catch (int err)
  {
    return err;
  }
  int ret = -1;
  if (ini_p)
  {
    ret = app->run();
    ret = app->unInitialize(ret);
  }
  return 0;
}
